/**<PRE>
 * SHA1.java - An implementation of the SHA-1 Algorithm
 * This version integrated into Freenet by Ian Clarke (02-02-2000) 
 * (i.clarke@dynamicblue.com) from a previous public domain version by 
 * Chuck McManis (cmcmanis@netcom.com) which was public domain
 * Tweaked by Mr.Tines<tines@windsong.demon.co.uk> for pegwit, June 1997
 * - added method 'frob()' which wipes the contents, so as to match
 * the bizarre behaviour of Pegwit's double barreled hashing.
 *
 * Based on the C code that Steve Reid wrote his header
 * was :
 *      SHA-1 in C
 *      By Steve Reid <steve@edmweb.com>
 *      100% Public Domain
 *
 *      Test Vectors (from FIPS PUB 180-1)
 *      "abc"
 *      A9993E36 4706816A BA3E2571 7850C26C 9CD0D89D
 *      "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq"
 *      84983E44 1C3BD26E BAAE4AA1 F95129E5 E54670F1
 *      A million repetitions of "a"
 *      34AA973C D4C4DAA4 F61EEB2B DBAD2731 6534016F
 </pre>*/
package Freenet.crypt;

public final class SHA1 implements Digest
{
    private int state[] = new int[ 5 ];
    private long count;
    private byte[] digestBits;
    private boolean digestValid;

    public int digestSize()
    {
      return 160;
    }

    /**
    * Simple constructor for SHA-1
    */
    public SHA1()
    {
      init();
    }

    public SHA1(boolean NSA)
    {
      init();
      /* always NSA */
    }

    /**
     *
     * Initializes new context
     */
    public void init()
    {
      state[ 0 ]  = 0x67452301;
      state[ 1 ]  = 0xEFCDAB89;
      state[ 2 ]  = 0x98BADCFE;
      state[ 3 ]  = 0x10325476;
      state[ 4 ]  = 0xC3D2E1F0;
      count       = 0;
      blockIndex  = 0;
      digestValid = false;
      block       = new byte[ 16 * 4 ];
      digestBits  = new byte[ 20 ];
    }

    /**
     * This takes a string and return a hash of that string.
     * @deprecated screws on anything but ascii text...
     **/
    public String doHash(String s)
    {
      this.init();
      for (int i=0; i<s.length(); i++)
          {
              this.update((byte) s.charAt(i));
          }
      this.finish();
      return this.digout();
    }

    /**
    * retrieve the value of a hash
    * @param digest int[] into which to place 5 elements
    * @param offset index of first of the 5 elements
    */
    public void extract(int[] digest, int offset)
    {
      for(int i=0; i<5; ++i) {
          digest[i + offset] = ((digestBits[4*i+0]<<24) & 0xFF000000) |
                               ((digestBits[4*i+1]<<16) & 0x00FF0000) |
                               ((digestBits[4*i+2]<< 8) & 0x0000FF00) |
                               ((digestBits[4*i+3]    ) & 0x000000FF);
      }
    }


    /*
     * The following array forms the basis for the transform^M
     * buffer. Update puts bytes into this buffer and then^M
     * transform adds it into the state of the digest.^M
     */
    private byte[] block;
    private int blockIndex;

    /*
     * Steve's original code and comments :
     *
     * blk0() and blk() perform the initial expand.
     * I got the idea of expanding during the round function from SSLeay
     *
     * #define blk0(i) block->l[i]
     * #define blk(i) (block->l[i&15] = rol(block->l[(i+13)&15]^block->l[(i+8)&15] \
     *   ^block->l[(i+2)&15]^block->l[i&15],1))
     *
     * (R0+R1), R2, R3, R4 are the different operations used in SHA1
     * #define R0(v,w,x,y,z,i) z+=((w&(x^y))^y)+blk0(i)+0x5A827999+rol(v,5);w=rol(w,30);
     * #define R1(v,w,x,y,z,i) z+=((w&(x^y))^y)+blk(i)+0x5A827999+rol(v,5);w=rol(w,30);
     * #define R2(v,w,x,y,z,i) z+=(w^x^y)+blk(i)+0x6ED9EBA1+rol(v,5);w=rol(w,30);
     * #define R3(v,w,x,y,z,i) z+=(((w|x)&y)|(w&x))+blk(i)+0x8F1BBCDC+rol(v,5);w=rol(w,30);
     * #define R4(v,w,x,y,z,i) z+=(w^x^y)+blk(i)+0xCA62C1D6+rol(v,5);w=rol(w,30);
     */

    /**
     * Hash a single 512-bit block. This is the core of the algorithm.
     *
     * Note1 that working with arrays is very inefficent in Java as it
     * does a class cast check each time you store into the array.
     * 
     * Note2, heh, fixed Note1
     */
    private void transform( byte[] block, int off )
    {
      int b0, b1, b2, b3, b4, b5, b6, b7,
          b8, b9, b10, b11, b12, b13, b14, b15,
          a, b, c, d, e;

      a = state[ 0 ];
      b = state[ 1 ];
      c = state[ 2 ];
      d = state[ 3 ];
      e = state[ 4 ];

      b0 = ((block[ off ] & 0xFF) << 24) |
           ((block[ off + 1 ] & 0xFF) << 16) |
           ((block[ off + 2 ] & 0xFF) <<  8) |
            (block[ off + 3 ] & 0xFF);           off += 4;
      b1 = ((block[ off ] & 0xFF) << 24) |
           ((block[ off + 1 ] & 0xFF) << 16) |
           ((block[ off + 2 ] & 0xFF) <<  8) |
            (block[ off + 3 ] & 0xFF);           off += 4;
      b2 = ((block[ off ] & 0xFF) << 24) |
           ((block[ off + 1 ] & 0xFF) << 16) |
           ((block[ off + 2 ] & 0xFF) <<  8) |
            (block[ off + 3 ] & 0xFF);           off += 4;
      b3 = ((block[ off ] & 0xFF) << 24) |
           ((block[ off + 1 ] & 0xFF) << 16) |
           ((block[ off + 2 ] & 0xFF) <<  8) |
            (block[ off + 3 ] & 0xFF);           off += 4;
      b4 = ((block[ off ] & 0xFF) << 24) |
           ((block[ off + 1 ] & 0xFF) << 16) |
           ((block[ off + 2 ] & 0xFF) <<  8) |
            (block[ off + 3 ] & 0xFF);           off += 4;
      b5 = ((block[ off ] & 0xFF) << 24) |
           ((block[ off + 1 ] & 0xFF) << 16) |
           ((block[ off + 2 ] & 0xFF) <<  8) |
            (block[ off + 3 ] & 0xFF);           off += 4;
      b6 = ((block[ off ] & 0xFF) << 24) |
           ((block[ off + 1 ] & 0xFF) << 16) |
           ((block[ off + 2 ] & 0xFF) <<  8) |
            (block[ off + 3 ] & 0xFF);           off += 4;
      b7 = ((block[ off ] & 0xFF) << 24) |
           ((block[ off + 1 ] & 0xFF) << 16) |
           ((block[ off + 2 ] & 0xFF) <<  8) |
            (block[ off + 3 ] & 0xFF);           off += 4;
      b8 = ((block[ off ] & 0xFF) << 24) |
           ((block[ off + 1 ] & 0xFF) << 16) |
           ((block[ off + 2 ] & 0xFF) <<  8) |
            (block[ off + 3 ] & 0xFF);           off += 4;
      b9 = ((block[ off ] & 0xFF) << 24) |
           ((block[ off + 1 ] & 0xFF) << 16) |
           ((block[ off + 2 ] & 0xFF) <<  8) |
            (block[ off + 3 ] & 0xFF);           off += 4;
      b10 = ((block[ off ] & 0xFF) << 24) |
            ((block[ off + 1 ] & 0xFF) << 16) |
            ((block[ off + 2 ] & 0xFF) <<  8) |
             (block[ off + 3 ] & 0xFF);           off += 4;
      b11 = ((block[ off ] & 0xFF) << 24) |
            ((block[ off + 1 ] & 0xFF) << 16) |
            ((block[ off + 2 ] & 0xFF) <<  8) |
             (block[ off + 3 ] & 0xFF);           off += 4;
      b12 = ((block[ off ] & 0xFF) << 24) |
            ((block[ off + 1 ] & 0xFF) << 16) |
            ((block[ off + 2 ] & 0xFF) <<  8) |
             (block[ off + 3 ] & 0xFF);           off += 4;
      b13 = ((block[ off ] & 0xFF) << 24) |
            ((block[ off + 1 ] & 0xFF) << 16) |
            ((block[ off + 2 ] & 0xFF) <<  8) |
             (block[ off + 3 ] & 0xFF);           off += 4;
      b14 = ((block[ off ] & 0xFF) << 24) |
            ((block[ off + 1 ] & 0xFF) << 16) |
            ((block[ off + 2 ] & 0xFF) <<  8) |
             (block[ off + 3 ] & 0xFF);           off += 4;
      b15 = ((block[ off ] & 0xFF) << 24) |
            ((block[ off + 1 ] & 0xFF) << 16) |
            ((block[ off + 2 ] & 0xFF) <<  8) |
             (block[ off + 3 ] & 0xFF);           off += 4;

      e +=(( b &( c ^ d ))^ d )+ (b0  ) +0x5A827999+ (((  a  ) << ( 5 )) | ((  a  ) >>> (32 - ( 5 )))) ;  b = (((  b  ) << ( 30 )) | ((  b  ) >>> (32 - ( 30 )))) ; ;
      d +=(( a &( b ^ c ))^ c )+ (b1  ) +0x5A827999+ (((  e  ) << ( 5 )) | ((  e  ) >>> (32 - ( 5 )))) ;  a = (((  a  ) << ( 30 )) | ((  a  ) >>> (32 - ( 30 )))) ; ;
      c +=(( e &( a ^ b ))^ b )+ (b2  ) +0x5A827999+ (((  d  ) << ( 5 )) | ((  d  ) >>> (32 - ( 5 )))) ;  e = (((  e  ) << ( 30 )) | ((  e  ) >>> (32 - ( 30 )))) ; ;
      b +=(( d &( e ^ a ))^ a )+ (b3  ) +0x5A827999+ (((  c  ) << ( 5 )) | ((  c  ) >>> (32 - ( 5 )))) ;  d = (((  d  ) << ( 30 )) | ((  d  ) >>> (32 - ( 30 )))) ; ;
      a +=(( c &( d ^ e ))^ e )+ (b4  ) +0x5A827999+ (((  b  ) << ( 5 )) | ((  b  ) >>> (32 - ( 5 )))) ;  c = (((  c  ) << ( 30 )) | ((  c  ) >>> (32 - ( 30 )))) ; ;
      e +=(( b &( c ^ d ))^ d )+ (b5  ) +0x5A827999+ (((  a  ) << ( 5 )) | ((  a  ) >>> (32 - ( 5 )))) ;  b = (((  b  ) << ( 30 )) | ((  b  ) >>> (32 - ( 30 )))) ; ;
      d +=(( a &( b ^ c ))^ c )+ (b6  ) +0x5A827999+ (((  e  ) << ( 5 )) | ((  e  ) >>> (32 - ( 5 )))) ;  a = (((  a  ) << ( 30 )) | ((  a  ) >>> (32 - ( 30 )))) ; ;
      c +=(( e &( a ^ b ))^ b )+ (b7  ) +0x5A827999+ (((  d  ) << ( 5 )) | ((  d  ) >>> (32 - ( 5 )))) ;  e = (((  e  ) << ( 30 )) | ((  e  ) >>> (32 - ( 30 )))) ; ;
      b +=(( d &( e ^ a ))^ a )+ (b8  ) +0x5A827999+ (((  c  ) << ( 5 )) | ((  c  ) >>> (32 - ( 5 )))) ;  d = (((  d  ) << ( 30 )) | ((  d  ) >>> (32 - ( 30 )))) ; ;
      a +=(( c &( d ^ e ))^ e )+ (b9  ) +0x5A827999+ (((  b  ) << ( 5 )) | ((  b  ) >>> (32 - ( 5 )))) ;  c = (((  c  ) << ( 30 )) | ((  c  ) >>> (32 - ( 30 )))) ; ;
      e +=(( b &( c ^ d ))^ d )+ (b10  ) +0x5A827999+ (((  a  ) << ( 5 )) | ((  a  ) >>> (32 - ( 5 )))) ;  b = (((  b  ) << ( 30 )) | ((  b  ) >>> (32 - ( 30 )))) ; ;
      d +=(( a &( b ^ c ))^ c )+ (b11  ) +0x5A827999+ (((  e  ) << ( 5 )) | ((  e  ) >>> (32 - ( 5 )))) ;  a = (((  a  ) << ( 30 )) | ((  a  ) >>> (32 - ( 30 )))) ; ;
      c +=(( e &( a ^ b ))^ b )+ (b12  ) +0x5A827999+ (((  d  ) << ( 5 )) | ((  d  ) >>> (32 - ( 5 )))) ;  e = (((  e  ) << ( 30 )) | ((  e  ) >>> (32 - ( 30 )))) ; ;
      b +=(( d &( e ^ a ))^ a )+ (b13  ) +0x5A827999+ (((  c  ) << ( 5 )) | ((  c  ) >>> (32 - ( 5 )))) ;  d = (((  d  ) << ( 30 )) | ((  d  ) >>> (32 - ( 30 )))) ; ;
      a +=(( c &( d ^ e ))^ e )+ (b14  ) +0x5A827999+ (((  b  ) << ( 5 )) | ((  b  ) >>> (32 - ( 5 )))) ;  c = (((  c  ) << ( 30 )) | ((  c  ) >>> (32 - ( 30 )))) ; ;
      e +=(( b &( c ^ d ))^ d )+ (b15  ) +0x5A827999+ (((  a  ) << ( 5 )) | ((  a  ) >>> (32 - ( 5 )))) ;  b = (((  b  ) << ( 30 )) | ((  b  ) >>> (32 - ( 30 )))) ; ;
      d +=(( a &( b ^ c ))^ c )+ (b0 = ((( b13^b8^b2^b0 ) << ( 1 )) | (( b13^b8^b2^b0 ) >>> (32 - ( 1 )))) )    +0x5A827999+ (((  e  ) << ( 5 )) | ((  e  ) >>> (32 - ( 5 )))) ;  a = (((  a  ) << ( 30 )) | ((  a  ) >>> (32 - ( 30 )))) ; ;
      c +=(( e &( a ^ b ))^ b )+ (b1 = ((( b14^b9^b3^b1 ) << ( 1 )) | (( b14^b9^b3^b1 ) >>> (32 - ( 1 )))) )    +0x5A827999+ (((  d  ) << ( 5 )) | ((  d  ) >>> (32 - ( 5 )))) ;  e = (((  e  ) << ( 30 )) | ((  e  ) >>> (32 - ( 30 )))) ; ;
      b +=(( d &( e ^ a ))^ a )+ (b2 = ((( b15^b10^b4^b2 ) << ( 1 )) | (( b15^b10^b4^b2 ) >>> (32 - ( 1 )))) )    +0x5A827999+ (((  c  ) << ( 5 )) | ((  c  ) >>> (32 - ( 5 )))) ;  d = (((  d  ) << ( 30 )) | ((  d  ) >>> (32 - ( 30 )))) ; ;
      a +=(( c &( d ^ e ))^ e )+ (b3 = ((( b0^b11^b5^b3 ) << ( 1 )) | (( b0^b11^b5^b3 ) >>> (32 - ( 1 )))) )    +0x5A827999+ (((  b  ) << ( 5 )) | ((  b  ) >>> (32 - ( 5 )))) ;  c = (((  c  ) << ( 30 )) | ((  c  ) >>> (32 - ( 30 )))) ; ;
      e +=( b ^ c ^ d )+ (b4 = ((( b1^b12^b6^b4 ) << ( 1 )) | (( b1^b12^b6^b4 ) >>> (32 - ( 1 )))) )    +0x6ED9EBA1+ (((  a  ) << ( 5 )) | ((  a  ) >>> (32 - ( 5 )))) ;  b = (((  b  ) << ( 30 )) | ((  b  ) >>> (32 - ( 30 )))) ; ;
      d +=( a ^ b ^ c )+ (b5 = ((( b2^b13^b7^b5 ) << ( 1 )) | (( b2^b13^b7^b5 ) >>> (32 - ( 1 )))) )    +0x6ED9EBA1+ (((  e  ) << ( 5 )) | ((  e  ) >>> (32 - ( 5 )))) ;  a = (((  a  ) << ( 30 )) | ((  a  ) >>> (32 - ( 30 )))) ; ;
      c +=( e ^ a ^ b )+ (b6 = ((( b3^b14^b8^b6 ) << ( 1 )) | (( b3^b14^b8^b6 ) >>> (32 - ( 1 )))) )    +0x6ED9EBA1+ (((  d  ) << ( 5 )) | ((  d  ) >>> (32 - ( 5 )))) ;  e = (((  e  ) << ( 30 )) | ((  e  ) >>> (32 - ( 30 )))) ; ;
      b +=( d ^ e ^ a )+ (b7 = ((( b4^b15^b9^b7 ) << ( 1 )) | (( b4^b15^b9^b7 ) >>> (32 - ( 1 )))) )    +0x6ED9EBA1+ (((  c  ) << ( 5 )) | ((  c  ) >>> (32 - ( 5 )))) ;  d = (((  d  ) << ( 30 )) | ((  d  ) >>> (32 - ( 30 )))) ; ;
      a +=( c ^ d ^ e )+ (b8 = ((( b5^b0^b10^b8 ) << ( 1 )) | (( b5^b0^b10^b8 ) >>> (32 - ( 1 )))) )    +0x6ED9EBA1+ (((  b  ) << ( 5 )) | ((  b  ) >>> (32 - ( 5 )))) ;  c = (((  c  ) << ( 30 )) | ((  c  ) >>> (32 - ( 30 )))) ; ;
      e +=( b ^ c ^ d )+ (b9 = ((( b6^b1^b11^b9 ) << ( 1 )) | (( b6^b1^b11^b9 ) >>> (32 - ( 1 )))) )    +0x6ED9EBA1+ (((  a  ) << ( 5 )) | ((  a  ) >>> (32 - ( 5 )))) ;  b = (((  b  ) << ( 30 )) | ((  b  ) >>> (32 - ( 30 )))) ; ;
      d +=( a ^ b ^ c )+ (b10 = ((( b7^b2^b12^b10 ) << ( 1 )) | (( b7^b2^b12^b10 ) >>> (32 - ( 1 )))) )    +0x6ED9EBA1+ (((  e  ) << ( 5 )) | ((  e  ) >>> (32 - ( 5 )))) ;  a = (((  a  ) << ( 30 )) | ((  a  ) >>> (32 - ( 30 )))) ; ;
      c +=( e ^ a ^ b )+ (b11 = ((( b8^b3^b13^b11 ) << ( 1 )) | (( b8^b3^b13^b11 ) >>> (32 - ( 1 )))) )    +0x6ED9EBA1+ (((  d  ) << ( 5 )) | ((  d  ) >>> (32 - ( 5 )))) ;  e = (((  e  ) << ( 30 )) | ((  e  ) >>> (32 - ( 30 )))) ; ;
      b +=( d ^ e ^ a )+ (b12 = ((( b9^b4^b14^b12 ) << ( 1 )) | (( b9^b4^b14^b12 ) >>> (32 - ( 1 )))) )    +0x6ED9EBA1+ (((  c  ) << ( 5 )) | ((  c  ) >>> (32 - ( 5 )))) ;  d = (((  d  ) << ( 30 )) | ((  d  ) >>> (32 - ( 30 )))) ; ;
      a +=( c ^ d ^ e )+ (b13 = ((( b10^b5^b15^b13 ) << ( 1 )) | (( b10^b5^b15^b13 ) >>> (32 - ( 1 )))) )    +0x6ED9EBA1+ (((  b  ) << ( 5 )) | ((  b  ) >>> (32 - ( 5 )))) ;  c = (((  c  ) << ( 30 )) | ((  c  ) >>> (32 - ( 30 )))) ; ;
      e +=( b ^ c ^ d )+ (b14 = ((( b11^b6^b0^b14 ) << ( 1 )) | (( b11^b6^b0^b14 ) >>> (32 - ( 1 )))) )    +0x6ED9EBA1+ (((  a  ) << ( 5 )) | ((  a  ) >>> (32 - ( 5 )))) ;  b = (((  b  ) << ( 30 )) | ((  b  ) >>> (32 - ( 30 )))) ; ;
      d +=( a ^ b ^ c )+ (b15 = ((( b12^b7^b1^b15 ) << ( 1 )) | (( b12^b7^b1^b15 ) >>> (32 - ( 1 )))) )    +0x6ED9EBA1+ (((  e  ) << ( 5 )) | ((  e  ) >>> (32 - ( 5 )))) ;  a = (((  a  ) << ( 30 )) | ((  a  ) >>> (32 - ( 30 )))) ; ;
      c +=( e ^ a ^ b )+ (b0 = ((( b13^b8^b2^b0 ) << ( 1 )) | (( b13^b8^b2^b0 ) >>> (32 - ( 1 )))) )    +0x6ED9EBA1+ (((  d  ) << ( 5 )) | ((  d  ) >>> (32 - ( 5 )))) ;  e = (((  e  ) << ( 30 )) | ((  e  ) >>> (32 - ( 30 )))) ; ;
      b +=( d ^ e ^ a )+ (b1 = ((( b14^b9^b3^b1 ) << ( 1 )) | (( b14^b9^b3^b1 ) >>> (32 - ( 1 )))) )    +0x6ED9EBA1+ (((  c  ) << ( 5 )) | ((  c  ) >>> (32 - ( 5 )))) ;  d = (((  d  ) << ( 30 )) | ((  d  ) >>> (32 - ( 30 )))) ; ;
      a +=( c ^ d ^ e )+ (b2 = ((( b15^b10^b4^b2 ) << ( 1 )) | (( b15^b10^b4^b2 ) >>> (32 - ( 1 )))) )    +0x6ED9EBA1+ (((  b  ) << ( 5 )) | ((  b  ) >>> (32 - ( 5 )))) ;  c = (((  c  ) << ( 30 )) | ((  c  ) >>> (32 - ( 30 )))) ; ;
      e +=( b ^ c ^ d )+ (b3 = ((( b0^b11^b5^b3 ) << ( 1 )) | (( b0^b11^b5^b3 ) >>> (32 - ( 1 )))) )    +0x6ED9EBA1+ (((  a  ) << ( 5 )) | ((  a  ) >>> (32 - ( 5 )))) ;  b = (((  b  ) << ( 30 )) | ((  b  ) >>> (32 - ( 30 )))) ; ;
      d +=( a ^ b ^ c )+ (b4 = ((( b1^b12^b6^b4 ) << ( 1 )) | (( b1^b12^b6^b4 ) >>> (32 - ( 1 )))) )    +0x6ED9EBA1+ (((  e  ) << ( 5 )) | ((  e  ) >>> (32 - ( 5 )))) ;  a = (((  a  ) << ( 30 )) | ((  a  ) >>> (32 - ( 30 )))) ; ;
      c +=( e ^ a ^ b )+ (b5 = ((( b2^b13^b7^b5 ) << ( 1 )) | (( b2^b13^b7^b5 ) >>> (32 - ( 1 )))) )    +0x6ED9EBA1+ (((  d  ) << ( 5 )) | ((  d  ) >>> (32 - ( 5 )))) ;  e = (((  e  ) << ( 30 )) | ((  e  ) >>> (32 - ( 30 )))) ; ;
      b +=( d ^ e ^ a )+ (b6 = ((( b3^b14^b8^b6 ) << ( 1 )) | (( b3^b14^b8^b6 ) >>> (32 - ( 1 )))) )    +0x6ED9EBA1+ (((  c  ) << ( 5 )) | ((  c  ) >>> (32 - ( 5 )))) ;  d = (((  d  ) << ( 30 )) | ((  d  ) >>> (32 - ( 30 )))) ; ;
      a +=( c ^ d ^ e )+ (b7 = ((( b4^b15^b9^b7 ) << ( 1 )) | (( b4^b15^b9^b7 ) >>> (32 - ( 1 )))) )    +0x6ED9EBA1+ (((  b  ) << ( 5 )) | ((  b  ) >>> (32 - ( 5 )))) ;  c = (((  c  ) << ( 30 )) | ((  c  ) >>> (32 - ( 30 )))) ; ;
      e +=((( b | c )& d )|( b & c ))+ (b8 = ((( b5^b0^b10^b8 ) << ( 1 )) | (( b5^b0^b10^b8 ) >>> (32 - ( 1 )))) )    +0x8F1BBCDC+ (((  a  ) << ( 5 )) | ((  a  ) >>> (32 - ( 5 )))) ;  b = (((  b  ) << ( 30 )) | ((  b  ) >>> (32 - ( 30 )))) ; ;
      d +=((( a | b )& c )|( a & b ))+ (b9 = ((( b6^b1^b11^b9 ) << ( 1 )) | (( b6^b1^b11^b9 ) >>> (32 - ( 1 )))) )    +0x8F1BBCDC+ (((  e  ) << ( 5 )) | ((  e  ) >>> (32 - ( 5 )))) ;  a = (((  a  ) << ( 30 )) | ((  a  ) >>> (32 - ( 30 )))) ; ;
      c +=((( e | a )& b )|( e & a ))+ (b10 = ((( b7^b2^b12^b10 ) << ( 1 )) | (( b7^b2^b12^b10 ) >>> (32 - ( 1 )))) )    +0x8F1BBCDC+ (((  d  ) << ( 5 )) | ((  d  ) >>> (32 - ( 5 )))) ;  e = (((  e  ) << ( 30 )) | ((  e  ) >>> (32 - ( 30 )))) ; ;
      b +=((( d | e )& a )|( d & e ))+ (b11 = ((( b8^b3^b13^b11 ) << ( 1 )) | (( b8^b3^b13^b11 ) >>> (32 - ( 1 )))) )    +0x8F1BBCDC+ (((  c  ) << ( 5 )) | ((  c  ) >>> (32 - ( 5 )))) ;  d = (((  d  ) << ( 30 )) | ((  d  ) >>> (32 - ( 30 )))) ; ;
      a +=((( c | d )& e )|( c & d ))+ (b12 = ((( b9^b4^b14^b12 ) << ( 1 )) | (( b9^b4^b14^b12 ) >>> (32 - ( 1 )))) )    +0x8F1BBCDC+ (((  b  ) << ( 5 )) | ((  b  ) >>> (32 - ( 5 )))) ;  c = (((  c  ) << ( 30 )) | ((  c  ) >>> (32 - ( 30 )))) ; ;
      e +=((( b | c )& d )|( b & c ))+ (b13 = ((( b10^b5^b15^b13 ) << ( 1 )) | (( b10^b5^b15^b13 ) >>> (32 - ( 1 )))) )    +0x8F1BBCDC+ (((  a  ) << ( 5 )) | ((  a  ) >>> (32 - ( 5 )))) ;  b = (((  b  ) << ( 30 )) | ((  b  ) >>> (32 - ( 30 )))) ; ;
      d +=((( a | b )& c )|( a & b ))+ (b14 = ((( b11^b6^b0^b14 ) << ( 1 )) | (( b11^b6^b0^b14 ) >>> (32 - ( 1 )))) )    +0x8F1BBCDC+ (((  e  ) << ( 5 )) | ((  e  ) >>> (32 - ( 5 )))) ;  a = (((  a  ) << ( 30 )) | ((  a  ) >>> (32 - ( 30 )))) ; ;
      c +=((( e | a )& b )|( e & a ))+ (b15 = ((( b12^b7^b1^b15 ) << ( 1 )) | (( b12^b7^b1^b15 ) >>> (32 - ( 1 )))) )    +0x8F1BBCDC+ (((  d  ) << ( 5 )) | ((  d  ) >>> (32 - ( 5 )))) ;  e = (((  e  ) << ( 30 )) | ((  e  ) >>> (32 - ( 30 )))) ; ;
      b +=((( d | e )& a )|( d & e ))+ (b0 = ((( b13^b8^b2^b0 ) << ( 1 )) | (( b13^b8^b2^b0 ) >>> (32 - ( 1 )))) )    +0x8F1BBCDC+ (((  c  ) << ( 5 )) | ((  c  ) >>> (32 - ( 5 )))) ;  d = (((  d  ) << ( 30 )) | ((  d  ) >>> (32 - ( 30 )))) ; ;
      a +=((( c | d )& e )|( c & d ))+ (b1 = ((( b14^b9^b3^b1 ) << ( 1 )) | (( b14^b9^b3^b1 ) >>> (32 - ( 1 )))) )    +0x8F1BBCDC+ (((  b  ) << ( 5 )) | ((  b  ) >>> (32 - ( 5 )))) ;  c = (((  c  ) << ( 30 )) | ((  c  ) >>> (32 - ( 30 )))) ; ;
      e +=((( b | c )& d )|( b & c ))+ (b2 = ((( b15^b10^b4^b2 ) << ( 1 )) | (( b15^b10^b4^b2 ) >>> (32 - ( 1 )))) )    +0x8F1BBCDC+ (((  a  ) << ( 5 )) | ((  a  ) >>> (32 - ( 5 )))) ;  b = (((  b  ) << ( 30 )) | ((  b  ) >>> (32 - ( 30 )))) ; ;
      d +=((( a | b )& c )|( a & b ))+ (b3 = ((( b0^b11^b5^b3 ) << ( 1 )) | (( b0^b11^b5^b3 ) >>> (32 - ( 1 )))) )    +0x8F1BBCDC+ (((  e  ) << ( 5 )) | ((  e  ) >>> (32 - ( 5 )))) ;  a = (((  a  ) << ( 30 )) | ((  a  ) >>> (32 - ( 30 )))) ; ;
      c +=((( e | a )& b )|( e & a ))+ (b4 = ((( b1^b12^b6^b4 ) << ( 1 )) | (( b1^b12^b6^b4 ) >>> (32 - ( 1 )))) )    +0x8F1BBCDC+ (((  d  ) << ( 5 )) | ((  d  ) >>> (32 - ( 5 )))) ;  e = (((  e  ) << ( 30 )) | ((  e  ) >>> (32 - ( 30 )))) ; ;
      b +=((( d | e )& a )|( d & e ))+ (b5 = ((( b2^b13^b7^b5 ) << ( 1 )) | (( b2^b13^b7^b5 ) >>> (32 - ( 1 )))) )    +0x8F1BBCDC+ (((  c  ) << ( 5 )) | ((  c  ) >>> (32 - ( 5 )))) ;  d = (((  d  ) << ( 30 )) | ((  d  ) >>> (32 - ( 30 )))) ; ;
      a +=((( c | d )& e )|( c & d ))+ (b6 = ((( b3^b14^b8^b6 ) << ( 1 )) | (( b3^b14^b8^b6 ) >>> (32 - ( 1 )))) )    +0x8F1BBCDC+ (((  b  ) << ( 5 )) | ((  b  ) >>> (32 - ( 5 )))) ;  c = (((  c  ) << ( 30 )) | ((  c  ) >>> (32 - ( 30 )))) ; ;
      e +=((( b | c )& d )|( b & c ))+ (b7 = ((( b4^b15^b9^b7 ) << ( 1 )) | (( b4^b15^b9^b7 ) >>> (32 - ( 1 )))) )    +0x8F1BBCDC+ (((  a  ) << ( 5 )) | ((  a  ) >>> (32 - ( 5 )))) ;  b = (((  b  ) << ( 30 )) | ((  b  ) >>> (32 - ( 30 )))) ; ;
      d +=((( a | b )& c )|( a & b ))+ (b8 = ((( b5^b0^b10^b8 ) << ( 1 )) | (( b5^b0^b10^b8 ) >>> (32 - ( 1 )))) )    +0x8F1BBCDC+ (((  e  ) << ( 5 )) | ((  e  ) >>> (32 - ( 5 )))) ;  a = (((  a  ) << ( 30 )) | ((  a  ) >>> (32 - ( 30 )))) ; ;
      c +=((( e | a )& b )|( e & a ))+ (b9 = ((( b6^b1^b11^b9 ) << ( 1 )) | (( b6^b1^b11^b9 ) >>> (32 - ( 1 )))) )    +0x8F1BBCDC+ (((  d  ) << ( 5 )) | ((  d  ) >>> (32 - ( 5 )))) ;  e = (((  e  ) << ( 30 )) | ((  e  ) >>> (32 - ( 30 )))) ; ;
      b +=((( d | e )& a )|( d & e ))+ (b10 = ((( b7^b2^b12^b10 ) << ( 1 )) | (( b7^b2^b12^b10 ) >>> (32 - ( 1 )))) )    +0x8F1BBCDC+ (((  c  ) << ( 5 )) | ((  c  ) >>> (32 - ( 5 )))) ;  d = (((  d  ) << ( 30 )) | ((  d  ) >>> (32 - ( 30 )))) ; ;
      a +=((( c | d )& e )|( c & d ))+ (b11 = ((( b8^b3^b13^b11 ) << ( 1 )) | (( b8^b3^b13^b11 ) >>> (32 - ( 1 )))) )    +0x8F1BBCDC+ (((  b  ) << ( 5 )) | ((  b  ) >>> (32 - ( 5 )))) ;  c = (((  c  ) << ( 30 )) | ((  c  ) >>> (32 - ( 30 )))) ; ;
      e +=( b ^ c ^ d )+ (b12 = ((( b9^b4^b14^b12 ) << ( 1 )) | (( b9^b4^b14^b12 ) >>> (32 - ( 1 )))) )    +0xCA62C1D6+ (((  a  ) << ( 5 )) | ((  a  ) >>> (32 - ( 5 )))) ; b = (((  b  ) << ( 30 )) | ((  b  ) >>> (32 - ( 30 )))) ; ;
      d +=( a ^ b ^ c )+ (b13 = ((( b10^b5^b15^b13 ) << ( 1 )) | (( b10^b5^b15^b13 ) >>> (32 - ( 1 )))) )    +0xCA62C1D6+ (((  e  ) << ( 5 )) | ((  e  ) >>> (32 - ( 5 )))) ; a = (((  a  ) << ( 30 )) | ((  a  ) >>> (32 - ( 30 )))) ; ;
      c +=( e ^ a ^ b )+ (b14 = ((( b11^b6^b0^b14 ) << ( 1 )) | (( b11^b6^b0^b14 ) >>> (32 - ( 1 )))) )    +0xCA62C1D6+ (((  d  ) << ( 5 )) | ((  d  ) >>> (32 - ( 5 )))) ; e = (((  e  ) << ( 30 )) | ((  e  ) >>> (32 - ( 30 )))) ; ;
      b +=( d ^ e ^ a )+ (b15 = ((( b12^b7^b1^b15 ) << ( 1 )) | (( b12^b7^b1^b15 ) >>> (32 - ( 1 )))) )    +0xCA62C1D6+ (((  c  ) << ( 5 )) | ((  c  ) >>> (32 - ( 5 )))) ; d = (((  d  ) << ( 30 )) | ((  d  ) >>> (32 - ( 30 )))) ; ;
      a +=( c ^ d ^ e )+ (b0 = ((( b13^b8^b2^b0 ) << ( 1 )) | (( b13^b8^b2^b0 ) >>> (32 - ( 1 )))) )    +0xCA62C1D6+ (((  b  ) << ( 5 )) | ((  b  ) >>> (32 - ( 5 )))) ; c = (((  c  ) << ( 30 )) | ((  c  ) >>> (32 - ( 30 )))) ; ;
      e +=( b ^ c ^ d )+ (b1 = ((( b14^b9^b3^b1 ) << ( 1 )) | (( b14^b9^b3^b1 ) >>> (32 - ( 1 )))) )    +0xCA62C1D6+ (((  a  ) << ( 5 )) | ((  a  ) >>> (32 - ( 5 )))) ; b = (((  b  ) << ( 30 )) | ((  b  ) >>> (32 - ( 30 )))) ; ;
      d +=( a ^ b ^ c )+ (b2 = ((( b15^b10^b4^b2 ) << ( 1 )) | (( b15^b10^b4^b2 ) >>> (32 - ( 1 )))) )    +0xCA62C1D6+ (((  e  ) << ( 5 )) | ((  e  ) >>> (32 - ( 5 )))) ; a = (((  a  ) << ( 30 )) | ((  a  ) >>> (32 - ( 30 )))) ; ;
      c +=( e ^ a ^ b )+ (b3 = ((( b0^b11^b5^b3 ) << ( 1 )) | (( b0^b11^b5^b3 ) >>> (32 - ( 1 )))) )    +0xCA62C1D6+ (((  d  ) << ( 5 )) | ((  d  ) >>> (32 - ( 5 )))) ; e = (((  e  ) << ( 30 )) | ((  e  ) >>> (32 - ( 30 )))) ; ;
      b +=( d ^ e ^ a )+ (b4 = ((( b1^b12^b6^b4 ) << ( 1 )) | (( b1^b12^b6^b4 ) >>> (32 - ( 1 )))) )    +0xCA62C1D6+ (((  c  ) << ( 5 )) | ((  c  ) >>> (32 - ( 5 )))) ; d = (((  d  ) << ( 30 )) | ((  d  ) >>> (32 - ( 30 )))) ; ;
      a +=( c ^ d ^ e )+ (b5 = ((( b2^b13^b7^b5 ) << ( 1 )) | (( b2^b13^b7^b5 ) >>> (32 - ( 1 )))) )    +0xCA62C1D6+ (((  b  ) << ( 5 )) | ((  b  ) >>> (32 - ( 5 )))) ; c = (((  c  ) << ( 30 )) | ((  c  ) >>> (32 - ( 30 )))) ; ;
      e +=( b ^ c ^ d )+ (b6 = ((( b3^b14^b8^b6 ) << ( 1 )) | (( b3^b14^b8^b6 ) >>> (32 - ( 1 )))) )    +0xCA62C1D6+ (((  a  ) << ( 5 )) | ((  a  ) >>> (32 - ( 5 )))) ; b = (((  b  ) << ( 30 )) | ((  b  ) >>> (32 - ( 30 )))) ; ;
      d +=( a ^ b ^ c )+ (b7 = ((( b4^b15^b9^b7 ) << ( 1 )) | (( b4^b15^b9^b7 ) >>> (32 - ( 1 )))) )    +0xCA62C1D6+ (((  e  ) << ( 5 )) | ((  e  ) >>> (32 - ( 5 )))) ; a = (((  a  ) << ( 30 )) | ((  a  ) >>> (32 - ( 30 )))) ; ;
      c +=( e ^ a ^ b )+ (b8 = ((( b5^b0^b10^b8 ) << ( 1 )) | (( b5^b0^b10^b8 ) >>> (32 - ( 1 )))) )    +0xCA62C1D6+ (((  d  ) << ( 5 )) | ((  d  ) >>> (32 - ( 5 )))) ; e = (((  e  ) << ( 30 )) | ((  e  ) >>> (32 - ( 30 )))) ; ;
      b +=( d ^ e ^ a )+ (b9 = ((( b6^b1^b11^b9 ) << ( 1 )) | (( b6^b1^b11^b9 ) >>> (32 - ( 1 )))) )    +0xCA62C1D6+ (((  c  ) << ( 5 )) | ((  c  ) >>> (32 - ( 5 )))) ; d = (((  d  ) << ( 30 )) | ((  d  ) >>> (32 - ( 30 )))) ; ;
      a +=( c ^ d ^ e )+ (b10 = ((( b7^b2^b12^b10 ) << ( 1 )) | (( b7^b2^b12^b10 ) >>> (32 - ( 1 )))) )    +0xCA62C1D6+ (((  b  ) << ( 5 )) | ((  b  ) >>> (32 - ( 5 )))) ; c = (((  c  ) << ( 30 )) | ((  c  ) >>> (32 - ( 30 )))) ; ;
      e +=( b ^ c ^ d )+ (b11 = ((( b8^b3^b13^b11 ) << ( 1 )) | (( b8^b3^b13^b11 ) >>> (32 - ( 1 )))) )    +0xCA62C1D6+ (((  a  ) << ( 5 )) | ((  a  ) >>> (32 - ( 5 )))) ; b = (((  b  ) << ( 30 )) | ((  b  ) >>> (32 - ( 30 )))) ; ;
      d +=( a ^ b ^ c )+ (b12 = ((( b9^b4^b14^b12 ) << ( 1 )) | (( b9^b4^b14^b12 ) >>> (32 - ( 1 )))) )    +0xCA62C1D6+ (((  e  ) << ( 5 )) | ((  e  ) >>> (32 - ( 5 )))) ; a = (((  a  ) << ( 30 )) | ((  a  ) >>> (32 - ( 30 )))) ; ;
      c +=( e ^ a ^ b )+ (b13 = ((( b10^b5^b15^b13 ) << ( 1 )) | (( b10^b5^b15^b13 ) >>> (32 - ( 1 )))) )    +0xCA62C1D6+ (((  d  ) << ( 5 )) | ((  d  ) >>> (32 - ( 5 )))) ; e = (((  e  ) << ( 30 )) | ((  e  ) >>> (32 - ( 30 )))) ; ;
      b +=( d ^ e ^ a )+ (b14 = ((( b11^b6^b0^b14 ) << ( 1 )) | (( b11^b6^b0^b14 ) >>> (32 - ( 1 )))) )    +0xCA62C1D6+ (((  c  ) << ( 5 )) | ((  c  ) >>> (32 - ( 5 )))) ; d = (((  d  ) << ( 30 )) | ((  d  ) >>> (32 - ( 30 )))) ; ;
      a +=( c ^ d ^ e )+ (b15 = ((( b12^b7^b1^b15 ) << ( 1 )) | (( b12^b7^b1^b15 ) >>> (32 - ( 1 )))) )    +0xCA62C1D6+ (((  b  ) << ( 5 )) | ((  b  ) >>> (32 - ( 5 )))) ; c = (((  c  ) << ( 30 )) | ((  c  ) >>> (32 - ( 30 )))) ; ;

      /* Add the working vars back into context.state[] */
      state[ 0 ] += a;
      state[ 1 ] += b;
      state[ 2 ] += c;
      state[ 3 ] += d;
      state[ 4 ] += e;
    }

    /**
     * Add one byte to the digest. When this is implemented
     * all of the abstract class methods end up calling
     * this method for types other than bytes.
     * @param b byte to add
     */
    public void update( byte b )
    {
      block[ blockIndex++ ] = b;
      count += 8;
      if ( blockIndex == 64 ) {
        transform( block, 0 );
        blockIndex = 0;
      }
    }

    /**
     * Add many bytes to the digest.
     * @param data byte data to add
     * @param offset start byte
     * @param length number of bytes to hash
     */
    public void update( byte[] data, int offset, int length )
    {
      int bytesLeft;

      if ( length == 0 )
        return;

      count += length * 8;

      for (;;) {
        /* transform directly from data */
        if ( blockIndex == 0 && length >= 64 ) {
          do {
            transform( data, offset );
            offset += 64;
            length -= 64;
          } while ( length >= 64 );

          if ( length == 0 ) {
            return;
          }
        }

        /* copy bytes into block */
        bytesLeft = 64 - blockIndex;

        if ( bytesLeft > length ) {
          System.arraycopy( data, offset, block, blockIndex, length );
          blockIndex += length;
          return;
        }

        /* have enough data to transform */
        System.arraycopy( data, offset, block, blockIndex, bytesLeft );
        offset     += bytesLeft;
        length     -= bytesLeft;
        blockIndex  = 0;

        transform( block, 0 );
      }
    }


    /**
     * Return completed digest filled into the given buffer.
     * @return the byte array result
     * @param reset If true, the hash function is reinitialized
     */
    public void digest(boolean reset, byte[] buffer, int offset) {
        finish();
        System.arraycopy(digestBits, 0, buffer, offset, digestBits.length);
        if (reset) init();
    }

    /**
     * Return completed digest.
     * @return the byte array result
     * @param reset If true, the hash function is reinitialized
     */
    public byte[] digest(boolean reset)
    {
        byte [] out = new byte[20];
        digest(reset, out, 0);
        return out;
    }

    /**
     * Return completed digest.
     * @return the byte array result
     */
    public byte[] digest() {
        return digest(true);
    }


    /**
     * Complete processing on the message digest.
     */
    public void finish() {
      byte[] bits = new byte[ 8 ];
      int    shift,
             bitsOff,
             stateOff;

      bitsOff = 0;
      shift   = 64;
      do {
        shift -= 8;
        bits[ bitsOff++ ] = (byte) ( ( count >>> shift ) & 0xFF );
      } while ( shift > 0 );

      update( (byte) 128 );

      while ( blockIndex != 56 )
        update( (byte) 0 );

      /* This should cause a transform to happen */
      update( bits, 0, 8 );

      stateOff = 0;
      bitsOff  = 0;
      do {
        digestBits[ bitsOff++ ] = (byte) ( ( state[ stateOff ] >> 24 ) & 0xFF );
        digestBits[ bitsOff++ ] = (byte) ( ( state[ stateOff ] >> 16 ) & 0xFF );
        digestBits[ bitsOff++ ] = (byte) ( ( state[ stateOff ] >> 8 ) & 0xFF );
        digestBits[ bitsOff++ ] = (byte) ( state[ stateOff ] & 0xFF );
      } while ( ++stateOff < state.length );

      digestValid = true;
    }


    /**
     * Print out the digest in a form that can be easily compared
     * to the test vectors.
     */
    protected String digout() {
      StringBuffer sb = new StringBuffer();
      for (int i = 0; i < 20; i++) {
        char c1, c2;

        c1 = (char) ((digestBits[i] >>> 4) & 0xf);
        c2 = (char) (digestBits[i] & 0xf);
        c1 = (char) ((c1 > 9) ? 'A' + (c1 - 10) : '0' + c1);
        c2 = (char) ((c2 > 9) ? 'A' + (c2 - 10) : '0' + c2);
        sb.append(c1);
        sb.append(c2);
      }
      return sb.toString();
    }


    public static void main( String[] args )
      throws java.io.IOException
    {
      SHAselfTest();
    }

    /**
     * perfoms a self-test - spits out the test vector outputs on System.out
     */
    public static void SHAselfTest()
    {
      int i, j;
      SHA1 s = new SHA1();
      // This line may be safely deleted, its to make it easy to see
      // the output of the program.
      System.out.println("SHA-1 Test PROGRAM.");
      System.out.println("This code runs the test vectors through the code.");

/*      "abc"
      A9993E36 4706816A BA3E2571 7850C26C 9CD0D89D */

      System.out.println("First test is 'abc'");
      String z = "abc";
      s.init();
      s.update((byte) 'a');
      s.update((byte) 'b');
      s.update((byte) 'c');
      s.finish();
      System.out.println(s.digout());
      System.out.println("A9993E364706816ABA3E25717850C26C9CD0D89D");

/*      "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq"
      84983E44 1C3BD26E BAAE4AA1 F95129E5 E54670F1 */

      System.out.println("Next Test is 'abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq'");
      z = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq";
      s.init();
      for( i=0; i<z.length(); ++i)
      {
              s.update((byte) z.charAt(i));
      }
      s.finish();
      System.out.println(s.digout());
      System.out.println("84983E441C3BD26EBAAE4AA1F95129E5E54670F1");

/*      A million repetitions of "a"
      34AA973C D4C4DAA4 F61EEB2B DBAD2731 6534016F */
        long startTime = 0 - System.currentTimeMillis();

      System.out.println("Last test is 1 million 'a' characters.");
      s.init();
      for (i = 0; i < 1000000; i++)
          s.update((byte) 'a');
      s.finish();
      System.out.println(s.digout());
      System.out.println("34AA973CD4C4DAA4F61EEB2BDBAD27316534016F");
      startTime += System.currentTimeMillis();
      double d = ((double)startTime)/1000.0;
      System.out.println(" done, elapsed time = "+d);
    }
}
