package Freenet.fs;

import java.io.*;
import java.util.*;
import Freenet.support.*;
import Freenet.Key;
import Freenet.FieldSet;
import Freenet.crypt.*;
import Freenet.crypt.ciphers.Twofish;

/**
 * Implements a simple filesystem in a file.  The filesystem itself has the 
 * following properties:
 *
 *  - Flat, unchanging file as the filesystem base.
 *  - Fragmenation resistance through a choosy allocation algorithm
 *  - Corruption resistance through atomic directory operations
 *  - Strong data protection via two-stage locking
 *  - One-byte block size (no blocks, so this must run on an OS filesystem 
 *    for good performance).
 *  - Optional encryption to thwart ciphertext matching attacks
 * 
 * @author scott
 */
public class DatastoreFS extends Thread {
    protected static final int FS_MAX=5, SYNC_INTERVAL=60000;

    protected Directory directory;
    protected File dsf, dsdf;
    protected long dsSize;
    protected Logger logger;
    //    protected Random r;
    protected Stack fileStack=new Stack();
    protected Vector 
	locks=new Vector();
    
    private byte[] cryptoKey;
    
    public DatastoreFS(File data, File directory, 
		       Logger log) throws IOException {
	//	this.r=r;
	dsf=data;
	dsSize=dsf.length();
	dsdf=directory;
	this.directory=new Directory(dsdf, dsSize);

	logger=(log == null ? new StandardLogger(System.err, 5, 
						 Logger.MINOR) : log);
	setDaemon(true);
	start();
    }

    /* ----------- Directory functions ----------------------------------- */

    public FSElement stat(String key) {
	return directory.get(key);
    }

    protected void sync() {
	directory.sync();
    }

    protected void listDir() {
	directory.list();
    }

    public long[][] getRanges(Object key) {
	return directory.getRanges(key);
    }

    public void free(Object key) throws FileNotFoundException {
	directory.free(key);
    }

    public void commit(WriteToken t, FSElement e) throws IOException {
	t.stream.close();
	for (int i=0; i<t.locks.length; i++) 
	    t.locks[i].unlock();
	directory.commit(e);
    }

    public void rollback(WriteToken t, FSElement e) {
	for (int i=0; i<t.locks.length; i++) 
	    t.locks[i].unlock();
	directory.rollback(t, e);
    }

    public void index(WriteToken l, FSElement e) throws IOException {
	directory.index(e);
	e.ranges=l.ranges;
    }

    public void run() {
	while (true) {
	    try {
		if (directory.isDirty()) {
		    logger.log(this, 
			       "Syncing datastore directory with disk.", 
			       Logger.NORMAL);
		    directory.sync();
		} else
		    logger.log(this, 
			       "No need to sync datastore, datastore is clean.", 
			       Logger.MINOR);
		
		sleep(SYNC_INTERVAL);
	    } catch (InterruptedException ie) {}
	}
    }

    /* ------------  Locking functions ------------------------------------ */

    public WriteLock writeLock(long[] range) throws IOException {
	RandomAccessFile f=new RandomAccessFile(dsf, "rw");
	f.seek(range[0]);
	WriteLock l=new WriteLock(this, range[0], range[1], f);
	lockObtain(l);
	waitForUpgrade(l);
	return l;
    }
    
    protected void waitForUpgrade(WriteLock l) {
	l.protstatus=Lock.ALL;
	spinLock(l);
	l.upgrade();
    }

    public ReadLock readLock(long[] range) throws IOException {
	RandomAccessFile f=getReadStream(range[0]);
	Lock wl=locked(Lock.WRITE, range);
	ReadLock l=null;
	if (wl!=null) 
	    l=new SlidingLock(this, range[0], range[1], f, (WriteLock)wl);
	else
	    l=new ReadLock(this, range[0], range[1], f);
	lockObtain(l);
	return l;
    }

    public Lock locked(int prot, long[] range) {
	for (Enumeration enum=locks.elements();
	     enum.hasMoreElements();) {
	    Lock l=(Lock)enum.nextElement();
	    if (l.isLocked() && 
		(l.getType() & prot) != 0) 
		return l;
	}
	return null;
    }
	
    public void unlock(Lock l) {
	logger.log(this, "Releasing lock "+l, Logger.DEBUGGING);
	locks.removeElement(l);
	synchronized(locks) {
	    locks.notifyAll();
	}
    }

    private void spinLock(Lock lck) {
	long start=lck.getStart();
	long end=lck.getEnd();
	for (Enumeration enum=locks.elements();
	     enum.hasMoreElements();) {
	    Lock l=(Lock)enum.nextElement();
	    if (l!=lck && l.isLocked() && 
		(l.getType() & lck.getProtection()) != 0) {
		long s=l.getStart(), e=l.getEnd();
		if (s<=start && e>=end) {
		    while (l.isLocked()) {
			try {
			    synchronized(locks) {
				locks.wait();
			    }
			} catch (InterruptedException ie) {}
		    }
		    spinLock(lck);
		    return;
		}
	    }
	}
    }

    private void lockObtain(Lock l) {
	spinLock(l);
	locks.addElement(l);
    }

    /* ------------ Access file pooling ------------------------------- */

    public void retire(RandomAccessFile f) {
	if (fileStack.size() < FS_MAX) 
	    fileStack.push(f);
    }

    protected RandomAccessFile getReadStream(long start) throws IOException {
	RandomAccessFile rf=null;
	synchronized(fileStack) {
	    if (!fileStack.isEmpty()) 
		rf=(RandomAccessFile)fileStack.pop();
	    else
		rf=new RandomAccessFile(dsf, "r");
	}
	rf.seek(start);
	return rf;
    }

    /* ------------ File allocation ------------------------------------ */

    /**
     * Attempts to allocate and lock a number of bytes to store a file.
     * If successful, a WriteLock is issued locking the area in the store.
     * If unsucessful, null is returned, indicating that the datastore is full.
     */
    public boolean allocate(long bytes, WriteToken t) throws IOException {
	long[][] ranges=directory.allocate(bytes);
	if (ranges==null) {
	    logger.log(this, "Failed to allocate "+bytes+" bytes, datastore full.", Logger.NORMAL);
	    return false;
	}
	logger.log(this, "Allocated and write-locking pages "+
		   directory.rangeList(ranges),
		   Logger.MINOR);
	WriteLock[] locks=new WriteLock[ranges.length];
	for (int i=0; i<ranges.length; i++) {
	    locks[i]=writeLock(ranges[i]);
	}
	t.locks=locks;
	t.ranges=ranges;
	return true;
    }

    public InputStream read(Object key) throws IOException {
	long[][] ranges=getRanges(key);
	logger.log(this, "Read-locking pages "+directory.rangeList(ranges),
		   Logger.MINOR);
	Vector locks=new Vector(ranges.length);
	for (int i=0; i<ranges.length; i++) 
	    locks.addElement(readLock(ranges[i]).getStream());
	return new SequenceInputStream(locks.elements());
    }
	
    public WriteToken create(long bytes) throws IOException {
	WriteToken rv=new WriteToken();
	if (!allocate(bytes, rv)) 
	    return null;
	LockSequenceOutputStream out=new LockSequenceOutputStream();
	for (int i=0; i<rv.locks.length; i++) {
	    out.addStream(rv.locks[i]);
	}
	rv.stream=out;
	return rv;
    }

    /* ------------ Encryption functions ----------------------------- */

    InputStream decrypt(Lock l, InputStream in) throws IOException {
	if (directory.dsKey.length==0) return in;
	BlockCipher c=new Twofish();
	return new CipherInputStream(c, in, 
				     directory.setupCipher(c, 
							   l.getStart(),
							   l.getEnd()));
    }

    OutputStream encrypt(Lock l, OutputStream out) throws IOException {
	if (directory.dsKey.length==0) return out;
	BlockCipher c=new Twofish();
	return new CipherOutputStream(c, out, 
				      directory.setupCipher(c, 
							    l.getStart(),
							    l.getEnd()));
    }
								    

    /* ------------ Maintenance functions ---------------------------- */
    public static void create(File data, File directory, long size, 
			      boolean force, Random r) 
	throws IOException {
	if (!force) {
	    BufferedReader rd=new BufferedReader(new InputStreamReader(System.in));
	    System.err.print("This will destroy any existing datastore of the same name.   Are you sure you\nwish to proceed? ");
	    String resp=rd.readLine();
	    if (resp.length()<1 ||
		resp.trim().toUpperCase().charAt(0) != 'Y') 
		return;
	}
	
	if (r!=null) System.err.println("Using Twofish encryption...");
	
	System.err.println("Initializing directory...");
	Directory dir=new Directory(directory, size);
	dir.reset(r);
	dir.sync();

	System.err.println("Initializing datastore...");
	Freenet.crypt.Yarrow y=new Freenet.crypt.Yarrow();
	FileOutputStream out=new FileOutputStream(data);

	int x=0, max=1024, maxval=0;
	long total=0;
	for (int i=1024; i<=65536; i*=2) {
	    byte[] buffer=new byte[i];
	    long start=System.currentTimeMillis();

	    long tot=0;
	    while (tot < size/8) {
		total+=buffer.length;
		tot+=buffer.length;
		y.nextBytes(buffer);
		out.write(buffer, 0, buffer.length);
	    }
	    long timing=(size/8)/(System.currentTimeMillis()-start);
	    if (timing > maxval) {
		max = i;
		maxval = (int)timing;
	    }
	    //	    System.err.println(i+" "+timings[x]);
	    x++;
	}
	System.out.println("Optimal write-size is "+max+" bytes.");
	byte[] buffer=new byte[4096];
	while (total < size) {
	    y.nextBytes(buffer);
	    out.write(buffer, 0, buffer.length);
	    total+=buffer.length;
	}
	out.close();
	System.err.println("Done");
    }	

    protected void reset(Random r) {
	directory.reset(r);
	sync();
    }

    /* ------------ Debugging ----------------------------------------- */

    public void listLocks() {
	System.err.println("Locks: ");
	for (Enumeration enum=locks.elements(); 
	     enum.hasMoreElements();) {
	    Lock l=(Lock)enum.nextElement();
	    System.err.println(l);
	}
    }

    public void listFree() {
	directory.listFree();
    }

    public static void main(String[] args) throws Exception {
	if (args[0].equals("create")) {
	    create(new File(args[1]), new File(args[2]), Long.parseLong(args[3]), false, (args.length>=5 && args[4].equals("encrypted") ? new Freenet.crypt.Yarrow() : null));
	    System.exit(0);
	} else if (args[0].equals("bm")) {
	    long[] timings=new long[7];
	    int x=0;
	    FileInputStream in=new FileInputStream(args[1]);
	    for (int i=1024; i<=65536; i*=2) {
		byte[] buffer=new byte[i];
		long start=System.currentTimeMillis();
		long tot=0;
		while (tot < 1024768)
		    tot+=in.read(buffer, 0, buffer.length);
		timings[x]=tot/(System.currentTimeMillis()-start);
		System.err.println(i+" "+timings[x]);
		x++;
	    }
	    System.exit(0);
	}

	DatastoreFS fs=new DatastoreFS(new File(args[0]), new File(args[1]),
				       null);

	if (args[2].equals("reset")) {
	    fs.reset((args.length>=4 && args[3].equals("encrypted")) ? new Freenet.crypt.Yarrow() : null);
	} else if (args[2].equals("retrieve")) {
	    OutputStream fout=System.out;
	    if (args.length>=5) 
		fout=new FileOutputStream(args[4]);
	    
	    InputStream data=fs.read(args[3]);
	    
	    long len=Long.MAX_VALUE;
	    byte[] buffer=new byte[65536];
	    int rc=0;
	    do {
		rc=data.read(buffer, 0, (int)Math.min(len, buffer.length));
		if (rc>0) {
		    len-=rc;
		    fout.write(buffer, 0, rc);
		}
	    } while (rc!=-1 && len!=0);	    
	    fout.close();
	} else if (args[2].equals("free")) {
	    fs.free(args[3]);
	    fs.sync();
	} else if (args[2].equals("list")) {
	    fs.listDir();
	    fs.listFree();
	}
    }
}	    

