package Freenet.fs;

import java.io.*;
import java.util.*;
import Freenet.crypt.Yarrow;

public class StoreTool {
    protected DatastoreFS fs;

    public StoreTool(DatastoreFS fs) {
	this.fs=fs;
    }

    public void write(File data) throws IOException {
	write(data.getName(), data);
    }

    public void write(String key, File data) throws IOException {
	write(key, new BufferedInputStream(new FileInputStream(data)),
	      data.length());
    }

    public void write(String key, InputStream data, long bytes) 
	throws IOException {
	
	WriteToken t=fs.create(bytes);
	if (t == null) 
	    throw new IOException("Not enough space to allocate "+
				  bytes+ " bytes");
	
	DataDescriptor d=new DataDescriptor(key);
	
	fs.index(t, d);
	OutputStream f=t.getStream();

	copy(data, f, bytes);
	f.close();
	fs.commit(t, d);
    }

    protected void copy(InputStream from, OutputStream to, long bytes) 
	throws IOException {
	long len=bytes;
	byte[] buffer=new byte[65536];
	int rc=0;

	do {
	    rc=from.read(buffer, 0, (int)Math.min(len, buffer.length));
	    if (rc>0) {
		len-=rc;
		to.write(buffer, 0, rc);
	    }
	} while (rc!=-1 && len!=0);
	to.flush();
    }

    public void copy(String key, File out) throws IOException {
	copy(key, new BufferedOutputStream(new FileOutputStream(out)));
    }

    public void copy(String key, OutputStream out) throws IOException {
	long bytes=size(key);
	InputStream in=fs.read(key);
	copy(in, out, bytes);
    }

    public long free() {
	return fs.directory.free();
    }

    public long size(String key) {
	long[][] ranges=fs.getRanges(key);
	return size(ranges);
    }

    public String[] directory() {
	return fs.directory.keys();
    }

    protected long size(long[][] ranges) {
	long total=0;
	for (int i=0; i<ranges.length; i++) 
	    total+= (ranges[i][1] - ranges[i][0]) + 1;
	return total;
    }

    public static void main(String[] args) throws Exception {
	DatastoreFS fs=new DatastoreFS(new File(args[0]), new File(args[1]),
				       null);
	StoreTool t=new StoreTool(fs);
	if (args[2].equals("dir")) {
	    String[] keys=t.directory();
	    long total=0;
	    for (int i=0; i<keys.length; i++) {
		total+=t.size(keys[i]);
		System.out.print(t.size(keys[i]));
		System.out.print('\t');
		FSElement e=fs.stat(keys[i]);
		if (e instanceof DataDescriptor) 
		    System.out.print(((DataDescriptor)e).timestamp+"\t");
		System.out.println(keys[i]);
	    }
	    System.out.println(keys.length+" entries, "+total+" bytes.");
	} else if (args[2].equals("write")) {
	    for (int i=3; i<args.length; i++) {
		try {
		    t.write(args[i], new File(args[i]));
		} catch (IOException e) {
		    e.printStackTrace();
		    fs.sync();
		}
	    }
	} else if (args[2].equals("read")) {
	    OutputStream out=System.out;
	    if (args.length > 4)
		out=new BufferedOutputStream(new FileOutputStream(args[4]));
	    t.copy(args[3], out);
	} else if (args[2].equals("remove")) {
	    fs.free(args[3]);
	} else if (args[2].equals("list-free")) {
	    fs.listFree();
	} else if (args[2].equals("defrag")) {
	    Yarrow y=new Yarrow();
	    File storeFile=new File(args[0]+".defrag");

	    System.out.println("Creating blank datastore...");
	    BufferedOutputStream out=
		new BufferedOutputStream(new FileOutputStream(storeFile));
	    
	    byte[] buffer=new byte[65536];
	    long size=new File(args[0]).length();
	    while (size > 0) {
		out.write(buffer, 0, (int)Math.min(buffer.length, size));
		size-=buffer.length;
	    }
	    out.close();
	    
	    DatastoreFS defrag=new DatastoreFS(storeFile,
					       new File(args[1]+".defrag"),
					       null);
	    defrag.reset((defrag.directory.dsKey.length == 0 ? null :y));
	    StoreTool dt=new StoreTool(defrag);

	    size=storeFile.length();
	    String[] keys=t.directory();
	    for (int i=0; i<keys.length; i++) {
		System.out.println("Copying file "+(i+1)+
				   "/"+keys.length+"...");
		WriteToken wt=defrag.create(t.size(keys[i]));
		DataDescriptor d=new DataDescriptor(keys[i]);
		defrag.index(wt, d);
		OutputStream cp=wt.getStream();
		t.copy(keys[i], cp);
		size-=t.size(keys[i]);
		defrag.commit(wt, d);
		defrag.listFree();
	    }
	    
	    System.out.println("Done copying files.");

	    WriteToken wt=defrag.create(dt.free());
	    out=new BufferedOutputStream(wt.getStream());
	    System.out.println("Randomizing freespace...");
	    while (size > 0) {
		y.nextBytes(buffer);
		out.write(buffer, 0, (int)Math.min(buffer.length, size));
		size-=buffer.length;
	    }
	    defrag.sync();
	    System.out.println("Done.");
	}
	fs.sync();
    }
	    
}


