package Freenet.keys;

import Freenet.*;
import Freenet.support.Fields;
import Freenet.support.io.VerifyingInputStream;
import Freenet.support.io.DataNotValidIOException;
import Freenet.crypt.*;
import java.math.BigInteger;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.util.Hashtable;
import java.util.Random;

/**
 * SVK stands for Signature Verifiable Key.  This key is composed using a 
 * DSS public key, and is thus verifyable by any node in the network.
 * 
 * KHKs are keyType 02 01.  ?
 * @author Scott G. Miller
 **/
public class SVK extends Key {
    public static final int SVK_MAXSIZE=32768;

    private DSAKeyPair kp;

    public static void main(String[] agrs) {
	SVK s=makeSVK(new Yarrow(), null, null);
	System.out.println("Random SVK : " + s); 
    }

    public static int keyNumber = 0x0201;

    /**
     * Constructs an SVK by generating a random SVK private key.
     *
     * @param r A random number generator
     **/
    public static SVK makeSVK(Random r, String documentName,
			      FieldSet storables) {
	BigInteger x=Util.generateLargeRandom(128,256,r);
	DSAKeyPair keyPair=new DSAKeyPair(x, Global.DSAgroupB);
	return makeSVK(keyPair, documentName, storables);
    }

    public static SVK makeSVK(DSAKeyPair keyPair, String documentName,
			      FieldSet storables) {
	byte[] keyVal=makeSVK(keyPair, documentName, storables, keyNumber);
	SVK s=new SVK(keyVal, keyNumber, keyPair);
	return s;
    }

    /**
     * The constructor for an SSK
     */
    public static SVK makeSVK(byte[] subspace_root, String documentName) {
	SHA1 ctx= new SHA1();
	ctx.update(subspace_root, 0, subspace_root.length - 3);
	byte[] docNameBytes = Util.hashString(new SHA1(), documentName);
	ctx.update(docNameBytes, 0, docNameBytes.length);

	byte[] keytmp = ctx.digest();
	byte[] keyval = new byte[23];
	System.arraycopy(keytmp, 0, keyval, 0, 20);

	keyval[20] = lengthLimit(SVK_MAXSIZE);
	keyval[21] = (byte) (keyNumber>>8 & 0xFF);
	keyval[22] = (byte) (keyNumber & 0xFF);

	return new SVK(keyval);
    }

    protected static byte[] makeSVK(DSAKeyPair keyPair,
				    String dn, 
				    FieldSet storables, int keyNumber) {
	byte[] pubkeyBytes = Util.MPIbytes(keyPair.getY());
	byte[] docNameBytes = null;
	SHA1 ctx= new SHA1();
	ctx.update(pubkeyBytes, 0, pubkeyBytes.length);

	if (dn!=null) {
	    byte[] imm=ctx.digest();
	    ctx.update(imm, 0, imm.length);
	    docNameBytes = Util.hashString(new SHA1(), dn);
	    if (storables!=null)
		storables.add("Document-name", 
			      Freenet.support.Fields.bytesToHex(docNameBytes,
								0,
								docNameBytes.length));
	    ctx.update(docNameBytes, 0, docNameBytes.length);
	}

	byte[] keytmp = ctx.digest();
	byte[] keyval = new byte[23];
	System.arraycopy(keytmp, 0, keyval, 0, 20);

	pubkeyBytes=keyPair.getY().toByteArray();
	if (storables != null) 
	    storables.add("Public-key",
			  Freenet.support.Fields.bytesToHex(pubkeyBytes,
							    0,
							    pubkeyBytes.length));
	keyval[20] = lengthLimit(SVK_MAXSIZE);
	keyval[21] = (byte) (keyNumber>>8 & 0xFF);
	keyval[22] = (byte) (keyNumber & 0xFF);

	return keyval;
    }

    /**
     * Wraps an SVK around this byte array.
     * @param keyval The buffer
     * @throws KeyException if the contents of the array are not a SVK
     **/
    public SVK(byte[] keyval) throws KeyException {
	this(keyval, keyNumber, null);	
    }

    protected SVK(byte[] keyval, int keyNumber) {
	this(keyval, keyNumber, null);
    }

    protected SVK(byte[] keyval, int keyNumber, DSAKeyPair keyPair) {
	super(keyval);
	if (val.length != 23 || val[21] != (keyNumber>>8 & 0xFF) || val[22] != (keyNumber & 0xFF))
	    throw new KeyException("byte array did not contain an SVK or KSK");
	this.kp=keyPair;
    }

    /**
     * Return the generated keypair
     */
    public DSAKeyPair getKeyPair() {
	return kp;
    }

    /**
     * Adds a DSS signature to a given storables FieldSet, by signing the 
     * provided hash
     *
     * @param hash The hash of the document being signed
     * @param r A random number generator
     * @param storables The storables in which to store the signature
     */
    public void sign(byte[] hash, Random r, FieldSet storables) {
	BigInteger k=Util.generateLargeRandom(80,160, r);
	BigInteger m=Util.byteArrayToMPI(hash);
	DSASignature sig=DSA.sign(kp, k, m);
	storables.add("Signature", sig.toString());
    }
    
    public VerifyingInputStream verifyStream(InputStream data,
					     FieldSet storables,
					     long docLength)
	throws DataNotValidIOException {
	return this.verifyStream(data, storables, docLength, keyNumber);
    }

    /**
     * Verifies a DSS-SVK by performing a signature verification.
     **/
    protected VerifyingInputStream verifyStream(InputStream data,
					     FieldSet storables,
					     long docLength, 
					     int keyNumber) 
	throws DataNotValidIOException {
	try {
	    if (val[20] != 0 && docLength > (1 << (val[20] & 0x3F))) {  // check length
		System.err.println(docLength + " > " + (1 >> (val[20] & 0x3F)));
		throw new DataNotValidIOException(Presentation.CB_BAD_KEY);
	    }


	    DSAKeyPair keyPair=null;
	    DSASignature sig=null;

	    SHA1 sigctx=new SHA1();
	    
	    // Handle SVK specific fields:
	    String docName = null;
	    String grp = null;
	    if ((keyNumber & 0xff) == 0x01) {
		grp = storables.get("Group");
		docName = storables.get("Document-name");
	    }

	    try {
		keyPair=
		    new DSAKeyPair(grp == null ? 
				   ((keyNumber & 0xff) == 0x01 ? 
				    Global.DSAgroupB : 
				    Global.DSAgroupA) :
				   DSAGroup.parse(grp),
				   Util.byteArrayToMPI(Util.hexToBytes(storables.get("Public-key"))));
				   
		sig=new DSASignature(storables.get("Signature"));
		
	    } catch (NullPointerException np) {
		np.printStackTrace();
		throw new DataNotValidIOException(Presentation.CB_BAD_KEY);
	    }

	
	    //Verify that the SVK itself is valid
	    byte[] shouldBe = new byte[23];
	    shouldBe[20]=val[20];
	    shouldBe[21]=(byte)(keyNumber >> 8);
	    shouldBe[22]=(byte) keyNumber;
	
	    SHA1 ctx=new SHA1();
	    byte[] pubkeyBytes=Util.MPIbytes(keyPair.getY());
	    ctx.update(pubkeyBytes, 0, pubkeyBytes.length);
	    if (docName != null) {
		byte[] imm=ctx.digest();
		ctx.update(imm, 0, imm.length);
		byte[] docNameBytes=Util.hexToBytes(docName);
		ctx.update(docNameBytes, 0, docNameBytes.length);
		sigctx.update(docNameBytes, 0, docNameBytes.length);
	    }
	
	    byte[] keyBytes=ctx.digest();
	    System.arraycopy(keyBytes, 0, shouldBe, 0, keyBytes.length);
	
	    if (!Util.byteArrayEqual(shouldBe, val)) 
		throw new DataNotValidIOException(Presentation.CB_BAD_KEY);
	    
	    //Create the verification stream

	    return new DSSVerifyingInputStream(data, keyPair, sig, 
					       docLength, sigctx);
	} catch (DataNotValidIOException e) {
	    throw e;
	} catch (Exception e) {
	    e.printStackTrace(Core.logger.getOut());
	    throw new DataNotValidIOException(Presentation.CB_OK);
	}
    }

    /**
     * Returns whether to cache the data for this key on this node.
     *
     * The implementation of isCachable in SVK always returns true.
     **/
    public boolean isCachable() {
	return true;
    }


    /**
     * Prints the the key in the form expected by Key.readKey(),
     * KHK/&lt;hex value&gt;
     **/
    public String toString() {
	return Fields.bytesToHex(val,0,23);
    }
}








