/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.

  Explanation of Code Versions: 
    0.0.0      = Initial Description
    0.0.1      = API Specified
    0.x (x>0)  = Partial Implementation
    x.0 (x>0)  = Operational
		
  Requires Classes: Node (1.0)
                    Address (1.0)
		    Message (1.0)
 */

/**
 * This is the DataRequest message
 *
 * @see Node
 * @see Address
 * @author Brandon Wiley (blanu@uts.cc.utexas.edu)
 * @author Ian Clarke (I.Clarke@strs.co.uk)
 **/

package Freenet.message;
import Freenet.*;
import Freenet.node.*;
import Freenet.support.*;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.IOException;
import java.net.*;

public class DataRequest extends Request
{

    public static final String messageName = "DataRequest";

  public DataRequest(long idnum, long htl, long depth, Key key)
  {
      super(idnum, htl, depth, key);
  }

  public DataRequest(RawMessage raw) throws InvalidMessageException
  {
      super(raw);
  }
  	
  public RawMessage toRawMessage(Presentation t)
  {
    RawMessage raw=super.toRawMessage(t);
    raw.messageType="DataRequest";
    return raw;
  }

    protected void refFound(NodeReference ref, Node n, KeyedMM kmm) 
	throws RequestAbortException {
	// requests don't do anything about found refs.
    } 

    protected void dataFound(Entity doc, Node n, KeyedMM kmm) 
	throws RequestAbortException {
        Core.logger.log(this,"Found " + searchKey + " replying",Logger.MINOR);
	
	DataReply drm=null;
	long repDepth = (long)(Math.abs(n.randSource.nextInt() % 200));
	try {
	    drm=new DataReply(id,
			      depth, // new htl
			      repDepth, // new depth
			      doc);
        } catch(Exception e) {
	    Core.logger.log(this,"Error loading "+searchKey+" from DataStore: "+e,Logger.ERROR);
	    return;
	}

	try {
	    DataSent ds = new DataSent(repDepth, kmm.rr, drm.in);
	    ConnectionHandler ch;
	    if (receivedWith == null) {
		ch = n.makeConnection(source);
		drm.keepAlive = false; // close again after sending message
	    } else if (!receivedWith.isOpen())
		ch = n.makeConnection(source);
	    else 
		ch = receivedWith;
	    drm.sending(n, ch);
	    ch.sendMessage(drm,ds);

	} catch(ConnectFailedException cfe) {
	    Core.logger.log(this,"Error connecting to send DataReply to " + cfe.peer,Logger.MINOR) ;
	} catch(SendFailedException sfe) {
	    Core.logger.log(this,"Error sending DataReply to " + sfe.peer,Logger.MINOR) ;
	}
	throw new RequestAbortException(kmm); // My job is done. Abort and forget about this.
    }

    // Private class
    
    private class DataSent implements NodeMessageObject {
	
	private MessageObject mo; // callback for restarting message
	private InputStream in;
	private long repDepth;
	private Exception e;
	
	public DataSent(long repDepth, MessageObject mo, InputStream in) {
	    this.mo = mo;
	    this.repDepth = repDepth;
	    this.in = in;
	}

	
	public long id() {
	    return DataRequest.this.id;
	}
  
	public MessageMemory received(Node n, MessageMemory mm) {
	    if (mm == null || !(mm instanceof KeyedMM)) {
		Core.logger.log(this,"DataSent object called incorrectly",Logger.ERROR);
		return mm;
	    }

	    KeyedMM kmm = (KeyedMM) mm;

	    try {
		in.close();
	    } catch (IOException e) {
	    }
	    n.sh.removeReader(searchKey,id);
	    
	    if (e == null) {
		Core.logger.log(this,"data sent correctly",Logger.DEBUGGING);
		// sendback the storedata message.
		try {
		    StoreData sd = new StoreData(id, depth, repDepth,null);
		    sd.resetDataSource(true);
		    ConnectionHandler ch;
		    if (kmm.replyCon == null || !kmm.replyCon.isOpen()) {
			ch = n.makeConnection(kmm.origRec);
		    } else {
			ch = kmm.replyCon;
		    }
		    sd.sending(n, ch);
		    ch.unlockedSend(sd,null);
		} catch (ConnectFailedException cfe) {
		    Core.logger.log(this,"Failed to send back StoreData message to " + cfe.peer, Logger.MINOR);
		} catch (SendFailedException sfe) {
		    Core.logger.log(this,"Failed to send back StoreData message to " + sfe.peer, Logger.MINOR);
		}
		kmm.state = kmm.DONE;
	    } else {
		if (e instanceof ConduitException) {
		    ConduitException ce = (ConduitException) e;
		    if (ce.inRead()) { // data was bad or disk corrupt
			try {
			    ce.getOutStream().write(Presentation.CB_RESTARTED);
			} catch (IOException ioe) {
			}
			n.sh.removeReader(searchKey,id);
			if (mo == null)
			    n.timer.add(0,mo);
		    } else {
			Core.logger.log(this,"Send of data for reply failed when connection to next node died!",Logger.MINOR);
		    }
		} else {
		    Core.logger.log(this,"Send failed on unknown exception " + e,Logger.NORMAL);
		}
	    }
	    return kmm;
	}

	public void setException(Exception e) {
	    this.e = e;
	}
    }
}
