package Freenet.message;
import java.io.*;
//import java.net.*;
import java.util.*;
import Freenet.*;
import Freenet.node.*;
import Freenet.support.*;
import Freenet.support.io.*;
/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.

  Explanation of Code Versions: 
    0.0.0      = Initial Description
    0.0.1      = API Specified
    0.x (x>0)  = Partial Implementation
    x.0 (x>0)  = Operational
		
  Requires Classes: Node (1.0)
                    Address (1.0)
		    Message (1.0)
 */

/**
 * This is the DataSend message
 *
 * @see Node
 * @see Address
 * @author Brandon Wiley (blanu@uts.cc.utexas.edu)
 * @author Ian Clarke (I.Clarke@strs.co.uk)
 * @author oskar (fingered everything)
 **/

public abstract class DataSend extends Message
{

    public InputStream in =null;
    public long length = 0;

    private Entity cache;
    private SplitOutputStream datatunnel;
    
    public DataSend(long idnum, long htl, long depth, FieldSet otherfields) {
	super(idnum, htl, depth, otherfields);
    }
    
    public DataSend(long idnum, long htl, long depth, Entity doc) throws BadDataException, IOException {
	super(idnum, htl, depth, new FieldSet());

	// restore Storable.* fields from data properties
	if (doc.props() != null) {
	    try {
		FieldSet storables = doc.props().restore();
		otherFields.add("Storable",storables);
	    } catch (DataProperties.WrongOrderException e) {
		// nothing to restore if it was not saved
	    }
	}

	//open data file for reading 
	length = doc.data().getLength();
	in = doc.data().getInputStream();
    }

    public DataSend(RawMessage raw)
	throws InvalidMessageException, BadAddressException {

	super(raw);

	if (raw.trailingFieldLength != 0)
	    length = raw.trailingFieldLength;
	else
	    throw new InvalidMessageException("Data sending message requires the trailing field length to be specified");
	if (raw.trailingFieldName != null)
	in = raw.trailingFieldStream;
	
	if (in==null) throw new InvalidMessageException("Data sending message requires a trailing field called Data");
    }

    public RawMessage toRawMessage(Presentation t) {
	RawMessage raw=super.toRawMessage(t);
	
	if (in != null) {
	    raw.trailingFieldStream = in;
	    raw.trailingFieldLength = length;
	raw.trailingFieldName="Data";
	}

	return raw;
    }


  /**
   * Called by a node after it receives this message
   * @param n The node that called this message.  This should be used
   *          to make any nescessary modifications to the DataStore etc.
   * @param sb Null unless this node has been seen before.  If non-null it
   *           is the Object returned by the received method of the
   *           last message seen with this ID.
   * @return The object to be passed to any messages received in the near
   *         future with the same ID as this one.  Null if no object should
   *         be passed.
   **/
  public MessageMemory pReceived(Node n, MessageMemory sb) {
      //      Core.logger.log(this,"Got a DataSend!",Logger.MINOR);
      if(sb==null || !(sb instanceof KeyedMM)) {// No one requested this, or I forgot about it
	  if (receivedWith != null)
	      receivedWith.close(); // we have bad data blocking this connection
	  in.notify();
	  return null;
      } else {	
	  KeyedMM kmm = (KeyedMM)sb;

	  // Cancel any timer waiting for this message
	  if (kmm.rr != null)
	      kmm.rr.cancel();
	  
	  if (kmm.state > kmm.AWAITING_DATA) { 
	      // note ">" - kmm.PENDING is fine
	      Core.logger.log(this,"Received a second DataSend for the same chain",Logger.MINOR);

	      return kmm;
	  }
	    
	  // cache the data pending a StoreData method
	  try {
	      cacheData(n, kmm);
	  } catch (DataNotValidIOException e) {
	      return kmm;
	  } catch (IOException e) {
	      Core.logger.log(this,"Error while caching data: " + e,Logger.ERROR);
	      return kmm;
	  }

	if (kmm.dataref != null || kmm.replyCon != null) {
	    // the old inputstream is piped into the file, so we pipe out a new one
	    try {
		in = cache.data().getInputStream();
	    } catch (IOException e) {
		// restart the request
		if (kmm.rr != null)
		    n.timer.add(0,kmm.rr);
		Core.logger.log(this,"Could not restore data from cache, restarting request",Logger.NORMAL);
		return kmm;
	    }
	    // still got places to go
	    try {
		if (kmm.dataCon == null || !(kmm.dataCon.isOpen())) {
		    kmm.dataCon = n.makeConnection(kmm.dataref);
		}
		sending(n, kmm.dataCon);
		SendDone sd = getSendDone(in);
		kmm.dataCon.sendMessage(this, sd);
	    } catch(ConnectFailedException cfe) {
		Core.logger.log(this,"Couldn't restore connection to proxy data to " + cfe.peer,Logger.NORMAL);
	    } catch(SendFailedException sfe) {
		Core.logger.log(this,"Couldn't proxy data to " + sfe.peer,Logger.NORMAL);
	    }
	} else
	    Core.logger.log(this,"Message ends here",Logger.DEBUGGING);

	return kmm;
    }
  }

    public void sending(Core n, Address peer, Address myAddress) throws SendFailedException
    {
	if (myAddress.equals(peer))
	    throw new SendFailedException(peer); // don't send to yourself
	source = myAddress;
    }

      // Protected/Private Methods

    protected MessageMemory timeOut(Node n, MessageMemory sb)
    {

	if (sb != null && sb instanceof KeyedMM && 
	    n.ds.searchData(((KeyedMM) sb).searchKey) == null) {
	    KeyedMM kmm = (KeyedMM) sb;

	    if (kmm.rr != null)
		kmm.rr.cancel();

	    // save the data
	    try {
		cacheData(n, kmm);
	    } catch (IOException e) {
		Core.logger.log(this,"Error while caching data: " + e,Logger.ERROR);
		return null;
	    }

	} else {
	    Core.logger.log(this,"Unknown DataSend, not saving",Logger.DEBUGGING);
	    if (receivedWith != null)
		receivedWith.close(); // we have bad data blocking this connection
	    in.notify();
	}

	return sb;
    }

    /**
     * Creates the local cache for the data and starts the transfer to
     * it.
     * @param n    The current node.
     * @param kmm  The current MessageMemory
     * @exception DataNotValidIOException  if the the Node could resolve
     *            already here that the data is invalid. The rcb will have
     *            been started, so the caller can return-
     * @exception IOException if some other error occured while writing data to
     *            disk (this is bad).
     **/
    private void cacheData(Node n, KeyedMM kmm) 
	throws DataNotValidIOException, IOException {

	datatunnel = new SplitOutputStream();

	FieldSet fs = otherFields.getSet("Storable");

	long dataPart = (fs != null && fs.get("PartSize") != null ?
			 Fields.stringToLong(fs.get("PartSize")) :
			 0);
			 

	cache = n.ds.newEntity(datatunnel, length,dataPart);
	kmm.data = cache;

	// save Storable.* fields in data properties
	DataProperties props = cache.props();

	// wrap verifying stream from key around the data
	VerifyingInputStream vis;
	try {
	    vis = kmm.searchKey.verifyStream(new PadOnErrorInputStream(in),
					     fs,length);
	} catch (DataNotValidIOException e) {
	    Core.logger.log(this,"Caching failed on invalid data, attempting to restart request",Logger.NORMAL);
	    if (kmm.rr != null)
		n.timer.add(0, kmm.rr);
	    throw e;
	}

	if (fs != null) {
	    try {
		props.store(fs);
	    } catch (DataProperties.WrongOrderException e) {
		// this should not happen here
		Core.logger.log(this,"Tried to write to full DataProperties element?!?!?",Logger.MINOR);
	    }
	}

	// Start a thread that reads from the socket and writes to the
	// data file. The ReceivedCB will put the file into the datastore
	// once we have to whole thing.  
	kmm.state = kmm.RECEIVING_DATA;
	ReceivedData rd = new ReceivedData(in);
	(new Conduit(vis, datatunnel, n.mh)).asyncFeed(rd,length);
    }

    /**
     * This method is called to return a MessageObject for when the data has 
     * finished sending. Implementations of SendDone returned
     * must be able to handle both the normal callback (stream finished) 
     * the exception callback (stream didn't).
     * @param in   The stream reading from the cache (so it can be closed).
     */
    protected abstract SendDone getSendDone(InputStream in);
 
    // Internal classes

    private class ReceivedData implements NodeMessageObject {

	private InputStream fromCon;
	private Exception e;

	public ReceivedData(InputStream fromCon) {
	    this.fromCon = fromCon;
	}

	public long id() {
	    return DataSend.this.id;
	}

	public MessageMemory received(Node n, MessageMemory mm) {

	    Core.logger.log(this,"Data signaled as receive done",
			    Logger.DEBUGGING);

	    if (mm == null || !(mm instanceof KeyedMM)
		|| (((KeyedMM) mm).state < KeyedMM.RECEIVING_DATA)) {
		Core.logger.log(this,
				"Data receive object handled at wrong time!",
				Logger.ERROR);
		return mm;
	    }

	    KeyedMM kmm = (KeyedMM) mm;

	    if (e == null) {
		Core.logger.log(this,"Data received successfully!",Logger.DEBUGGING);
		kmm.state = kmm.RECEIVED_DATA;
		try {
		    cache.data().stop();
		} catch (IOException e) {
		    Core.logger.log(this,"Error closing cache object although "
				    + "stream signaled finished",Logger.ERROR);
		}

		synchronized (fromCon) {
		    fromCon.notifyAll();
		}

		// if storedata was already received and postponed, handle now
		return ((kmm.storeData != null) ?
			kmm.storeData.pReceived(n,kmm) :
			kmm);
	    } else {
		if (!(e instanceof ConduitException))
		    return kmm; // defensive

		ConduitException ce = (ConduitException) e;
	    
		if (ce.inRead()) {
		    try {
			if (ce.getIOException() instanceof DataNotValidIOException) {
			    // send an interrupt to stream
			    int cb = ((DataNotValidIOException) 
				      ce.getIOException()).getCode();
			    Core.logger.log(this,"Got DNV("+cb+")", 
					    Logger.DEBUGGING);
			    if (cb == (Presentation.CB_OK & 0xff)) {
				if (receivedWith != null)  // if it is null then it will close
				    receivedWith.close();
				ce.getOutStream().write(Presentation.CB_BAD_DATA);
				cache.data().stopStreams(Presentation.CB_BAD_DATA);
			    } else if (cb == (Presentation.CB_RESTARTED & 0xff)) {
				ce.getOutStream().write(cb);
				cache.data().stopStreams(cb);		    
			    }
			} else if (e instanceof IOException) {
			    Core.logger.log(this,
				       "Read from CIS threw IOException: " + 
				       e + 
				       ". This should never happen (padding).",
				       Logger.ERROR);
			    if (receivedWith != null)
				receivedWith.close();
			    ce.getOutStream().write(Presentation.CB_CONN_DIED);
			    cache.data().stopStreams(Presentation.CB_CONN_DIED);
			}
			cache.data().stop();
		    } catch (IOException ioe) {
			Core.logger.log(this,"Error writing CB to cache on error in stream",Logger.ERROR);
		    }
		} else {
		    Core.logger.log(this,"Error writing to cache!",Logger.ERROR);
		}
		synchronized (fromCon) {
		    fromCon.notifyAll();
		}
		kmm.state = kmm.BROKEN_DATA;
		// call senddone if it has been postponed
		return ((kmm.sendDone != null) ?
			kmm.sendDone.received(n,kmm) :
			kmm);
	    }
	}

	public void setException(Exception e) {
	    this.e = e;
	}
    }
}










