package Freenet.message;
import Freenet.*;
import Freenet.node.*;
import Freenet.support.*;

/**
 * RequestRestarted MessageObjects are used to restart a chain on some
 * error in execution, and generally send back a QueryRestarted after which 
 * they act as if a RequestFailed had been received.
 *
 * @author oskar
 **/

public class RequestRestarted implements Schedulable, NodeMessageObject {

    /**
     * @return The upper bound of a one sided 90% confidence interval
     *         for the time it should take to get a reply based on the
     *         the hopTimeExpected and hopTimeDeviation values from
     *         the config (and the assumption that the time is normal).
     *         In milliseconds.
     **/
    public static long getTime(long htl) {
	return (long) (Core.hopTimeExpected * htl + 1.28 * Math.sqrt(htl) * Core.hopTimeDeviation);
    }

    private long id;
    private long hopsToLive;
    private TickerToken tt;

    /**
     * Creates a new RequestRestarted MessageObject.
     * @param id          The id of the request message chain.
     * @param hopsToLive  The number of hops to live left on the request.
     */
    public RequestRestarted(long id, long hopsToLive) {
	this.id = id;
	this.hopsToLive = hopsToLive;
    }

    public long id() {
	return id;
    }

    public long hopsToLive() {
	return hopsToLive;
    }

    public MessageMemory received(Node n, MessageMemory mm) {
	if (mm == null || !(mm instanceof KeyedMM)) {
	    Core.logger.log(this,"No MM available to restart message",Logger.MINOR);
	    return mm;
	}

	KeyedMM kmm = (KeyedMM) mm;
	
	/* note that the if the send fails, the query won't be restarted, 
	   since if it can't send the notification, it probably won't be able 
	   to send the reply */
	try {
	    Core.logger.log(this,"Restarting request " + Long.toHexString(id),Logger.DEBUGGING);
	    // remove this ref since it didn't reply
	    n.ds.removeRef(kmm.lastAttempt); 
	    
	    QueryRestarted qrm = new QueryRestarted(id, mm.depth);
	    qrm.sendBack(n,mm);

	    /* note the null, unknown fields are not restored on timeone since
	       they are not saved. This means that Requests should actually 
	       not carry unknown fields... */
	    RequestFailed rf = new RequestFailed(id, hopsToLive,null);

	    kmm.state = kmm.PENDING;
	    kmm.lastAttempt = kmm.prevAttempt;

	    // handle as if a failed message were received
	    return rf.received(n,kmm); 

	} catch (SendFailedException sfe) {
	    Core.logger.log(this,"Couldn't restart on timeout because send failed to " + sfe.peer, Logger.NORMAL);
	    return mm;
	}
    }

    public void setException(Exception e) {
	// could be useful some day
    }

    public void getToken(TickerToken tt) {
	this.tt = tt;
    }

    /**
     * Cancels this RequestRestarted from the last Ticker it was added to.
     * @return True if it was cancled, false otherwise (unscheduled or already
     *         executed).
     */
    public boolean cancel() {
	return tt != null && tt.cancel();
    }

    /**
     * Sets the HopsToLive remaining on this request.
     * @param hopsToLive   The number of hopsToLive to give restarted requests
     */
    public void setHopsToLive(long hopsToLive) {
	this.hopsToLive = hopsToLive;
    }
}










