package Freenet.message;
import Freenet.*;
import Freenet.node.*;
import Freenet.support.*;
import java.io.*;

/**
 * This a MessageObject for  when data has finished tunneling though the node.
 * It should handle both the successful and failed states of tunnel. Each data 
 * carrying message needs to implement the abstract methods of this class
 * to react correctly when finished or interrupted while tunneling data.
 *
 * @author oskar
 **/

abstract class SendDone implements NodeMessageObject {

    /** The id of the message chain moving data **/
    protected long id;
    private InputStream in;

    private Exception e = null;

    /**
     * Create new SendDone
     * @param id   The Request message chains id.
     * @param in   The stream reading from the cache (so it can be closed).
     **/
    protected SendDone(long id, InputStream in) {
	this.id = id;
	this.in = in;
    }

    public long id() {
	return id;
    }
	
    /**
     * If the data was sent correctly, this will make sure the StoreData 
     * message was received (otherwise postponing itself until it has)
     **/
    public final MessageMemory received(Node n, MessageMemory mm) {
	Core.logger.log(this,"Send signaled ended.",Logger.DEBUGGING);
	try {
	    in.close();
	} catch (IOException ioe) {
	    Core.logger.log(this,"IOException when closing input: " + ioe,Logger.DEBUGGING);
	}

	if (mm == null || !(mm instanceof KeyedMM) 
	    || (((KeyedMM) mm).state < KeyedMM.RECEIVING_DATA)
	    || (((KeyedMM) mm).state >= KeyedMM.DONE)) {
	    Core.logger.log(this,"Called to handle SendDone incorrectly",Logger.ERROR);
	    return mm;
	} else {
	    KeyedMM kmm = (KeyedMM) mm;
	    n.sh.removeReader(kmm.searchKey,id);
	    if (e == null) {
		Core.logger.log(this,"Send Successful!",Logger.DEBUGGING);
		if (kmm.state < kmm.STORED_DATA) {
		    Core.logger.log(this,"SendDone handeling being postponed.",
				    Logger.DEBUGGING);
		    kmm.sendDone = this;
		    return kmm;
		}	    
		return sendFinished(n,kmm);
	    } else {
		Core.logger.log(this,"Exception occured during send!",
				Logger.DEBUGGING);
		if (!(e instanceof ConduitException)) {
		    Core.logger.log(this,
				    "Got exception " + e + 
				    " which I don't know what to do with.",
				    Logger.MINOR);
		    return kmm;
		}

		ConduitException ce = (ConduitException) e;

		if (kmm.state < kmm.BROKEN_DATA) {
		    Core.logger.log(this,
				    "SendDone handeling being postponed.",
				    Logger.DEBUGGING);
		    kmm.sendDone = this;
		    return kmm;
		} else if (kmm.state >= kmm.RECEIVED_DATA && ce.inRead())
		    Core.logger.log(this,
				    "Read from cache failed although the data"+
				    " was received fully, in state " + 
				    kmm.state,Logger.ERROR);
		
		MessageMemory rmm = sendFailed(n,kmm,ce);
		try {
		    ce.getInStream().close();
		} catch (IOException e) {
		}
		
		Core.logger.log(this, "Returning", Logger.DEBUGGING);
		return rmm;
	    }
	}
    }

    public void setException(Exception e) {
	this.e = e;
    }

    /**
     * Called when the data is finished sending and the StoreData method has
     * arrived.
     * 
     * The thread that calls this is synchronized on kmm.data
     **/
    protected abstract MessageMemory sendFinished(Node n, KeyedMM kmm);

    /**
     * Called when the data fails sending.
     *
     * The thread that calls this is synchronized on kmm.data
     * @param ce  The ConduitException describing the problem.
     **/ 
    protected abstract MessageMemory sendFailed(Node n, KeyedMM kmm, 
						ConduitException ce);
}






