package Freenet.message;
import Freenet.*;
import Freenet.node.*;
import Freenet.support.Logger;
/**
 * This message is sent after the DataSend has been completed and actually
 * commits the data to the store.
 *
 * @author oskar
 **/

public class StoreData extends Message {

    public static final String messageName = "StoreData";

    public NodeReference dataSource;

    private boolean resetDS;

    public StoreData(long idnum, long htl, long depth) {
        this(idnum, htl, depth, null);
    }

    public StoreData(long idnum, long htl, long depth,
                     NodeReference dataSource) {
        super(idnum, htl,depth, null);
        this.dataSource = dataSource;
        keepAlive = false;
    }

    public StoreData(RawMessage raw) throws InvalidMessageException, BadAddressException {
        super(raw);
        String sourcestring = otherFields.get("DataSource");
        otherFields.remove("DataSource");
        dataSource = sourcestring == null ? null : new NodeReference(sourcestring);
        keepAlive = false;
    }

    public RawMessage toRawMessage(Presentation t) {
        RawMessage raw=super.toRawMessage(t);
        if (dataSource != null)
            raw.fs.add("DataSource",dataSource.toString());
        raw.messageType = messageName;
        return raw;
    }

    public MessageMemory pReceived(Node n, MessageMemory mm) {
        if (mm == null || !(mm instanceof KeyedMM)) {
            Core.logger.log(this,"Got unexpected StoreData message!",Logger.NORMAL);
            return mm;
        } else {
            KeyedMM kmm = (KeyedMM) mm;

            /* check state, and wait for the data to finish
               (the Store data) should be sent after the data,
               but some race conditions could still cause it to handled
               before the receive is finalized */

            if (kmm.data == null || kmm.state < kmm.RECEIVING_DATA) {
                Core.logger.log(this,"Got StoreData message for data that I have not received yet",Logger.DEBUGGING);
                return kmm;
            } else if (kmm.state > kmm.STORED_DATA) {
                Core.logger.log(this,"Got StoreData message for data that I have already stored",Logger.MINOR);
                return kmm;
            } else if (kmm.state < kmm.RECEIVED_DATA) {
                Core.logger.log(this,"Postponing store while waiting for data to be received",Logger.DEBUGGING);
                kmm.storeData = this;
                return kmm;
            }

            if (kmm.data.data().isReadable() && (n.ds.searchData(kmm.searchKey) == null)){

                Core.logger.log(this,"Received data for " + Long.toHexString(id) + " adding to DataStore",Logger.DEBUGGING);

                NodeReference nodeRef;

                // What's the use of putting a null in, I ask you?
                if (dataSource != null) {
                    nodeRef = dataSource;
                } else if (kmm.lastAddr != null) { // yeah, what he said
                    nodeRef = new NodeReference(kmm.lastAddr);
                } else {
                    nodeRef = null;
                }

                n.ds.put(kmm.searchKey, nodeRef, kmm.data);

            } else {
                Core.logger.log(this,"Not caching data for " + Long.toHexString(id) + ", bad data stream or data I already have.",Logger.NORMAL);
                Core.logger.log(this,"cache.isReadable: " + kmm.data.data().isReadable() + ", n.ds.searchData(key): " + n.ds.searchData(kmm.searchKey) + ", kmm.state: " + kmm.state,Logger.DEBUGGING);

            }
            kmm.state = kmm.STORED_DATA;
            kmm.storeData = this;
            return ((kmm.sendDone != null) ?
                    kmm.sendDone.received(n,kmm) :
                    kmm);
        }
    }

    public MessageMemory timeOut(Node n, MessageMemory mm) {
        return pReceived(n,mm);
    }

    /**
     * This method is called to set whether the DataSource of this message
     * should be reset to that of the current node during the next time
     * sending() is called.
     * @param b  Whether the DataSource should be reset at this node.
     * @return   Whether dataSource was <b>previously</b> set to be reset or
     *           not.
     **/
    public boolean resetDataSource(boolean b) {
        boolean r = resetDS;
        resetDS = b;
        return r;
    }

    public void sending(Core n, Address peer, Address myAddress) throws SendFailedException {

        if (myAddress.equals(peer))
            throw new SendFailedException(peer); // don't send to yourself
        source = myAddress;

        /* if asked to, or moving from one network to another, reset DataSource
         *  to here (unless the Node is transient).
         */
        if (resetDS ||
            (receivedAt != null && receivedAt.equalsHost(myAddress))) {

            if (n.getTransience()) {
                Core.logger.log(this,"Not resetting DataSource, Node is transient.",Logger.DEBUGGING);
            }

            if (n instanceof Node &&((Node) n).myRef != null) {
                Core.logger.log(this,"Not resetting DataSource, I'm not a Node!",Logger.DEBUGGING);
            }

            //I'm not fond of using instanceof like this, but I will do for now
            this.dataSource = (n.getTransience() ? null :
                               (n instanceof Node &&((Node) n).myRef != null ?
                                ((Node) n).myRef :
                                new NodeReference(myAddress)));
        }

    }
}





























