package Freenet.node;
import Freenet.*;
import Freenet.support.Logger;
import Freenet.support.Fields;
import Freenet.support.io.*;
import java.io.*;
import java.util.*;
/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.
*/

/**
 * This class is used to store data.  The object creates a file on disk
 * in a (statically specified) directory to store the data in.  In general
 * the data is accessed using streams.
 *
 * @author <A HREF="mailto:I.Clarke@strs.co.uk">Ian Clarke</A>
 * @author oskar
 **/

public class FileData implements Data, Serializable {

    // Internal classes

    private class FDataInputStream extends FilterInputStream {
	private long read = 0;
	private boolean closed = false;
	//	private FileInputStream fis;

	public FDataInputStream() throws IOException {
	    super(new BufferedInputStream(new FileInputStream(FileData.this.name)));
	}

	/*
	 * Somebody note: I am lazy with reads here, and just doing a 50 ms
	 * sleep on nothing available (since this is FileInputStream, nothing
	 * available always means EOF in the JVMs I have used). 
	 * If performance seems bad, synchronizing, waiting, and notifying 
	 * might be faster.
	 */
	public int read() throws IOException {
	    if ((!closed) && (read < length)) {
		while (in.available() == 0) {
		    if (incomplete) {
			remove();
			throw new BadDataException();
		    }
		    try {
			Thread.sleep(50);
		    } catch (InterruptedException e) {
			throw new InterruptedIOException();
		    }
		}
		read++;
		if (read == length)
		    remove();
		return in.read();
	    } else {
		return -1;
	    }
	}

	public int read(byte[] b, int off, int len) throws IOException, BadDataException {
	    if (read >= length || closed)
		return -1;
	    else if (read + len <= length) {
		while (in.available() == 0) {
		    if (incomplete) {
			remove();
			throw new BadDataException();
		    }
		    try {
			Thread.sleep(50);
		    } catch (InterruptedException e) {
			throw new InterruptedIOException();
		    }
		}
		int n = in.read(b, off, len);
		read += n;
		if (read == length)
		    remove();
		return n;
	    } else {
		// cast is safe, difference smaller then len
		return read(b, off, (int) (length - read));
	    }
	}
	
	public void close() throws IOException {
	    if (closed)
		throw new IOException();
	    closed = true;
	    remove();
	    in.close();
	}

	public void remove() {
	    FileData.this.readers.removeElement(this);
	}
    }

    // Public Fields

    /** The path where files should be stored **/
    public static File path;

    // Private fields
    /* Indicates whether this class is ready for reading */
    private boolean incomplete = false;
    /* index of InputStream reading from the data */
    private transient Vector readers = new Vector();

    // Protected Fields
    protected File name;
    protected long length;
    protected long partLength;
    protected transient OutputStream toFile;
    

    // Constructors
    /**
     * This creates a data item from the specified file.
     * @param name The file in which the data is stored
     **/
    public FileData(File name)
    {
	this.name = name;
	this.length = name.length();
    }

    /**
     * This creates a new data item by reading data off the stream
     * @param datatunnel  The data stream
     * @param length      The size of the item
     * @param partLength  The number of bytes between interleaving control
     *                    bytes in the data (0 indicates no interleaving 
     *                    control bytes).
     **/
    public FileData(SplitOutputStream datatunnel, long length, long partLength) throws IOException
    {
	this.length = length;
	this.partLength = partLength;
	try {
	    FileOutputStream out = newFile();
	    toFile = out;
	    datatunnel.addOutput(out);
	} catch (Exception e) {
	    Core.logger.log(this,"Exception creating file for caching stream!",
			    Logger.ERROR);
	    throw new IOException();
	}
    }
    

    /*
     * Creates a new file for this data object. Will return null if
     * there already is a file
     */
    private FileOutputStream newFile() throws IOException {
	if (name != null)
	    return null; // only run if data object has no file
	String tname;
	// Get a list of files in this directory so that there is no duplication
	String[] filelist = path.list();
	// Choose a name for this data item
	boolean used;
	do {
	    tname = "t"+Math.abs(Core.randSource.nextInt() % 10000000);
	    used = false;
	    if(filelist!=null)
		for (int x=0; x<filelist.length; x++) {
		    if (tname.equals(filelist[x]))
			used = true;
		}
	} while (used);
	name = new File(path, tname);
	Core.logger.log(this,"Created new file " + name,Logger.DEBUGGING);
	return new FileOutputStream(name);
    }
    
    /**
     * @return Whether this object contains readable data
     */
    public boolean isReadable() {
	return !incomplete;
    }

    /**
     * Returns a long which specifies the length of the data stored in this
     * object
     **/
    public long getLength() throws BadDataException {
	
	if (incomplete)
	    throw new BadDataException();
	else
	    return length;
    }
    
    /**
     * Returns an InputStream which supplies the data stored in this
     * object. 
     **/
    public ControlInputStream getInputStream() throws BadDataException, IOException {
	Core.logger.log(this,"Returning new stream from data, length: 0x" 
			+ Fields.longToString(length) + " partLength: 0x"
			+ Fields.longToString(partLength), Logger.DEBUGGING);
	if (incomplete)
	    throw new BadDataException();
	else {
	    ControlInputStream i = 
		new ControlInputStream(new FDataInputStream(),
				       partLength,
				       length);
	    readers.addElement(i);
	    return i;
	}
    }

    /**
     * Stop the <b>writing</b> of data to this stream.
     */
    public void stop() throws IOException {
	try {
	    toFile.close();
	} finally {
	    if (name.length() < length)
		incomplete = true;
	}
    }

    public void stopStreams(int endChar) {
	for (Enumeration e = readers.elements() ; e.hasMoreElements() ;) {
	    ControlInputStream cis = (ControlInputStream) e.nextElement();
	    cis.endWithNextControl(endChar);
	}
    }

     public String toString()
    {
	return name.toString();
    }

    public String fileName() {
	return name.toString();
    }

    protected void finalize() throws Throwable
    {
	name.delete();
	super.finalize();
    }
}





