package Freenet.node;
import Freenet.*;
import Freenet.crypt.*;
import Freenet.support.*;
import Freenet.support.io.*;
import Freenet.presentation.FreenetProtocol;
import Freenet.keys.KHK;
import java.util.*;
import java.net.*;
import java.io.*;

/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.
*/

/**
 * This is a Wrapper object that contains the components of a Node in the
 * Adaptive Network.  It uses a Network object to communicate with other
 * Nodes in the Network.
 *
 * @author <A HREF="mailto:I.Clarke@strs.co.uk">Ian Clarke</A>
 * @author <a href="mailto:blanu@uts.cc.utexas.edu">Brandon Wiley</a>
 **/

public class Node extends Core
{
    public static Node n;
    public static boolean run = true;
    static final String BUILD = "LATEST_BUILD="; // Look in inform.php
    public static Params params;

    // node specific options
    public static File dataPath;
    public static int maxHopsToLive;
    
    // option defaults (if not specified on command line or in config file)
    static public final String defaultDataPath = ".freenet";
    static public final String defaultDataPropertiesPath = ".freenet";
    static public final String defaultDataStorePath = ".freenet";
    static public final long defaultDiskCache = 100000000;
    static public final int defaultDataStoreSize = 500;
    static public final int defaultMessageStoreSize = 500;
    static public final int defaultBandwidthLimit = 50000;
    static public final int defaultInputBandwidthLimit = 0; /* disabled */
    static public final int defaultOutputBandwidthLimit = 0; /* disabled */
    static public final boolean defaultInformRead = true;
    static public final boolean defaultInformWrite = true;
    static public final long defaultInformDelay = 86400;
    static public final String defaultInformUrl = "http://www.octayne.com/inform.php";
    static public final String defaultNodeFile = "nodes.config";
    static public final int defaultMaxHopsToLive = 100;
    static public final int defaultListenPort = 0;
    static public final int defaultLogging = Logger.NORMAL;
    static public final String defaultLogFile = "freenet.log"; // "no" for stdout;
    static public final int defaultVerbosity = 4;
    static public final long defaultCheckpointInterval = 300;

    public static void main(String[] args) {
	// set revision number for handshake

	StringTokenizer cvsId = new StringTokenizer ("$Id: Node.java,v 1.66.2.5 2001/06/19 04:09:01 misterbad Exp $", " ");
	cvsId.nextToken();
	cvsId.nextToken();
	serverRevision = cvsId.nextToken();
	
	try {
	    // startup logging to System.out
	    Logger log;
	    log = new StandardLogger(System.out,defaultVerbosity,
				     defaultLogging);
	    setLogger(log);
	    
	    // parse command line and read config file
	    //	    Logger.threshold = Logger.ERROR;
	    try {
		params = new Params(Params.defaultRCFiles, args);
	    } catch (FileNotFoundException e) {
		params = new Params(args);
	    }

	    // check for help
	    if (params.getParam("help") != null) {
		usage();
		return;
	    } else if (params.getParam("version") != null) {
		version();
		return;
	    }
	    int port = params.getint("listenPort",defaultListenPort);
	    initNode(params, "" + port);
	    startNode(params, port);

	} catch (Exception e) {
	    e.printStackTrace(System.err);
	}
    }

    public static void initNode(Params params, String port) {
	// set runtime logging
	
	String logthresh = params.getParam("logging");
	int thresh;
	if (logthresh==null)
	    thresh = defaultLogging;
	else
	    thresh = StandardLogger.priorityOf(logthresh);
	
	int verb = params.getint("verbosity",defaultVerbosity);
	
	String fname = params.getParam("logFile",defaultLogFile);
	Logger log;
	try {
	    if (!fname.equalsIgnoreCase("NO"))
		log = new StandardLogger(fname,verb,thresh);
	    else
		log = new StandardLogger(System.err,verb,thresh);
	} catch (IOException e) {
	    System.err.println("Writing to log failed!");
		log = new SuppressedLogger();
	}
	setLogger(log);
	
	// Init the core settings
	init(params);
	
	maxHopsToLive = params.getint("maxHopsToLive",
				      defaultMaxHopsToLive);    
	
	// make directories for data store
	String dataPath = params.getParam("dataPath", defaultDataPath) + 
	    File.separator + port;
	String dataPropertiesPath =  
	    params.getParam("dataPropertiesPath", 
			    defaultDataPropertiesPath) + 
	    File.separator + port;
	FileData.path = new File(dataPath);
	if (!FileData.path.exists())
	    FileData.path.mkdirs();
	FileDataProperties.path = new File(dataPropertiesPath);
	if (!FileDataProperties.path.exists())
	    FileDataProperties.path.mkdirs();

        /* bandwidth throttle variables */
        int inputBandwidthLimit,
            outputBandwidthLimit,
            bandwidthLimit;
        Bandwidth bandwidth;

        /* if either of these are set, ignore the bandwidthLimit parameter */
	inputBandwidthLimit = params.getint("inputBandwidthLimit",
                                            defaultInputBandwidthLimit);
	outputBandwidthLimit = params.getint("outputBandwidthLimit",
                                            defaultOutputBandwidthLimit);

        /* create separate bandwidth limiters for input and output */
        if (inputBandwidthLimit != 0 || outputBandwidthLimit != 0) {
            if (inputBandwidthLimit != 0) {
                bandwidth = new Bandwidth(inputBandwidthLimit);
                ThrottledInputStream.setThrottle(bandwidth);
            }
            if (outputBandwidthLimit != 0) {
                bandwidth = new Bandwidth(outputBandwidthLimit);
                ThrottledOutputStream.setThrottle(bandwidth);
            }
        }
        /* if a bandwidthLimit is set, throttle both input and output through
           a single bandwith object, so the sum of input + output bandwidth
           may not exceed the available bandwidth */
        else {
            bandwidthLimit = params.getint("bandwidthLimit", 
                                           defaultBandwidthLimit);
            if (bandwidthLimit != 0) {
                bandwidth = new Bandwidth(bandwidthLimit);
                ThrottledInputStream.setThrottle(bandwidth);
                ThrottledOutputStream.setThrottle(bandwidth);
            }
        }
    }

    public static void startNode(Params param, int port) {
	try {
	    // get node settings
	    String refS = params.getParam("nodeAddress");
	    NodeReference myRef = null;
	    if (refS != null) {
		myRef = new NodeReference("tcp",refS + ":" + port);
	    }

	    HandshakeHandler hh = params.getboolean("doHandshake", true) ?
		(HandshakeHandler) new StandardHandshakeHandler() :
		(HandshakeHandler) new SuppressedHandshakeHandler();
	    Presentation t = new FreenetProtocol();

	    // create datastore
	    String dataStorePath = params.getParam("dataStorePath", defaultDataStorePath);

	    StandardDataStore.filename = dataStorePath + File.separator + "store_" + port;
	    StandardDataStore.dsminz = params.getlong("dsminsz", 0);
	    StandardDataStore.checkpointInterval = 
		params.getlong("checkpointInterval", 
			       defaultCheckpointInterval);

	    StandardDataStore ds = 
		StandardDataStore.makeDataStore(params.getlong("diskCache", 
							       defaultDiskCache), 
						params.getint("dataStoreSize",
							      defaultDataStoreSize));

	    // create node

	    try {
		n = new Node(ds,
			     new ListeningAddress ("tcp/" + port), 
			     t,
			     hh,
			     myRef);
	    } catch (ListenException e) {
		System.out.println("Could not bind to port " + port
				   + " - maybe another node is running?");
		if (port < 1024) {
		    System.out.println("Or, you might not have privileges " +
				       "to bind to a port < 1024.");
		}
		System.exit(1);
		return;
	    }
	    // a Hashtable to store addresses in when they are read from various sources
	    Hashtable at = new Hashtable();

	    // do inform read/write
	    String informUrl = params.getParam("informUrl", defaultInformUrl);
	    
	    if (informUrl != null && (params.getboolean("informRead", defaultInformRead))
		    || ((params.getboolean("informWrite", defaultInformWrite)))) {
		InformThread it = new InformThread(at, informUrl, params, n);
	    }
	    
	    try {
		String nodeFile = params.getParam("nodeFile", defaultNodeFile);
		BufferedReader br=new BufferedReader(new FileReader(nodeFile));
		Key tname;
		SHA1 sha=new SHA1(true);
		while(br.ready()) {
		    String addrstr=br.readLine();
		    addrstr=addrstr.trim();
		    if(!addrstr.equals("")) {
			tname = KHK.makeKHK(addrstr);
			if(n.ds.searchRef(tname)==null)
			    at.put(addrstr, tname);
		    }
		}
	    } catch(IOException e) {}
        
	    Enumeration iterator = at.keys();
	    while(iterator.hasMoreElements()) {
		String addr=(String)iterator.nextElement();
		try {
		    n.ds.put((Key)at.get(addr), new NodeReference(addr), null);
		} catch (IllegalArgumentException e) {
		    Core.logger.log(null, "Bad node reference read from "
				    + "datastore initialization file.",
				    Logger.MINOR);
		}catch(Exception e) {
		    e.printStackTrace();
		}
	    }
	    if (run) {
		// create message handler
		int messStoreSize = params.getint("messageStoreSize", 
						  defaultMessageStoreSize);
		StandardMessageHandler mh =
		    new StandardMessageHandler(n, messStoreSize);
		
		// clean DataStore and start maintence thread
		DataStoreMaintence dsm = 
		    new DataStoreMaintence(Core.randSource.nextLong());
		dsm.cleardir(ds);
		dsm.schedule(n);

		NodeReference serviceRef = new NodeReference("tcp/127.0.0.1:" + port);

                ServiceLauncher sl = new ServiceLauncher(serviceRef, params.getset("services"));
                sl.launch();

		// start accepting connections
		n.acceptConnections(mh);
	    }
	} catch(Exception e) {
	    e.printStackTrace();
	}
	finally { 
	    randSource.close();
	}
    }

    /**
     * Print version information
     **/
    public static void version() {
	System.out.println("Freenet server version " +  freenetVersion + 
			   " using protocol " + protocolVersion + " (build " + 
			   buildNumber + " revision " +  serverRevision + ")");
    }


    /**
     * Print usage information
     **/
    public static void usage() {
	version();

	System.out.println("Usage: freenet_server [options]");
	System.out.println("");
	System.out.println("  -connectTimeout msec       Time to wait for connection to remote node");
	System.out.println("  -dataStoreSize n           Max number of items to keep in store");
	System.out.println("  -diskCache bytes           Max size of disk cache");
	System.out.println("  -handshakeLife msec        Time to treat old handshakes as good");
	System.out.println("  -handshakeTimeout msec     Time to wait for handshake from remote node");
	System.out.println("  -informRead yes|no         Read initial nodes from list on webserver");
	System.out.println("  -informWrite yes|no        Add this node to list on webserver");
	System.out.println("  -informUrl url             URL of node list");
	System.out.println("  -listenPort port           Local port to listen on");
	System.out.println("  -logFile file              Name of log file (`no' to log to console)");
	System.out.println("  -logging error|normal|minor|debugging");
	System.out.println("                             Logging level");
	System.out.println("  -messageStoreSize n        Max number of pending messages to track");
	System.out.println("  -timePerHop msec           Time to wait for replies (per hop)");
	System.out.println("  -tunnel yes|no             Start forwarding messages while incomplete");
	System.out.println("  -verbosity 1-5             Verbosity of log messages");
	System.out.println("  -help                      Display this help and exit");
	System.out.println("  -version                   Output version information and exit");
	System.out.println("");
	System.out.println("Send bug reports to freenet-dev@lists.sourceforge.net");
    }

    public DataStore ds;
    public StreamHandler sh;
    public NodeReference myRef = null;

    /**
     * Creates a new Node
     * @param ds     The store where cached data and references are kept. And
     *               which is used to resolve the next step in the route.
     * @param lstaddr  The listening address this node should preside on.
     *                 This is gives what transport will be used.
     * @param t        The Presentation used when reading and writing data.
     * @param hh       A HandshakeHandler module to request and remember
     *                 handshake iformation about other nodes.
     **/
    public Node(DataStore ds, ListeningAddress lstaddr,
		Presentation t, HandshakeHandler hh) throws ListenException {
	this(ds,lstaddr,t,hh,null);
    }

    /**
     * Creates a new Node
     * @param ds     The store where cached data and references are kept. And
     *               which is used to resolve the next step in the route.
     * @param lstaddr  The listening address this node should preside on.
     *                 This is gives what transport will be used.
     * @param t        The Presentation used when reading and writing data.
     * @param hh       A HandshakeHandler module to request and remember
     *                 handshake iformation about other nodes.
     * @param myRef    The optional NodeReference name to send to other 
     *                 nodes (if given it will be used rather then the Address
     *                 resolved from the socket).
     **/
    public Node(DataStore ds, ListeningAddress lstaddr, 
		Presentation t, HandshakeHandler hh,
		NodeReference myRef) throws ListenException {
        super(lstaddr, t, hh);
	sh = new StreamHandler(this);
	this.ds = ds;
	this.myRef = myRef;
    }
}
