/* SearchTreeNode.java
 */

package Freenet.node;

import Freenet.Key;

import java.lang.Math;
import java.util.Enumeration;
import java.util.Vector;

/**
 * A class implementing a simple binary search tree for keys.
 * (Note: You could pretty much change "Key" to Comparable in this class, but
 * since we retain jdk 1.1 compatibility, we can't use Comparable in Fred.)
 * 
 * So this class just uses Keys.
 *
 * Note also that no duplicates are allowed.
 *
 * @author Mr. Bad <mr.bad@pigdog.org>
 */

public class SearchTreeNode {

    /* Left child. */
    protected SearchTreeNode left = null; 
    /* Right child. */
    protected SearchTreeNode right = null;
    /* Key-value pair owned by this node. */
    protected Key key = null;
    protected Object value = null;

    /**
     * Default constructor.
     */

    public SearchTreeNode() {
    }

    /**
     * Basic constructor.
     * 
     * @param key Key to use for looking up this node again.
     * @param value Value to store.
     */

    public SearchTreeNode(Key key, Object value) {
	this(key, value, null, null);
    }

    /**
     * Full constructor.
     * 
     * @param key Key to use for looking up this node again.
     * @param value Value to store.
     * @param left Left child.
     * @param value Value to store.
     */

    public SearchTreeNode(Key key, Object value, 
			  SearchTreeNode left, 
			  SearchTreeNode right)
    {
	setKey(key);
	setValue(value);
	setLeft(left);
	setRight(right);
    }

    /**
     * Insert a new key-value pair into the tree. This will put the
     * pair into the tree, sorted by key.
     * 
     * @param key Key of new item.
     * @param value Value of new item.
     */

    public synchronized void insert(Key key, Object value) {
	if (this.key == null) {
	    setKey(key);
	    setValue(value);
	} else {
	    int res = key.compareTo(this.key);
	    if (res == 0) {
		setValue(value);
	    } else if (res < 0) {
		if (left == null) {
		    setLeft(new SearchTreeNode(key, value));
		} else {
		    left.insert(key, value);
		}
	    } else if (res > 0) {
		if (right == null) {
		    setRight(new SearchTreeNode(key, value));
		} else {
		    right.insert(key, value);
		}
	    }
	}
    }

    /**
     * Returns the minimum key value of this Node or its children.
     *
     * @return Minimum key value, or null if there are no nodes in the tree.
     */

    public synchronized Key min() {
	if (key == null) {
	    return null;
	} else {
	    if (left != null) {
		return left.min();
	    } else {
		return key;
	    }
	}
    }

    /**
     * Returns the maximum key value of this Node or its children.
     *
     * @return Maximum key value, or null if there are no nodes in the tree.
     */

    public synchronized Key max() {
	if (key == null) {
	    return null;
	} else {
	    if (right != null) {
		return right.max();
	    } else {
		return key;
	    }
	}
    }
	
    /**
     * Get the value mapped to the given key.
     *
     * @param key Key to look up.
     * @return Value mapped to key, or null if key isn't in the tree.
     */

    public synchronized Object get(Key key) {
	if (this.key == null) {
	    return null;
	} else {
	    int res = key.compareTo(this.key);
	    if (res == 0) {
		return value;
	    } else if (res < 0) {
		if (left == null) {
		    return null;
		} else {
		    return left.get(key);
		}
	    } else if (res > 0) {
		if (right == null) {
		    return null;
		} else {
		    return right.get(key);
		}
	    } else {
		return null;
	    }
	}
    }

    /**
     * Remove the value from the tree mapped to key.
     *
     * Note that, since the node at the top of the tree may be
     * deleted, this method returns a new top value. This method
     * should always be called as:
     * 
     *     node = node.remove(key);
     * 
     * ...or weird things will happen.
     * 
     * @param key Key to look up.
     * @return New top for the tree after removing value mapped to key.
     */

    public synchronized SearchTreeNode remove(Key key) {
	if (this.key == null) {
	    return this;
	} else {
	    int res = key.compareTo(this.key);
	    if (res == 0) {
		int lcount = (left == null) ? 0 : left.count();
		int rcount = (right == null) ? 0 : right.count();
		
		if (lcount > rcount) {
		    if (rcount > 0) {
			left.insert(right);
		    }
		    return left;
		} else if (lcount < rcount) {
		    if (lcount > 0) {
			right.insert(left);
		    }
		    return right;
		} else {
		    if (lcount == 0) {
			return null;
		    } else {
			if (Math.random() < 0.5) {
			    left.insert(right);
			    return left;
			} else {
			    right.insert(left);
			    return right;
			}
		    }
		}
	    } else if (res < 0) {
		if (left != null) {
		    setLeft(left.remove(key));
		}
		return this;
	    } else if (res > 0) {
		if (right != null) {
		    setRight(right.remove(key));
		}
		return this;
	    } else {
		return this;
	    }
	}
    }

    /**
     * Gets the count of nodes in the tree.
     *
     * @return How many nodes there are in the tree (including this one).
     */

    public synchronized int count() {
	if (key == null) {
	    return 0;
	} else {
	    int cnt = 1;
	    if (left != null) {
		cnt += left.count();
	    }
	    if (right != null) {
		cnt += right.count();	    
	    }
	    return cnt;
	}
    }

    /**
     * Balances the tree. A balanced tree is much better for
     * insertion, searching, and removal.
     *
     * Note that after balancing, the top node is probably not the top any more.
     * It's therefore important to always call this method as follows:
     *
     *     node = node.balance();
     *
     * @return New top node for the tree.
     */

    public synchronized SearchTreeNode balance() {
	int leftCount = (left == null) ? 0 : left.count();
	int rightCount = (right == null) ? 0 : right.count();
	
	if (leftCount - rightCount > 1) {
	    SearchTreeNode newTop = left;
	    setLeft(null);
	    newTop.insert(this);
	    return newTop.balance();
	} else if (rightCount - leftCount > 1) {
	    SearchTreeNode newTop = right;
	    setRight(null);
	    newTop.insert(this);
	    return newTop.balance();
	} else {
	    // we're already balanced
	    if (left != null) {
		setLeft(left.balance());
	    }
	    if (right != null) {
		setRight(right.balance());
	    }
	    return this;
	}
    }

    /**
     * Given a key, returns the next smaller (or equal) and next larger (or equal) keys in the tree.
     * 
     * @param key Key to compare against.
     * @return Next smaller and next larger keys in the tree. If the key is in the tree,
     *         returns [key, key]. If the key is smaller than anything in the tree, returns
     *         [null, min()]. If the key is larger than anything in the tree, returns
     *         [max(), null].
     */

    public synchronized Key[] frame(Key key) {
	Key[] c = null;
	if (this.key != null) {
	    int res = key.compareTo(this.key);
	    if (res == 0) {
		c = new Key[2];
		c[0] = c[1] = this.key;
	    } else if (res < 0) {
		if (left == null) {
		    c = new Key[2];
		    c[0] = null;
		    c[1] = this.key;
		} else {
		    Key leftMax = left.max();
		    int leftRes = key.compareTo(leftMax);
		    if (leftRes > 0) {
			c = new Key[2];
			c[0] = leftMax;
			c[1] = this.key;
		    } else {
			c = left.frame(key);
		    }
		}
	    } else {
		if (right == null) {
		    c = new Key[2];
		    c[0] = this.key;
		    c[1] = null;
		} else {
		    Key rightMin = right.min();
		    int rightRes = key.compareTo(rightMin);
		    if (rightRes < 0) {
			c = new Key[2];
			c[0] = this.key;
			c[1] = rightMin;
		    } else {
			c = right.frame(key);
		    }
		}
	    }
	}
	return c;
    }

    /**
     * Given a key, returns the next smaller (strictly) key in the tree.
     * 
     * @param key Key to compare against.
     * @return Next smaller key in the tree, or null if the key is
     *         smaller than everything in the tree.
     */

    public synchronized Key nextLeft(Key key) {
	if (this.key == null) {
	    return null;
	} else {
	    int res = key.compareTo(this.key);
	    if (res <= 0) {
		if (left == null) {
		    return null;
		} else {
		    Key leftMax = left.max();
		    int leftRes = key.compareTo(leftMax);
		    if (leftRes > 0) {
			return leftMax;
		    } else {
			return left.nextLeft(key);
		    }
		}
	    } else {
		if (right == null) {
		    return null;
		} else {
		    Key rightMin = right.min();
		    int rightRes = key.compareTo(rightMin);
		    if (rightRes <= 0) {
			return this.key;
		    } else {
			return right.nextLeft(key);
		    }
		}
	    }
	}
    }

    /**
     * Given a key, returns the next larger (strictly) key in the tree.
     * 
     * @param key Key to compare against.
     * @return Next larger key in the tree, or null if the key is
     *         larger than everything in the tree.
     */

    public synchronized Key nextRight(Key key) {
	if (this.key == null) {
	    return null;
	} else {
	    int res = key.compareTo(this.key);
	    if (res < 0) {
		if (left == null) {
		    return null;
		} else {
		    Key leftMax = left.max();
		    int leftRes = key.compareTo(leftMax);
		    if (leftRes >= 0) {
			return this.key;
		    } else {
			return left.nextRight(key);
		    }
		}
	    } else {
		if (right == null) {
		    return null;
		} else {
		    Key rightMin = right.min();
		    int rightRes = key.compareTo(rightMin);
		    if (rightRes < 0) {
			return rightMin;
		    } else {
			return right.nextRight(key);
		    }
		}
	    }
	}
    }

    /**
     * Debugging output.
     * 
     * @param depth Depth to indent this key and children.
     */

    protected synchronized void dump(int depth) {
	if (left != null) {
	    left.dump(depth + 1);
	}
	for (int i = 0; i < depth; i++) {
	    System.out.print(" ");
	}
	System.out.println("Key = " + key + ", Value = " + value);
	if (right != null) {
	    right.dump(depth + 1);
	}
    }

    /**
     * Debugging output.
     * 
     * @param depth Depth to indent this key and children.
     */

    public synchronized void dump() {
	dump(0);
    }

    /**
     * Returns an enumeration of the keys in this tree.
     * 
     * @return Enumeration of keys.
     */

    // XXX: Make a real enumeration instead of being a putz

    public synchronized Enumeration keys() {
	Vector v = new Vector();
	addKeys(v);
	return v.elements();
    }

    /**
     * Returns an enumeration of the values in this tree.
     * 
     * @return Enumeration of values.
     */

    // XXX: Make a real enumeration instead of being a putz

    public synchronized Enumeration elements() {
	Vector v = new Vector();
	addValues(v);
	return v.elements();
    }

    /**
     * Add a sub-node to the tree. This is mainly used for removing or
     * balancing.
     *
     * @param other node to add.
     */

    protected void insert(SearchTreeNode other) {
	// key should not be null.
	int res = other.key.compareTo(key);
	if (res < 0) {
	    if (left == null) {
		setLeft(other);
	    } else {
		left.insert(other);
	    }
	} else if (res > 0) {
	    if (right == null) {
		setRight(other);
	    } else {
		right.insert(other);
	    }
	} else if (res == 0) {
	    // this shouldn't happen
	}
    }

    /**
     * Add all values in this tree to the given vector.
     * Mainly useful for making the bogus enumeration for elements().
     *
     * @param v Vector to add stuff to.
     */

    protected void addValues(Vector v) {
	if (key != null) {
	    if (left != null) {
		left.addValues(v);
	    }
	    v.addElement(this.value);
	    if (right != null) {
		right.addValues(v);
	    }
	}
    }

    /**
     * Add all keys in this tree to the given vector.
     * Mainly useful for making the bogus enumeration for keys().
     *
     * @param v Vector to add stuff to.
     */

    protected void addKeys(Vector v) {
	if (key != null) {
	    if (left != null) {
		left.addKeys(v);
	    }
	    v.addElement(this.key);
	    if (right != null) {
		right.addKeys(v);
	    }
	}
    }

    /**
     * Accessor for key.
     * 
     * @param key New key.
     */

    protected void setKey(Key key) {
	this.key = key;
    }

    /**
     * Accessor for value.
     * 
     * @param value New value.
     */

    protected void setValue(Object value) {
	this.value = value;
    }

    /**
     * Accessor for left child.
     * 
     * @param node New left child.
     */

    protected void setLeft(SearchTreeNode node) {
	this.left = node;
    }

    /**
     * Accessor for right child.
     * 
     * @param node New right child.
     */

    protected void setRight(SearchTreeNode node) {
	this.right = node;
    }
}
