/*
  This code is part of the Java Adaptive Network Client by Ian Clarke.
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.
 */

/**
 * DataStore.java
 *
 * This class is used by a node to store references and data.
 *
 * @version 0.0
 * @author Ian Clarke (I.Clarke@strs.co.uk)
 **/

package Freenet.node;
import Freenet.*;
import Freenet.support.*;
import Freenet.support.io.*;
import java.io.*;
import java.util.*;

public class StandardDataStore implements DataStore {

    static private long write_interval;
    static public String filename = ".freenet" + File.separator + "store";
    public static long dsminz = 0;
    public static long checkpointInterval = 300;

    public static StandardDataStore makeDataStore(long maxData, int maxItems) {
	StandardDataStore result = 
	    new StandardDataStore(maxData, maxItems);
    
	if (!readV5DataStore(result)) {
	    Core.logger.log(null, 
			    "Couldn't load DataStore from "+filename, 
			    Logger.NORMAL);
	}

	return result;
    }
  
   public static boolean readV5DataStore(StandardDataStore ds) {
	try {
	    FileInputStream fstream = new FileInputStream(filename);
	    ReadInputStream in = 
		new ReadInputStream(new BufferedInputStream(fstream));
	    String s = "";
	    try {
		s = in.readTo('\n','\r');
	    } catch (IOException e) {
		e.printStackTrace();
	    }
	    if (!s.equals("VERSION 5")) {
		Core.logger.log(null, "Not a verson 4 datastore",
				Logger.MINOR);
		return false;
	    }
	    FieldSet fs;
	    try {
		boolean cont=true;
		while(cont) {
		    try {
			fs = new FieldSet();
			if (fs.parseFields(in,'\n','=','.') != null) {
			    DataStoreItem dsi = new DataStoreItem(fs,ds);
			    if (dsi.isValid()) {
				ds.put(dsi);
			    } else {
				Core.logger.log(null,"Bad entry while reading version "
					+ "4 datastore file, skipping.",
					Logger.MINOR);
			    }
			} else 
			    cont=false;
		    } catch (BadEntryException e) {
			Core.logger.log(null,"Bad entry while reading version "
					+ "4 datastore file, skipping.",
					Logger.MINOR);
		    }
		}
	    } catch (EOFException e) {
	    } catch (IOException e) {
		Core.logger.log(null,"Problem reading V4 datastore: " + e, 
				Logger.NORMAL);
		return false;
	    }
	} catch (FileNotFoundException e) {
	    return false;
	}
	return true;
    }

    /**
     * Writes the DataStore to disk if has changed since last time this was
     * called
     * @return true if the DataStore was written, otherwise false
     **/
    public boolean tofile() throws IOException {
	if (!needsWriting) {
	    Core.logger.log(this,"No reason to write out datastore.",Logger.DEBUG);
	    return false;
	}
	// need to do this periodically
	synchronized(this) {
	    t = t.balance();
	}

	(new DoWrite()).doWrite(this,filename);
	return true;
    }
  
    protected static class DoWrite {
	public void doWrite(StandardDataStore store, String filename) throws IOException {
	    Core.logger.log(this,"Writing datastore to disk",Logger.MINOR);
	    
	    OutputStream fstream = 
		new BufferedOutputStream(new FileOutputStream(filename+".new"));
	    WriteOutputStream p = new WriteOutputStream(fstream);
	    
	    p.writeUTF("VERSION 5",'\n');
	    
	    synchronized(store) {
		CyclicArray ar = store.ar;
		for (int i = ar.length() - 1; i >= 0; i--) {
		    DataStoreItem item = (DataStoreItem) ar.get(i);
		    if (item == null)
			continue;
		    if (!item.isValid()) {
			Core.logger.log(this, "Invalid DataStoreItem (key = " + item.key + ") during save, skipped", Logger.MINOR);
			continue;
		    }

		    FieldSet fs = item.fields();
		    fs.writeFields(p,"StandardDataStore",'\n','=','.');
		}		
		store.needsWriting = false;
	    }
	    p.close();
	    fstream.close();
	    
	    if (!new File(filename).delete()) {
	    	Core.logger.log(this,"Could not delete old datastore",Logger.MINOR);
	    }
	    
	    if (!new File(filename+".new").renameTo( new File(filename) )) {
	    	Core.logger.log(this,"Could not rename new datastore",Logger.NORMAL);
	    } else {
	    	Core.logger.log(this,"Datastore replaced successfully.",Logger.MINOR);
	    }
	}
    }

    // Public Fields
    public long maxData;
    
    // Protected Fields
    protected CyclicArray ar;
    protected SearchTreeNode t;
    protected boolean needsWriting = false;		// empty data store need not be written

    // Constructor

    protected StandardDataStore(long maxData, int maxItems) {
	ar = new CyclicArray(maxItems);
	t = new SearchTreeNode();
	this.maxData = maxData;
    }

    // Public Methods

    public Enumeration keys() {
	return t.keys();
    }

    public Enumeration data() {
	return new EntityEnumeration(this);
    }

  /**
   * Searches for a key in the DataStore and if found returns
   * any data associated with that key.  If none is found,
   * null is returned.
   * @param k The key to search for
   * @return The data associated with the key or null if not found
   **/

  public Entity searchData(Key k) {
    Core.logger.log(this, "searchData("+k + ")", Logger.DEBUG);
    synchronized (this)
    {
      DataStoreItem dsi = (DataStoreItem)t.get(k);
      if(dsi==null || dsi.doc==null)
      {
        Core.logger.log(this, "Not found.", Logger.DEBUG);
        return null;
      }
      // We have a hit, raise its priority
      Core.logger.log(this, "Data found.", Logger.DEBUG);
      Entity doc = dsi.doc;
      NodeReference addr = dsi.ref;
      remove(k);
      put(k, addr, doc);
      return doc;
    }
  }

  /**
    * Searches for a key in the DataStore and if found returns
    * any reference associated with that key.  If none is found,
    * null is returned.
    * @param k The key to search for
    * @return The reference associated with the key or null
    *         if not found
    **/

  public NodeReference searchRef(Key k) {
      if (k == null)
	  return null;
    Core.logger.log(this, "called searchRef("+k + ")", Logger.DEBUG);
    synchronized (this)
    {
      DataStoreItem dsi = (DataStoreItem)t.get(k);
      if(dsi==null)
      {
        Core.logger.log(this, "Not found.", Logger.DEBUG);
        return null;
      }
      Core.logger.log(this, "Ref found.", Logger.DEBUG);
      return dsi.ref;
    }
  }

  /**
      * Removes an item from the DataStore given its key
    * @param k The key of the item to remove
    **/

  public void remove(Key k) {
    synchronized (this) {
      for (int x = 0; x < ar.length(); x++) {
        DataStoreItem dsi = (DataStoreItem) ar.get(x);
	/* this is too annoying even for debugging
	  Core.logger.log(this,
			"Testing "+dsi + " against "+k, Logger.DEBUGGING);
	*/
        if (dsi != null)
          if (dsi.key.equals(k)) {
            ar.remove(x);
            t = t.remove(k);
            needsWriting = true;
            return;
            }
        }
      }
    }

    /**
     * Removes the reference for a given key. If there is no
     * document stored under this key, the entry will be removed
     * completely
     * @param k The key of item who's reference to remove
     **/
    public void removeRef(Key k) {
	synchronized(this) {
	    DataStoreItem dsi = (DataStoreItem) t.get(k);
	    if (dsi != null)
	      {
		if (dsi.ref != null)
		  Core.logger.log(this,"Removing ref to " + dsi.ref,Logger.DEBUG);
		if (dsi.doc != null) {
		  Core.logger.log(this,"Removing ref",Logger.DEBUG);
		  dsi.ref = null;
		} else {
		  remove(k);
		}
	      }
	}
    }

  /**
    * Returns the reference associated with the closest key
    * to k.
    * @param k The key to search for
    * @return The closest Key to k
    **/
  public synchronized Key findClosestKey(Key k) {
      Key[] frame = t.frame(k);
      if (frame == null) {
	  return null;
      } else {
	  return chooseLeftOrRight(k, (Key) frame[0], (Key) frame[1]);
      }
  }

    protected Key chooseLeftOrRight(Key k, Key left, Key right) {
	if (left == null) {
	    if (right != null) {
		return right;
	    } else {
		return null;
	    }
	} else if (right == null) {
	    return left;
	} else {
	    return ((k.isCloserTo_Ordered(left, right)) ? left : right);
	}
    }

  /**
    * Returns the reference associated with the closest key
    * to k.  The masking key allows the 'next best' key to
    * be found.  Imagine the entries in the datastore being
    * sorted in terms of closeness to the key you specify.
    * With no mask key the reference associated with the
    * top-most item will be returned.  However, with a mask
    * key any references associated with keys above the mask
    * key (including the mask key itself) will be ignored.
    * This facility allows best-first backtracking.
    * @param k The key to search for
    * @param maskKey The masking key (or null for no mask)
    * @return The closest Key to k (excluding masked keys) or
    *         null if all keys are masked.
    **/

  public synchronized Key findClosestKey(Key k, Key maskKey) {
      if (maskKey == null) {
	  return findClosestKey(k);
      }

      Key[] frame = t.frame(k);
      if (frame == null) {
	  return null;
      } 
      Key left = frame[0];
      Key right = frame[1];

      int res = maskKey.compareTo(k);

      if (res == 0) {
	  left = t.nextLeft(maskKey);
	  right = t.nextRight(maskKey);
      } else if (res < 0) {
	  left = t.nextLeft(maskKey);
	  while (right != null && !k.isCloserTo(maskKey, right)) {
	      right = t.nextRight(right);
	  }
      } else {
	  right = t.nextRight(maskKey);
	  while (left != null && !k.isCloserTo(maskKey, left)) {
	      left = t.nextLeft(left);
	  }
      }

      return chooseLeftOrRight(k, left, right);
  }

  /**
    * Adds an item to the DataStore
    * @param k The key to add
    * @param r The address to which requests should be directed for
    *          this key if the data is removed
    * @param p The properties associated with this key
    * @param d The document associated with this key
    **/

  public void put(Key k, NodeReference r, Entity d) {
      if (d != null && false) {
	  Core.logger.log(this,"Objects for StandardDataStore must be created using StandardDataStore.newEntity()",Logger.ERROR);
	  return;
      }
      put(new DataStoreItem(k, r, d));
  }

    /**
     * Inserts a DataStoreItem into the datastore.  The new item will be placed
     * in the DataStore at a random position such that the total length of all 
     * items "above" it (where new items are added at the "top") is smaller 
     * than the length of this DSI.  If an item with no document is encountered
     * then the item is inserted immedately.
     **/
    synchronized public void put(DataStoreItem dsi) {
	Key k = dsi.key;

	long tp=0;
	int pos=0;
	while((pos < ar.length()) && (tp<dsi.length()) && 
	      (ar.get(pos)!=null)) {
	    long l = ((DataStoreItem) ar.get(pos)).length();
	    if (dsminz != 0 && l < dsminz)
		l = dsminz;
	    tp += l;
	    pos++;
	}
	DataStoreItem old;
	if (pos < ar.length()) {
	    if ((ar.get(pos) == null) ||
		(((DataStoreItem) ar.get(pos)).doc == null))
		old = (DataStoreItem) ar.insert(pos, dsi);
	    else {
		if (pos != 0)
		    old = (DataStoreItem) ar.insert(Math.abs(Core.randSource.nextInt()) %
						    pos, dsi);
		else 
		    old = (DataStoreItem) ar.insert(0, dsi);
	    }
	    if (old != null) {
		t = t.remove(old.key);
	    }
	    t.insert(k, dsi);
	    needsWriting = true;
	    cleanUpData();
	}
    }

    /**
     * Class for creating documents. Overwrite this to create documents
     * of other types then files.
     **/
    protected static class MyEntity extends FileEntity {
	public MyEntity(FieldSet fs) {
	    super(fs);
	}
	public MyEntity(DataInputStream stream) throws IOException {
	    super(stream);
	}
	public MyEntity(SplitOutputStream datatunnel, long length, long partLength) throws IOException{
	    super(datatunnel, length, partLength);
	}
    }

    /**
     * Reloads a saved Entity item.
     * @param   fs  A FieldSet describing the entity to load
     * @return  A new Data object
     */
    public Entity newEntity(FieldSet fs) {
	return new MyEntity(fs);
    }

    /**
     * Reloads a saved Entity item.
     * @param stream A stream containing the information needed to load the 
     *               item.
     * @return A new Data object
     **/

    public Entity newEntity(DataInputStream stream) 
	throws IOException {

	return new MyEntity(stream);
    }

    /**
     * Creates an Entity that reads data off a
     * SplitOutputStream as it is being written too.
     * @param datatunnel SplitOutputStream to read data from
     * @param length     Number of bytes to read into this object
     * @param partLength After how many bytes to expect a control
     *                   byte in this object. If 0 a byte should only
     *                   be expected last.
     * @return           A new Data object.
     */
    public Entity newEntity(SplitOutputStream datatunnel, long length, long partLength) throws IOException {
	return new MyEntity(datatunnel, length, partLength);
    }
  
  // Protected Methods

  /**
    * Removes excess data (ie. if there is more data than is
    * permitted by the maxData field).
    **/
  protected void cleanUpData() {
    long total = 0;
    Vector toRemove = new Vector();
    for (int x = 0; x < ar.length(); x++) {
      Object o = ar.get(x);
      if (o != null) {
        DataStoreItem dsi = (DataStoreItem) o;
        if (dsi.doc != null) {
          if (total + dsi.doc.length() > maxData) {
	      if (dsi.ref == null) {
		  toRemove.addElement(dsi.key);
	      } else {
		  dsi.doc = null;
	      }
	      needsWriting = true;
          } else
            total += dsi.doc.length();
          }
        }
      }

    // We do this so we don't do any removal during the traverse,
    // which would probly make life difficult.

    Enumeration keys = toRemove.elements();
    while (keys.hasMoreElements()) {
	Key key = (Key) keys.nextElement();
	remove(key);
    }
  }

    public Hashtable refList()
    {
        Object dummy = new Object();
	Hashtable ret = new Hashtable();
	for (int x=0; x<ar.length(); x++)
	    {
		ret.put(((DataStoreItem) ar.get(x)).ref.toString(), dummy);
	    }
	return ret;
    }


  // Mainly for debugging
  public String toString()
  {
    return t.toString();
  }
}

class BadEntryException extends Exception {
}

class DataStoreItem implements Serializable {
    public Key key;
    public NodeReference ref;
    public Entity doc;

    public DataStoreItem(Key k, NodeReference a, Entity d) {
	key = k;
	ref = a;
	doc = d;
    }

    public DataStoreItem(DataInputStream stream, StandardDataStore ds) throws IOException, BadEntryException {
	boolean bad = false;
	try {
	    key = Key.readKey(stream.readUTF());
	} catch (KeyException e) {
	    e.printStackTrace();
	    bad = true;
	}
	try {
	    String refString = stream.readUTF();
	    if (!refString.equals(""))
		ref = new NodeReference(refString);
	} catch (IllegalArgumentException e) {
	    e.printStackTrace();
	    bad = true;
	}
	if (stream.readBoolean()) {
	    doc = ds.newEntity(stream);
	}
	if (bad)
	    throw new BadEntryException();
    }

    public DataStoreItem(FieldSet fs, StandardDataStore ds) 
	throws BadEntryException {
	try {
	    String k = fs.get("Key");
	    if (k == null)
		throw new BadEntryException();
	   else
	       key = Key.readKey(k);
	    String r = fs.get("Ref");
	    if (r != null)
		ref = new NodeReference(r);
	    String e = fs.get("HasEntity");
	    if (e != null && Fields.stringToBool(e,false)) {
		doc = ds.newEntity(fs);
	    }
	} catch (KeyException e) {
	    throw new BadEntryException();
	} catch (IllegalArgumentException e) {
	    throw new BadEntryException();
	}
    }

    public long length()
    {
	return (doc == null ? 0 : doc.length());
    }
    
    public FieldSet fields() {
	FieldSet fs = new FieldSet();
	fs.add("Key",key.toString());
	if (ref != null)
	    fs.add("Ref",ref.toString());
	if (doc != null) {
	    fs.add("HasEntity",Fields.boolToString(true));
	    doc.write(fs);
	}
	return fs;
    }

    public boolean isValid() {
	return ((key != null) && ((ref != null) || (doc != null)));
    }

    public String toString() {
	return "Key: " + key + ", Ref: " + ref + ", Doc: " + doc;
    }
}

/**
 * Makes an enumeration of entities in this DataStore
 **/

class EntityEnumeration implements Enumeration {

    Enumeration e;
    Entity next;

    public EntityEnumeration(StandardDataStore ds) {
	e = ds.t.elements();
	fillNext();
    }

    private void fillNext() {
	while (next == null && e.hasMoreElements()) {
	    DataStoreItem dsi = (DataStoreItem) e.nextElement();
	    next = dsi.doc;
	}
    }

    public boolean hasMoreElements() {
	return next != null;
    }

    public Object nextElement() {
	Object r = next;
	next = null;
	fillNext();
	return r;
    }
}
