package Freenet.node;
import Freenet.*;
import java.util.*;
import Freenet.support.*;
/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.

  Explanation of Code Versions: 
    0.0.0      = Initial Description
    0.0.1      = API Specified
    0.x (x>0)  = Partial Implementation
    x.0 (x>0)  = Operational
		
 */

/**
 * This class handles incoming messages.
 *
 * @author <A HREF="mailto:I.Clarke@strs.co.uk">Ian Clarke</A>
 **/

public class StandardMessageHandler implements MessageHandler
{
  // Public Fields
  // Protected/Private Fields

  public Node node;
  private LimitedHashtable messageMemories;
  private LimitedHashtable tickets;
  
  public static void main(String[] args)
    {
      System.out.println("Testing limitedHashtable");
      LimitedHashtable t = new LimitedHashtable(3);
      t.put(new Integer(1), new String("one"));
      System.out.println(t.toString());
      System.out.println("Removing 2");
      t.remove(new Integer(2));
      System.out.println(t.toString());   
      t.put(new Integer(2), new String("two"));
      System.out.println(t.toString());
      t.put(new Integer(3), new String("three"));
      System.out.println(t.toString());
      t.put(new Integer(4), new String("four"));
      System.out.println(t.toString());
      System.out.println("Removing 2");
      t.remove(new Integer(2));
      System.out.println(t.toString());
      t.put(new Integer(5), new String("five"));
      System.out.println(t.toString());
      t.put(new Integer(6), new String("six"));
      System.out.println(t.toString());
    }

  // Constructors

  /**
   * @param n The node that should be used to send messages
   *          and for which the datastore should be updated
   * @param m The number of message objects that will be
   *          remembered
   **/
  public StandardMessageHandler(Node n, int m)
    {
      node = n;
      messageMemories = new LimitedHashtable(m);
      tickets = new LimitedHashtable(m);
    }

  // Public Methods

  /**
   * Handle a message
   * @param mo The message to handle
   **/
  public void handle(MessageObject mo) {
      if (!(mo instanceof NodeMessageObject)) {
	  Core.logger.log(this, "Received a MessageObject that the node cannot handler",Logger.ERROR);
	  return;
      }
      NodeMessageObject message = (NodeMessageObject) mo;
      try {
	  Long longid = new Long(message.id());
	  Object ticket = getTicket(longid);
	  synchronized(ticket) {
	      MessageMemory mm = (MessageMemory)messageMemories.get(longid);
	      MessageMemory newmm = message.received(node, mm);
	      if (newmm != mm) {
		  messageMemories.remove(longid);
		  if (newmm != null) {
		      Object[] lmm = messageMemories.limput(longid, newmm);
		      if (lmm != null) { // dispose of lost mm
			  ((MessageMemory) lmm[1]).lost((Long) lmm[0]);
		      }
		  } else {
		      remTicket(longid);
		      Core.logger.log(this, "Ending chain : " + 
				      Fields.longToString(longid.longValue()),
				      Logger.DEBUGGING);
		  }
	      }
	  }
      } catch(Exception e) {e.printStackTrace();}
  }

  // Protected/Private Methods

  // Return the canonical object corresponding to the messageid
  // Create one if it does not already exist.  Synchronize to
  // protect against race conditions.
  synchronized Object getTicket(Long longid)
  {
    Object ticket = tickets.get(longid);
    if (ticket == null) {
      ticket = new Object();
      tickets.put(longid, ticket);
    }
    return ticket;
  }

  synchronized void remTicket(Long id) {
      tickets.remove(id);
  }

}


/**
 * A variant on a Hashtable which only holds a limited number of
 * messages.  When more messages than that are put in, old ones are
 * deleted, oldest first.
 *
 * @version Rrevision$
 * @author <A HREF="mailto:I.Clarke@strs.co.uk">Ian Clarke</A>
 **/
class LimitedHashtable
{
    private class Container {
	public Object o;
	public Object key;
	public Container prev, succ;
	public Container(Object key, Object o) {
	    this.o = o;
	    this.key = key;
	}

	public void add(Container c) {
	    prev.succ = c;
	    c.prev = prev;
	    c.succ = this;
	    this.prev = c;
	}

	public void replace(Container c) {
	    prev.succ = c;
	    succ.prev = c;
	    c.succ = succ;
	    c.prev = prev;
	    prev = null;
	    succ = null;
	}

	public void remove() {
	    prev.succ = succ;
	    succ.prev = prev;
	    succ = null;
	    prev = null;
	}
	public String toString() {
	    if (o != null && key != null)
		return key + " = " + o;
	    else
		return null;
	}
    }

    Hashtable ht;
    Container ptr = null;
    int size;
    int currentsize;

    public LimitedHashtable(int size)
    {
      this.ht = new Hashtable(size);
      this.size = size;
      this.currentsize = 0;
    }

    /**
     * @return  If an old object was pushed out of Hashtable by this
     *          insert, it's an array containing it's key (0) and data (1) 
     *           is returned. Otherwise null. Observe that 
     *          any object previously under this key will _not_ be
     *          returned (use normal put for that).
     **/
    public synchronized Object[] limput(Object key, Object value)
    {
	if (size == 0) return null;

	Container c = new Container(key,value);
	Object[] old = null;
	if (ptr == null) {
	    c.prev = c;
	    c.succ = c;
	    ptr = c;
	    currentsize++;
	} else if (currentsize < size) {
	    ptr.add(c);
	    currentsize++;
	} else {
	    ptr.replace(c);
	    ht.remove(ptr.key);
	    old = new Object[2];
	    old[0] = ptr.key;
	    old[1] = ptr.o;
	    ptr = c.succ;
	}
	ht.put(key,c);
	return old;
    }

  public synchronized Object put(Object key, Object value)
    {
	if (size == 0)
	    return null;
	Container c = new Container(key, value);
	if (ptr == null) {
	    c.prev = c;
	    c.succ = c;
	    ptr = c;
	    currentsize++;
	} else if (currentsize < size) {
	    ptr.add(c);
	    currentsize++;
	} else {
	    ptr.replace(c);
	    ht.remove(ptr.key);
	    ptr = c.succ;
	}
	Container old = (Container) ht.put(key,c);
	return old == null ? null : old.o;
    }

    public synchronized Object get(Object key) {
	Container c = (Container)ht.get(key);
	return c == null ? null : c.o;
    }

    public synchronized Object remove(Object key) {
	Container c = (Container) ht.remove(key);
	if (c != null) {
	    if (currentsize > 1) {
		if (c == ptr)
		    ptr = ptr.succ;
		c.remove();
	    } else {
		ptr = null;
	    }
	    currentsize--;
	    return  c.o;
	} else
	    return null;
    }

    public String toString()
    {
      return ht.toString() + ", size: " + currentsize + ", max: "+size;
    }
}
