package Freenet.node.gui;

import Freenet.thread.ThreadManager;
import Freenet.node.Node;
import Freenet.node.DataStoreMaintence;
import Freenet.node.StandardDataStore;
import java.awt.*;
import java.awt.event.*;

/**
 * Runs the Freenet Node as a GUI Frame rather than a command line
 * application or daemon.  It's done by simply wrapping Freenet.node.Node.
 *
 * @author <a href="mailto:anthony@takahe.net">Anthony Jones</a>
 **/

public class GUINode extends Frame {
    
    /** Message shown at the bottom of the frame. **/
    private Label message;
            
    public static void main(String[] args) {
        GUINode guiNode = new GUINode(args);
    }
    
    public GUINode(String[] args) {
        setLayout(new BorderLayout());
        setTitle("Freenet node " + Node.freenetVersion);
        setSize(250, 150);
        show();
        
        addWindowListener(new WindowAdapter() {
            public void windowClosing(WindowEvent e) {
                actionExit();
            }
        });

        MenuBar mb = createMenuBar();
        setMenuBar(mb);
        
        message = new Label("Node running");
        add(message, BorderLayout.SOUTH);
        validate();

        Node.main(args);
    }

    /** Open configurator then display a message telling the user to restart the 
     * node
     **/

    protected void actionConfigure() {
	Config config = new Config() {
	    protected void actionAccept() {
		if (!verify())
		    return;
		update();
		showDialog();
		this.dispose();
	    }

	    public void showDialog() {
		Dialog d = new Dialog(this, true);
		
		TextArea ta = new TextArea("You must restart the node for the configuration to take effect.", 5, 30, TextArea.SCROLLBARS_NONE);
		ta.setEditable(false);
		d.add(ta, BorderLayout.CENTER);
		Button buttonAccept = new Button("OK");
		final Window parent = this;
		buttonAccept.addActionListener(new ActionListener() {
		    public void actionPerformed(ActionEvent e) {
			parent.hide();
			GUINode.this.dispose();
		    }
		});
		d.add(buttonAccept, BorderLayout.SOUTH);
		d.pack();
		Dimension size = d.getSize();
		Dimension parentSize = parent.getSize();
		Point parentPos = parent.getLocation();
		d.setLocation(parentPos.x + (parentSize.width - size.width) / 2,
		    parentPos.y + (parentSize.height - size.height) / 2);
		d.show();
	    }
	};
	config.show();
    }
 
    /** Shutdown cleanly then exit. **/ 
    protected void actionExit() {
        message.setText("Shutting down node");
        message.setForeground(Color.red);
	Node n = Node.n;
	try {
	    if (n != null) {
		try {
		    ThreadManager tm = n.threadManager;
		    if (tm != null) {
			tm.flush();
			tm.halt();
			((Thread) tm).join();
		    }
		} catch (InterruptedException e) {
		    System.err.println("Darn, interrupted!");
		} 
		DataStoreMaintence dsm = 
		    new DataStoreMaintence(n.randSource.nextLong());
		n.mh.handle(dsm);
		if (n.ds instanceof StandardDataStore)
		    dsm.cleardir((StandardDataStore) n.ds);
	    }
	} finally {
            dispose();
            System.exit(0);
        }
    }
    
    /** Show an about window. **/
    protected void actionAbout() {
        Dialog d = new AboutDialog(this);
        d.show();
    }
    
    /**
     * Create the file menu.
     * @return File menu.
     **/
    protected Menu createFileMenu() {
        Menu m = new Menu("File");
	MenuItem miConfigure = new MenuItem("Configure");
	miConfigure.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		actionConfigure();
	    }
	});
	m.add(miConfigure);
        MenuItem miExit = new MenuItem("Exit");
        miExit.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                actionExit();
            }
        });
        m.add(miExit);
        return m;
    }
    
    /** 
     * Create the help menu.
     * @return Help menu.
     **/
    protected Menu createHelpMenu() {
        Menu m = new Menu("Help");
        MenuItem miAbout = new MenuItem("About");
        miAbout.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                actionAbout();
            }
        });
        m.add(miAbout);
        return m;
    }
    
    /**
     * Create the menu bar.
     * @return The menu bar. 
     **/
    protected MenuBar createMenuBar() {
        MenuBar mb = new MenuBar();
        mb.add(createFileMenu());
        mb.add(createHelpMenu());
        return mb;
    }
}
