/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU General Public Licence (GPL) 
  version 2.  See http://www.gnu.org/ for further details of the GPL.
 */

/**
 * This class represents a raw message in the Freenet Protocol
 * expected by Freenet.  It can be created either from
 * an InputStream or manually created from scratch.  It can
 * then by piped out to an OutputStream.  Methods are provided
 * for manipulation of normal fields, however the type of the
 * message and the trailing field should be set by direct
 * manipulation of fields.
 *
 * @author <A HREF="mailto:I.Clarke@strs.co.uk">Ian Clarke</A>
 * @author <A HREF="mailto:blanu@uts.cc.utexas.edu">Brandon Wiley</A>
 **/
package Freenet.presentation;
import Freenet.*;
import Freenet.support.*;
import Freenet.support.io.*;
import java.io.*;
import java.io.EOFException;
import java.util.*;

public class FNPRawMessage extends RawMessage {
    
    // Constructors

    static public void main(String[] args) throws Exception
    {
      RawMessage r=new FNPRawMessage(System.in);
      System.out.println(r);
      r.writeMessage(System.out);
    }

    /**
     * Constructs a new RawMessage off an FNP Stream
     * @param i An InputStream of decrypted FNP data
     **/
    public FNPRawMessage(InputStream i) throws InvalidMessageException, EOFException {
	Core.logger.log(this,"Reading message",Logger.DEBUGGING);
	ReadInputStream ris = new ReadInputStream(i);
	//	PushbackInputStream in = new PushbackInputStream(i);
	fs = new FieldSet();

	try {
	    // Read message type
	    messageType = ris.readToEOF('\n','\r');

	    //	    System.out.println(messageType);
	    trailingFieldName = fs.parseFields(ris,
					       '\n', // ends a field
					       '\r', // ignore before end
					       '=', // delimits field
					       '.'); // delimits subset
	    setFields(ris);
	} catch (EOFException e) {
	    if (messageType != null) {
		Core.logger.log(this, "Stream died while reading message of type: " + messageType, Logger.ERROR); 
	    } else {
		// stream closed without getting a new message
		Core.logger.log(this, "Stream closed", Logger.DEBUGGING);
	    }
	    throw e;
	} catch (IOException e) {
	    throw new EOFException("Could not parse message from stream : " + e);
	} catch(Exception e) {
	    Core.logger.log(this, "Exception in RawMessage()", Logger.ERROR);
	    e.printStackTrace();
        }
    }

    private void setFields(ReadInputStream in) {
	// Read and set the presentation related fields

	// setting KeepAlive
	String kavalue = fs.get("KeepAlive");
	keepAlive = kavalue == null || Fields.stringToBool(kavalue, true);
	if (kavalue != null) 
	    fs.remove("KeepAlive");

	String pvalue = fs.get("Persist");
	persist = (pvalue == null ?
		   0 : (Fields.stringToBool(pvalue,false) ?
			1 : -1));

	// setting Source
	String srcvalue = fs.get("Source");
	fs.remove("Source");

	// XXX Fixed to work with Kaffe
	if (srcvalue != null && !"".equals(srcvalue))
	{
	    try {
		source = new Address(srcvalue);
	    } catch (BadAddressException e) {
		source = null;
	    }
	}
	else
	{
	    source = null;
	}

	// setting DataLength and trailing
	String dlvalue = fs.get("DataLength");
	trailingFieldLength = dlvalue == null ? 0 : Fields.stringToLong(dlvalue);
	if (dlvalue != null) 
	    fs.remove("DataLength");
	if (trailingFieldLength != 0) {// we have a trailing
	    trailingFieldStream = in;
	} else {
	    trailingFieldName = null; // no trailing
	}
    }

    /**
     * Constructs a new RawMessage for sending
     * @param messageType   The name of the message type.
     * @param source        The address from which this message claims to come.
     *                      may be null.
     * @param keepAlive     Whether to keep alive the connection after 
     *                      receiving or sending this message.
     * @param fs            A set of message specific fields. This may be
     *                      null if there are no such fields on creation.
     * @param trailingLength The length of the trailing data, or 0 if there 
     *                       is no trailing data
     * @param trailingName   The name of the trailing field, or null if there
     *                       is no trailing data
     * @param trailing       An inputstream containing the trailing data,
     *                       in a format that can be copied to destination
     *                       of this message.
     **/
    protected FNPRawMessage(String messageType, Address source, boolean keepAlive, FieldSet fs, long trailingLength, String trailingName, InputStream trailing) {
	super(messageType, source, keepAlive, 0, fs == null ? new FieldSet() : fs, trailingLength, trailingName, trailing);
    }
    
    // Public Methods

    public void writeMessage(OutputStream out) throws IOException
    {

	String key, value;

	WriteOutputStream writer=new WriteOutputStream(out);
      
	// Output message type
	writer.writeUTF(messageType, '\n');

	// Output tansport options

	if (source != null)
	    fs.add("Source",source.toString());
	if (!keepAlive)
	    fs.add("KeepAlive",Fields.boolToString(keepAlive));
	if (persist != 0)
	    fs.add("Persist",Fields.boolToString(persist > 0));
	if (trailingFieldLength != 0)
	    fs.add("DataLength",Fields.longToString(trailingFieldLength));

	// Output message fields

	fs.writeFields(writer,trailingFieldName == null ? "EndMessage" : trailingFieldName,
		       '\n', // ends a pair
		       '=',  // delimits a pair
		       '.'); // delimits a subset
	// empty writer
	writer.flush();
    }

    public String toString() {
	return messageType + "\n" + "{Source=" + source + ",KeepAlive=" + keepAlive + ",DataLength=" + trailingFieldLength + "," + fs.toString() + "}";
    }
}
