package Freenet.presentation;
import Freenet.*;
import Freenet.support.Logger;
import Freenet.crypt.*;
import Freenet.crypt.ciphers.*;
import java.io.*;

/**
 * This is the implementation of the standard protocol used by Freenet
 * to communicate messages. Freenet Protocol (FNP) uses text based messages
 * with a simple header and fieldname=fieldvalue format. FNP utilizes 
 * encryption, currently by doing a DH exchange.
 **/

public class FreenetProtocol extends Presentation {

    // Fields
    private DiffieHellman kE;

    public FreenetProtocol() {
	this(false);
    }

    public FreenetProtocol(boolean oneShot) {
	kE=new DiffieHellman(Global.DHgroupA, Core.randSource, 
			     oneShot ? 0 : 5);
    }

    /**
     * Performs a DH key exchange of a new connection (should) and sets it
     * up.
     * @param c   The Connection to initialize
     **/
    public void initConnection(Connection c) throws IOException {
	//Simple change the right hand side of the next line to change crypto
	BlockCipher cipher=new Rijndael(128,128);
	int oldTimeout=c.getSoTimeout();
	c.setSoTimeout(Core.authTimeout);
	try {
	    CipherLink negotiator=new CipherLink(Core.randSource,
						 kE,
						 cipher,
						 c.in, c.out);
	    c.info=negotiator;
	    c.setReady();
	} catch (InterruptedIOException e) {
	    Core.logger.log(this, "Authentication timed out", Logger.ERROR);
	    c.close();
	    throw e;
	} finally {
	    c.setSoTimeout(oldTimeout);	
	}
    }

    /**
     * Return an InputStream that reads data off the connection with
     * this protocol.
     * param c The connection you wish to read the message to
     * return  An inputStream from which the the RawMessage can be read.
     **/
    public InputStream getReceiveStream(Connection c) throws IOException {
	if (c.ready()) 
	    return ((CipherLink)c.info).getInputStream();
	else {
	    synchronized(c.initLock) {
		try {
		    c.initLock.wait();
		} catch (InterruptedException e) {}
	    }
	    return getReceiveStream(c);
	}
    }


    /**
     * Return an OutputStream that writes data to the connection with
     * this protocol.
     * param c  The connection you wish to write the message to
     * return  An outputstream that to which the RawMessage and trailing
     *         field can be written.
     **/
    public OutputStream getSendStream(Connection c) throws IOException {
        if (c.ready())  
	    return ((CipherLink)c.info).getOutputStream();	
        else {
            synchronized(c.initLock) {
		try {
		    c.initLock.wait();
		} catch (InterruptedException e) {}
            }
            return getSendStream(c);
        }
    }

    /**
     * Creates a new RawMessage of a connection by reading this presentation. This
     * method locks until an entire message has been read (excluding trailing)
     * @param c     The connectio to read
     * @return      A new raw message
     **/
    public RawMessage readMessage(InputStream in) throws InvalidMessageException, EOFException{
	PushbackInputStream pin;
	if (in instanceof PushbackInputStream)
	    pin = (PushbackInputStream) in;
	else
	    pin = new PushbackInputStream(in);
	int i;
	do {
	    try {
		i = pin.read();
	    } catch (IOException e) {
		try {
		    in.close();
		} catch (IOException e2) {
		    e2.printStackTrace();
		}
		throw new InvalidMessageException("Error while waiting for message");
	    }
	} while (i == 0);
	if (i == -1)
	    throw new EOFException();
	else {
	    try {
		pin.unread(i);
	    } catch (IOException e) {
		try {
		    in.close();
		} catch (IOException e2) {
		    e2.printStackTrace();
		}
		throw new RuntimeException("Problem while pushing back read data in FreenetProtocol");
	    }
	    return new FNPRawMessage(pin);
	}
    }

    /** 
     * Creates a new RawMessage of a given type that uses this presentation
     * @param messageType   The name of the message type
     * @param source        The address from which this message claims to come.
     *                      May be null.
     * @param keepAlive     Whether to keep alive the connection after 
     *                      receiving or sending this message.
     * @param fs            A set of message specific fields.
     * @param trailingLength The length of the trailing data, or 0 if there 
     *                       is no trailing data
     * @param trailingName   The name of the trailing field, or null if there
     *                       is no trailing data
     * @param data           An inputstream containing the trailing data,
     *                       straight off the DataStore (decrypted).
     * @return     A new raw message
     **/   
    public RawMessage newMessage(String messageType, Address source, boolean keepAlive, FieldSet fs, long trailingLength, String trailingName, InputStream data) {
	//should not need any crypto here
	return new FNPRawMessage(messageType, source, keepAlive, fs, trailingLength, trailingName, data);
    }
}
