package Freenet.scripts;
import Freenet.support.io.ReadInputStream;
import Freenet.support.Logger;
import Freenet.support.StandardLogger;
import Freenet.Core;
import Freenet.Params;
import Freenet.node.Node;
import java.net.ServerSocket;
import java.util.Vector;
import java.lang.reflect.Method;
import java.io.FileOutputStream;
import java.io.PrintStream;
import java.io.IOException;
import java.io.FileNotFoundException;
import java.text.DateFormat;
import java.util.Date;
/**
 * A simple script that produces a .freenetrc. Note the simple, this
 * is not meant to be some generic autogenerating config file maker -
 * when a new entry to .freenetrc is added, you'll have to code it.
 *
 * When main() is run, it will set up the static fields, and then call all
 * static argumentless methods whose name begin with "setParam" in 
 * alphabetical order. So to add the new setting "fooBar", all you need to do
 * is add a method called "setParamFooBar" that uses the provided streams and
 * methods to write the settings to the config file.
 *
 * @author oskar;
 */


public class Setup {

    /** Output stream to the file **/
    public static PrintStream out;
    /** Output stream to the user **/
    public static PrintStream qout;
    /** InputStream from the user **/
    public static ReadInputStream in;
    /** Whether we are in expert mode **/
    public static boolean expert = false;
    /** Are we in silent creation mode? **/
    public static boolean silent = false;
    /** Are we in default creation mode? **/
    public static boolean defaultMode = false;
    /** Load the existing defaults to params **/
    public static Params params;

    public static void main(String[] args) {
      boolean newFile=false;

	if (args.length > 0) {
	    //Read in existing prefs first
	    try {
		params = new Params(args[0]);	          	
	    }
	    catch (FileNotFoundException e) {
		//create file and reinitialize params later again
		newFile = true;
	    }
	    //then create the file new
	    try {	
		out = new PrintStream(new FileOutputStream(args[0]));
		if  (newFile) params = new Params(args[0]);
	    } catch (IOException e) {
		System.out.println("Could not open file for writing");
		System.exit(1);
	    }
	} else {
	    System.out.println("Usage: Freenet.scripts.setup outfile [simple|expert|silent|default]");
	    System.exit(1);
	}

	in = new ReadInputStream(System.in);
	qout = System.out;

	qout.println("Freenet Configuration");
	
	if  (args.length == 1)
	  expert = getBool("Create Freenet ini file in expert mode (asks about everything)?",false);
	else {
		expert = args[1].equalsIgnoreCase("expert");
		defaultMode = args[1].equalsIgnoreCase("default");
		silent = (defaultMode) ? true : args[1].equalsIgnoreCase("silent");
	}

	if (!expert) qout.println("Running in simple mode. Some preferences will be skipped.");

	qout.println("You can choose the default preferences by just hitting <ENTER>");
	out.println("[Freenet node]");
	out.println("# Freenet configuration file");
	out.println("# Note that all properties may be overridden from the command line,");
	out.println("# so for example, java Freenet.Node -listenPort 10000 will cause");
	out.println("# the setting in this file to be ignored");
	out.println();
	DateFormat df = DateFormat.getDateTimeInstance();

	out.println("# This file was automatically generated by Freenet setup (at " + df.format(new Date()) + ")");
	out.println();
	out.println();
	
	qout.println();
	Method[] ms = Setup.class.getMethods();
        	quickSort(ms,0,ms.length - 1);

	for (int i = 0 ; i < ms.length ; i++) {
	    if (ms[i].getName().startsWith("setParam")) {
		try {
		    if (expert) qout.println("Setting: " + ms[i].getName().substring(8));
		    ms[i].invoke(null,null);
		    out.println();
		    qout.println();
		} catch (Exception e) {
		    e.printStackTrace(System.out);
		}
	    }
	}
	qout.println("Setup finished, exiting.");
    }

    private static void quickSort(Method[] ms, int off, int last) {
	if (last - off < 1)
	    return;
	Method def = ms[off];
	int f = off;
	int l = last;
	boolean end = true;
	Method sel = ms[l];
	Method tmp;
	int i;
	while (f < l) {
	    i = def.getName().compareTo(sel.getName());
	    if (end) {
		if (i > 0) {
		    ms[f] = sel;
		    sel = ms[f + 1];
		    f++;
		    end = false;
		} else {
		    l--;
		    sel = ms[l];
		}
	    } else {
		if (i < 0) {
		    ms[l] = sel;
		    sel = ms[l-1];
		    l--;
		    end = true;
		} else {
		    f++;
		    sel = ms[f];
		}
	    }	
	}
	ms[f] = def;
	quickSort(ms,off,f-1);
	quickSort(ms,f+1 ,last);
    }


    /**
     * reads all buffered input from the user and throws it away
     */
    public static void clearInput() throws IOException {
	while (in.available() > 0)
	    in.read();
    }

    /**
     * Ends to program in the case of EOF on the user input
     */
    public static void qeof() {
	qout.println("Input interrupted");
	System.exit(1);
    }

    /**
     * Prints a comment to the file and user.
     * @param s The comment to print.
     */
    public static void comment(String s) {
	out.println("# " + s);
	qout.println(s);
    }

    /**
     * Prints a comment to the file, but only the user if expert is set
     * @param s The comment to print.
     */
    public static void expComment(String s) {
	out.println("# " + s);
	if (expert)
	    qout.println(s);
    }
    
    /**
     * Gets a boolean value from the user.
     * @param query   The text to print as the input query
     * @param default  The value to return on EOL from the user
     * @return the input or the default if nothing was entered.
     */
    public static boolean getBool(String query, boolean dfault) {
	try {
	    int i;
	    qout.print(query + " ");
	    do {
		qout.println(dfault ? "[Y/n]" : " [y/N]");
		if (silent) return dfault;
		i = in.read();
		if (i < 0)
		    qeof();
	    } while (i != 'y' && i != 'Y' && i != 'n' && i != 'N' 
		     && i != '\r' && i != '\n');
	    clearInput();
	    return i == '\r' || i == '\n' ? dfault : (i == 'y' || i == 'Y'); 
	} catch (IOException e) {
	    qeof();
	    return false;
	}
    }

    /**
     * Gets a String value from the user.
     * @param query The text to print as the input query
     * @param default  The value to return on EOL from the user
     * @return the input or the default if nothing was entered.
     */
    public static String getString(String query, String dfault) {
	try {
	    String s;
	    qout.println(query + " [" + dfault + ']');
	    if (silent) return dfault;
	    s = in.readTo('\n','\r');
	    clearInput();
	    return "".equals(s) ? dfault : s;
	} catch (IOException e) {
	    qeof();
	    return null;
	}
    }

    /**
     * Gets a number value from the user.
     * @param query The text to print as the input query
     * @param default  The value to return on EOL from the user
     * @return the input or the default if nothing was entered.
     */
    public static long getNumber(String query, long dfault) {
	boolean failed;
	long l = 0;
	do {
	    failed = false;
	    String s = getString(query, Long.toString(dfault));
	    try {
		l = Long.parseLong(s);
	    } catch (NumberFormatException e) {
		failed = true;
	    }
	} while (failed);
	return l;
    }


    ///////////////////////////////////////////////////////////////////
    //                                                               //
    //                           Param Methods                       //
    //                                                               //
    ///////////////////////////////////////////////////////////////////

    public static void setParamListenPort() {
	String id = "listenPort";

	comment("The port to which the Freenet Server should listen for connections");
	qout.println("Setup will attempt to find a free port that Freenet can use");
	ServerPortSearch sps = new ServerPortSearch();
	boolean b = false;
	int i = params.getint(id,Core.defaultListenPort);
	if (i != Core.defaultListenPort) {
		qout.println("You have already another port ("+i+") than the default configured.");
		b=getBool("Do you want to keep it?",true);
	} else if (i == Core.defaultListenPort && defaultMode) {
	    b = true;
	}

	while (!b) {
	    qout.println("Searching...");
	    b = false;
	    i = sps.getFreePort();
	    b = getBool("Port " + i + " is free. Should we use it?", true);
	    if (!b && getBool("Input port manually?", false)) {
		int j;
		do {
		    j = (int) getNumber("Port value to use?",i);
		} while (i < 0 || i > (1 << 16));
		i = j;
		b = true;
	    }
	    
	}
	out.println(id + "=" +  i);
    }

    public static void setParamTransient() {
	String id = "transient";
	expComment("Transient nodes will not send their address to other nodes, and therefore");
	expComment("will not get any requests for data (except if you activate the \"inform\" CGI");
	expComment("service below). Use this only if you are on a slow, non permanent connection.");
	qout.println();
	boolean b = params.getboolean(id,false);
	if (expert) 
	b = getBool("Should your node be transient?", b);
	out.println(id + "=" + (b ? "yes":"no"));
    }
    
    public static void setParamCheckPointInterval() {
	String id = "checkPointInterval";
	long l = params.getlong(id,Node.defaultCheckpointInterval);
	expComment("How often to write the datastore to disk (in seconds)");
	if (expert)
	    l = getNumber("?",l);
	out.println(id + "="+l);
    }
    
    public static void setParamNodeAddress() {
	String id = "nodeAddress";

	comment("The address to use when advertising this node to  the network. If this is ");
	comment("unset, which is prefered, the node's address will be read from the socket,"); 
	comment("allowing the node to function seamlessly on multihomed machines (and even");
	comment("functioning a Freenet proxy over firewalls where necessary). However, people");
	comment("who have volatile IP addresses but constant DNS names (such as people using");
	comment("Dynamic DNS services) may wish to set this so their node can still be found");
	comment("after the ip changes.");
	qout.println();
	String s = getString("Enter the node address, or press enter to let the node autodetect.\n",params.getParam(id,""));
	out.println((s.equals("")?"#":"") + id + "=" + s);
    }

    public static void setParamDataStoreSettings() {
	String id = "diskCache";
	String id2 = "dataStoreSize";
	String id3 = "dataStorePath";

	out.println("# The size of the disk cache is specified here in bytes");
	long l = params.getlong(id,Node.defaultDiskCache);
	l = getNumber("How much diskspace, in Megabytes, can Freenet use?", l/(1024*1024));
	out.println(id+"=" + l*(1024*1024));
	out.println("");
	expComment("The number of items permitted in the dataStore");
	long n = params.getlong(id2,Node.defaultDataStoreSize);
	if (expert)
	    n = getNumber("How many?",n);
	out.println(id2+"=" + n);	
	expComment("The name of the directory containing the dataStore (store_port) file");
	String s = params.getParam(id3,Node.defaultDataStorePath);
	if (expert)
	    s = getString("Directory name (relative)?",s);
	out.println(id3+"=" + s);	
    }

    public static void setParamDataPath() {
	String id = "dataPath";

	expComment("The name of the directory containing the data files");
	String s = params.getParam(id,Node.defaultDataPath);
	if (expert)
	    s = getString("Directory name (relative)?",s);
	out.println(id+ "="+s);
    }

    public static void setParamDataPropertiesPath() {
	String id = "dataPropertiesPath";
	expComment("The name of the directory containing the data properties files");
	String s = params.getParam(id,Node.defaultDataPropertiesPath);
	if (expert)
	    s = getString("Directory name (relative)?",s);
	out.println(id+"="+s);
    }

    public static void setParamMessageStoreSize() {
	String id = "messageStoreSize";
	expComment("The number of outstanding message replies the node will");
	expComment("wait for before it starts to \"forget\" them");
	long l = params.getlong(id,Node.defaultMessageStoreSize);
	if (expert)
	    l = getNumber("?",l);
	out.println(id+"="+l);
    }

    public static void setParamNodeFile() {
	String id = "nodeFile";
	expComment("The name of a file containing an initial set of nodes to connect to");
	String s = params.getParam(id,Node.defaultNodeFile);
	if (expert)
	    s = getString("File name?",s);
	out.println(id + "="+s);
    }
    
    public static void setParamInform() {
	String id = "informRead";
	expComment("Should we read some information about other nodes from the informUrl?");
	boolean b = params.getboolean(id,Node.defaultInformRead);
	if (expert) getBool("?",b);
	out.println(id + "=" + (b ? "yes" : "no"));
	id = "informWrite";
	expComment("Should we write our address to the inform URL so that others might");
	expComment("find out about us more easily?");
	boolean c = params.getboolean(id,Node.defaultInformWrite);
	if (expert) c = getBool("?",c);
	out.println(id+ "=" + (c ? "yes" : "no"));
	id = "informDelay";
	expComment("After which time (seconds) of node uptime should we add the presence to the informURL?");
	long l = params.getlong(id,Node.defaultInformDelay);
	if (expert)   l = getNumber("?",l);
	out.println(id+"=" + l);

	expComment("The URL of a CGI script which will allow the server");
	expComment("to learn about other servers in the Freenet, and");
	expComment("to inform other servers of its presence.");
	id = "informUrl";
	String s = params.getParam(id,Node.defaultInformUrl);
	if (expert)
	    s = getString("Full URL?",s);
	out.println(id+"=" + s);
    }

    public static void setParamBandWidthLimit() {
        long l;
        String id;

	id = "bandwidthLimit";
	comment("The maximum rate data passes through the node, measured in");
        comment("bytes per second.");
        comment("If parameter is set to 0, bandwidth limiting is disabled");
	/*qout.println("Enter 0 to disable bandwidth limiting");*/
	l = params.getlong(id,Node.defaultBandwidthLimit);
	l = getNumber("?",l);
	out.println(((l==0) ? "# " : "") + id +"="+l);

	id = "inputBandwidthLimit";
	expComment("The maximum rate incoming data passes through the node,");
        expComment("measured in bytes per second.");
	expComment("This parameter disables bandwidthLimit if non-zero.");
	l = params.getlong(id,Node.defaultInputBandwidthLimit);
	if (expert) {
            qout.println("Enter bytes/sec to enable input bandwidth limiting");
            l = getNumber("?",l);
        }
	out.println(((l==0) ? "# " : "") + id +"="+l);

	id = "outputBandwidthLimit";
	expComment("The maximum rate outgoing data passes through the node,");
        expComment("measured in bytes per second.");
	expComment("This parameter disables bandwidthLimit if non-zero.");
	l = params.getlong(id,Node.defaultOutputBandwidthLimit);
	if (expert) {
            qout.println("Enter bytes/sec to enable output bandwidth limiting");
            l = getNumber("?",l);
            if (l != 0 && l < 2500) {
                qout.println(l + " bytes/sec is too small to be useful to " +
                                 "the network");
                qout.println("Consider setting your Node's transient instead " +
                             "of limiting output bandwidth");
            }
        }
	out.println(((l==0) ? "# " : "") + id +"="+l);
    }

    public static void setParamTickerTime() {
	String id = "tickerTime";
	expComment("The number of milliseconds between ticks on the timer");
	long l = params.getlong(id,Core.defaultTickerTime);
	if (expert)
	    l = getNumber("?",l);
	out.println(id+"="+l);
    }

    public static void setParamHopsTimes() {
	String id = "hopTimeExpected";
	String id2 = "hopTimeDeviation";
	expComment("The expected time and standard deviation, in milliseconds, that it takes");
	expComment("a Freenet node to pass a message. These are used to calculate how long");
	expComment("the node should wait before assuming that a passed message is lost.");
	expComment("These are mostly here for debugging reasons - changing them will NOT");
	expComment("make requests come back faster for you.");
	long exp = params.getlong(id,Core.defaultHopTimeExpected);
	long dev = params.getlong(id2,Core.defaultHopTimeDeviation);
	if (expert) {
	    exp = getNumber("Expected value?",exp);
	    dev = getNumber("Standard Deviation?",dev);
	}
	out.println(id+"=" + exp);
	out.println(id2+"="+ dev);
    }

    public static void setParamConnectTimeout() {
	String id = "connectTimeout";
	expComment("How long to wait to connect to a host before giving up (in milliseconds)");
	long l = params.getlong(id,Core.defaultConnectTimeout);
	if (expert)
	    l = getNumber("?",l);
	out.println(id+ "=" + l);
    }

    public static void setParamConnectionTimeout() {
	String id= "connectionTimeout";
	expComment("How long to wait listen on an inactive connection before closing (if");
	expComment("reply address is known).");
	long l = params.getlong(id,Core.defaultConnectionTimeout);
	if (expert)
	    l = getNumber("?",l);
	out.println(id+"=" + l);
    }

    public static void setParamDoHandshake() {
	String id= "doHandshake";
	expComment("Leave this as \"yes\" unless you are doing timeout debugging.");
	boolean b = true;
	if (expert)
	    b = getBool("Should we handshake other nodes?",b);
	out.println(id+"=" + (b ? "yes" : "no"));
    }

    public static void setParamAuthTimeout() {
	String id  = "authTimeout";
	expComment("How long to wait for a handshake (in milliseconds)");
	long l = params.getlong(id,Core.defaultAuthTimeout);
	if (expert)
	    l = getNumber("?",l);
	out.println(id+"="+l);
    }

    public static void setParamHandshakeTimeout() {
	String id  = "handshakeTimeout";
	expComment("How long to wait for a handshake (in milliseconds)");
	long l = params.getlong(id,Core.defaultHandshakeTimeout);
	if (expert)
	    l = getNumber("?",l);
	out.println(id+"=" + l);
    }

    public static void setParamHandshakeLife() {
	String id = "handshakeLife";
	expComment("How long before a handshake expires (in milliseconds)");
	long l = params.getlong(id,Core.defaultHandshakeLife);
	if (expert)
	    l = getNumber("?",l);
	out.println(id+"=" + l);
    }

    public static void setParamMaximumConnectionThreads() {
	String id = "maximumConnectionThreads";
	expComment("Should we use thread-management?  If this number is defined and non-zero,");
	expComment("this specifies how many inbound connections can be active at once.");
	long l = params.getlong(id,Core.defaultMaximumConnectionThreads);
	if (expert)
	    l = getNumber("?",l);
	out.println(id+"=" + l);
    }

    public static void setParamMaxHopsToLive() {
	String id ="maxHopsToLive";
	expComment("The maximum number of hops to live to tolerate on Requests the node");
	expComment("passes. This does not directly effect your node, only the total load");
	expComment("on the network, so there is no reason to change it.");
	long l = params.getlong(id,Node.defaultMaxHopsToLive);
	if (expert)
	    l = getNumber("?",l);
	out.println(id+"=" + l);
    }

    public static void setParamKeyTypes() {
	out.println("# A list of keytype number class pairs for plugin keytypes");
	out.println("#keyTypes=");
    }

    public static void setParamLogging() {
	int[] vals = {Logger.ERROR, Logger.NORMAL, Logger.MINOR, 
		      Logger.DEBUGGING};
	String[] levels = new String[vals.length];
	StringBuffer sb;
	for (int i = 0 ; i < vals.length ; i++) {
	    for(sb = new StringBuffer(StandardLogger.priorityOf(vals[i]));
		sb.length() < 16;
		sb.append(' '));
	    levels[i] = sb.toString();
	}

	expComment("The error reporting threshold, one of:");
	if (expert)
	    qout.print("0");
	expComment("  " + levels[0] + "Errors only");
	if (expert)
	    qout.print("1");
	expComment("  " + levels[1] + "Report significant events");
	if (expert)
	    qout.print("2");
	expComment("  " + levels[2] + "Report minor events");
	if (expert)
	    qout.print("3");
	expComment("  " + levels[3] + "Report events only of relevance when debugging");
	long l=1 ;
	if (expert)
                  do {
	    l = getNumber("What level should we use?",
			  1);
	  } while (l < 0 || l >= levels.length);

	out.println("logging=" + levels[(int)l]);
    }

    public static void setParamLogFile() {
	String id = "logFile";
	expComment("The name of the log file ( \"no\" to log to screen (standard out))");
	String s = params.getParam(id,Node.defaultLogFile);
	if (expert)
	    getString("?",s);
	out.println(id + "="+s);
    }

    public static void setParamVerbosity() {
	String id = "verbosity";
	expComment("How verbose should the logging be? (1-5)");
	long l = params.getlong(id,Node.defaultVerbosity);
	if (expert) {
	    do {
		l = getNumber("?",l);
	    } while (l < 1 || l > 5);
	}
	out.println(id + "=" + l);
    }

    public static void setParamBufferSize() {
	String id = "bufferSize";
	expComment("What size should the buffers have when moving data?");
	long l = params.getlong(id,Core.defaultBufferSize);
	if (expert)
	    l = getNumber("?",l);
	out.println(id+"=" + l);

    }
    
    public static void setParamServiceFProxy() {
	String id = "services.fproxy.class";
	String def = "Freenet.contrib.fproxy.HttpHandlerServlet";
	String id1 = "services.fproxy.allowedhosts";
	String def1 = "127.0.0.1";
	String id2 = "services.fproxy.port";
	String id3 = "services.fproxy.doFiltering";
	String id4 = "services.fproxy.passthroughMimeTypes";
	String def3 = "text/plain,image/jpeg,image/gif,image/png,application/octet-stream," +
	    "application/zip,audio/basic,audio/midi,audio/mpeg,audio/x-aiff,audio/x-wav,image/cgm," +
	    "image/g3fax,image/tiff,image/x-xbinmap,image/x-xpixmap,image/x-xwindowdump,application/unknown," +
	    "image/pjpeg,application/x-debian-package";

	out.println("# Set to Freenet.contrib.fproxy.HttpHandlerServlet to start FProxy automatically with the node");
	boolean b = getBool("Should FProxy automatically be started with the node?",true);
	String s = b ? ((expert) ? getString("Location of FProxy ?",def):def ) : "";
  	out.println(id+"=" + s);

	if (b) {
	    String h = expert ? getString("Allowed hosts (comma delimited list)?",def1) : def1;
	    out.println(id1+"=" + h);
	
	    long l  = params.getlong(id2,8081);
	    if  (expert) {
		l = getNumber("FProxy port to use?", l);
	    }
	    out.println(id2+"=" + l);

	    boolean filter = params.getboolean(id3, true);
	    if (expert) {
		filter = getBool("Should FProxy do anonymity filtering?", filter);
	    }
	    out.println(id3 + "=" + filter);

	    String passthrough = params.getParam(id4, def3);
	    if (expert) {
		passthrough = getString("MIME types to skip for filtering (comma-delimited list)?", passthrough);
	    }
	    out.println(id4 + "=" + passthrough);
	}
    }

    public static void setParamServiceFCP() {
	String id = "services.fcp.class";
	String def = "Freenet.contrib.fcp.FCPHandler";
	String id1 = "services.fcp.allowedhosts";
	String def1 = "127.0.0.1";
	String id2 = "services.fcp.port";

	out.println("# Set to Freenet.contrib.fcp.FCPHandler to start FCP automatically with the node.");
	boolean b = getBool("Should FCP automatically be started with the node?",true);
	String s = b ? ((expert) ? getString("Location of FCP?",def):def ) : "";
	out.println(id+"=" + s);

	if (b) {
	    String h = expert ? getString("Allowed hosts (comma delimited list)?",def1) : def1;
	    out.println(id1+"=" + h);

	    long l = params.getlong(id2, 8082);
	    if  (expert) {
		l = getNumber("FCP port to use?", l);
	    }
	    out.println(id2+"=" + l);
	}
    }

    public static void setParamServiceXMLRPC() {
	String id = "services.xmlrpc.class";
	String def = "Freenet.contrib.xmlrpc.XmlRpcServlet";
	String id1 = "services.xmlrpc.allowedhosts";
	String def1 = "127.0.0.1";
	String id2 = "services.xmlrpc.port";

	out.println("# Set to Freenet.contrib.xmlrpc.XmlRpcServlet to start XML-RPC automatically with the node.");
	boolean b = getBool("Should XML-RPC automatically be started with the node?",true);
	String s = b ? ((expert) ? getString("Location of XML-RPC?",def):def ) : "";
	out.println(id+"=" + s);

	if (b) {
	    String h = expert ? getString("Allowed hosts (comma delimited list)?",def1) : def1;
	    out.println(id1+"=" + h);

	    long l = params.getlong(id2, 6690);
	    if  (expert) {
		l = getNumber("XML-RPC port to use?", l);
	    }
	    out.println(id2+"=" + l);
	}
    }
}


