package Freenet.support;
import Freenet.Core;

import java.io.Serializable;
/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.
 */

/**
 * This class represents an array which is of finite length
 * and can have information added to it in such a way that
 * if the array fills to capacity, the first information added
 * will be deleted.
 * <p>
 * It can be considered to be a FIFO of fixed length, where the
 * FIFO's top gets pushed off when it is filled to capacity.
 *
 * @version 2.0
 * @author <A HREF="mailto:ian@octayne.com">Ian Clarke</A>
 **/

public class CyclicArray implements Serializable
{
    public static void main(String[] args)
    {
	CyclicArray ca = new CyclicArray(10);
	for (int x=0; x<10; x++)
	    {
		ca.put(new Integer(x));
	    }
	System.out.println("Initial:\n"+ca);
	ca.insert(4, new Integer(99));
	System.out.println("insert 99 at 4:\n"+ca);
	ca.remove(4);
	System.out.println("remove at 4:\n"+ca);
    }

  /**
   * The array that contains the objects
   **/
  Object[] ar;
  /**
   * A pointer to the next item to be deleted, moves backwards
   * through array
   **/
  int ptr;

  /**
   *  Construct a CyclicArray of the given size.
   *
   * @param size the size of the array.
   */

  public CyclicArray(int size)
    {
      ar = new Object[size];
      ptr = size-1;
    }

  /**
   * Obtain the length of this CyclicArray
   * @return Length of array
   **/
  public int length()
    {
      return ar.length;
    }

  /**
   * Return the object at a given position in this CyclicArray.
   * @param objectPos The position of the object to return (0 is the most
   *                  recently added
   * @return The object at that position (or null if no object)
   **/
  public Object get(int objectPos)
    {
	return ar[objectPos];
    }

  /**
   * Adds an object to the top of this CyclicArray.  This may result in an
   * old object being deleted from this array.
   * @param newObj The object to add to this CyclicArray
   * @return Any object deleted from this array as a result of this call (or
   *         null if no object deleted)
   **/
  public Object put(Object newObj)
    {
        Object ret = ar[ar.length-1];
	if (ar[ar.length-1] != null)
	  Core.logger.log(this, "Object "+ar[ar.length-1]+" pushed off stack", Logger.DEBUG);
	System.arraycopy(ar, 0, ar, 1, ar.length-1);
	ar[0] = newObj;
	return ret;
    }

    /**
     * Inserts an object into the CyclicArray.  This may result in an old
     * object being deleted from the array.
     * @param pos The position where the object should be inserted (0 is
     *            the youngest object)
     * @param obj The object to be inserted
     **/
    public Object insert(int pos, Object obj)
    {
	Object ret = ar[ar.length-1];
	try
	    {
	      Core.logger.log(this, "Inserted "+obj+" to position "+pos+"/"+ar.length,
			      Logger.DEBUG);
	      if (ar[ar.length-1] != null)
		Core.logger.log(this, "Object "+ar[ar.length-1]+" pushed off stack", Logger.DEBUG);
	      System.arraycopy(ar, pos, ar, pos+1, (ar.length - pos)-1);
	      ar[pos] = obj;
	    }
	catch (ArrayIndexOutOfBoundsException e)
	    {
		if ((pos < 0) || (pos >= ar.length))
		    {
			Core.logger.log(this, "Attempted to insert to an invalid position "+
				   pos+" in array of length "+ar.length, Logger.ERROR);
		    }
	    }
	return ret;
    }

  /**
   * Removes an object from the specified position in this CyclicArray.
   * @param pos The position of the object to remove (0 is most
   *                  recently added)
   **/
  public void remove(int pos)
    {

	try
	    {
		System.arraycopy(ar, pos+1, ar, pos, (ar.length - pos)-1);
		ar[ar.length-1] = null;
	    }
	catch (ArrayIndexOutOfBoundsException e)
	    {
		if ((pos < 0) || (pos >= ar.length))
		    {
			Core.logger.log(this, "Attempted to remove from an invalid position "+
				   pos+" in array of length "+ar.length, Logger.ERROR);
		    }
	    }
    }

    public String toString()
    {
	StringBuffer sb = new StringBuffer();
	for (int x=0; x<ar.length; x++)
	    {
		sb.append(" "+x+":"+get(x)+"\n");
	    }
	return sb.toString();
    }
}


