package Freenet.thread;

/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.
*/

/**
 * This is the atomic element of the e-threads package.  An EThread is an 
 * extended thread; it should only be used in conjunction with a
 * {@link ThreadManager ThreadManager}.
 * EThreads cooperate with the ThreadManager to provide additional
 * helpful behaviors such as thread recycling, pooling, and priority management.
 * <p>
 * Currently an EThread will only work with a {@link ThreadPool ThreadPool}.
 *
 * @author Scott G. Miller
 */
public class EThread implements Runnable {
    protected Runnable code;
    protected ThreadManager host;
    protected Thread executor;

    /**
     * Constructs an EThread controlled by the given ThreadManager.
     *
     * @param host The ThreadPool controlling this EThread.
     */

    public EThread(ThreadPool host) {
	super();
	this.host=host;
    }

    /**
     * Gets the Thread instance for this EThread.
     *
     * @return The Thread instance for this EThread.
     */
    public Thread getThread() {
	return executor;
    }

    /**
     * Gets the ThreadManager controlling this EThread.
     *
     * @return The ThreadManager controlling this EThread.
     */
    public ThreadManager getHost() {
	return host;
    }

    /**
     * Allocates a new Thread with this EThread as the target and starts running it.
     */
    protected void begin() {
	if (executor==null){
	    executor=new Thread(this);
	    executor.start();
	}
	synchronized (this) {
	    notifyAll();
	}
    }

    /**
     * Sets a Runnable as the target of this EThread.
     *
     * @param r The Runnable to which to set the target.
     */
    protected void setCode(Runnable r) {
	code=r;
    }

    /**
     * Runs the target of this EThread. If the target of this EThread is an
     * {@link ERunnable ERunnable} then the target's {@link
     * ERunnable.setExecutionInstance setExecutionInstance} method is called
     * with this EThread as the parameter.
     * <p>
     * In any case, the target's run method is called.
     * <p>
     * When the target's run method returns (or throws a Throwable exception),
     * this EThread returns itself to its controlling ThreadManager, allowing
     * other targets to be assigned to it.
     */
    public void run() {
	for (;;) {
	    while (code == null) {
		synchronized(this) {
		    try {   
			wait();
		    } catch (InterruptedException ie) {}
		}
	    }

	    if (code instanceof ERunnable)
		((ERunnable)code).setExecutionInstance(this);
	    try {
		code.run();
	    } catch (Throwable t) {
	    } finally {
		code=null;
		if (!host.reclaim(this))
		    return;
	    }
	}
    }

    /**
     * Returns a String representation of this EThread.
     *
     * @return A String representation of this EThread.
     */
    public String toString() {
	return "EThread["+executor+','+code+','+host+']';
    }
}
