package Freenet.thread;

/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.
*/

import java.util.*;
import Freenet.support.*;

/**
 * Provides a bounded pool of EThreads and a means to request that jobs
 * be run on them.  Note that the term "job" is synonymous with Runnable.
 *
 * A ThreadPool represents a set of EThreads that are made available
 * for processing N jobs simultaneously, where N is the size of the pool.
 * If more jobs are inserted into the ThreadPool, they are put on hold
 * until one of the N running jobs completes, at which point the job
 * may run.  
 *
 * In this way one can bound the number of concurrently executing jobs
 * to a managable amount, which improves the efficiency of running jobs
 * and prevents a system from being brought down with an overload of tasks
 * (such as in a Denial of Service attack).  Unlike unmanaged spawning
 * of threads (1:1 thread to task), a ThreadPool will effiently work on
 * N jobs at once, putting additional requests on hold.
 *
 * @author Scott G. Miller
 */
public class ThreadPool extends Thread implements ThreadManager {
    protected int maxThreads, maxJobs, maxPoolThreads, runningCount; 
    protected ThreadGroup poolGroup;
    protected BlockingQueue jobs;
    protected BlockingStack threads;
    protected boolean run=true;

    /**
     * Creates a ThreadPool with a fixed number of concurrently
     * executable jobs.  This also allocates a job queue, so that
     * an unlimited number of new jobs can be inserted in a non-blocking
     * manner even if all
     * maxThreads threads are occupied.
     *
     * @param maxThreads The number of jobs that can be active at any
     * one time.
     */
    public ThreadPool(int maxThreads) {
	this(5, maxThreads, Integer.MAX_VALUE);
    }

    /**
     * Creates a ThreadPool with a fixed number of concurrently
     * executable jobs.  This also allocates a job queue, so that
     * an unlimited number of new jobs can be inserted in a non-blocking
     * manner even if all maxThreads threads are occupied.
     *
     * @param tg      The ThreadGroup to use for the pool and all it's 
     * children.
     * @param maxThreads The number of jobs that can be active at any
     * one time.
     */
    public ThreadPool(ThreadGroup tg, int maxThreads) {
	this(tg, 5, maxThreads, Integer.MAX_VALUE);
    }


    /**
     * Creates a ThreadPool with a fixed number of concurrently
     * executable jobs.  This also allocates a job queue, so that
     * new jobs can be inserted in a non-blocking manner even if all
     * maxThreads threads are occupied. This ThreadPool allows up to a 
     * total of maxJobs running and queued jobs.
     *
     * @param maxThreads The number of jobs that can be active at any
     * one time.
     * @param maxJobs The maximum total number of allowable jobs.  {@link
     * run run()} returns false to turn away additional jobs past this point.
     */
    public ThreadPool(int maxPool, int maxThreads, int maxJobs) {
	this(new ThreadGroup("ThreadPool"),maxPool, maxThreads, 
	     maxJobs);
    }

    /**
     * Creates a ThreadPool with a fixed number of concurrently
     * executable jobs.  This also allocates a job queue, so that
     * new jobs can be inserted in a non-blocking manner even if all
     * maxThreads threads are occupied. This ThreadPool allows up to a 
     * total of maxJobs running and queued jobs.
     *
     * @param tg      The ThreadGroup to use for the pool and all it's 
     * children.
     * @param maxThreads The number of jobs that can be active at any
     * one time.
     * @param maxJobs The maximum total number of allowable jobs.  {@link
     * run run()} returns false to turn away additional jobs past this point.
     */
    public ThreadPool(ThreadGroup tg, int maxPool, int maxThreads, 
		      int maxJobs) {
	super("ThreadPool");
        setDaemon(true);
	this.poolGroup=tg;
	jobs=new BlockingQueue();
	threads=new BlockingStack();
	this.maxPoolThreads=maxPool;
	this.maxThreads=maxThreads;
	this.maxJobs=maxJobs;
	fillThreadstack();
    }

    //(Re)fills the thread-queue with maxThreads EThreads
    private void fillThreadstack() {
	while (threads.size()<maxPoolThreads) {
	    threads.push(new EThread(this));
	}
    }

    /**
     * Called by an EThread to indicate that it has finished processing
     * a job and may be reclaimed by another job.
     *
     * @param e The EThread to reclaim.
     */
    public boolean reclaim(EThread e) {
	boolean rv=false;
	if (maxThreads == 0 || threads.size()<maxPoolThreads) {
	    threads.push(e);
	    rv=true;
	} 
	synchronized(this) {
	    runningCount--;
	}
	synchronized(threads) {
	    threads.notifyAll();
	}
	return rv;
    }

    private int runningThreads() {
	return runningCount;
    }

    /**
     * Queue a job for running.
     *
     * @param r A job to queue.
     *
     * @return True if the job was successfully queued, or false
     * if this ThreadPool is not running or if the maximum number of
     * allowable jobs for this ThreadPool has been reached.
     */
    public boolean run(Runnable r) {
	if (run) {
	    //	    System.err.println("["+runningThreads()+','+jobs.size()+']');
	    if (runningThreads() + jobs.size() < maxJobs) {
		jobs.enqueue(r);
		return true;
	    }
	}
	return false;
    }

    public void blockingRun(Runnable r) {
	if (run) {
	    //	    System.err.println("["+runningThreads()+','+jobs.size()+']');
	    while ( runningThreads() + jobs.size() >= maxJobs) {
		synchronized(threads) {
		    try {
			threads.wait();
		    } catch (InterruptedException ie) {}
		}
	    }
	    jobs.enqueue(r);
	}
    }

    /**
     * Wait for all queued jobs to start running.
     */

    public void flush() {
	while (!jobs.isEmpty()) {
	    synchronized(jobs) {
		try {
		    jobs.wait();
		} catch (InterruptedException ie) {}
	    }
	}
    }

    /**
     * Halts this thread manager. Causes all running jobs to stop running.
     */
    public void halt() {
	run=false;
        interrupt();
    }

    /**
     * Activates this ThreadPool. This ThreadPool will grab a queued job,
     * associate it with an EThread, and tell the EThread to begin. It does
     * this over and over again, until it is {@link halt halt}ed.
     */

    public void run() {
	while (run) {
	    try {

		// Block until both an EThread is available and a job is
		// queued.


		EThread host=null;
		if (threads.isEmpty() && 
		    ((runningThreads() < maxThreads) || (maxThreads==0))) { 
		    // Make a new thread.
		    host=new EThread(this);
		}
		else {
		    // Use an old thread, possibly waiting if none
		    // is immediately available.
		    host=(EThread)threads.pop();
		}

		Runnable job=(Runnable)jobs.dequeue();
		host.setCode(job);
		host.begin();

		synchronized(this) {
		    runningCount++;
		}

		// Sends a signal to anyone waiting in flush that
		// a job has started to run.

		synchronized(jobs) {
		    jobs.notifyAll();
		}
	    } catch (InterruptedException ie) {}
	}
    } 
}    

