/*
	This file is part of OpenFuniter.

	complex.c V2.2.0 - complex math routines.

	Copyright (C) 1995-2002 Stijn Wolters.
    Original idea: Ernic Kamerich (University of Nijmegen)

	See README for contact information.

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version. See

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

	See COPYING for more information.
*/

#include "complex.h"

/*
**	Constructor: create a complex number
*/

TComplex Complex(double RealPart, double ImagPart)
{
	TComplex c;

	c.r = RealPart;
	c.i = ImagPart;

	return(c);
}

/*
**	Return the imaginary-part of a complex-number.
*/

double Imag(TComplex c)
{
	return(c.i);
}

/*
**	Return the real-part of a complex-number.
*/

double Real(TComplex c)
{
	return(c.r);
}

/*
**	Complex negation.
*/

TComplex CNeg(TComplex c)
{
	TComplex 	Res;

	Res.r = -c.r;
	Res.i = -c.i;

	return(Res);
}

/*
**	Complex addition (c1 + c2).
*/

TComplex CAdd(TComplex c1, TComplex c2)
{
	TComplex 	Res;

	Res.r = c1.r + c2.r;
	Res.i = c1.i + c2.i;

	return(Res);
}

/*
**	Complex subtraction (c1 - c2).
*/

TComplex CSub(TComplex c1, TComplex c2)
{
	TComplex 	Res;

	Res.r = c1.r - c2.r;
	Res.i = c1.i - c2.i;

	return(Res);
}

/*
**	Multiply two complex-numbers (c1 * c2).
*/

TComplex CMul(TComplex c1, TComplex c2)
{
	TComplex 	Res;

	Res.r = c1.r * c2.r - c1.i * c2.i;
	Res.i = c2.r * c1.i + c2.i * c1.r;

	return(Res);
}

/*
**	Divide two complex numbers (c1 / c2).
*/

TComplex CDiv(TComplex c1, TComplex c2)
{
	TComplex 	Res;
	double		Absc2;

	Absc2 = c2.r * c2.r + c2.i * c2.i;

	if(Absc2 > 0) {
		Res.r = c1.r * c2.r + c1.i * c2.i / Absc2;
		Res.i = c1.i * c2.r - c1.r * c2.i / Absc2;
	}
	else {
		Res.r = 1.0;
		Res.i = 1.0;
	}

	return(Res);
}

/*
**	Return the absolute value of c.
*/

double CAbs(TComplex c)
{
	return(c.r * c.r + c.i * c.i);
}

/*
**	Complex root.
*/

TComplex CSqrt(TComplex c)
{
	double r, p;

	if(c.r == 0.0 && c.i == 0.0) return(c);
	if(c.r == 0.0) return(c);

	p = atan2(c.i, c.r) / 2.0;
	r = sqrt(sqrt(c.r * c.r + c.i * c.i));
	return(Complex(r * cos(p), r * sin(p)));
}
