/*
	This file is part of XFuniter.

	coords.c V2.2.0 - drawing a grid, converting coordinates and zoom.

	Copyright (C) 1995-2002 Stijn Wolters.
    Original idea: Ernic Kamerich (University of Nijmegen)

	See README for contact information.

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version. See

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

	See COPYING for more information.
*/

#include "coords.h"

TCoords *CreateCoords(double XMin, double YMin, double XMax, double YMax,
	unsigned int XStart, unsigned int YStart, unsigned int XEnd,
	unsigned int YEnd, int Mode)
{
	TCoords *Coords;

	if((Coords = (TCoords *) malloc(sizeof(TCoords))) == NULL) return NULL;

    SetCoords(Coords, XMin, YMin, XMax, YMax);
	Coords->XStart = XStart;
	Coords->YStart = YStart;
	Coords->XEnd = XEnd;
	Coords->YEnd = YEnd;
	Coords->Mode = Mode;

	return(Coords);
}

int SetCoords(TCoords *Coords, double XMin, double YMin, double XMax, 
    double YMax)
{
	if(Coords != NULL) {
        Coords->XMin = XMin;
	    Coords->YMin = YMin;
	    Coords->XMax = XMax;
	    Coords->YMax = YMax;
	    Coords->Width = Coords->XMax - Coords->XMin;
	    Coords->Height = Coords->YMax - Coords->YMin;
        return 1;
    }
    return 0;
}
    
void FreeCoords(TCoords *Coords)
{
	if(Coords != NULL) free(Coords);
}

int ZoomBox(BITMAP *Bitmap, FONT *Font, TCoords *Coords, char *HelpText[])
{
	int             MouseXLast, MouseYLast, DragBoxCol, Steps = 1, ch, Key, 
                    ExtKey, XMickeys, YMickeys;
	double          AspectRatio, x1, y1, x2, y2, old_x1, old_x2, old_y1,
					old_y2;
	TCoords			*NewCoords;

	/* Show help text */
    
    if(HelpText != NULL) {
        InfoBox(HelpText, 4);
        HelpText[0] = HelpText[5];
    }
    
	/* Set color for dragbox to white (last color anyway) */

	DragBoxCol = makecol(255, 255, 255);
    
	/* Calculate aspectratio for rect */

	AspectRatio = ((double) Coords->YEnd + 1.0) / ((double) Coords->XEnd + 1.0);

	/* Initialize some shortcuts for often used variables */

	x1 = Coords->XStart; 	y1 = Coords->YStart;
	x2 = Coords->XEnd; 		y2 = Coords->YEnd;

    /* Show mouse */
    
    show_mouse(screen);
    
	/*
	**	when no key is pressed we assume the mouse is used to select an area.
	*/
    
	while(!keypressed()) {
	    
    	poll_mouse(); /* without braces? */
    
		if(mouse_b & 1) {
			xor_mode(TRUE);
            scare_mouse();
			rect(Bitmap, x1, y1, x2, y2, makecol(255, 255, 255));
            unscare_mouse();
			poll_mouse();
			x1 = mouse_x;
			y1 = mouse_y;
			while(mouse_b & 1) {
				MouseXLast = mouse_x;
				MouseYLast = mouse_y;
			    get_mouse_mickeys(&XMickeys, &YMickeys);
                if((XMickeys != 0) || (YMickeys != 0)) {
				    poll_mouse();
				    xor_mode(TRUE);
                    scare_mouse();
                    acquire_bitmap(Bitmap);
				    rect(Bitmap, x1, y1, mouse_x, mouse_y, 
                    makecol(255, 255, 255));
                    rect(Bitmap, x1, y1, MouseXLast, MouseYLast,
                    makecol(255, 255, 255));
                    release_bitmap(Bitmap);
                    vsync();
                    unscare_mouse();
                }
                release_screen();
                vsync();
			}
			x2 = (double) mouse_x;

			if(mouse_y > y1) {
				y2 = y1;
				y1 = (double) mouse_y;
			}
			else
				y2 = (double) mouse_y;
		}
	}
    
	do {
		if(!keypressed()) {

			poll_mouse();

			if(mouse_b & 1) {
				xor_mode(TRUE);
                scare_mouse();
				rect(Bitmap, x1, y1, x2, y2, makecol(255, 255, 255));
                unscare_mouse();
				poll_mouse();
				x1 = mouse_x;
				y1 = mouse_y;
				while(mouse_b & 1) {
                    acquire_bitmap(Bitmap);
					MouseXLast = mouse_x;
					MouseYLast = mouse_y;
			        get_mouse_mickeys(&XMickeys, &YMickeys);
                    if((XMickeys != 0) || (YMickeys != 0)) {
					    poll_mouse();
				        xor_mode(TRUE);
                        scare_mouse();
                        acquire_bitmap(Bitmap);
					    rect(Bitmap, x1, y1, mouse_x, mouse_y, 
                        makecol(255, 255, 255));
				        xor_mode(TRUE);
					    rect(Bitmap, x1, y1, MouseXLast, MouseYLast, 
                        makecol(255, 255, 255));
                        release_bitmap(Bitmap);
                        vsync();
                        unscare_mouse();
                    }
				}
				x2 = (double) mouse_x;
				if(mouse_y > y1) {
					y2 = y1;
					y1 = (double) mouse_y;
				}
				else
					y2 = (double) mouse_y;
			}
		}
		else {

			Key = readkey(); 
            ExtKey = Key >> 8;
            
			if(ExtKey != KEY_F1) {
				old_x1 = x1; old_x2 = x2; old_y1 = y1; old_y2 = y2;
				rect(Bitmap, old_x1, old_y1, old_x2, old_y2,
                makecol(255, 255, 255));
			}

			xor_mode(TRUE);
			switch(ExtKey) {
            	case KEY_PGUP:
					x1+=Steps; y1+=(AspectRatio * (double) Steps);
					x2-=Steps; y2-=(AspectRatio * (double) Steps);
					break;
				case KEY_PGDN:
					x1-=Steps; y1-=(AspectRatio * (double) Steps);
					x2+=Steps; y2+=(AspectRatio * (double) Steps);
					break;
				case KEY_UP:
					y1-=Steps; y2-=Steps;
					break;
				case KEY_DOWN:
					y1+=Steps; y2+=Steps;
					break;
				case KEY_RIGHT:
					x1+=Steps; x2+=Steps;
					break;
				case KEY_LEFT:
					x1-=Steps; x2-=Steps;
					break;
				case KEY_F1:
					xor_mode(FALSE);
                    if(HelpText != NULL) InfoBox(HelpText, 4);
					xor_mode(TRUE);
					break;
				default:
					if((ExtKey >= KEY_0) && (ExtKey <= KEY_9))
						Steps = ((ExtKey == KEY_0) ? (KEY_9 + 1) : (ExtKey)) - 
                        KEY_0;
					break;
			}
            
			if(ExtKey != KEY_F1) {
				rect(Bitmap, x1, y1, x2, y2,
                makecol(255, 255, 255));
				xor_mode(TRUE);
			}
		}
	} while(ExtKey != KEY_ESC && ExtKey != KEY_ENTER);

	xor_mode(FALSE);

	if(ExtKey == KEY_ENTER) {
		if((NewCoords = (TCoords *) malloc(sizeof(TCoords))) != NULL) {
			memcpy(NewCoords, Coords, sizeof(TCoords));
			NewCoords->XMin = R2M_X(*Coords, x1);
			NewCoords->XMax = R2M_X(*Coords, x2);

			if(y1 < y2) {
				NewCoords->YMin = R2M_Y(*Coords, y2);
				NewCoords->YMax = R2M_Y(*Coords, y1);
			}
			else if(y1 > y2) {
				NewCoords->YMin = R2M_Y(*Coords, y1);
				NewCoords->YMax = R2M_Y(*Coords, y2);
			}
			NewCoords->Width = NewCoords->XMax - NewCoords->XMin;
			NewCoords->Height = NewCoords->YMax - NewCoords->YMin;

			memcpy(Coords, NewCoords, sizeof(TCoords));
			free(NewCoords);
		}
		else return 0;
	}
    show_mouse(NULL);
	return 1;
}

int DrawGrid(BITMAP *Bitmap, const FONT *Font, TCoords *Coords)
{
	double		TmpXMin, TmpYMin, TmpXMax, TmpYMax, TmpXSize, TmpYSize,
				XFac, YFac, XStep, YStep, XPos, YPos;
	int			XMid, YMid, PowN, PowM, XDigits, YDigits, XReal, YReal,
                UsedColor, OldTextMode;
    BITMAP      *BitmapDotLine;
    
    acquire_bitmap(Bitmap);
    vsync();

	Coords->Width = Coords->XMax - Coords->XMin;
	Coords->Height = Coords->YMax - Coords->YMin;
    
	/* 
    **  Set the max. color, default it's white in this case 
    **  Set textmode to transparent.
    **  Create a color 'white'.
    **  Create a dot-pattern for the grid.
    */

	OldTextMode = text_mode(-1);
    UsedColor = makecol(255, 255, 255);
    
    if(Coords->Mode & CO_GRID) {
        BitmapDotLine = create_bitmap(2, 2);
        putpixel(BitmapDotLine, 0, 0, UsedColor);
        putpixel(BitmapDotLine, 1, 1, UsedColor);
    }
    
	/* Calculate position for axes, preferably they should be at 0.0 */

	if(Coords->XMin < 0.0 && Coords->XMax > 0.0)
		XMid = (int) M2R_X(*Coords, 0.0);
	else
		XMid = 0;

	if(Coords->YMin < 0.0 && Coords->YMax > 0.0)
		YMid = (int) M2R_Y(*Coords, 0.0);
	else
		YMid = 0;

	/* Draw lines for X and Y axes */

	if(Coords->Mode & CO_YCOORDS)
		hline(Bitmap, (int) M2R_X(*Coords, Coords->XMin), YMid, 
        (int) M2R_X(*Coords, Coords->XMax), UsedColor);

	if(Coords->Mode & CO_XCOORDS)
		vline(Bitmap, XMid, (int) M2R_Y(*Coords, Coords->YMin), 
        (int) M2R_Y(*Coords, Coords->YMax), UsedColor);
        
	TmpXSize = Coords->Width;
	PowN = 0;
	if(Coords->Width <= 10.0) {
		while(((int) TmpXSize) == 0) {
			TmpXSize *= 10.0;
			PowN++;
		}
	}
	else {
		TmpXSize = (long int) (TmpXSize + 0.5);
		while(((long int) TmpXSize) != 0L) {
			TmpXSize /= 10.0;
			PowN--;
		}
		TmpXSize *= 10.0;
		PowN++;
	}

	TmpYSize = Coords->Height;
	PowM = 0;
	if(Coords->Height <= 10.0) {
		while(((int) TmpYSize) == 0) {
			TmpYSize *= 10.0;
			PowM++;
		}
	}
	else {
		TmpYSize = (long int) (TmpYSize + 0.5);
		while(((long int) TmpYSize) != 0L) {
			TmpYSize /= 10.0;
			PowM--;
		}
		TmpYSize *= 10; PowM++;
	}

	/* Round the original coordinates and put them in temp. variables */

	TmpXMin = ((long int) (Coords->XMin * pow(10.0, PowN))) *
		pow(10.0, -PowN) - 1.0 * pow(10.0, -PowN);
	TmpYMin = ((long int) (Coords->YMin * pow(10.0, PowM))) *
		pow(10.0, -PowM) - 1.0 * pow(10.0, -PowM);

	TmpXMax = ((long int) (Coords->XMax * pow(10.0, PowN))) *
		pow(10.0, -PowN) + 1.0 * pow(10.0, -PowN);
	TmpYMax = ((long int) (Coords->YMax * pow(10.0, PowM))) *
		pow(10.0, -PowM) + 1.0 * pow(10.0, -PowM);

	/* Determine the multiply factor which is used to calc. the stepsize */

	if(TmpXSize <= 2.0) XFac = 5.0; else if(TmpXSize <= 5.0) XFac = 2.0; else
		XFac = 1.0;
	if(TmpYSize <= 2.0) YFac = 5.0; else if(TmpYSize <= 5.0) YFac = 2.0; else
		YFac = 1.0;

	XStep = (TmpXSize / (XFac * TmpXSize)) * pow(10.0, (double) -PowN);
	if(Coords->Height <= 10.0) XDigits = PowN + 1; else XDigits = 0;
	YStep = (TmpYSize / (YFac * TmpYSize)) * pow(10.0, (double) -PowM);
	if(Coords->Height <= 10.0) YDigits = PowM + 1; else YDigits = 0;

	if(Coords->Mode & CO_XCOORDS) {
		XPos = TmpXMin;
		while(XPos <= TmpXMax) {
			if(XPos >= Coords->XMin && XPos <= Coords->XMax) {
				XReal = M2R_X(*Coords, XPos);
				if(Coords->Mode & CO_GRID) {
                    drawing_mode(DRAW_MODE_COPY_PATTERN, BitmapDotLine, 0, 0);
                    vline(Bitmap, XReal, Coords->YStart, Coords->YEnd, 
                    UsedColor);
                    solid_mode();
				}
				if(Coords->Mode & CO_TEXT) {
					textprintf(Bitmap, Font, XReal, YMid + 6, 
                    UsedColor, "%.*f", XDigits, XPos);
				}
				vline(Bitmap, XReal, YMid - 3, YMid + 3, UsedColor);
			}
			XPos += XStep;
		}
	}

	if(Coords->Mode & CO_YCOORDS) {
		YPos = TmpYMin;
		while(YPos <= TmpYMax) {
			if(YPos >= Coords->YMin && YPos <= Coords->YMax) {
				YReal = M2R_Y(*Coords, YPos);
				if(Coords->Mode & CO_GRID) {
                    drawing_mode(DRAW_MODE_COPY_PATTERN, BitmapDotLine, 
                    0, 0);
					hline(Bitmap, Coords->XStart, YReal, Coords->XEnd, 
                    UsedColor);
                    printf("X: %d , %d, %d\n", Coords->XStart, YReal,
                    Coords->XEnd);
                    solid_mode();
				}
				if(Coords->Mode & CO_TEXT) {
					textprintf(Bitmap, Font, XMid + 6, YReal - 3, 
                    UsedColor, "%.*f", YDigits, YPos);
				}
				hline(Bitmap, XMid - 3, YReal, XMid + 3, UsedColor);
			}
			YPos += YStep;
		}
	}
    
    text_mode(OldTextMode);
    
    release_bitmap(Bitmap);
    vsync();

    if(Coords->Mode & CO_GRID) destroy_bitmap(BitmapDotLine);
    
	return 0;
}
