/*
	This file is part of XFuniter.

	menus.c V2.2.0 - all menu routines.

	Copyright (C) 1995-2002 Stijn Wolters.
    Original idea: Ernic Kamerich (University of Nijmegen).
    
	See README for contact information.

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version. See

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

	See COPYING for more information.
*/

#include "menus.h"
#include "texts.h"

static int              Functiontype;
static TRealExp         *RealExp;
static TComplexExp      *ComplexExp;

int ParametersCallback(DIALOG *d)
{
    static int      LastFocus = 0;
    unsigned int    RGraph = RealExp->Graph, CGraph = ComplexExp->Graph;

    if(LastFocus != find_dialog_focus(d)) {

        /* 'Real' parameters */

        if(!Functiontype) {
            switch(LastFocus) {
                case 1:
                    if(RGraph == GR_R_STEP || RGraph ==  GR_R_ORBITS_X) {
                        if(!RealExp->GraphPar[RGraph].EscapeValueChanged) {
                            sprintf(d[8].dp, "%g",
                            CalculateRealFnEscapeValue(RealExp));
                            object_message(&d[3], MSG_DRAW, 0);
                        }
                    }
                    break;
                case 2:
                    if(RGraph != GR_R_ORBITS_X) {
                        if(RealExp->GraphPar[RealExp->Graph].xbegin1 !=
                        atof(d[2].dp)) {
                            strcpy(d[3].dp, d[2].dp);
                            object_message(&d[3], MSG_DRAW, 0);
                        }
                    }
                    else {
                        if(!RealExp->GraphPar[RGraph].EscapeValueChanged) {
                            sprintf(d[8].dp, "%g",
                            CalculateRealFnEscapeValue(RealExp));
                            object_message(&d[8], MSG_DRAW, 0);
                        }
                    }
                    break;
                case 4:
                    if(RGraph == GR_R_STEP) {
                        strcpy(d[6].dp, d[4].dp);
                        if(!RealExp->GraphPar[RGraph].EscapeValueChanged) {
                            sprintf(d[8].dp, "%g",
                            CalculateRealFnEscapeValue(RealExp));
                            object_message(&d[8], MSG_DRAW, 0);
                        }
                    }
                    break;
                case 5:
                    if(RGraph == GR_R_STEP) {
                        strcpy(d[7].dp, d[5].dp);
                        if(!RealExp->GraphPar[RGraph].EscapeValueChanged) {
                            sprintf(d[8].dp, "%g",
                            CalculateRealFnEscapeValue(RealExp));
                            object_message(&d[8], MSG_DRAW, 0);
                        }
                    }
                    break;
                case 6:
                case 7:
                    if(!RealExp->GraphPar[RGraph].EscapeValueChanged) {
                        sprintf(d[8].dp, "%g",
                        CalculateRealFnEscapeValue(RealExp));
                        object_message(&d[8], MSG_DRAW, 0);
                    }
                    break;
                case 8:
                    /* Set to FALSE we force re-calculation of EscapeValue */

                    RealExp->GraphPar[RGraph].EscapeValueChanged = FALSE;

                    /*
                    **  If the calculated value isn't equal to that which the
                    **  user entered, the editfield/text is painted red and
                    **  the 'changed' flag is set to TRUE (user changed value).
                    */

                    if(CalculateRealFnEscapeValue(RealExp) != atof(d[8].dp)) {
                        d[8].bg = 1;
                        RealExp->GraphPar[RGraph].EscapeValueChanged = TRUE;
                    }

                    /*
                    **  In the other case the old color is restored, and so is
                    **  the flag.
                    */

                    else {
                        d[8].bg = 0;
                        RealExp->GraphPar[RGraph].EscapeValueChanged = FALSE;
                    }
                    object_message(&d[8], MSG_DRAW, 0);
                    break;
                default:
                    break;
            }
        }

        /* 'Complex' parameters */

        if(!Functiontype) {
            switch(LastFocus) {

                case 1:
                case 2:
                case 6:
                case 7:
                    if(!ComplexExp->GraphPar[CGraph].EscapeValueChanged) {
                        sprintf(d[8].dp, "%g",
                        CalculateComplexFnEscapeValue(ComplexExp));
                        object_message(&d[3], MSG_DRAW, 0);
                    }
                    break;
                case 8:
                    /* Set to FALSE we force re-calculation of EscapeValue */

                    RealExp->GraphPar[RGraph].EscapeValueChanged = FALSE;

                    /*
                    **  If the calculated value isn't equal to that which the
                    **  user entered, the editfield/text is painted red and
                    **  the 'changed' flag is set to TRUE (user changed value).
                    */

                    if(CalculateComplexFnEscapeValue(ComplexExp) !=
                    atof(d[8].dp)) {
                        d[8].bg = 1;
                        ComplexExp->GraphPar[CGraph].EscapeValueChanged = TRUE;
                    }

                    /*
                    **  In the other case the old color is restored, and so is
                    **  the flag.
                    */

                    else {
                        d[8].bg = 0;
                        ComplexExp->GraphPar[CGraph].EscapeValueChanged = FALSE;
                    }
                    object_message(&d[8], MSG_DRAW, 0);
                    break;
                default:
                    break;
            }
        }
    }
    LastFocus = find_dialog_focus(d);
    return D_O_K;
}

int MainMenu(TRealExp *RealExp0, TComplexExp *ComplexExp0)
{
    int     Result;
    MENU    MnuMainMenuItems[] = {

                {   MainMenuStr[ 0], MnuFunctiontype,    NULL, 0, NULL },
                {   MainMenuStr[ 1], MnuGraph,           NULL, 0, NULL },
                {   MainMenuStr[ 2], MnuDiagram,         NULL, 0, NULL },
                {   MainMenuStr[ 3], MnuFunction,        NULL, 0, NULL },
                {   MainMenuStr[ 4], MnuParameters,      NULL, 0, NULL },
                {   MainMenuStr[ 5], MnuDraw,            NULL, 0, NULL },
                {   MainMenuStr[ 6], MnuSettings,        NULL, 0, NULL },
                {   MainMenuStr[ 7], NULL,               NULL, 0, NULL },
                {   MainMenuStr[ 8], MnuAbout,           NULL, 0, NULL },
                {   MainMenuStr[ 9], MnuQuit,            NULL, 0, NULL },
                {   MainMenuStr[10], NULL,               NULL, 0, NULL }, { 0 }
            },

            MnuMainMenu[] = {

                {   MainMenuStr[11], NULL, MnuMainMenuItems, 0, NULL }, { 0 }
            };

    DIALOG  DlgMainMenu[] = {

    {
        d_menu_proc,
        1, SCREEN_H - (text_height(font) + 5),
        0, 0, 0, 0, 0, 0, 0, 0, MnuMainMenu, NULL, NULL
    },
    
    { 0 } };

    /* Set pointer to real and complex experiments */

    RealExp = RealExp0;
    ComplexExp = ComplexExp0;

    /* Draw statusbar */

    StatusBar(-1, -1, NULL);

    Result = do_dialog(DlgMainMenu, find_dialog_focus(DlgMainMenu));
    if(Result == -1) {
        if(alert("Quit?", NULL, NULL, "OK", "Cancel", 13, 27) == 2)
            MainMenu(RealExp0, ComplexExp0);
    }
    return(Result);
}

int MnuFunctiontype(void)
{
    int     i, w = 540, h = 320, rh = text_height(font) * 2, fgcol = 0, 
            bkcol = 15, MaxItemLen = GetHighestStrLen(FnTypeMenuStr, 1, 2);
    DIALOG  DlgFunctiontype[4] = { { 0 }, { 0 }, { 0 }, { 0 } };
                
    DlgFunctiontype[0].proc = d_alt_box_proc;
    DlgFunctiontype[0].x = (SCREEN_W - w) / 2;
    DlgFunctiontype[0].y = (SCREEN_H - h) / 2;
    DlgFunctiontype[0].w = w;
    DlgFunctiontype[0].h = h;
    DlgFunctiontype[0].fg = fgcol;
    DlgFunctiontype[0].bg = bkcol;
    DlgFunctiontype[0].dp = FnTypeMenuStr[0];

    for(i = 1; i <= 2; i++) {
        DlgFunctiontype[i].proc = d_menubar_proc;
        DlgFunctiontype[i].x = DlgFunctiontype[0].x + (w - MaxItemLen) / 2;
        DlgFunctiontype[i].y = DlgFunctiontype[0].y + (h - rh * 2) / 2 + i * rh;
        DlgFunctiontype[i].w = MaxItemLen;
        DlgFunctiontype[i].h = text_height(font);
        DlgFunctiontype[i].fg = fgcol;
        DlgFunctiontype[i].bg = bkcol;
        DlgFunctiontype[i].key = FnTypeHkStr[i - 1];
        DlgFunctiontype[i].d2 = 0;
        DlgFunctiontype[i].dp = FnTypeMenuStr[i];
    }

    /* Set default value and show dialog */

    DlgFunctiontype[(Functiontype) ? 2 : 1].flags |= D_SELECTED;
    
    popup_dialog(DlgFunctiontype, (Functiontype) ? 2 : 1);

    /* Process result */

    Functiontype = !(DlgFunctiontype[1].flags & D_SELECTED);

    return(0);
}

int MnuGraph(void)
{
    int     i, w = 540, h = 320, rh = text_height(font) * 2, fgcol = 0, 
            bkcol = 15, MaxItemLen = GetHighestStrLen(GraphMenuStr, 1, 4);
    DIALOG  DlgGraph[] = { { 0 }, { 0 }, { 0 }, { 0 }, { 0 }, { 0 } };
        
    DlgGraph[0].proc = d_alt_box_proc;
    DlgGraph[0].x = (SCREEN_W - w) / 2;
    DlgGraph[0].y = (SCREEN_H - h) / 2;
    DlgGraph[0].w = w;
    DlgGraph[0].h = h;
    DlgGraph[0].fg = fgcol;
    DlgGraph[0].bg = bkcol;
    DlgGraph[0].dp = GraphMenuStr[0];
    
    for(i = 1; i <= 4; i++) {
        DlgGraph[i].proc = d_menubar_proc;
        DlgGraph[i].x = DlgGraph[0].x + (w - MaxItemLen) / 2;
        DlgGraph[i].y = DlgGraph[0].y + (h - rh * 4) / 2 + i * rh;
        DlgGraph[i].w = MaxItemLen;
        DlgGraph[i].h = text_height(font);
        DlgGraph[i].fg = fgcol;
        DlgGraph[i].bg = bkcol;
        DlgGraph[i].key = GraphHkStr[(Functiontype && (i == 3)) ? 4 : i - 1];
        DlgGraph[i].d2 = 0;
        DlgGraph[i].dp = GraphMenuStr[(Functiontype && (i == 3)) ? 5 : i];
    }
 
    /* 
    **  Set menu items for real graph
    **  Because item 2 is hidden for real graphs, I move the first (which is
    **  identical for real and complex in name) to the second item to prevent
    **  a gap (0, 2, 3) which causes a problem when the item-number is used in
    **  the GraphPar array.
    */

    if(!Functiontype) {
        DlgGraph[2].dp = DlgGraph[1].dp;
        DlgGraph[1].flags |= D_HIDDEN;
    }

    /* Set current value as default and show dialog */
    
    DlgGraph[((Functiontype) ? (ComplexExp->Graph + 1) : (RealExp->Graph + 2)) 
    ].flags |= D_SELECTED;

    popup_dialog(DlgGraph, 
    (Functiontype) ? (ComplexExp->Graph + 1) : (RealExp->Graph + 2));

    /* Process result */
    
    for(i = 1; i <= 4; i++) {
        if(DlgGraph[i].flags & D_SELECTED) {
            (Functiontype) ? (ComplexExp->Graph = i - 1) : (RealExp->Graph = i -
            2);
            break;
        }
    }

    /* Just to be safe... */

    if(i < 0) i = 0;
    
    return 0;
}    

int MnuDiagram(void)
{
    int     w = 540, h = 320, rh = text_height(font) * 2, fgcol = 0, bkcol = 15,
            j, i = ((Functiontype) ? ((ComplexExp->Graph < 2) ? 4 : 8) : 0), 
            MaxItemLen = (Functiontype) ? 
            ((ComplexExp->Graph < 2) ? GetHighestStrLen(DiagramMenuStr, 6, 9) :
            GetHighestStrLen(DiagramMenuStr, 10, 13)) : 
            GetHighestStrLen(DiagramMenuStr, 2, 5);
    DIALOG  DlgDiagram[6] = { { 0 }, { 0 }, { 0 }, { 0 }, { 0 }, { 0 } };
        
    DlgDiagram[0].proc = d_alt_box_proc;
    DlgDiagram[0].x = (SCREEN_W - w) / 2;
    DlgDiagram[0].y = (SCREEN_H - h) / 2;
    DlgDiagram[0].w = w;
    DlgDiagram[0].h = h;
    DlgDiagram[0].fg = fgcol;
    DlgDiagram[0].bg = bkcol;
    DlgDiagram[0].dp = DiagramMenuStr[(Functiontype && 
    ((ComplexExp->Graph == GR_C_STEP) || (ComplexExp->Graph == GR_C_STEPINV))) ? 
    1 : 0];
    
    for(j = 1; j <= 4; j++) {
        DlgDiagram[j].proc = d_menubar_proc;
        DlgDiagram[j].x = DlgDiagram[0].x + (w - MaxItemLen) / 2;
        DlgDiagram[j].y = DlgDiagram[0].y + (h - rh * 4) / 2 + j * rh;
        DlgDiagram[j].w = MaxItemLen;
        DlgDiagram[j].h = text_height(font);
        DlgDiagram[j].fg = fgcol;
        DlgDiagram[j].bg = bkcol;
        DlgDiagram[j].key = DiagramHkStr[i + j];
        DlgDiagram[j].d2 = 0;
        DlgDiagram[j].dp = DiagramMenuStr[i + 1 + j];
    }
    
    /*
    **  For R -> R with Graph 'orbitdiagram var. x' or 'orbitdiagram var. c',
    **  this menu should be skipped.
    */
    
    if(!Functiontype && (RealExp->Graph == GR_R_ORBITS_X || RealExp->Graph ==
    GR_R_ORBITS_C))  
        return 0; 
        
    /* 
    **  Set menu items for complex graph.
    **  For "Orbits for variable c (Mandelbrot)", "Inverse iteration method" is
    **  hidden (doesn't make sense there). 
    */

    if(Functiontype && (ComplexExp->Graph == GR_C_ORBITS_C)) {
        DlgDiagram[4].flags |= D_HIDDEN;    
    }
    
    /*
    **  Set current value as default and show dialog. Since "Orbits for var. c" 
    **  is the only diagram with only three options so if this one is selected 
    **  after a graph with four, Complex diagram is decreased with one to avoid
    **  a diagram that makes no sense for the selected graph.
    */

    if(ComplexExp->Graph == GR_C_ORBITS_C && ComplexExp->Diagram == DM_INVERSE)
        ComplexExp->Diagram--;

    DlgDiagram[((Functiontype) ? (ComplexExp->Diagram) : 
    (RealExp->Diagram)) + 1].flags |= D_SELECTED;
    
    popup_dialog(DlgDiagram, ((Functiontype) ? ComplexExp->Diagram : 
    RealExp->Diagram) + 1);

    /* Process result */

    for(i = 1; i <= 4; i++) {
        if(DlgDiagram[i].flags & D_SELECTED) {
            (Functiontype) ? (ComplexExp->Diagram) : (RealExp->Diagram) = i - 1;
            break;
        }
    }

    return 0;
}    

int MnuColoring(void)
{
    int     i, w = 540, h = 320,rh = text_height(font) * 2, fgcol = 0, 
            bkcol = 15, MaxItemLen = GetHighestStrLen(ColMenuStr, 1, 3);             
    DIALOG  DlgColoring[5] = { { 0 }, { 0 }, { 0 }, { 0 }, { 0 } };

    DlgColoring[0].proc = d_alt_box_proc;
    DlgColoring[0].x = (SCREEN_W - w) / 2;
    DlgColoring[0].y = (SCREEN_H - h) / 2;
    DlgColoring[0].w = w;
    DlgColoring[0].h = h;
    DlgColoring[0].fg = fgcol;
    DlgColoring[0].bg = bkcol;
    DlgColoring[0].dp = ColMenuStr[0];
    
    for(i = 1; i <= 3; i++) {
        DlgColoring[i].proc = d_menubar_proc;
        DlgColoring[i].x = DlgColoring[0].x + (w - MaxItemLen) / 2;
        DlgColoring[i].y = DlgColoring[0].y + (h - rh * 3) / 2 + i * rh;
        DlgColoring[i].w = MaxItemLen;
        DlgColoring[i].h = text_height(font);
        DlgColoring[i].fg = fgcol;
        DlgColoring[i].bg = bkcol;
        DlgColoring[i].key = ColHkStr[i - 1];
        DlgColoring[i].d2 = 0;
        DlgColoring[i].dp = ColMenuStr[i];
    }
    
    /* Set current value as default and show dialog */
    
    DlgColoring[ComplexExp->Coloring + 1].flags |= D_SELECTED;
    
    popup_dialog(DlgColoring, ComplexExp->Coloring + 1);

    /* Process result */

    for(i = 1; i <= 3; i++) {
        if(DlgColoring[i].flags & D_SELECTED) {
            ComplexExp->Coloring = i - 1;
            break;
        }
    }

    return(0);
}    

int MnuFunction(void)
{
    int     i, w = 540, h = 320, rh = text_height(font) * 2, fgcol = 0, 
            bkcol = 15, MaxItemLen = GetHighestStrLen(FnMenuStr, 1, 8);             
    DIALOG  DlgFn[10] = {   { 0 }, { 0 }, { 0 }, { 0 }, { 0 },
                            { 0 }, { 0 }, { 0 }, { 0 }, { 0 } 
                        };
    
    DlgFn[0].proc = d_alt_box_proc;
    DlgFn[0].x = (SCREEN_W - w) / 2;
    DlgFn[0].y = (SCREEN_H - h) / 2;
    DlgFn[0].w = w;
    DlgFn[0].h = h;
    DlgFn[0].fg = fgcol;
    DlgFn[0].bg = bkcol;
    DlgFn[0].dp = FnMenuStr[0];
    
    for(i = 1; i <= 1; i++) {
        DlgFn[i].proc = d_menubar_proc;
        DlgFn[i].x = DlgFn[0].x + (w - MaxItemLen) / 2;
        DlgFn[i].y = DlgFn[0].y + (h - rh * 8) / 2 + i * rh;
        DlgFn[i].w = MaxItemLen;
        DlgFn[i].h = text_height(font);
        DlgFn[i].fg = fgcol;
        DlgFn[i].bg = bkcol;
        DlgFn[i].key = 0;
        DlgFn[i].d2 = 0;
        DlgFn[i].dp = FnMenuStr[i];
    }
    
    /* Set current value as default and show dialog */
    
    DlgFn[RealExp->Function + 1].flags |= D_SELECTED;
    
    popup_dialog(DlgFn, RealExp->Function + 1);

    /* Process result */

    for(i = 1; i <= 1; i++) {
        if(DlgFn[i].flags & D_SELECTED) {
            RealExp->Function = i - 1;
            break;
        }
    }
    return 0;
}

int MnuParameters(void)
{
    char        String[11][MAX_EDIT_LEN];
    int         w = 540, h = 320, fgcol = 15, bkcol = 0,
                MaxItemLen = GetHighestStrLen(ParMenuStr, 1, 25) +
                MAX_EDIT_LEN * text_height(font);
	TCoords		RCoords;
	TRGraphPar	RGraphPar;
	TCoords		CCoords;
	TCGraphPar	CGraphPar;
    DIALOG      DlgPar[] = {

    {   
        d_alt_box_proc,
        (SCREEN_W - w) / 2,
        (SCREEN_H - h) / 2, w, h, bkcol, fgcol, 0, 0, 0, 0, 
        ParMenuStr[0], NULL, NULL
    },
    
    {
        d_alt_edit_proc,
        (SCREEN_W - w) / 2 + (w - MaxItemLen) / 2, 
        (SCREEN_H - h) / 2 + (h - 10) / 3,
        MAX_EDIT_LEN * text_height(font), text_height(font),
        fgcol, bkcol, 0, 1 << 8, MAX_EDIT_LEN - 1, 0, String[0], 
        ParMenuStr[1], "f"
    },

    {
        d_alt_edit_proc,
        (SCREEN_W - w) / 2 + (w - MaxItemLen) / 2,
        (SCREEN_H - h) / 2 + (h - 10) / 3 + 10,
        MAX_EDIT_LEN * text_height(font), text_height(font),
        fgcol, bkcol, 0, 2 << 8, MAX_EDIT_LEN - 1, 0, String[1],
        ParMenuStr[2], "f"
    },

    {
        d_alt_edit_proc,
        (SCREEN_W - w) / 2 + (w - MaxItemLen) / 2,
        (SCREEN_H - h) / 2 + (h - 10) / 3 + 20, 
        MAX_EDIT_LEN * text_height(font), text_height(font),
        fgcol, bkcol, 0, 3 << 8, MAX_EDIT_LEN - 1, 0, String[2], 
        ParMenuStr[3], "f"
    },

    {
        d_alt_edit_proc,
        (SCREEN_W - w) / 2 + (w - MaxItemLen) / 2, 
        (SCREEN_H - h) / 2 + (h - 10) / 3 + 30,
        MAX_EDIT_LEN * text_height(font), text_height(font),
        fgcol, bkcol, 0, 4 << 8, MAX_EDIT_LEN - 1, 0, String[3],
        ParMenuStr[4], "f"
    },

    {
        d_alt_edit_proc,
        (SCREEN_W - w) / 2 + (w - MaxItemLen) / 2,
        (SCREEN_H - h) / 2 + (h - 10) / 3 + 40,
        MAX_EDIT_LEN * text_height(font), text_height(font),
        fgcol, bkcol, 0, 5 << 8, MAX_EDIT_LEN - 1, 0, String[4], 
        ParMenuStr[5], "f"
    },

    {
        d_alt_edit_proc,
        (SCREEN_W - w) / 2 + (w - MaxItemLen) / 2, 
        (SCREEN_H - h) / 2 + (h - 10) / 3 + 50,
        MAX_EDIT_LEN * text_height(font), text_height(font),
        fgcol, bkcol, 0, 6 << 8, MAX_EDIT_LEN - 1, 0, String[5], 
        ParMenuStr[6], "f"
    },

    {
        d_alt_edit_proc,
        (SCREEN_W - w) / 2 + (w - MaxItemLen) / 2, 
        (SCREEN_H - h) / 2 + (h - 10) / 3 + 60,
        MAX_EDIT_LEN * text_height(font), text_height(font),
        fgcol, bkcol, 0, 7 << 8, MAX_EDIT_LEN - 1, 0, String[6], 
        ParMenuStr[7], "f"
    },

    {
        d_alt_edit_proc,
        (SCREEN_W - w) / 2 + (w - MaxItemLen) / 2, 
        (SCREEN_H - h) / 2 + (h - 10) / 3 + 70, 
        MAX_EDIT_LEN * text_height(font), text_height(font),
        fgcol, bkcol, 0, 8 << 8, MAX_EDIT_LEN - 1, 0, String[7], 
        ParMenuStr[8], "f"
    },

    {
        d_alt_edit_proc,
        (SCREEN_W - w) / 2 + (w - MaxItemLen) / 2,
        (SCREEN_H - h) / 2 + (h - 10) / 3 + 80,
        MAX_EDIT_LEN * text_height(font), text_height(font),
        fgcol, bkcol, 0, 9 << 8, MAX_EDIT_LEN - 1, 0, String[8], 
        ParMenuStr[9], "i"
    },

    {
        d_alt_edit_proc,
        (SCREEN_W - w) / 2 + (w - MaxItemLen) / 2,
        (SCREEN_H - h) / 2 + (h - 10) / 3 + 90,
        MAX_EDIT_LEN * text_height(font), text_height(font),
        fgcol, bkcol, 0, 10 << 8, MAX_EDIT_LEN - 1, 0, String[9],
        ParMenuStr[10], "i"
    },

    {
        d_alt_edit_proc,
        (SCREEN_W - w) / 2 + (w - MaxItemLen) / 2,
        (SCREEN_H - h) / 2 + (h - 10) / 3 + 100, 
        MAX_EDIT_LEN * text_height(font), text_height(font),
        fgcol, bkcol, 0, 11 << 8, MAX_EDIT_LEN - 1, 0, String[10],
        ParMenuStr[11], "i"
    },

    { 0 } 
    
    };
    
    /*
    **  Set correct values
    */

    /* Create shortcuts for coordinates and graph parameters */

	RCoords = *(RealExp->GraphPar[RealExp->Graph].Coords);
	RGraphPar = RealExp->GraphPar[RealExp->Graph];
	CCoords = *(ComplexExp->GraphPar[ComplexExp->Graph].Coords);
	CGraphPar = ComplexExp->GraphPar[ComplexExp->Graph];
    
    /* R -> R */
    
    if(!Functiontype) {
        
        /*
        **  Hide fields that are not required/make no sense for the selected
        **  graph, and change names for some fields.
        */
        
        switch(RealExp->Graph) {
            case GR_R_STEP:
                DlgPar[6].flags |= D_HIDDEN;    /* Y-Coordinate (lowest)    */
                DlgPar[7].flags |= D_HIDDEN;    /* Y-Coordinate (highest)   */
                DlgPar[10].flags |= D_HIDDEN;   /* Total number of iter...  */
                DlgPar[11].dp2 = ParMenuStr[14];                
                break;
            case GR_R_ORBITS_X:
                DlgPar[2].flags |= D_HIDDEN;    /* 1st starting value...    */
                DlgPar[3].flags |= D_HIDDEN;    /* 2nd starting value...    */
                DlgPar[4].dp2 = ParMenuStr[12];
                DlgPar[5].dp2 = ParMenuStr[13];
                DlgPar[9].flags |= D_HIDDEN;    /* Number of steps/iter...  */
                break;
            case GR_R_ORBITS_C:
                DlgPar[1].flags |= D_HIDDEN;    /* Value for parameter c    */
                DlgPar[3].flags |= D_HIDDEN;    /* 2nd starting value...    */
                DlgPar[4].dp2 = ParMenuStr[15];
                DlgPar[5].dp2 = ParMenuStr[16];
                DlgPar[9].flags |= D_HIDDEN;    /* Number of steps/iter...  */
                break;
        }
        
        /* Copy values to the input strings */
        
        sprintf(String[ 0], "%g", RGraphPar.c);        
        sprintf(String[ 1], "%g", RGraphPar.xbegin1);
        sprintf(String[ 2], "%g", RGraphPar.xbegin2);
        sprintf(String[ 3], "%g", RCoords.XMin);        
        sprintf(String[ 4], "%g", RCoords.XMax);
        sprintf(String[ 5], "%g", RCoords.YMin);
        sprintf(String[ 6], "%g", RCoords.YMax);        
        sprintf(String[ 7], "%g", RGraphPar.EscapeValue);
        sprintf(String[ 8], "%ld", RGraphPar.IterSteps);        
        sprintf(String[ 9], "%ld", RGraphPar.IterMax);        
        sprintf(String[10], "%ld", RGraphPar.IterSkip);
        if(RealExp->Graph == GR_R_STEP)
            sprintf(String[10], "%d", RGraphPar.Iterated);
    }
    else {

        /*
        **  Hide fields that are not required/make no sense for the selected
        **  graph, and change names for some fields.
        */
         
        DlgPar[1].flags |= D_HIDDEN;
        DlgPar[2].dp2 = ParMenuStr[19];
        DlgPar[3].dp2 = ParMenuStr[20];
        DlgPar[4].dp2 = ParMenuStr[21];
        DlgPar[5].dp2 = ParMenuStr[22];
        DlgPar[6].dp2 = ParMenuStr[23];
        DlgPar[7].dp2 = ParMenuStr[24];

        switch(ComplexExp->Graph) {
            case GR_C_STEP:
            case GR_C_STEPINV:
                DlgPar[10].flags |= D_HIDDEN;
                break;
            case GR_C_ORBITS_Z:
                DlgPar[9].flags |= D_HIDDEN;
                break;
            case GR_C_ORBITS_C:
                DlgPar[2].dp2 = ParMenuStr[25];
                DlgPar[3].dp2 = ParMenuStr[26];
                DlgPar[9].flags |= D_HIDDEN;
                break;
        }

        /* Copy values to the input strings */

        if(ComplexExp->Graph == GR_C_ORBITS_Z) {
            sprintf(String[ 1], "%g", CGraphPar.cReal);
            sprintf(String[ 2], "%g", CGraphPar.cImag);
        }
        else {
            sprintf(String[ 1], "%g", CGraphPar.zReal);
            sprintf(String[ 2], "%g", CGraphPar.zImag);
        }
        sprintf(String[ 3], "%g", CCoords.XMin);
        sprintf(String[ 4], "%g", CCoords.XMax);        
        sprintf(String[ 5], "%g", CCoords.YMin);
        sprintf(String[ 6], "%g", CCoords.YMax);        
        sprintf(String[ 7], "%g", CGraphPar.EscapeValue);
        sprintf(String[ 8], "%d", CGraphPar.IterSteps);
        sprintf(String[ 9], "%d", CGraphPar.IterMax);
        sprintf(String[10], "%d", CGraphPar.IterSkip);
    }

    /* Show dialog */

    do_alt_dialog(DlgPar, find_dialog_focus(DlgPar), ParametersCallback);

    /* Process data: Copy values to the input strings */

    if(!Functiontype) {
        RGraphPar.c = atof(String[ 0]);
        RGraphPar.xbegin1 = atof(String[ 1]);
        RGraphPar.xbegin2 = atof(String[ 2]);
        RCoords.XMin = atof(String[ 3]);
        RCoords.XMax = atof(String[ 4]);
        RCoords.YMin = atof(String[ 5]);
        RCoords.YMax = atof(String[ 6]);
	    RCoords.Width = RCoords.XMax - RCoords.XMin;
	    RCoords.Height = RCoords.YMax - RCoords.YMin;
        RGraphPar.EscapeValue = atof(String[ 7]);
        RGraphPar.IterSteps = atol(String[ 8]);
        RGraphPar.IterMax = atol(String[ 9]);
        RGraphPar.IterSkip = atol(String[10]);
        if(RealExp->Graph == GR_R_STEP)
            RGraphPar.Iterated = atoi(String[10]);

        /* Copy values back */

	    *(RealExp->GraphPar[RealExp->Graph].Coords) = RCoords;
        RealExp->GraphPar[RealExp->Graph] = RGraphPar;
    }
    else {
        if(ComplexExp->Graph != GR_C_ORBITS_Z) {
            CGraphPar.cReal = atof(String[ 1]);
            CGraphPar.cImag = atof(String[ 2]);
        }
        else {
            CGraphPar.zReal = atof(String[ 1]);
            CGraphPar.zImag = atof(String[ 2]);
        }
        CCoords.XMin = atof(String[ 3]);
        CCoords.XMax = atof(String[ 4]);
        CCoords.YMin = atof(String[ 5]);
        CCoords.YMax = atof(String[ 6]);
	    CCoords.Width = CCoords.XMax - CCoords.XMin;
	    CCoords.Height = CCoords.YMax - CCoords.YMin;
        CGraphPar.EscapeValue = atof(String[ 7]);
        CGraphPar.IterSteps = atol(String[ 8]);        
        CGraphPar.IterMax = atol(String[ 9]);        
        CGraphPar.IterSkip = atol(String[10]);

        /* Copy values back */
        
	    *(ComplexExp->GraphPar[ComplexExp->Graph].Coords) = CCoords;
        ComplexExp->GraphPar[ComplexExp->Graph] = CGraphPar;         
    }
    return 0;
}

int MnuShowData(void)
{
    return 0;
}

int MnuDraw(void)
{
    scare_mouse();
    if(!Functiontype) 
        RealDraw(RealExp, RealStepHelpStr); 
    else 
        ComplexDraw(ComplexExp);
    unscare_mouse();
    if(!Functiontype) DoRCentralMenu(); else DoCCentralMenu();
    return 0;
}

int MnuSettings(void)
{
    return 0;
}

int MnuLoad(void)
{
    return 0;
}

int MnuSave(void)
{
    char        FilePath[2048] = "default.ps";
    FILE        *fp;
    
    if(file_select_ex("Save file: ", FilePath, "PS", 2047,  300, 400) != 0) {
        if((fp = fopen(FilePath, "w")) != NULL) {
            scare_mouse();
            WriteEPS(screen, SCREEN_W, SCREEN_H, fp, 0);
            unscare_mouse();
            fclose(fp);
        }
    }
    return 0;
}

int MnuAbout(void)
{
    int     w = 540, h = 320, fgcol = 15, bkcol = 0;
    DIALOG  DlgAbout[] = { 
    
    {   
        d_alt_box_proc,
        (SCREEN_W - w) / 2, (SCREEN_H - h) / 2, w, h,
        fgcol, bkcol, 0, 0, 0, 0, AboutMenuStr[0], NULL, NULL
    },
    
    {
        d_textbox_proc,
        (SCREEN_W - w) / 2 + 10, (SCREEN_H - h) / 2 + h / 2,
        w - 20, h / 2 - 10, fgcol, bkcol, 0, 0, 0, 0, LicenseText,  
        NULL, NULL
    },
    
    { 0 } };
    
    popup_dialog(DlgAbout, 0);
    return 0;
}
    
int MnuQuit(void)
{
    if(alert("Quit?", NULL, NULL, "OK", "Cancel", 13, 27) == 2) return(D_O_K);
    return(D_CLOSE);
}

int DoRCentralMenu(void)
{
    char        *RJMenuStr[] = {

                    /* Zoom is not for the combination 'web/step' */

                    "ENTER    - Select area (zoom) ",
		            "<T>      - Change diagram type",
		            "Space    - Orbit diagram var.x",
		            "<S>      - Save menu          ",
		            "ESCAPE   - To parameter menu  ",

                    /* 1 & 2 are changed for 'orbit diagram var. x' */

		            "BkSp     - Step-by-step iter. ",
                    "BkSp     - Orbit diagram var.x",
		            "Space    - Orbit diagram var.c"
                };
    int         AltKey, OldGraph, NewGraph;
    static int  ShowMenu = 0;

    /*
    **  Set correct menu texts.
    */
    
    if(RealExp->Graph == GR_R_ORBITS_X) {
        RJMenuStr[1] = RJMenuStr[5];
        RJMenuStr[2] = RJMenuStr[7];
    }
    if(RealExp->Graph == GR_R_ORBITS_C) {
        RJMenuStr[1] = RJMenuStr[6];
        RJMenuStr[2] = "";
    }
    
    if(ShowMenu == 0) {
        AltKey = InfoBox(RJMenuStr, 4) >> 8;

        if(AltKey == KEY_ESC) { ShowMenu = 0; return 1; }
            
        switch(AltKey) {
            case KEY_ENTER:
                ZoomBox(screen, font, RealExp->GraphPar[RealExp->Graph].Coords,
                ZoomHelpStr);
                clear_keybuf();
                MnuDraw();
                break;
            case KEY_S:
                MnuSave();
                clear_keybuf();
                DoRCentralMenu();
                break;
            case KEY_SPACE: 
                if(RealExp->Graph == GR_R_ORBITS_C) {
                    clear_keybuf();
                    ShowMenu = 0;
                    DoRCentralMenu();
                }
                break;
            case KEY_BACKSPACE:
                if(RealExp->Graph == GR_R_STEP) {
                    clear_keybuf();
                    ShowMenu = 0;
                    DoRCentralMenu();
                }
                break;         
            case KEY_T:
                ShowMenu = 1;
                break;
            default:
                clear_keybuf();
                ShowMenu = 0;
                DoRCentralMenu();
                break;
        }
    }

    OldGraph = RealExp->Graph;
        
    /*
    **	Step by step iteration    --> Orbit diagram for variable x
    */
        
    if(AltKey == KEY_SPACE && RealExp->Graph == GR_R_STEP) {
        NewGraph = RealJump(RealExp, GR_R_ORBITS_X, NULL, F1KeyStr, NULL);
        if(NewGraph != -1) {
            RealExp->Graph = NewGraph;
            clear_keybuf();
            ShowMenu = 0;
            MnuDraw();
        }    
    }
                
   	/*
    **	Orbit diagram for variable x  --> Banendiagram for variabele c
    */
        
    else if(AltKey == KEY_SPACE && RealExp->Graph == GR_R_ORBITS_X) {
        NewGraph = RealJump(RealExp, GR_R_ORBITS_C, NULL, F1KeyStr, NULL);
        if(NewGraph != -1) {
            RealExp->Graph = NewGraph;
            clear_keybuf();
            ShowMenu = 0;
            MnuDraw();
        }
    }
        
    /*
    **	Banendiagram variabele x  --> Step by step iteration
	*/

	else if(AltKey == KEY_BACKSPACE && RealExp->Graph == GR_R_ORBITS_X) {
        NewGraph = RealJump(RealExp, GR_R_STEP, NULL, F1KeyStr, NULL);
        if(NewGraph != -1) {
            RealExp->Graph = NewGraph;
            ShowMenu = 1;
        }
    }

    /*
	**	Orbit diagram variable c  --> Banendiagram variabele x
	*/

	else if(AltKey == KEY_BACKSPACE && RealExp->Graph == GR_R_ORBITS_C) {
        NewGraph = RealJump(RealExp, GR_R_ORBITS_X, NULL, F1KeyStr, NULL);
        if(NewGraph != -1) {
            RealExp->Graph = NewGraph;
            clear_keybuf();
            ShowMenu = 0;
            MnuDraw();
        }
	}
    
    /* Diagram selection for 'Step by step iteration' */
        
    if(ShowMenu == 1) {
        AltKey = InfoBox(RJSubMenuStr, 4) >> 8;

        switch(AltKey) {
            case KEY_W:
                RealExp->Diagram = DM_WEB;
                break;
            case KEY_I:
                RealExp->Diagram = DM_ITERVALUES;
                break;
            case KEY_S:
                RealExp->Diagram = DM_STEP;
                break;
            case KEY_C:
                RealExp->Diagram = DM_COMBIWEBSTEP;
                break;
            default:
                clear_keybuf();
                if(AltKey == KEY_ESC) {
                    RealExp->Graph = OldGraph;
                    ShowMenu = 0; 
                }
                else 
                    ShowMenu = 1;
                DoRCentralMenu();
                break;
        }
        
        if(AltKey == KEY_W || AltKey == KEY_I || 
        AltKey == KEY_S || AltKey == KEY_C) {
            clear_keybuf();
            ShowMenu = 0;
            MnuDraw();
        }
    }
    return 0;               
}

int DoCCentralMenu(void)
{
    char        *CJMenuStr[] = {
    
	                "ENTER  - Select area (zoom)         ",
	                "<T>    - Change diagram type        ",
	                "BkSp   - Step-by-step iteration     ",
	                "Space  - Mandelbrot set             ",
                    "<S>    - Save menu                  ",
                    "ESC    - To parameter menu          ",

                    "Space  - Julia set                  ",
                    "BkSp   - Julia set                  "
                },
    
                *CJStepMenuStr[] = {
		
                    "<1>    - Iteration              ",
	                "<2>    - Inverse iteration      ",
	                "ESC    - Previous menu",
	            },
    
                *CJStepShapeMenuStr[] = {
    
                    "<P>    - Point         ",
                    "<L>    - Line          ",
                    "<R>    - Rectangle     ",
                    "<C>    - Circle        ",
                    "ESC    - Previous menu "
        
                },
                
                *CJStepColMenuStr[] = {        
                    
                    "<S> - Color depends on startingpoint     ",
                    "<I> - Color depends on iteration steps   ",
                    "<A> - All iteration steps the same color ",
                    "ESC - Previous menu"
                },
    
                *CJOrbitsMenuStr[] = {
    
	                "Julia:                                 ",
	                "<F> -  Filled-in                       ",
	                "<E> -  Escape time                     ",
	                "<B> -  Boundary trace                  ",
	                "<I> -  Inverse iteration method        ",
	                "ESC -  Previous menu                   "
	            };
    int         AltKey, OldGraph, NewGraph;
    static int  ShowMenu = 0, StepJump = FALSE;
    
    /*
    **  Set correct menu texts.
    */

    if(ComplexExp->Graph == GR_C_STEP || ComplexExp->Graph == GR_C_STEPINV) {
        CJMenuStr[2] = CJMenuStr[6];
        CJMenuStr[3] = "";
    }
    
    if(ComplexExp->Graph == GR_C_ORBITS_C) {
        CJMenuStr[2] = CJMenuStr[7];
        CJMenuStr[3] = "";
        CJOrbitsMenuStr[0] = "Mandelbrot:";
        
        /* Inverse is only valid for the Julia set */
        
        CJOrbitsMenuStr[4] = "";
    }

    if(ShowMenu == 0) {
        AltKey = InfoBox(CJMenuStr, 5) >> 8;

        if(AltKey == KEY_ESC) { ShowMenu = 0; return 1; }
            
        switch(AltKey) {
            case KEY_ENTER:
                ZoomBox(screen, font, 
                ComplexExp->GraphPar[ComplexExp->Graph].Coords, ZoomHelpStr);
                clear_keybuf();
                MnuDraw();
                break;
            case KEY_S:
                MnuSave();
                clear_keybuf();
                DoCCentralMenu();
                break;
            case KEY_SPACE:
            case KEY_BACKSPACE:
                break;         
            case KEY_T:
                ShowMenu = 1;
                break;
            default:
                clear_keybuf();
                ShowMenu = 0;
                DoCCentralMenu();
                break;
        }
    }
    
    OldGraph = ComplexExp->Graph;
    
  	/*
    **	Step by step iteration (inv)    --> Orbit diagram for variable z
    */
        
    if(AltKey == KEY_SPACE && ((ComplexExp->Graph == GR_C_STEP) ||
    (ComplexExp->Graph == GR_C_STEPINV))) {
        NewGraph = ComplexJump(ComplexExp, GR_C_ORBITS_Z, F1KeyStr,
        NULL, FALSE);
        if(NewGraph != -1) { ComplexExp->Graph = NewGraph; ShowMenu = 1; }
    }

   	/*
    **	Orbit diagram for variable z  --> Banendiagram for variabele c
    */
        
    else if(AltKey == KEY_SPACE && ComplexExp->Graph == GR_C_ORBITS_Z) {
        NewGraph = ComplexJump(ComplexExp, GR_C_ORBITS_C, F1KeyStr,
        NULL, FALSE);
        if(NewGraph != -1) { ComplexExp->Graph = NewGraph; ShowMenu = 1; }
    }

   	/*
    **	Orbit diagram for variable c  --> Banendiagram for variabele z
    */
        
    else if(AltKey == KEY_BACKSPACE && ComplexExp->Graph == GR_C_ORBITS_C) {
        NewGraph = ComplexJump(ComplexExp, GR_C_ORBITS_Z, F1KeyStr,
        NULL, TRUE);
        if(NewGraph != -1) { ComplexExp->Graph = NewGraph; ShowMenu = 1; }
    }
    
   	/*
    **	Orbit diagram for variable z  --> Step by step iteration (inv)
    */
        
    else if(AltKey == KEY_BACKSPACE && ComplexExp->Graph == GR_C_ORBITS_Z) {
        StepJump = TRUE; 
        OldGraph = ComplexExp->Graph; 
        ComplexExp->Graph = GR_C_STEP; 
        ShowMenu = 1;
    }

    /*
    **  Show submenu's.
    */
    
    if(ShowMenu == 1 || ShowMenu == 2) {
    
        /* 
        **  Show 1st Submenu for 'orbit diagram for var. z' and  
        **  'orbit diagram for var. c'.
        */
        
        if((ComplexExp->Graph == GR_C_ORBITS_Z || 
        ComplexExp->Graph == GR_C_ORBITS_C) && ShowMenu == 1) {

            /*
            **  Set correct menu texts.
            */
                    
            if(ComplexExp->Graph == GR_C_ORBITS_C) {
                CJOrbitsMenuStr[0] = "Mandelbrot:";
                CJOrbitsMenuStr[4] = "";
            }    
            else {
                CJOrbitsMenuStr[0] = "Julia:";
                CJOrbitsMenuStr[4] = "<I> -  Inverse iteration method        ";
            }
            
            AltKey = InfoBox(CJOrbitsMenuStr, 4) >> 8;
            
            switch(AltKey) {
                case KEY_F:
                    ComplexExp->Diagram = DM_FILLED;
                    break;
                case KEY_E:
                    ComplexExp->Diagram = DM_ETD;
                    break;
                case KEY_B:
                    ComplexExp->Diagram = DM_BOUNDARY;
                    break;
                case KEY_I:
                    ComplexExp->Diagram = DM_INVERSE;
                    break;
                default:
                    clear_keybuf();
                    if(AltKey == KEY_ESC) {
                        ComplexExp->Graph = OldGraph;
                        ShowMenu = 0; 
                    }
                    else 
                        ShowMenu = 1;
                    DoCCentralMenu();
                    break;
            }
        
            if(AltKey == KEY_F || AltKey == KEY_E || 
            AltKey == KEY_B || AltKey == KEY_I) {
                clear_keybuf();
                ShowMenu = 0;
                MnuDraw();
            }
        }
        
        /* 
        **  Show 1st Submenu for 'step-by-step iteration' and  
        **  'step-by-step inverse iteration'.
        */
       
        if((ComplexExp->Graph == GR_C_STEP || 
        ComplexExp->Graph == GR_C_STEPINV) && ShowMenu == 1) {
            AltKey = InfoBox(CJStepMenuStr, 2) >> 8;
                        
            switch(AltKey) {
                case KEY_1:
                case KEY_1_PAD:
                    ComplexExp->Graph = GR_C_STEP;
                    break;
                case KEY_2:
                case KEY_2_PAD:
                    ComplexExp->Graph = GR_C_STEPINV;
                    break;
                default:
                    clear_keybuf();
                    if(AltKey == KEY_ESC) {
                        ComplexExp->Graph = OldGraph;
                        ShowMenu = 0; 
                    }
                    else 
                        ShowMenu = 1;
                    DoCCentralMenu();
                    break;
            }
        
            if(AltKey == KEY_1 || AltKey == KEY_2 || 
            AltKey == KEY_1_PAD || AltKey == KEY_2_PAD) {
                clear_keybuf();
                ShowMenu = 2;
            }
        }
        
        /* 
        **  Show 2nd Submenu for 'step-by-step iteration' and  
        **  'step-by-step inverse iteration'.
        */

        if((ComplexExp->Graph == GR_C_STEP || 
        ComplexExp->Graph == GR_C_STEPINV) && ShowMenu == 2) {
            AltKey = InfoBox(CJStepShapeMenuStr, 4) >> 8;
                        
            switch(AltKey) {
                case KEY_P:
                    ComplexExp->Diagram = SH_POINT;
                    break;
                case KEY_L:
                    ComplexExp->Diagram = SH_LINE;
                    break;
                case KEY_R:
                    ComplexExp->Diagram = SH_RECT;
                    break;
                case KEY_C:
                    ComplexExp->Diagram = SH_CIRCLE;
                    break;
                default:
                    clear_keybuf();
                    if(AltKey == KEY_ESC) ShowMenu = 1; else ShowMenu = 2;
                    DoCCentralMenu();
                    break;
            
            }

            if(AltKey == KEY_P || AltKey == KEY_L || 
            AltKey == KEY_R || AltKey == KEY_C) {
                clear_keybuf();
                ShowMenu = 3;
            }
        }
        if((ComplexExp->Graph == GR_C_STEP || 
        ComplexExp->Graph == GR_C_STEPINV) && ShowMenu == 3) {
            AltKey = InfoBox(CJStepColMenuStr, 3) >> 8;
                        
            switch(AltKey) {
                case KEY_S:
                    ComplexExp->Coloring = CM_START;
                    break;
                case KEY_I:
                    ComplexExp->Coloring = CM_ITERSTEP;
                    break;
                case KEY_A:
                    ComplexExp->Coloring = CM_SAME;
                    break;
                default:
                    clear_keybuf();
                    if(AltKey == KEY_ESC) ShowMenu = 2; else ShowMenu = 3;
                    DoCCentralMenu();
                    break;
            }

            if(AltKey == KEY_S || AltKey == KEY_I || AltKey == KEY_A) {
                clear_keybuf();
                ShowMenu = 0;
                if(StepJump) {
                    StepJump = FALSE;
                    NewGraph = ComplexExp->Graph;
                    ComplexExp->Graph = OldGraph;
                    ComplexJump(ComplexExp, NewGraph, F1KeyStr, NULL, FALSE);
                }
                MnuDraw();
            }
        }
    }
    return 0;
}
