/*
	This file is part of OpenFuniter.

	cplexp.c V2.2.3 - a complex experiment (functiontype: from C to C)

    Copyright (C) 1995-2003 Stijn Wolters.
    Original idea: Ernic Kamerich (University of Nijmegen)

	See README for contact information.

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version. See

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

	See COPYING for more information.
*/

#include "cplexp.h"

static int      JuliaVisible = FALSE;

/*!
**	Create GraphPar.
*/

TCGraphPar *CreateComplexGraphPar(TCoords *Coords, double zReal, double zImag,
	double cReal, double cImag, double EscapeValue, unsigned int IterSteps,
	unsigned int IterSkip, unsigned int IterMax)
{
	TCGraphPar		*GraphPar;

	if((GraphPar = (TCGraphPar *) malloc(sizeof(TCGraphPar))) == NULL)
		return NULL;

	GraphPar->Coords = Coords;
	GraphPar->zReal = zReal;
	GraphPar->zImag = zImag;
	GraphPar->cReal = cReal;
	GraphPar->cImag = cImag;
	GraphPar->EscapeValue = EscapeValue;
	GraphPar->IterSteps = IterSteps;
	GraphPar->IterSkip = IterSkip;
	GraphPar->IterMax = IterMax;

	return(GraphPar);
}

/*
**  Destroy (Free) memory for a GraphPar.
*/

void FreeComplexGraphPar(TCGraphPar *GraphPar)
{
	if(GraphPar != NULL) free(GraphPar);
}

/*
**  Load ComplexGraphPar.
*/

int LoadComplexGraphPar(FILE *fp, TCGraphPar *GraphPar)
{
    TCoords     Coords;
    int         Res = 0;
    
	Coords = *(GraphPar->Coords);
    
    Res = fscanf(fp, "%lg,%lg,%lg,%lg\n", &GraphPar->zReal, &GraphPar->zImag, 
    &GraphPar->cReal, &GraphPar->cImag);
        
    Res = fscanf(fp, "%lg,%lg,%lg,%lg\n", &Coords.XMin, &Coords.XMax, 
    &Coords.YMin, &Coords.YMax);
        
    Res = fscanf(fp, "%lg\n", &GraphPar->EscapeValue);

    Res = fscanf(fp, "%d,%d,%d\n", &GraphPar->IterMax, &GraphPar->IterSteps,
    &GraphPar->IterSkip);

    Res = fscanf(fp, "%d\n", &GraphPar->EscapeValueChanged);

    *(GraphPar->Coords) = Coords;
    
    return Res;

}

/*
**  Save a ComplexGraphPar.
*/

int SaveComplexGraphPar(FILE *fp, TCGraphPar *GraphPar)
{
    TCoords     Coords;

	Coords = *(GraphPar->Coords);

    fprintf(fp, "%.*g,%.*g,%.*g,%.*g\n", DBL_DIG,  GraphPar->zReal, 
    DBL_DIG, GraphPar->zImag, DBL_DIG, GraphPar->cReal, DBL_DIG, 
    GraphPar->cImag);
        
    fprintf(fp, "%.*g,%.*g,%.*g,%.*g\n", DBL_DIG, Coords.XMin, 
    DBL_DIG, Coords.XMax, DBL_DIG, Coords.YMin, DBL_DIG, 
    Coords.YMax);
        
    fprintf(fp, "%.*g\n", DBL_DIG, GraphPar->EscapeValue);

    fprintf(fp, "%d,%d,%d\n", GraphPar->IterMax, GraphPar->IterSteps,
    GraphPar->IterSkip);
        
    fprintf(fp, "%d\n", GraphPar->EscapeValueChanged);
    
    return 0;
}

/*
**  Creates a complex experiment including the four GraphPar's.
*/

TComplexExp *CreateComplexExp(unsigned int XStart, unsigned int YStart,
	unsigned int XEnd, unsigned int YEnd)
{
	TComplexExp		*CplExp;
	TCoords			*StepCoords, *StepInvCoords, *OrbitsZCoords,
				    *OrbitsCCoords;

    /* Allocate memory for 1 GraphPar */
    
	if((CplExp = (TComplexExp *) malloc(sizeof(TComplexExp))) == NULL)
		return NULL;

	/* Step by step */

	StepCoords =
		CreateCoords(-3, -2.1875, 3, 2.1875, XStart, YStart, XEnd, YEnd, 56);
	CplExp->GraphPar[GR_C_STEP] =
		*CreateComplexGraphPar(StepCoords, 0, 0, 0, 0, 2, 1, 0, 250);

	/* Step by step inverse */

	StepInvCoords =
		CreateCoords(-3, -2.1875, 3, 2.1875, XStart, YStart, XEnd, YEnd, 56);
	CplExp->GraphPar[GR_C_STEPINV] =
		*CreateComplexGraphPar(StepInvCoords, 0, 0, 0, 0, 2, 1, 0, 250);

	/* Orbits for variable z */

	OrbitsZCoords =
		CreateCoords(-2, -1.4375, 2, 1.4375, XStart, YStart, XEnd, YEnd, 56);
	CplExp->GraphPar[GR_C_ORBITS_Z] =
		*CreateComplexGraphPar(OrbitsZCoords, 0, 0, -1, 0, 2, 1, 0, 250);

	/* Orbits for variable c */

	OrbitsCCoords =
		CreateCoords(-2, -1.4375, 2, 1.4375, XStart, YStart, XEnd, YEnd, 56);
	CplExp->GraphPar[GR_C_ORBITS_C] =
		*CreateComplexGraphPar(OrbitsCCoords, 0, 0, 0, 0, 2, 1, 0, 250);

	/* Set escape values */
    
    CplExp->Graph = GR_C_STEP;
	CplExp->GraphPar[GR_C_STEP].EscapeValue =
        CalculateComplexFnEscapeValue(CplExp);
	CplExp->GraphPar[GR_C_STEP].EscapeValueChanged = FALSE; 
    
    CplExp->Graph = GR_C_STEPINV;
	CplExp->GraphPar[GR_C_STEPINV].EscapeValue = 
        CalculateComplexFnEscapeValue(CplExp);
	CplExp->GraphPar[GR_C_STEPINV].EscapeValueChanged = FALSE; 
    
    CplExp->Graph = GR_C_ORBITS_Z;
    CplExp->GraphPar[GR_C_ORBITS_Z].EscapeValue = 
        CalculateComplexFnEscapeValue(CplExp);
    CplExp->GraphPar[GR_C_ORBITS_Z].EscapeValueChanged = FALSE;
    
    CplExp->Graph = GR_C_ORBITS_C;
	CplExp->GraphPar[GR_C_ORBITS_C].EscapeValue = 
        CalculateComplexFnEscapeValue(CplExp);
    CplExp->GraphPar[GR_C_ORBITS_C].EscapeValueChanged = FALSE;

	CplExp->Settings = *CreateSettings();
    
    CplExp->Function = 0;
	CplExp->Graph = GR_C_STEP;
	CplExp->Diagram = SH_POINT;
	CplExp->Coloring = 0;
	CplExp->EscValueChanged = FALSE;

    CplExp->InvJuliaIterMax = 50000;
    return(CplExp);
}

void FreeComplexExp(TComplexExp *CplExp)
{
	if(CplExp != NULL) free(CplExp);
}

/*
**  Save a ComplexExperiment.
*/

int LoadComplexExp(TComplexExp *ComplexExp, char *FileName, int Mode)
{
    FILE    *fp;
    int     Res = 0, Functiontype = 1;
    
    if(Mode == 1) { /* read old par files */ }
        
	if((fp = fopen(FileName, "r")) == NULL) return 0;

    fscanf(fp, "%d", &Functiontype);
    if(Functiontype == 1) {
        Res = LoadComplexGraphPar(fp, &ComplexExp->GraphPar[GR_C_STEP]);
        Res = LoadComplexGraphPar(fp, &ComplexExp->GraphPar[GR_C_STEPINV]);
        Res = LoadComplexGraphPar(fp, &ComplexExp->GraphPar[GR_C_ORBITS_Z]);
        Res = LoadComplexGraphPar(fp, &ComplexExp->GraphPar[GR_C_ORBITS_C]);

        Res = fscanf(fp, "%d,%d,%d\n", &ComplexExp->Graph, &ComplexExp->Diagram, 
        &ComplexExp->Function); 

        LoadSettings(fp, &ComplexExp->Settings);    
    }
	fclose(fp);
	return Functiontype;
}

/*
**  
**  Save a ComplexExperiment.
*/

int SaveComplexExp(TComplexExp *ComplexExp, char *FileName)
{
	FILE        *fp;
    
	if((fp = fopen(FileName, "w+")) == NULL) return 0;
    
    fprintf(fp, "%d\n", 1);
    
    SaveComplexGraphPar(fp, &ComplexExp->GraphPar[GR_C_STEP]);
    SaveComplexGraphPar(fp, &ComplexExp->GraphPar[GR_C_STEPINV]);
    SaveComplexGraphPar(fp, &ComplexExp->GraphPar[GR_C_ORBITS_Z]);
    SaveComplexGraphPar(fp, &ComplexExp->GraphPar[GR_C_ORBITS_C]);
        
    fprintf(fp, "%d,%d,%d\n", ComplexExp->Graph, ComplexExp->Diagram, 
    ComplexExp->Function); 
    
    SaveSettings(fp, &ComplexExp->Settings);    
      
	fclose(fp);
	return 0;
}

void SetComplexExpDefaultCoords(TComplexExp *ComplexExp, int Graph)
{
    switch(Graph) {
        case GR_C_STEP:
        case GR_C_STEPINV:
            SetCoords(ComplexExp->GraphPar[Graph].Coords,
            -3, -2.1875, 3, 2.1875); 
            break;
        case GR_C_ORBITS_Z:
        case GR_C_ORBITS_C:
            SetCoords(ComplexExp->GraphPar[Graph].Coords,
            -2, -1.4375, 2, 1.4375); 
            break;
    }
}

/*
**	This function could be improved, it's inefficient but at least more
**  efficient than the original which used a cursor that walked across the
**	screen to scan for a particular color :-).
*/

int IsInList(int *XArray, int *YArray, int x, int y)
{
	int flag;

	flag = FALSE;
	do {
		if((*XArray == x) && (*YArray == y)) { flag = TRUE; break; }
	} while(((*XArray++) != NULL) && ((*YArray++) != NULL));

	return(flag);

}

/*
**	Since BGI doesn't support different write-modes for circles, we have to
**	draw the circle ourselves (OK, a polygon was a possibility...).
*/

void DrawCircle(BITMAP *Bitmap, TCoords *Coords, int x, int y, int r, 
    int Color)
{
	int			j, NumberOfPoints, xa, ya;
	double 		xr, yr, Angle, AspectRatio;

	AspectRatio = (double) (Coords->YEnd - Coords->YStart) /
	(double) (Coords->XEnd - Coords->XStart);
	xr = (double) r;
	yr = (double) r * AspectRatio / (Coords->Height / Coords->Width);
	NumberOfPoints = (int) (2.0 * M_PI * (double) max(xr, yr));
	for(j = 0; j < NumberOfPoints; j++) {
		Angle = M_PI / 180.0 * ((double) j * (360.0 / NumberOfPoints));
		xa = cos(Angle) * xr + (double) x;
		ya = sin(Angle) * yr + (double) y;
		putpixel(Bitmap, xa, ya, Color);
	}
}

/*
**	This function draws a line and stores all (converted) points in an array.
**	Simple linedrawing is used, no Bresenham-algorithm (yet).
**	PosInLine is used to continue counting when we call this routine multiple
**	times to draw the rectangle.
*/

int StoreLine(BITMAP *Bitmap, TCoords *Coords, int x1, int y1, int x2, int y2, 
    int Color, int PosInLine, TComplex **Points)
{
	int 	x, y, c, s, i, j;
	double 	r, ch, sh;
    
	s = (double) (x2 - x1);
	c = (double) (y2 - y1);
    
	r = sqrt(fabs(c) * fabs(c) + fabs(s) * fabs(s));
	if(r == 0.0) return 0;

	sh = s / r;
	ch = c / r;
	j = PosInLine;
	for(i = 0; i < (int) r; i++) {
		x = sh * (double) i + (double) x1;
		y = ch * (double) i + (double) y1;
		(*Points)[j].r = R2M_X(*Coords, abs((int) x));
		(*Points)[j].i = R2M_Y(*Coords, abs((int) y));
		j++;
		putpixel(Bitmap, x, y, Color);
	}
	return j;
}

/*
**	This function draws a rectangle and stores all (converted) points in an
**	array and requires StoreLine in order to do that :-).
*/

int StoreRectangle(BITMAP *Bitmap, TCoords *Coords, int x1, int y1, int x2, 
    int y2, int Color, TComplex **Points)
{
	int		Pos1, Pos2, Pos3, Pos4;

	Pos1 = StoreLine(Bitmap, Coords, x1, y1, x2, y1, Color, 0, Points);
	Pos2 = StoreLine(Bitmap, Coords, x2, y1, x2, y2, Color, Pos1, Points);
	Pos3 = StoreLine(Bitmap, Coords, x2, y2, x1, y2, Color, Pos2, Points);
	Pos4 = StoreLine(Bitmap, Coords, x1, y2, x1, y1, Color, Pos3, Points);
	return Pos4;
}

/*
**	This function draws a circle and stores all (converted) points in an
**	array.
*/

int StoreCircle(BITMAP *Bitmap, int NumPoints, TCoords *Coords, int x, int y, 
    int r, int Color, TComplex **Points)
{
	int 		j, xa, ya, NumberOfPoints;
	double		Angle, AspectRatio, xr, yr;

	AspectRatio = (double) (Coords->YEnd - Coords->YStart) /
	(double) (Coords->XEnd - Coords->XStart);
	xr = (double) r;
	yr = (double) r * AspectRatio / (Coords->Height / Coords->Width);
	NumberOfPoints = NumPoints;
    /*(int) (2.0 * M_PI * max(xr, yr)) + 1;*/
	for(j = 0; j < NumberOfPoints - 1; j++) {
		Angle = M_PI / 180.0 * ((double) j * (360.0 / NumberOfPoints));
		xa = cos(Angle) * xr + (double) x;
		ya = sin(Angle) * yr + (double) y;	
        (*Points)[j].r = R2M_X(*Coords, xa);
		(*Points)[j].i = R2M_Y(*Coords, ya);
        putpixel(Bitmap, xa, ya, Color);
	}
	return NumberOfPoints;
}

/*
**	Create a shape and return the number of points or an error.
**  If Mode is negative the user will be able to select a shape, otherwise 
**  a realloc takes place in which case NumPoints holds the new size.
*/

TComplex *CreateShape(BITMAP *Bitmap, TComplexExp *ComplexExp,  int Color, 
    int *NumPoints, char **HelpText)
{
	int		    x1, y1, x2, y2;
    TComplex    *Points;
	TCoords		Coords;

	Coords = *(ComplexExp->GraphPar[ComplexExp->Graph].Coords);
    
    *NumPoints = DefineShape(Bitmap, ComplexExp, &x1, &y1, &x2, &y2, Color, 
    HelpText);        
    
    /*if(*NumPoints < 0) return(NULL);*/ 
    
    if((Points = (TComplex *) malloc(sizeof(TComplex) * *NumPoints)) == NULL) 
        return(NULL);
    
    switch(ComplexExp->Diagram) {
        case SH_POINT:
		    Points[0].r = R2M_X(Coords, x1);
		    Points[0].i = R2M_Y(Coords, y1);
	        break;
        case SH_LINE:		
            StoreLine(Bitmap, &Coords, x1, y1, x2, y2, Color, 0, &Points);
            break;
        case SH_RECT:
		    StoreRectangle(Bitmap, &Coords, x1, y1, x2, y2, Color, &Points);
	        break;
	    case SH_CIRCLE:
            StoreCircle(Bitmap, *NumPoints, &Coords, x1, y1, x2, Color, 
                &Points);
	        break;
    }

	return Points;
}

int DefineShape(BITMAP *Bitmap, TComplexExp *ComplexExp, int *x1, int *y1, 
    int *x2, int *y2, int Color, char **HelpText)
{
    char        v1[40];
	int			XMickeys, YMickeys, Key, ExtKey;
	double		Perimeter, sp, cp, r;
	TCoords		Coords;

	Coords = *(ComplexExp->GraphPar[ComplexExp->Graph].Coords);
    
	/*
    **	Select first point which is the upperleft
    **	corner of a reactangle or the centre of a circle.
    **  In case of points, the routine will not wait for a mouse-click.
	*/

	do {
		poll_mouse();
		*x1 = mouse_x;
		*y1 = mouse_y;
        get_mouse_mickeys(&XMickeys, &YMickeys);
        		
        if((XMickeys != 0) || (YMickeys != 0)) {
			sprintf(v1, "(%2.2f, %2.2f)", R2M_X(Coords, *x1), 
            R2M_Y(Coords, *y1));
            StatusBar(ComplexExp->Settings.LAY_Statusline, -1, 
            ComplexExp->Graph, v1);
		}

		if(ComplexExp->Diagram == SH_POINT) {
            *x2 = *x1;
			*y2 = *y1;
            if(ExtKey == KEY_ENTER) {
                clear_keybuf(); 
                return 1;
            }
        }

        if(keypressed()) {
			Key = readkey();
			ExtKey = Key >> 8;
            if(ExtKey == KEY_F1) {
				scare_mouse();
				if(HelpText != NULL) InfoBox(HelpText, 3);
				unscare_mouse();
			}
			else if(ExtKey == KEY_ESC)
				return -2;
        }
	} while(!(mouse_b & 1)); 

	/*
	**	Second selection: not active for single dots, lower-right corner of
	**	a rectangle and radius of the circle.
	*/

	xor_mode(TRUE); 
	while((mouse_b & 1)) { /* was !mouse_b & 2 */
		*x2 = mouse_x;
		*y2 = mouse_y;
        get_mouse_mickeys(&XMickeys, &YMickeys);

        sp = fabs((double) (*x2 - *x1));
        cp = fabs((double) (*y2 - *y1));
		r = (int) (sqrt(sp * sp + cp * cp) + 0.5);

        if((XMickeys != 0) || (YMickeys != 0)) {
            solid_mode();
            
            if((ComplexExp->Diagram == SH_RECT) || 
            (ComplexExp->Diagram == SH_LINE)) 
			    sprintf(v1, "(%2.2f, %2.2f) - (%2.2f, %2.2f)", 
                R2M_X(Coords, *x1), R2M_Y(Coords, *y1),
                R2M_X(Coords, *x2), R2M_Y(Coords, *y2));
            else {
			    sprintf(v1, "(%2.2f, %2.2f) - (%2.2f)", 
                R2M_X(Coords, *x1), R2M_Y(Coords, *y1), 
                R2M_Y(Coords, *y1 + r));
            }
            StatusBar(ComplexExp->Settings.LAY_Statusline, -1, 
            ComplexExp->Graph, v1);
            xor_mode(TRUE);
		}
               
        poll_mouse();

		scare_mouse();
        switch(ComplexExp->Diagram) {
	        case 1:
			    line(Bitmap, *x1, *y1, mouse_x, mouse_y, Color);
				line(Bitmap, *x1, *y1, mouse_x, mouse_y, Color);
                Perimeter = r;
				break;
			case 2:
			    rect(Bitmap, *x1, *y1, mouse_x, mouse_y, Color);
				rect(Bitmap, *x1, *y1, mouse_x, mouse_y, Color);
                Perimeter = (sp + cp) * 2.0;
				break;
			case 3:
			    DrawCircle(Bitmap, &Coords, *x1, *y1, r, Color);
			    DrawCircle(Bitmap, &Coords, *x1, *y1, r, Color);
			    *x2 = (int) r;
			    *y2 = 0;
                Perimeter = 2.0 * 3.1415926535 * r;
			    break;
		}
        unscare_mouse();
        
		if(keypressed()) {
			Key = readkey();
			ExtKey = Key >> 8;
            if(ExtKey == KEY_F1) {
				scare_mouse();
				solid_mode();
				if(HelpText != NULL) InfoBox(HelpText, 4);
				xor_mode(TRUE);
				unscare_mouse();
			}
			else if(ExtKey == KEY_ESC)
				return -2;
		}
	}
	solid_mode();
	return (int) (Perimeter + 0.5);
}

/*
**	Draw a line between two points (which are represented by a small block).
*/

void IPDDrawLine(BITMAP *Bitmap, TComplexExp *ComplexExp, TComplex *Point,
	TComplex *PointLast, int Color, unsigned int Iter, int ClearMode)
{
    char                        v1[40];
	int 						xz, yz, xzlast, yzlast;
	TCoords						Coords;
	TCGraphPar					GraphPar;

	Coords = *(ComplexExp->GraphPar[ComplexExp->Graph].Coords);
	GraphPar = ComplexExp->GraphPar[ComplexExp->Graph];

	xz = M2R_X(Coords, Real(*Point));
	yz = M2R_Y(Coords, Imag(*Point));

	xzlast = M2R_X(Coords, Real(*PointLast));
	yzlast = M2R_Y(Coords, Imag(*PointLast));

	if(Iter == GraphPar.IterSkip)
		circlefill(Bitmap, xzlast, yzlast, 4, Color);
	else
		rectfill(Bitmap, xzlast - 2, yzlast - 2, xzlast + 2, yzlast + 2, Color);

	if(ComplexExp->Settings.SSI_Lines) 
        line(Bitmap, xzlast, yzlast, xz, yz, Color);

	rectfill(Bitmap, xz - 2, yz - 2, xz + 2, yz + 2, Color);

	rectfill(Bitmap, xz - 2, yz - 2, xz + 2, yz + 2, Color);
	rectfill(Bitmap, xzlast - 2, yzlast - 2, xzlast + 2, yzlast + 2, Color);

            
	sprintf(v1, "z(%5d) = (%2.2f + %2.2fi)", Iter + 1, Real(*Point), 
    Imag(*Point));
    StatusBar(ComplexExp->Settings.LAY_Statusline, -1, ComplexExp->Graph, v1);

	if(ComplexExp->Settings.SSI_Text && (!ClearMode)) {
		textprintf(Bitmap, font, M2R_X(Coords, Real(*Point)), M2R_Y(Coords, 
        Imag(*Point)), Color, "(%1.1f,%1.1f)", Real(*Point), Imag(*Point));
	}
}

int IPDPlot(BITMAP *Bitmap, TComplexExp *ComplexExp, TComplex **Points, 
    TComplex *LastPoint, unsigned int Iter, int Color, int IterColor, 
    int *Size, int KeyPressed)
{
    /*char            *CurHelpStr = "[F1]=Help.";*/
	int				ExitValue = TRUE, xc, yc, XScr, YScr, i, j, 
                    *PixelsX, *PixelsY;
    static int      TempColor = 1;
	TCoords			Coords;

	Coords = *(ComplexExp->GraphPar[ComplexExp->Graph].Coords);

	XScr = Coords.XEnd;
	YScr = Coords.YEnd;

	/*
	**	Allocate memory for the x and y pixels and set to 0.
	*/

	if((PixelsX = (int *) malloc(*Size * sizeof(int))) == NULL) return -2;
	if((PixelsY = (int *) malloc(*Size * sizeof(int))) == NULL) return -2;
	memset(PixelsX, 0, *Size * sizeof(int));
	memset(PixelsY, 0, *Size * sizeof(int));

	/* Set color for dots */
    
	if(ComplexExp->Coloring == CM_START) {
        if(Iter == 0) {
            TempColor++;
	        if(TempColor == 15) TempColor = 1; 
        }
        Color = TempColor;
    }
    else if(ComplexExp->Coloring == CM_ITERSTEP) {
		Color = (int) (Iter % ComplexExp->Settings.COL_Max) + 1;
		if(Color == 14) Color = 1;
	}
	else
		Color = IterColor;

	/* Step by step inverse iteration */

	if(ComplexExp->Graph == GR_C_STEP) {
		for(j = 0; j < *Size; j++) {
			xc = M2R_X(Coords, (*Points)[j].r);
			yc = M2R_Y(Coords, (*Points)[j].i);
			if(ComplexExp->Diagram == SH_POINT)
				IPDDrawLine(Bitmap, ComplexExp, Points[0], LastPoint, Color, 
                Iter, KeyPressed);
			else
				putpixel(Bitmap, xc, yc, Color);
		}
	}

	/* Step by step iteration */

	else if(ComplexExp->Graph == GR_C_STEPINV) {
		if(ComplexExp->Diagram == SH_POINT)
			IPDDrawLine(Bitmap, ComplexExp, Points[0], LastPoint, Color, Iter,
			KeyPressed);
		else {
			i = 0;
			for(j = 0; j < *Size; j++) {
				xc = (int) M2R_X(Coords, (*Points)[j].r);
				yc = (int) M2R_Y(Coords, (*Points)[j].i);
                putpixel(Bitmap, xc, yc, Color);
                
                /* TEST */               
                if(!IsInList(PixelsX, PixelsY, xc, yc)) {
					PixelsX[i] = xc; PixelsY[i] = yc;
					if(i > *Size - 2) {
                        /*
                        printf("1.Range: i: %d | NumPoints: %d\n", i, *Size);
                        */
                        ExitValue = FALSE;
                        break; 
                    }
					i++;
				}
                
                if(KeyPressed == KEY_EQUALS) {
					xc = (int) M2R_X(Coords, -((*Points)[j].r));
					yc = (int) M2R_Y(Coords, -((*Points)[j].i));
                    putpixel(Bitmap, xc, yc, Color);
                
					if(!IsInList(PixelsX, PixelsY, xc, yc)) {
						PixelsX[i] = xc; PixelsY[i] = yc;
						if(i > *Size - 2) {   
                            /*
                            printf("2.Range: i: %d | NumPoints: %d\n", i, *Size); 
                            */
                            ExitValue = FALSE; 
                            break;
                        }
                        i++;
					}
				}
                /* END TEST */
			}
            /* TEST */          
			if((KeyPressed == KEY_EQUALS) && ExitValue) {
				/*printf("Total new point: %d (Max. %d)\n", i, *Size);*/
                for(j = 0; j < i; j++) {
					(*Points)[j].r = R2M_X(Coords, PixelsX[j]);
					(*Points)[j].i = R2M_Y(Coords, PixelsY[j]);
				}
            }
            /* END TEST */
		}
    }
	if(PixelsX != NULL) free(PixelsX);
	if(PixelsY != NULL) free(PixelsY);
	return ExitValue;
}

int DoComplexStep(BITMAP *Bitmap, BITMAP *JuliaBitmap, TComplexExp *ComplexExp,
int JuliaVisible, char **StepHelpText, char **StepInvHelpText, 
char **JuliaNoMemStr, char **DefShapeHelpText)
{
	TCoords				Coords;
	TCGraphPar			GraphPar;
    TComplex            c, LastPoint;
	TComplex            *Points;
	double              PixelWidth, PixelHeight;
	int                 IterColor, Key = 0, ExtKey = 0, NumPoints = 1,
                        j, XMickeys, YMickeys, ShapeDefined = FALSE, EscStatus,
                        OldGraph, OldDiagram;
	unsigned int        l;

    /* Create some shortcuts */
    
	Coords = *(ComplexExp->GraphPar[ComplexExp->Graph].Coords);
	GraphPar = ComplexExp->GraphPar[ComplexExp->Graph];
    
	PixelWidth = (Coords.XMax - Coords.XMin) / (Coords.XEnd - Coords.XStart);
	PixelHeight = (Coords.YMax - Coords.YMin) / (Coords.YEnd - Coords.YStart);
    
    /* 
    **  Try to copy the image of the Julia-set on the screen to the JuliaBitmap.
    **  For 'delete' there is something added:
    **
    **  In case the Julia-set should be visible, but there was not enough mem.
    **  to create the JuliaBitmap, the user will be prompted with a question
    **  whether or not the Julia-set should be recalculated. Since I made the
    **  mistake of declaring Diagram not in GraphPar, this is limited to only
    **  the Julia inverse method.
    */
    
    if(JuliaVisible) {
        if(JuliaBitmap != NULL) 
            blit(screen, JuliaBitmap, 0, 0, 0, 0, Coords.XEnd, Coords.YEnd);
        else {
            if(alert(JuliaNoMemStr[0], JuliaNoMemStr[1], JuliaNoMemStr[2],
                     JuliaNoMemStr[3], JuliaNoMemStr[4], 13, 27) == 1) {
                rectfill(Bitmap, Coords.XStart, Coords.YStart, Coords.XEnd, 
                Coords.YEnd, 0);
                OldGraph = ComplexExp->Graph;
                OldDiagram = ComplexExp->Diagram;
                ComplexExp->Graph = GR_C_ORBITS_Z;
                ComplexExp->Diagram = DM_INVERSE;
                DrawComplexOrbits(Bitmap, ComplexExp);
                ComplexExp->Graph = OldGraph;
                ComplexExp->Diagram = OldDiagram;
            }
            else
                rectfill(Bitmap, Coords.XStart, Coords.YStart, Coords.XEnd, 
                Coords.YEnd, 0);
        }
    }
    else
        rectfill(screen, Coords.XStart, Coords.YStart, Coords.XEnd, 
        Coords.YEnd, 0);
    
    DrawGrid(Bitmap, font, &Coords);

    if(ComplexExp->Graph == GR_C_STEPINV) {
        if(ComplexExp->Diagram == SH_POINT)  
	        ExtKey = InfoBox(StepInvHelpText + 1, 4);
        else
            ExtKey = InfoBox(StepInvHelpText, 5);
    }
    else 
        ExtKey = InfoBox(StepHelpText, 2);
        
	if(ExtKey == KEY_ESC) return 1;

	c = Complex(GraphPar.cReal, GraphPar.cImag);
    show_mouse(Bitmap);
	
    if(ComplexExp->Coloring == CM_SAME)
        IterColor =
        ComplexExp->Settings.COL_List[ComplexExp->Settings.COL_Function];
    
    do {
		poll_mouse();
		if(!ShapeDefined) {
            Points = CreateShape(Bitmap, ComplexExp, 15, &NumPoints,
            DefShapeHelpText);
        }
        
        /* Points couldn't be allocated */
        
        if(Points == NULL) 
            return 0;        
        
        /* Esc-key */
        
        if(NumPoints == -2) { 
            show_mouse(NULL); 
            free(Points); 
            return 0; 
        }
		
        if(NumPoints >= 0) ShapeDefined = TRUE;
        
        /* Restart Creation of shape */
        
        if(NumPoints < 0) { printf("NumPoints < 0.\n"); continue; }
		
        l = 0L;
        do {
			if(ShapeDefined) {
                if(ComplexExp->Diagram == SH_POINT) LastPoint = Points[0];

			    if(ComplexExp->Graph == GR_C_STEPINV)
				    for(j = 0; j < NumPoints; j++)
                        IterateInvFn(&Points[j].r, &Points[j].i, c.r, c.i);
			    else
				    for(j = 0; j < NumPoints; j++)
			            CIterateFn(Points[j].r, Points[j].i,
                        c.r, c.i, GraphPar.EscapeValue, PixelWidth, PixelHeight,
			            1, 1, FALSE, &Points[j].r, &Points[j].i, &EscStatus);
            }
            
			if(l >= GraphPar.IterSkip) {

				if((l % GraphPar.IterSteps) == 0) {
                    
                    while(TRUE) {
                        if(ComplexExp->Diagram != SH_POINT) {
                            poll_mouse();
							if(mouse_b & 2) {
                                if(ShapeDefined) free(Points); 
                                ShapeDefined = FALSE;
                                break; 
                            } 
							if(keypressed()) {
								Key = readkey();
                                ExtKey = Key >> 8;
							}
                            else 
								continue;
						}
						else {
                            Key = readkey();
                            ExtKey = Key >> 8;
                            
                            if(ExtKey == KEY_EQUALS && 
                            ComplexExp->Graph == GR_C_STEPINV
							&& ComplexExp->Diagram == SH_POINT)
							    continue;
                        }

						if((ExtKey == KEY_ENTER) || (ExtKey == KEY_PLUS_PAD) ||
						(ExtKey == KEY_MINUS) || (ExtKey == KEY_EQUALS)) {
                            break;
                        }

						if(ExtKey == KEY_ESC) {
							show_mouse(NULL);
							if(ShapeDefined) free(Points);
                            ShapeDefined = FALSE;
						    clear_keybuf();
							return 1;
						}

						if(ExtKey == KEY_DEL) {
                            if(!JuliaVisible) {
                                rectfill(Bitmap, Coords.XStart, Coords.YStart,
                                Coords.XEnd, Coords.YEnd, 0);
                            }
                            else {
                                if(JuliaBitmap != NULL) {
                     	            scare_mouse();
                                    blit(JuliaBitmap, Bitmap, 0, 0, 0, 0, 
                                    Coords.XEnd, Coords.YEnd);
                               	    unscare_mouse();
                                }
                                else {
                                    if(alert(JuliaNoMemStr[0], 
                                    JuliaNoMemStr[1], JuliaNoMemStr[2],
                                    JuliaNoMemStr[3], JuliaNoMemStr[4], 13,
                                    27) == 1) {
                                        rectfill(Bitmap, Coords.XStart, 
                                        Coords.YStart, Coords.XEnd, Coords.YEnd,
                                        0);
                                        OldGraph = ComplexExp->Graph;
                                        OldDiagram = ComplexExp->Diagram;
                                        ComplexExp->Graph = GR_C_ORBITS_Z;
                                        ComplexExp->Diagram = DM_INVERSE;
                                        DrawComplexOrbits(Bitmap, ComplexExp);
                                        ComplexExp->Graph = OldGraph;
                                        ComplexExp->Diagram = OldDiagram;
                                    }
                                    else {
                     	                scare_mouse();
                                        rectfill(Bitmap, Coords.XStart, 
                                        Coords.YStart, Coords.XEnd, Coords.YEnd,
                                        0);
                                        unscare_mouse();
                                    }
                                }
                            }
                            if(ShapeDefined) free(Points);
                            ShapeDefined = FALSE;
                     	    scare_mouse();
                            DrawGrid(Bitmap, font, &Coords);
						    unscare_mouse();
                            l = 0L;
                            clear_keybuf();
                            break;
						}

						if(ExtKey == KEY_F1) {
                            if(ComplexExp->Graph == GR_C_STEPINV) {
                                if(ComplexExp->Diagram == SH_POINT)  
	                                InfoBox(StepInvHelpText + 1, 4);
                                else
                                    InfoBox(StepInvHelpText, 5);
                            }
                            else 
                                InfoBox(StepHelpText, 3);
                            clear_keybuf();
						}
					} /* end of while(TRUE) */

                    if(ShapeDefined) {
					    if(ComplexExp->Graph == GR_C_STEPINV) {
						    if(ExtKey == KEY_MINUS)
							    for(j = 0; j < NumPoints; j++)
								    Points[j] = CNeg(Points[j]);
						    else if(ExtKey == KEY_ENTER) {
							    if((rand() % 2) < 1)
								    for(j = 0; j < NumPoints; j++)
									    Points[j] = CNeg(Points[j]);
						    }
					    }
                    }

					poll_mouse();
                    if(ComplexExp->Diagram == SH_POINT) {                   
					    get_mouse_mickeys(&XMickeys, &YMickeys);
                        if((XMickeys != 0) || (YMickeys != 0)) {
                            if(ShapeDefined) {
                                ShapeDefined = FALSE;
                                free(Points);
                            }
                            clear_keybuf();
						    break;
					    }
                    }
                    else  
                        if(mouse_b & 2) break; 
                }
                
				scare_mouse();
				if(ShapeDefined) {
                    while(!IPDPlot(Bitmap, ComplexExp, &Points, &LastPoint, l, 
                    0, IterColor, &NumPoints, ExtKey)) {
                        NumPoints*=2;
                        if((Points = (TComplex *) realloc(Points, 
                        sizeof(TComplex) * NumPoints)) == NULL) 
                            return 0;
                    }
                }
                unscare_mouse();
            }
            
			if(EscStatus && ShapeDefined && (ComplexExp->Graph == GR_C_STEP)) 
                break;  
            
			l++;
		} while(l <= MAXINT);
	} while(TRUE);
}


/*
**	Draw orbits for variable x and c.
*/

int DrawComplexOrbits(BITMAP *Bitmap, TComplexExp *ComplexExp)
{
	int				Res, XStart, XEnd, YStart, YEnd, MYStart, MYEnd,
					XScr, YScr, Width, Height, MirrorState;
	unsigned long   Iter;
    double			PixelWidth, PixelHeight, EscapeValue;
	clock_t			StartTime, EndTime;
	TCGraphPar		GraphPar;
	TCoords			Coords;

	/* Init */

	Coords = *(ComplexExp->GraphPar[ComplexExp->Graph].Coords);
	GraphPar = ComplexExp->GraphPar[ComplexExp->Graph];
    
	EscapeValue = ComplexExp->GraphPar[ComplexExp->Graph].EscapeValue;
	ComplexExp->GraphPar[ComplexExp->Graph].EscapeValue =
		EscapeValue * EscapeValue;
    
	XStart = 0;
	YStart = 0;
	XEnd = Coords.XEnd - Coords.XStart - 1;
	YEnd = Coords.YEnd - Coords.YStart - 1;
	XScr = XEnd;
	YScr = YEnd;
	Width = XEnd + 1;
	Height = YEnd + 1;

	PixelWidth = (Coords.XMax - Coords.XMin) / Width;
	PixelHeight = (Coords.YMax - Coords.YMin) / Height;

    /* Julia-set: Inverse Iteration Method using random-method */
    
    if(ComplexExp->Graph == GR_C_ORBITS_Z && ComplexExp->Diagram == DM_INVERSE) 
    {
        for(Iter = 1; Iter <= ComplexExp->InvJuliaIterMax; Iter++) {
            
            /* Iterate function */
            
            IterateInvFn(&GraphPar.zReal, &GraphPar.zImag, 
            GraphPar.cReal, GraphPar.cImag);
            
            /* Choose random the positive/negative root */
            
            if((rand() % 2) < 1) {
                GraphPar.zReal = -GraphPar.zReal;
                GraphPar.zImag = -GraphPar.zImag;
            }
            
            /* Plot pixel */
            
            if(Iter >= GraphPar.IterSkip) {
                putpixel(Bitmap, M2R_X(Coords, GraphPar.zReal), 
                M2R_Y(Coords, GraphPar.zImag),
                ComplexExp->Settings.COL_Function);
            }
        }
	    DrawGrid(Bitmap, font, GraphPar.Coords);
        return 0;
    }
    
	StartTime = clock();

	MirrorState = MS_NONE;
    
	/*
	** Mirror in x.
	*/

	if(Coords.YMin < 0.0 && Coords.YMax > 0.0) {
		if(!(ComplexExp->Graph == GR_C_ORBITS_Z && GraphPar.cImag != 0.0)) {
			if(Coords.YMin + Coords.YMax > 0.0) {
				YStart = (Coords.YMax + Coords.YMin) / PixelHeight;
				YEnd = Coords.YMax / PixelHeight;
				MirrorState = MS_X;
			}
			if(Coords.YMin + Coords.YMax < 0.0) {
				YStart = 0;
				YEnd = Coords.YMax / PixelHeight;
				YScr = YEnd * 2;
				MirrorState = MS_X;
			}
		}
		if(Coords.YMin + Coords.YMax == 0.0) {
			YEnd = Height / 2;
			MirrorState = MS_X;
		}
	}

	/*
	**	Mirror in y.
	*/

	if(Coords.XMin < 0.0 && Coords.XMax > 0.0) {
		if(ComplexExp->Graph == GR_C_ORBITS_Z && GraphPar.cImag == 0.0) {
			if(Coords.XMin + Coords.XMax > 0.0) {
				XStart = 0;
				XEnd = (-Coords.XMin - Coords.XMin) / PixelWidth;
				XScr = XEnd - 1;
				MirrorState += MS_Y;
			}
			if(Coords.XMin + Coords.XMax < 0.0) {
				XStart = (-Coords.XMax - Coords.XMin) / PixelWidth;
				XEnd = (0.0 - Coords.XMin) / PixelWidth;
				MirrorState += MS_Y;
			}
			if(Coords.XMin + Coords.XMax == 0.0) {
				XEnd = Width / 2;
				MirrorState += MS_Y;
			}
		}
	}

	/* Draw area */

	Res = DrawComplexOrbitsArea(Bitmap, XStart, YStart, XEnd, YEnd, XScr, YScr,
	Width, Height, MirrorState, ComplexExp);

	/* Store coordinates for the central mirror part */

	MYStart = YStart;
	MYEnd = YEnd;

	/* Try to mirror remaining parts */

	if(ComplexExp->Graph == GR_C_ORBITS_Z) {

		if(Coords.XMin < 0.0 && Coords.XMax > 0.0) {
			if(YStart > 0) {
				YEnd = YStart;
				YStart = 0;
				MirrorState = MS_Y;
				Res = DrawComplexOrbitsArea(Bitmap, XStart, YStart, XEnd, YEnd,
				XScr, YScr, Width, Height, MirrorState, ComplexExp);
			}

			if(YScr < Height - 1) {
				YStart = YScr;
				YEnd = Height - 1;
				MirrorState = MS_Y;
				Res = DrawComplexOrbitsArea(Bitmap, XStart, YStart, XEnd, YEnd,
				XScr, YScr, Width, Height, MirrorState, ComplexExp);
			}
		}

		YStart = MYStart;
		YEnd = MYEnd;

		MirrorState = MS_NONE;

		if(XStart > 0) {
			XEnd = XStart;
			XStart = 0;
			if(Coords.YMin < 0.0 && Coords.YMax > 0.0)
				MirrorState = MS_X;
			else
				XScr = XEnd;
			Res = DrawComplexOrbitsArea(Bitmap, XStart, YStart, XEnd, YEnd, 
            XScr, YScr, Width, Height, MirrorState, ComplexExp);
		}
		else if(XScr < Width - 1) {
			XStart = XScr;
			XEnd = Width - 1;
			if(Coords.YMin < 0.0 && Coords.YMax > 0.0)
				MirrorState = MS_X;
			Res = DrawComplexOrbitsArea(Bitmap, XStart, YStart, XEnd, YEnd, 
            XScr, YScr, Width, Height, MirrorState, ComplexExp);
		}
	}


	/* The last part where no mirroring is possible */

	MirrorState = MS_NONE;
	if(YStart > 0) {
		YEnd = YStart;
		YStart = 0;
		Res = DrawComplexOrbitsArea(Bitmap, XStart, YStart, XEnd, YEnd, XScr, 
        YScr, Width, Height, MirrorState, ComplexExp);
	}
	else {
		YStart = YScr;
		YEnd = Height - 1;
		Res = DrawComplexOrbitsArea(Bitmap, XStart, YStart, XEnd, YEnd, XScr, 
        YScr, Width, Height, MirrorState, ComplexExp);
	}

	EndTime = clock();

	/*
    sprintf(TextStr, "Total calc. time: %4.4gs", 
    (float) ((EndTime - StartTime) / CLOCKS_PER_SEC));
    StatusBar(ComplexExp->Settings.LAY_Statusline, -1, ComplexExp->Graph, 
    TextStr);
    */

	ComplexExp->GraphPar[ComplexExp->Graph].EscapeValue = EscapeValue;

	DrawGrid(Bitmap, font, GraphPar.Coords);
	return Res;
}

int DrawComplexOrbitsArea(BITMAP *Bitmap, unsigned int XStart, 
    unsigned int YStart, unsigned int XEnd, unsigned int YEnd, 
    unsigned int XScr, unsigned int YScr, unsigned int Width, 
    unsigned int Height, unsigned int MirrorState, TComplexExp *ComplexExp)
{
	unsigned char	Col;
	unsigned int 	MemPrevYLine = FALSE, XPos, YPos,
					PrevYPixel, *PrevYLinePixel, LastIter, Iter, MaxPeriod,
                    tmpe;
    int             AltKey;
	double			zRealTemp, cRealTemp, zImagTemp, cImagTemp, PixelWidth,
					PixelHeight, XPixel, YPixel, tmpr, tmpi;
	TCGraphPar		GraphPar;
	TCoords			Coords;

	/* Init */

	Coords = *(ComplexExp->GraphPar[ComplexExp->Graph].Coords);
	GraphPar = ComplexExp->GraphPar[ComplexExp->Graph];

	PixelWidth = (Coords.XMax - Coords.XMin) / Width;
	PixelHeight = (Coords.YMax - Coords.YMin) / Height;

	MaxPeriod = (unsigned int) sqrt(GraphPar.IterMax);

	/*
	**	Array which holds the calculated pixels for the previous line.
	**	If there's not enough memory to hold the array, the pixel will be
	**	calculated. This function is just for the 'JM_BOUNDARY'.
	*/

	if((PrevYLinePixel = (unsigned int *) malloc(Width *
	sizeof(unsigned int) + 1)) != NULL) MemPrevYLine = TRUE;
    
	if(MemPrevYLine) memset(PrevYLinePixel, 0, Width);
    
	LastIter = 0L;

	for(YPos = YStart; YPos <= YEnd; YPos++) {
         
		YPixel = Coords.YMax - YPos * PixelHeight;
		if(ComplexExp->Graph == GR_C_ORBITS_C) {
			cImagTemp = YPixel;
			zImagTemp = GraphPar.zImag;
		}
		else {
			cImagTemp = GraphPar.cImag;
			cImagTemp = YPixel;
		}

		cImagTemp = (ComplexExp->Graph == GR_C_ORBITS_C) ? 
        YPixel : GraphPar.cImag;
		zImagTemp = (ComplexExp->Graph == GR_C_ORBITS_C) ?
		GraphPar.zImag : YPixel;

		for(XPos = XStart; XPos <= XEnd; XPos++) {

			XPixel = Coords.XMin + XPos * PixelWidth;
			
            cRealTemp = (ComplexExp->Graph == GR_C_ORBITS_C) ?
			XPixel : GraphPar.cReal;
			zRealTemp = (ComplexExp->Graph == GR_C_ORBITS_C) ?
			GraphPar.zReal : XPixel;
            
            /* Iterate function */
            
            Iter = CIterateFn(zRealTemp,
		    (ComplexExp->Graph == GR_C_ORBITS_C) ?
            zImagTemp : zImagTemp - PixelHeight, cRealTemp,
			(ComplexExp->Graph == GR_C_ORBITS_C) ?
            cImagTemp - PixelHeight : cImagTemp,
			GraphPar.EscapeValue, PixelWidth, PixelHeight,
			MaxPeriod, GraphPar.IterMax, TRUE, &tmpr, &tmpi, &tmpe);

			/*
			**	If there was not enough memory for the previous line, and the
			**	diagram is 'Boundary', then the previous y pixel is calculated,
			**	in the other case it is stored in the array of
			**	'previous y pixels'.
			*/

            if(ComplexExp->Diagram == DM_BOUNDARY) {
			    if(!MemPrevYLine)
			        PrevYPixel = CIterateFn(zRealTemp,
		            (ComplexExp->Graph == GR_C_ORBITS_C) ?
                    zImagTemp : zImagTemp - PixelHeight, cRealTemp,
				    (ComplexExp->Graph == GR_C_ORBITS_C) ?
                    cImagTemp - PixelHeight : cImagTemp,
				    GraphPar.EscapeValue, PixelWidth, PixelHeight,
				    MaxPeriod, GraphPar.IterMax, TRUE, &tmpr, &tmpi, &tmpe);
                else
				    PrevYPixel = PrevYLinePixel[XPos];
            }

			/* Filled set */

			if(ComplexExp->Diagram == DM_FILLED) {
				if(Iter == 0)
					Col = ComplexExp->Settings.COL_Function;
				else
					goto Quit;
            }

			/* Escape Time Diagram */

			if(ComplexExp->Diagram == DM_ETD) {
				if(Iter == 0)
					Col = ComplexExp->Settings.COL_Inside;
				else if(Iter == GraphPar.IterMax)
					Col = ComplexExp->Settings.COL_Outside;
				else {
					Col = ComplexExp->Settings.COL_Max - ((GraphPar.IterMax -
					Iter) % ComplexExp->Settings.COL_Max);
					if(Col == 0)
						Col++;
					else if(Col == ComplexExp->Settings.COL_Max)
						Col--;
				}
			}

			/*
			**	Boundary:
			**
			**	This is a diagram that requires some explanation, certainly
			**	since the names of the variables don't exactly represent what
			**	they do. LastIter is the iteration-value for the last calculated
			**	x pixel, and PrevYPixel the iteration value for the pixel just
			**	above the current. To determine the pixel that is closest to
			**	the M-sets boudary the current pixel is compared to the ones
			**	just mentioned. If the current pixel is not in the set and one
			**	of the others is, or the other way around, the pixel is set
			**	(which means we consider it on the boundary of the M-set).
			*/

			if(ComplexExp->Diagram == DM_BOUNDARY) {
				if((!LastIter && Iter) || (LastIter && !Iter) ||
				(!PrevYPixel && Iter) || (PrevYPixel && !Iter))
					Col = ComplexExp->Settings.COL_Function;
				else
					goto Quit;
            }
            
			/* Drawing */

			putpixel(Bitmap, XPos, YPos, ComplexExp->Settings.COL_List[Col]);
			if(MirrorState & MS_X) {
				if(GraphPar.cImag == 0)
					putpixel(Bitmap, XPos, YScr - YPos + YStart,
						ComplexExp->Settings.COL_List[Col]);
				else
					putpixel(Bitmap, XScr - XPos + XStart, YScr - YPos + YStart,
						ComplexExp->Settings.COL_List[Col]);
            }
            
			if(MirrorState & MS_Y)
				putpixel(Bitmap, XScr - XPos + XStart, YPos,
					ComplexExp->Settings.COL_List[Col]);

			if((MirrorState & MS_X) && (MirrorState & MS_Y))
				putpixel(Bitmap, XScr - XPos + XStart, YScr - YPos + YStart,
					ComplexExp->Settings.COL_List[Col]);

			/* I know... but it is probably somewhat quicker :-) */

			Quit:

			/* Store iterationvalue for the y pixel in the array */

			if(MemPrevYLine) PrevYLinePixel[XPos] = Iter;

			/* Store iterationvalue for the previous x pixel */

			LastIter = Iter;
		}
        
        /* Look for a keypress */
        
        if(keypressed()) {
            AltKey = readkey() >> 8;
            if(AltKey == KEY_ESC) { 
                if(MemPrevYLine) free(PrevYLinePixel);
                return 0;
            }
        }
	}

	if(MemPrevYLine) free(PrevYLinePixel);
	return 1;
}

/*
**	calculate complex escapevalue.
*/

double CalculateComplexFnEscapeValue(TComplexExp *ComplexExp)
{
	double		vr, vi, miny, maxy, cw, p;
    TCoords     Coords;
    TCGraphPar  GraphPar;

	Coords = *(ComplexExp->GraphPar[ComplexExp->Graph].Coords);
	GraphPar = ComplexExp->GraphPar[ComplexExp->Graph];

	cw = 0.0;

	if(ComplexExp->Graph == GR_C_ORBITS_C) {
		vr = GraphPar.zReal * GraphPar.zReal;
		vi = GraphPar.zImag * GraphPar.zImag;
	}
	else {
		vr = GraphPar.zReal * GraphPar.zReal;
		vi = GraphPar.zImag * GraphPar.zImag;
	}

	miny = Coords.YMin;
	maxy = Coords.YMax;

	cw = max(2.0, sqrt(vr + vi));

	if(cw == 0.0) p = 2.0; else p = 1.0;
	return(max(max(fabs(p * miny), fabs(p * maxy)), cw));
}

/*
**	Iterate function.
*/

unsigned int CIterateFn(double zReal, double zImag, double cReal, double cImag, 
    double EscapeValue, double PixelWidth, double PixelHeight, 
    unsigned int MaxPeriod, unsigned int MaxIter, int PeriodCheck, 
    double *RealRes, double *ImagRes, int *EscStatus)
{
	unsigned int	Iter, Period;
	double 			zRealTemp, wReal = zReal, wImag = zImag, zRealMul,
					zImagMul;
        
    *EscStatus = FALSE;
	Period = 0;
    
    for(Iter = MaxIter; Iter > 0; Iter--) {

		zRealMul = zReal * zReal;
		zImagMul = zImag * zImag;

		if(zRealMul + zImagMul > EscapeValue) {
            *EscStatus = TRUE;
            break;
        }
        
		zRealTemp = zReal;
		zReal = zRealMul - zImagMul + cReal;
		zImag = (zRealTemp + zRealTemp) * zImag + cImag;
    
		/*
		**	Check if the difference between z and z' is less than the size of
		**	one pixel, in which case the iteration cycle can end.
		*/

		if(PeriodCheck) {
			if(fabs(zReal - wReal) < PixelWidth &&
			fabs(zImag - wImag) < PixelHeight) {
				Iter = 0;
				break;
			}

			if(Period > MaxPeriod) {
				Period = 0;
				wReal = zReal;
				wImag = zImag;
			}
			Period++;
		}
	}
    *RealRes = zReal;
    *ImagRes = zImag;
	return Iter;
}

/*
**	Inverse function.
*/

void IterateInvFn(double *zReal, double *zImag, double cReal, double cImag)
{
	TComplex	z, c;

	z = Complex(*zReal, *zImag);
	c = Complex(cReal, cImag);
	z = CSqrt(CSub(z, c));
	*zReal = Real(z);
	*zImag = Imag(z);
}

/*
**  This routine can be used to jump from one graph to another.
**  For the current (source/original) graph the routine uses the member in 
**  TComplexExp, DestGraph is the graph to which should be jumped. 
**  InfoStr points to a text which should display the key to hit for help,
**  as HelpText should point to an array of lines for the actual help-text.
**  Mode selects whether to display a small image of the 'inverse Julia-set"
**  while a c is selected on the "Mandelbrot-set".
**  On 'cancel' the routine returns -1, on 'ok' it returns the selected graph. 
*/

int ComplexJump(TComplexExp *ComplexExp, int DestGraph, char *InfoStr, 
    char *HelpText[], int Mode)
{
    char        Str[255], *TmpInfoStr;
    int         Key, AltKey, XMickeys, YMickeys, tmp_xp, tmp_yp, tmp_b,
                JuliaPreview = TRUE;
    BITMAP      *JuliaBitmap, *MandelBkBitmap;
    TCoords     Coords;
    TComplexExp *SmallJuliaSet;
    
    Coords = *(ComplexExp->GraphPar[ComplexExp->Graph].Coords);
    
    if(ComplexExp->Graph == GR_C_ORBITS_C && DestGraph == GR_C_ORBITS_Z) {
        
        if(Mode) {
            SmallJuliaSet = CreateComplexExp(0, 0, 99, 99);
            
            /* 
            **  Restore palette which is altered by the Julia preview function. 
            */
    
            set_palette(ComplexExp->Settings.Palette);
            set_palette_range(ComplexExp->Settings.Palette, 0, 15, FALSE);

            SmallJuliaSet->Graph = GR_C_ORBITS_Z;
            SmallJuliaSet->Diagram = DM_INVERSE;
            SmallJuliaSet->GraphPar[GR_C_ORBITS_Z].IterMax = 500;
            
            MandelBkBitmap = create_bitmap(100, 100);
            blit(screen, MandelBkBitmap, 0, SCREEN_H - 120, 0, 0, 99, 99);
            
            JuliaBitmap = create_bitmap(100, 100);
            clear_bitmap(JuliaBitmap); 
        }

        if((TmpInfoStr = (char *) malloc(strlen(InfoStr) + 80)) == NULL) 
            return -2;
        
        strncpy(TmpInfoStr, InfoStr, strlen(InfoStr) + 1);

        if(DestGraph == GR_C_ORBITS_Z) 
            strcat(TmpInfoStr, " c = %2.*f + %2.*fi ");
        if(DestGraph == GR_C_ORBITS_C) 
            strcat(TmpInfoStr, " z = %2.*f + %2.*fi ");
        
        show_mouse(screen);
	    poll_mouse();
	    tmp_xp = mouse_x;
        tmp_yp = mouse_y;
        tmp_b = mouse_b;
        do {
            poll_mouse();
            tmp_b = mouse_b;
            get_mouse_mickeys(&XMickeys, &YMickeys);
            if((XMickeys != 0) || (YMickeys != 0)) {
                tmp_xp = mouse_x;
                tmp_yp = mouse_y;
                if(Mode && JuliaPreview) {
                    SmallJuliaSet->GraphPar[SmallJuliaSet->Graph].cReal = 
                    R2M_X(Coords, mouse_x);
                    SmallJuliaSet->GraphPar[SmallJuliaSet->Graph].cImag = 
                    R2M_Y(Coords, mouse_y);
                    SmallJuliaSet->GraphPar[SmallJuliaSet->Graph].Coords->Mode =
                    0;
                    SmallJuliaSet->InvJuliaIterMax = 150;
                    clear_bitmap(JuliaBitmap);
                    DrawComplexOrbits(JuliaBitmap, SmallJuliaSet);
                    acquire_screen();
                    blit(JuliaBitmap, screen, 0, 0, 0, SCREEN_H - 120, 99, 99);
                    release_screen();
                }
                sprintf(Str, TmpInfoStr, 4, R2M_X(Coords, mouse_x), 
                4, R2M_Y(Coords, mouse_y));
                StatusBar(ComplexExp->Settings.LAY_Statusline, -1, 
                ComplexExp->Graph, Str);
                
            }
            if(keypressed()) {
                Key = readkey();
                AltKey = Key >> 8;
                if(HelpText != NULL) 
			        if(AltKey == KEY_F1) InfoBox(HelpText, 5);
                if(Mode) { 
                    if(AltKey == KEY_P) { 
                        JuliaPreview = !JuliaPreview;
                        if(!JuliaPreview)
                            blit(MandelBkBitmap, screen, 0, 0, 0, 
                            SCREEN_H - 120, 99, 99);
                        else
                            blit(JuliaBitmap, screen, 0, 0, 0, SCREEN_H - 120, 
                            99, 99);
                    }
                }
            }
        } while(!(tmp_b & 1) && !(tmp_b & 2) && AltKey != KEY_ENTER && 
        AltKey != KEY_I && AltKey != KEY_ESC);
	
        show_mouse(NULL);

        free(TmpInfoStr);
        if(Mode) {
            blit(MandelBkBitmap, screen, 0, 0, 0, SCREEN_H - 120, 99, 99);
            FreeComplexExp(SmallJuliaSet);
            destroy_bitmap(JuliaBitmap);
            destroy_bitmap(MandelBkBitmap);
        }
        
	    if(AltKey == KEY_ESC) return(-1);
    }

    /* Step by step iteration  ->  Orbit diagram for var. z */

    if((ComplexExp->Graph == GR_C_STEP || ComplexExp->Graph == GR_C_STEPINV) &&
    DestGraph == GR_C_ORBITS_Z) {
        if(ComplexExp->Graph == GR_C_STEPINV) {
            ComplexExp->GraphPar[DestGraph].zReal = 
            ComplexExp->GraphPar[ComplexExp->Graph].zReal =
            ComplexExp->GraphPar[GR_C_STEPINV].zReal;         
            ComplexExp->GraphPar[DestGraph].zImag =  
            ComplexExp->GraphPar[ComplexExp->Graph].zReal =
            ComplexExp->GraphPar[GR_C_STEPINV].zReal;         
        }
        else { 
            ComplexExp->GraphPar[DestGraph].zReal = 
            ComplexExp->GraphPar[ComplexExp->Graph].zReal =
            ComplexExp->GraphPar[GR_C_STEP].zReal;         
            ComplexExp->GraphPar[DestGraph].zImag =  
            ComplexExp->GraphPar[ComplexExp->Graph].zReal =
            ComplexExp->GraphPar[GR_C_STEP].zReal;
        }
        ComplexExp->Graph = DestGraph;        
    }

    /* Orbit diagram for var. z  ->  Orbit diagram for var. c */

    if(ComplexExp->Graph == GR_C_ORBITS_Z && DestGraph == GR_C_ORBITS_C) {
        ComplexExp->Graph = DestGraph;   
        
        /* Set coordinates to default so that the entire M-set is displayed */

        SetComplexExpDefaultCoords(ComplexExp, DestGraph);
        
        /* Set z to 0 */
         
        ComplexExp->GraphPar[DestGraph].zReal = 0; 
        ComplexExp->GraphPar[DestGraph].zImag = 0; 
    }        

    /* Orbit diagram for var. c  ->  Orbit diagram for var. z */
    
    if(ComplexExp->Graph == GR_C_ORBITS_C && DestGraph == GR_C_ORBITS_Z) {
        ComplexExp->Graph = DestGraph;   
        
        /* If right mousebutton is clicked we switch to inverse */
        
        if((tmp_b & 2) || (AltKey == KEY_I)) ComplexExp->Diagram = DM_INVERSE;

        /* Set coordinates to default so that the entire J-set is displayed */
        
        SetComplexExpDefaultCoords(ComplexExp, DestGraph);
        
        /* Convert screen coordinates and copy them to real and imag c */
         
        ComplexExp->GraphPar[DestGraph].cReal = R2M_X(Coords, tmp_xp); 
        ComplexExp->GraphPar[DestGraph].cImag = R2M_Y(Coords, tmp_yp); 
    }
    
    /* Orbit diagram for var. z  ->  Step by step iteration */
    
    if(ComplexExp->Graph == GR_C_ORBITS_Z && 
    (DestGraph == GR_C_STEP || DestGraph == GR_C_STEPINV)) {
        
        /* Copy the entire GraphPar to destination graph */
        
        memcpy(&ComplexExp->GraphPar[DestGraph],
        &ComplexExp->GraphPar[GR_C_ORBITS_Z], sizeof(TCGraphPar));
        
        /* Set new graph */
        
        ComplexExp->Graph = DestGraph;
    }        
    return(ComplexExp->Graph);    
}

int ComplexDraw(TComplexExp *ComplexExp, char **StepHelpText, 
char **StepInvHelpText, char **JuliaNoMemStr, char **DefShapeHelpText)
{
    TCGraphPar          SGraphPar, SIGraphPar, JGraphPar;  
    BITMAP              *Bitmap;

    Bitmap = create_bitmap(SCREEN_W, SCREEN_H);
    clear_bitmap(Bitmap);
    
    SGraphPar = ComplexExp->GraphPar[GR_C_STEP];
    SIGraphPar = ComplexExp->GraphPar[GR_C_STEPINV];
    JGraphPar = ComplexExp->GraphPar[GR_C_ORBITS_Z];
    
    switch(ComplexExp->Graph) {
        case GR_C_STEP:
            if((memcmp(&SGraphPar, &JGraphPar, sizeof(TCGraphPar)) != 0) && 
            !JuliaVisible) 
                clear_bitmap(screen);
            DoComplexStep(screen, Bitmap, ComplexExp, JuliaVisible,
            StepHelpText, StepInvHelpText, JuliaNoMemStr, DefShapeHelpText);
            JuliaVisible = FALSE;
            break;
        case GR_C_STEPINV:
            if((memcmp(&SIGraphPar, &JGraphPar, sizeof(TCGraphPar)) != 0) && 
            !JuliaVisible) 
                clear_bitmap(screen);
            DoComplexStep(screen, Bitmap, ComplexExp, JuliaVisible,
            StepHelpText, StepInvHelpText, JuliaNoMemStr, DefShapeHelpText);
            JuliaVisible = FALSE;
            break;
        case GR_C_ORBITS_Z:
            JuliaVisible = TRUE;
            if(Bitmap != NULL) {
                DrawComplexOrbits(Bitmap, ComplexExp);
                blit(Bitmap, screen, 0,0, 0, 0, SCREEN_W, SCREEN_H); 
            } 
            else
                DrawComplexOrbits(screen, ComplexExp);
            break;
        case GR_C_ORBITS_C:
            JuliaVisible = FALSE;
            if(Bitmap != NULL) {
                DrawComplexOrbits(Bitmap, ComplexExp);
                blit(Bitmap, screen, 0,0, 0, 0, SCREEN_W, SCREEN_H); 
            } 
            else
                DrawComplexOrbits(screen, ComplexExp);
            break;
        default:
            break;            
    }            
    StatusBar(ComplexExp->Settings.LAY_Statusline, -1, ComplexExp->Graph, NULL);
    
    free(Bitmap);
    return 0;
}
