/*
	This file is part of XFuniter.

	realexp.c V2.2.3 - a real experiment (functiontype: from R to R)

	Copyright (C) 1995-2003 Stijn Wolters.
    Original idea: Ernic Kamerich (University of Nijmegen).
    
	See README for contact information.

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version. See

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

	See COPYING for more information.
*/

#include "realexp.h"

/*!\brief
**	Create GraphPar.
**
**  This routine returns a pointer to an instance of the TRGraphPar structure.
**
**\param    Coords Points to an instance of the TCoords structure, 
**          see coords.c for more details.
**\param    Iterated Represents how many times the function should be
**          iterated for a given x when drawing the function for
**          'step-by-step iteration' and the diagram 'combination
**          of web- and stepdiagram'.
**\param    IterMax Maximum number of iterations.
**\param    IterSkip Number of skipped iterations. These are calculated,
**          not displayed.
**\param    IterSteps The number of steps displayed at one, this value should
**          ofcourse be less than the value for IterMax.
**\param    xbegin1 First startingvalue for iteration.
**\param    xbegin2 Second startingvalue for iteration.
**\param    c Value for c.
**\param    EscapeValue The escapevalue, it is a number which is used
**          in CalculateRealFnEscapeValue to determine when a value tends to go
**          towards infinity.
*/

TRGraphPar *CreateRealGraphPar(TCoords *Coords, int Iterated,
	long int IterMax, long int IterSkip, long int IterSteps, double xbegin1,
	double xbegin2, double c, double EscapeValue)
{
	TRGraphPar		*GraphPar;

	if((GraphPar = (TRGraphPar *) malloc(sizeof(TRGraphPar))) == NULL)
		return NULL;

    SetRealGraphPar(GraphPar, Coords, c, xbegin1, xbegin2, EscapeValue, 
        Iterated, IterMax, IterSkip, IterSteps);

	return(GraphPar);
}

/*!\brief
**	Set values for GraphPar.
**
**  This routine returns nothing.
**
**\param    GraphPar Points to an instance of the TRGraphPar    
**          struct, see CreateGraphPar for more info.
**\param    Coords Points to an instance of the TCoords structure, 
**          see coords.c for more details.
**\param    c Value for c.
**\param    xbegin1 First startingvalue for iteration.
**\param    xbegin2 Second startingvalue for iteration.
**\param    EscapeValue The escapevalue, it is a number which is used
**          in CalculateRealFnEscapeValue to determine when a value tends to go
**          towards infinity.
**\param    Iterated Represents how many times the function should be
**          iterated for a given x when drawing the function for
**          'step-by-step iteration' and the diagram 'combination
**          of web- and stepdiagram'.
**\param    IterMax Maximum number of iterations.
**\param    IterSkip Number of skipped iterations. These are calculated,
**          not displayed.
**\param    IterSteps The number of steps displayed at one, this value 
**          should ofcourse be less than the value for IterMax.
*/

void SetRealGraphPar(TRGraphPar *GraphPar, TCoords *Coords, double c, 
    double xbegin1, double xbegin2, double EscapeValue, unsigned int Iterated,
    unsigned long int IterMax, unsigned long int IterSkip, 
    unsigned long int IterSteps)
{
	GraphPar->Coords = Coords;
	GraphPar->Iterated = Iterated;
	GraphPar->IterMax = IterMax;
	GraphPar->IterSkip = IterSkip;
	GraphPar->IterSteps = IterSteps;
	GraphPar->xbegin1 = xbegin1;
	GraphPar->xbegin2 = xbegin2;
	GraphPar->c = c;
	GraphPar->EscapeValue = EscapeValue;
}

/*!\brief
**  Destroy (free memory) for a GraphPar.
**
**  This routine returns nothing.
**
**\param    GraphPar Points to an instance of the TRGraphPar    
**          struct, see CreateGraphPar for more info.
*/

void FreeRealGraphPar(TRGraphPar *GraphPar)
{
	if(GraphPar != NULL) free(GraphPar);
}

/*!\brief
**  Load a RealGraphPar.
**
**  This routine returns 0 on success otherwise a value < 0.
**
**\param    fp File-pointer.
**\param    GraphPar Points to an instance of the TRGraphPar    
**          struct, see CreateGraphPar for more info.
*/

int LoadRealGraphPar(FILE *fp, TRGraphPar *GraphPar)
{
    TCoords     Coords;
    int         Res = 0;
    
	Coords = *(GraphPar->Coords);
    
    Res = fscanf(fp, "%lg,%lg,%lg\n", &GraphPar->c, &GraphPar->xbegin1, 
    &GraphPar->xbegin2);
    if(Res <= 0) return(Res);
        
    Res = fscanf(fp, "%lg,%lg,%lg,%lg\n", &Coords.XMin, &Coords.XMax, 
    &Coords.YMin, &Coords.YMax);
    if(Res <= 0) return(Res);
        
    Res = fscanf(fp, "%lg\n", &GraphPar->EscapeValue);
    if(Res <= 0) return(Res);
        
    Res = fscanf(fp, "%ld,%ld,%ld\n", &GraphPar->IterMax, &GraphPar->IterSteps, 
    &GraphPar->IterSkip);
    if(Res <= 0) return(Res);
        
    Res = fscanf(fp, "%d,%d\n", &GraphPar->Iterated, 
    &GraphPar->EscapeValueChanged);
    if(Res <= 0) return(Res);
    
    *(GraphPar->Coords) = Coords;
    
    return Res;
}

/*!\brief
**  Save a RealGraphPar.
**
**  This routine returns always 0. 
**
**\param    fp File-pointer.
**\param    GraphPar Points to an instance of the TRGraphPar    
**          struct, see CreateGraphPar for more info.
*/

int SaveRealGraphPar(FILE *fp, TRGraphPar *GraphPar)
{
    TCoords     Coords;

	Coords = *(GraphPar->Coords);

    fprintf(fp, "%.*g,%.*g,%.*g\n", DBL_DIG, GraphPar->c, 
    DBL_DIG, GraphPar->xbegin1, DBL_DIG, GraphPar->xbegin2);
        
    fprintf(fp, "%.*g,%.*g,%.*g,%.*g\n", DBL_DIG, Coords.XMin, 
    DBL_DIG, Coords.XMax, DBL_DIG, Coords.YMin, DBL_DIG, 
    Coords.YMax);
        
    fprintf(fp, "%.*g\n", DBL_DIG, GraphPar->EscapeValue);
        
    fprintf(fp, "%ld,%ld,%ld\n", GraphPar->IterMax, GraphPar->IterSteps, 
    GraphPar->IterSkip);
        
    fprintf(fp, "%d,%d\n", GraphPar->Iterated, GraphPar->EscapeValueChanged);
    
    return 0;
}

/*!\brief
**  Creates a RealExperiment including the three GraphPar's.
**
**  This function returns a pointer to an instance of the TRealExp structure.
**
**\param    XStart X begin of the field (screen coordinates).
**\param    YStart Y begin of the field (screen coordinates).   
**\param    XEnd X end of the field (screen coordinates). 
**\param    YEnd Y end of the field (screen coordinates).
**
**\note
**  All members are set to the default values, including the GraphPar's
*/

TRealExp *CreateRealExp(unsigned int XStart, unsigned int YStart,
	unsigned int XEnd, unsigned int YEnd)
{
	TRealExp		*RealExp;
	TCoords			*StepCoords, *OrbitsXCoords, *OrbitsCCoords;

	if((RealExp = (TRealExp *) malloc(sizeof(TRealExp))) == NULL) return NULL;

	/* Step by step iteration */

	StepCoords = CreateCoords(-2, -2, 2, 2, XStart, YStart, XEnd, YEnd, 56);
	RealExp->GraphPar[GR_R_STEP] = 
		*CreateRealGraphPar(StepCoords, 1, 100, 0, 1, 1.4, 1.4, -0.7, 2);

	/* Orbits for variable x */

	OrbitsXCoords =
		CreateCoords(-2.5, -2, 2.5, 2, XStart, YStart, XEnd, YEnd, 56);
    RealExp->GraphPar[GR_R_ORBITS_X] =	
		*CreateRealGraphPar(OrbitsXCoords, 1, 250, 0, 1, 1.4, 1.4, -1.41, 2);

	/* Orbits for variable c */

	OrbitsCCoords =
		CreateCoords(-2.5, -2, 0.5, 2, XStart, YStart, XEnd, YEnd, 56);
	RealExp->GraphPar[GR_R_ORBITS_C] = 
		*CreateRealGraphPar(OrbitsCCoords, 1, 250, 50, 1, 0, 0, -0.7, 2);

    /* Set default escapevalues for the graphs */
    
	RealExp->Graph = GR_R_STEP;
	RealExp->GraphPar[GR_R_STEP].EscapeValue = 
        CalculateRealFnEscapeValue(RealExp);
	RealExp->GraphPar[GR_R_STEP].EscapeValueChanged = FALSE;

	RealExp->Graph = GR_R_ORBITS_X;
	RealExp->GraphPar[GR_R_ORBITS_X].EscapeValue =
        CalculateRealFnEscapeValue(RealExp);
	RealExp->GraphPar[GR_R_ORBITS_X].EscapeValueChanged = FALSE;

	RealExp->Graph = GR_R_ORBITS_C;
	RealExp->GraphPar[GR_R_ORBITS_C].EscapeValue =
        CalculateRealFnEscapeValue(RealExp);
	RealExp->GraphPar[GR_R_ORBITS_C].EscapeValueChanged = FALSE;

	/* Settings */

	RealExp->Settings = *CreateSettings();

    /* Initial values for graph, diagram, function, etc */
    
	RealExp->Graph = GR_R_STEP;
	RealExp->Diagram = DM_WEB;
	RealExp->Function = 0;
    
    strcpy(RealExp->FunctionStr, "x*x+c");
    
	return(RealExp);
}

/*!\brief
**  Destroy (free memory) for a RealExperiment.
**
**  This function returns nothing.
**
**\param    RealExp Points to an instance of the TRealExp    
**          struct, see CreateGraphPar and CreateRealExp for more info.
*/

void FreeRealExp(TRealExp *RealExp)
{
	if(RealExp != NULL) free(RealExp);
}

/*!\brief
**  Load RealExperiment.
**
**  This function returns 0 or 1 on succes, which is the functiontype, 
**  otherwise a value less than 0.
**
**\param    RealExp Points to an instance of the TRealExp    
**          struct, see CreateGraphPar and CreateRealExp for more info.
**\param    FileName path + filename of the 'experiment' which should be
**          loaded.
**\param    Mode FALSE: experiments should be loaded in the new style, TRUE:
**          they should be loaded in old style. 
*/

int LoadRealExp(TRealExp *RealExp, char *FileName, int Mode)
{
    FILE    *fp;
    int     Res = 0, Index, Graph, EscapeValueChanged, Functiontype = 0;
    char    *fm = (Mode) ? "r" : "rb";
        
	if((fp = fopen(FileName, fm)) == NULL) return -1;

    /* Experimental code to read the old prm format */
    
    if(!Mode) {
        for(Index = 0; Index <= 2; Index++) {
           
            switch(Index) {
                case 0:
                    Graph = GR_R_STEP;  
                    break;
                case 1:
                    Graph = GR_R_ORBITS_X;
                    break;
                case 2:
                    Graph = GR_R_ORBITS_C;
                    break;
            }
        
            fread(&RealExp->GraphPar[Graph].c,1,sizeof(double),fp);
            fread(&RealExp->GraphPar[Graph].xbegin1,1,sizeof(double),fp);
            fread(&RealExp->GraphPar[Graph].xbegin2,1,sizeof(double),fp);
            fread(&RealExp->GraphPar[Graph].Coords->XMin,1,sizeof(double),fp);
            fread(&RealExp->GraphPar[Graph].Coords->XMax,1,sizeof(double),fp);
            fread(&RealExp->GraphPar[Graph].Coords->YMin,1,sizeof(double),fp);
            fread(&RealExp->GraphPar[Graph].Coords->YMax,1,sizeof(double),fp);
            fread(&RealExp->GraphPar[Graph].EscapeValue,1,sizeof(double),fp);
         
            fread(&RealExp->GraphPar[Graph].IterMax,1,sizeof(int),fp);
            fread(&RealExp->GraphPar[Graph].IterSkip,1,sizeof(int),fp);
            fread(&RealExp->GraphPar[Graph].IterSteps,1,sizeof(int),fp);

            fread(&RealExp->GraphPar[Graph].EscapeValue,1,sizeof(double),fp);
        }
        
        fseek(fp, 0x30e, SEEK_SET);
        
        fread(&RealExp->Graph, 1, sizeof(int), fp);
        
        fseek(fp, 0x318, SEEK_SET);
        
        fread(&RealExp->Function, 1, sizeof(int), fp);
        
        fseek(fp, 0x320, SEEK_SET);
        fread(&EscapeValueChanged, 1, sizeof(int), fp);
        RealExp->GraphPar[GR_R_STEP].EscapeValueChanged=EscapeValueChanged;
        RealExp->GraphPar[GR_R_ORBITS_X].EscapeValueChanged=EscapeValueChanged;
        RealExp->GraphPar[GR_R_ORBITS_C].EscapeValueChanged=EscapeValueChanged;
        
        fread(RealExp->FunctionStr, 100, sizeof(char), fp);
    }
    
    /* Read new par format */

    else {
        fscanf(fp, "%d", &Functiontype);
        
        if(Functiontype == 0) {
            Res = LoadRealGraphPar(fp, &RealExp->GraphPar[GR_R_STEP]);
            Res = LoadRealGraphPar(fp, &RealExp->GraphPar[GR_R_ORBITS_X]);
            Res = LoadRealGraphPar(fp, &RealExp->GraphPar[GR_R_ORBITS_C]);

            Res = fscanf(fp, "%d,%d,%d,%s\n", &RealExp->Graph, &RealExp->Diagram, 
            &RealExp->Function, RealExp->FunctionStr); 
    
            LoadSettings(fp, &RealExp->Settings);    
        }
    }
    
	fclose(fp);
	return Functiontype;
}

/*!\brief
**  Save RealExperiment.
**
**  This function returns 0 on succes, otherwise a value less than 0.
**
**\param    RealExp Points to an instance of the TRealExp    
**          struct, see CreateGraphPar and CreateRealExp for more info.
**\param    FileName path + filename of the 'experiment' which should be
**          loaded.
*/

int SaveRealExp(TRealExp *RealExp, char *FileName)
{
	FILE        *fp;
    
	if((fp = fopen(FileName, "w+")) == NULL) return -1;

    SaveRealGraphPar(fp, &RealExp->GraphPar[GR_R_STEP]);
    SaveRealGraphPar(fp, &RealExp->GraphPar[GR_R_ORBITS_X]);
    SaveRealGraphPar(fp, &RealExp->GraphPar[GR_R_ORBITS_C]);
        
    fprintf(fp, "%d,%d,%d,%s\n", RealExp->Graph, RealExp->Diagram, 
    RealExp->Function, RealExp->FunctionStr); 
    
    SaveSettings(fp, &RealExp->Settings);    
      
	fclose(fp);
	return 0;
}


/*!\brief
**  Calculate escapevalue for real experiments.
**
**  This function returns the escapevalue.
**
**\param    RealExp Points to an instance of the TRealExp    
**          struct, see CreateGraphPar and CreateRealExp for more info.	
**
*/

double CalculateRealFnEscapeValue(TRealExp *RealExp)
{
	double		cvalue, maxc, minc, miny, maxy, cw, p;
    TCoords     Coords;
    TRGraphPar  GraphPar;

	Coords = *(RealExp->GraphPar[RealExp->Graph].Coords);
	GraphPar = RealExp->GraphPar[RealExp->Graph];
    
    /*! 
    **  User changed escapevalue, unless the flag is reset it aborts this 
    **  function.
    */
    
    if(GraphPar.EscapeValueChanged) return(GraphPar.EscapeValue);

	cw = 0.0;

	cvalue = GraphPar.c;
	minc = maxc = cvalue;
	miny = Coords.YMin;
	maxy = Coords.YMax;

	/*!
	**	When the graph is 'Orbits for variable c', for each c
	**	the escape-value is calculated.
	*/

	if(RealExp->Graph == GR_R_ORBITS_C) {
		minc = Coords.XMin;
		maxc = Coords.XMax;
	}

	switch(RealExp->Function) {

		/*!
		**	x->x*x+c
		*/

		case 0:
			if(cvalue <= 0.25)
				cw = 1.0 / 2.0 * (1.0 + sqrt(1.0 - 4.0 * cvalue));
			else
				cw = 0.5;
            break;
		/*!
		**	x->frac(cx)
		*/

		case 1:
			cw = 1.0;
            break;

		/*!
		**	x->c * cos(x)
		*/

		case 2:
			if(RealExp->Graph == GR_R_STEP || RealExp->Graph == GR_R_ORBITS_X)
				cw = fabs(cvalue);
			else
				cw = max(fabs(minc), fabs(maxc));
            break;
            
		/*!
		**	x->cx(1-x)
		*/

		case 3:
			if(minc > 0.0)		cw = 1.0;
			else if(maxc < 0.0) cw = 0.5 * (1.0 + sqrt(1.0 - 4.0 / maxc));
			else				cw = 1000000000000000.0;
            break;
            
		/*!
		**	x->-cx^3+(c+1)x^2
		*/

		case 4:
			if(cvalue > 0.0)
				cw = max(1.0, fabs(2.0 / cvalue) + 1.0);
			else if((cvalue > -1.0) && (cvalue < 0))
				cw = -1.0 / cvalue;
			else if(cvalue <= -1.0)
				cw = 1.0;
		    break;
            
        /*!
		**	x->c(1-|2x-1|)
		*/

		case 5:
			cw = 1.0;
            break;
            
		/*
		**	userdefined function
		*/

		case 6:
			cw = 0.0;
			break;
			
            /*
			if(RealExp->GraphPar[RealExp->Graph].EscapeFunctionStr
			!= NULL)
				cw = evalx(RealExp->GraphPar[RealExp->Graph].
				EscapeFunctionStr, 3, 'C', cvalue, 'M', maxc, 'N', minc);
			else
				cw = 0.0;
			*/
	}

	if(cw == 0.0) p = 2.0; else p = 1.0;
	return(max(max(fabs(p * miny), fabs(p * maxy)), cw));
}

/*!\brief
**  Draw the function for 'step-by-step iteration' web and combi web, step.
**
**  This function returns nothing.
**
**\param    Bitmap Points to an instance of the Allegro BITMAP structure.
**\param    RealExp Points to an instance of the TRealExp    
**          struct, see CreateGraphPar and CreateRealExp for more info.	
**
*/

void DrawFunction(BITMAP *Bitmap, TRealExp *RealExp)
{
    unsigned int    XPos;
    double          x, y, XLast, YLast, XEnd, YStart;     
    TCoords         Coords;
    TRGraphPar      GraphPar;
    
	Coords = *(RealExp->GraphPar[RealExp->Graph].Coords);
	GraphPar = RealExp->GraphPar[RealExp->Graph];

    /* 
    **  Use only the lower half of the bitmap in case the diagram is
    **  Combination of web and step.
    */
    
    if(RealExp->Diagram == DM_COMBIWEBSTEP) {
        XEnd = Coords.XEnd;
        YStart = Coords.YStart;
        Coords.XEnd = Coords.XEnd / 2;
        Coords.YStart = Coords.YStart + Coords.YEnd / 2;
    }
    
    acquire_bitmap(Bitmap);
        
	if(RealExp->Diagram == DM_WEB || RealExp->Diagram == DM_COMBIWEBSTEP) {

		/* Draw the function */

		x = R2M_X(Coords, RealExp->GraphPar[RealExp->Graph].xbegin1);
		XLast = x;
		YLast = y = M2R_Y(Coords,
        IterateFn(RealExp->FunctionStr, RealExp->Function, GraphPar.Iterated,
		x, GraphPar.c));

		for(XPos = 0; XPos < Coords.XEnd; XPos++) {
			x = R2M_X(Coords, XPos);
        	y = IterateFn(RealExp->FunctionStr, RealExp->Function,
			GraphPar.Iterated, x, GraphPar.c);
			if(y <= Coords.YMax && y >= Coords.YMin)
				line(Bitmap, XPos, M2R_Y(Coords, y), XLast, YLast,
                RealExp->Settings.COL_List[RealExp->Settings.COL_Function]);
			XLast = XPos; YLast = M2R_Y(Coords, y);
		}

		/* Draw the line x=y */
        
		for(XPos = 0; XPos < Coords.XEnd; XPos++) {
            y = R2M_X(Coords, XPos);
            putpixel(Bitmap, XPos, (int) M2R_Y(Coords, y),
            RealExp->Settings.COL_List[RealExp->Settings.COL_XYLine]);
        }
	}

    release_bitmap(Bitmap);
    vsync();
    
    /* Draw grid */
    
    DrawGrid(Bitmap, font, &Coords);

    /* Restore old values. */
    
    if(RealExp->Diagram == DM_COMBIWEBSTEP) {
        Coords.XEnd = XEnd;
        Coords.YStart = YStart;
    }
    
}

/*!\brief
**  Draw one iteration step ('step-by-step iteration, all diagrams).
**
**  This function returns the current step's line position.
**
**\param    Bitmap Points to an instance of the Allegro BITMAP structure.
**\param    RealExp Points to an instance of the TRealExp    
**          struct, see CreateGraphPar and CreateRealExp for more info.	
**\param    CurStepLinePos The line position of the current step.
**\param    startx The startingvalue (like xbegin1 and xbegin2).
**\param    x 
**\param    y
**\param    Col Drawingcolor for the current step.
*/

int DrawIterStep(BITMAP *Bitmap, TRealExp *RealExp, int CurStepLinePos,
double startx, double x, double y, int Col)
{
	int         x1, y1, x2, y2, XEnd, YStart;
    TCoords     Coords;
        
    /* Some shortcuts */
    
	Coords = *(RealExp->GraphPar[RealExp->Graph].Coords);

    /* In case of combination web/step we use the lower half of the screen */
    
    if(RealExp->Diagram == DM_COMBIWEBSTEP) {
        XEnd = Coords.XEnd;
        YStart = Coords.YStart;
        Coords.XEnd = Coords.XEnd / 2;
        Coords.YStart = Coords.YStart + (Coords.YEnd / 2);
    }

    /* Convert coordinates */

	x1 = (int) M2R_X(Coords, x);
	y1 = (int) M2R_Y(Coords, x);
	x2 = (int) M2R_X(Coords, y);
	y2 = (int) M2R_Y(Coords, y);

    /* Draw lines */

    acquire_bitmap(Bitmap);
    if(RealExp->Diagram == DM_WEB || RealExp->Diagram == DM_COMBIWEBSTEP) {
	    vline(Bitmap, x1, y1, y2, Col);
		hline(Bitmap, x1, y2, x2, Col);
		if(RealExp->Diagram == DM_WEB)
		    hline(Bitmap, 0, y2, 10, Col);
		if(RealExp->Diagram == DM_COMBIWEBSTEP) {
			hline(Bitmap, Coords.XEnd + CurStepLinePos, y2, Coords.XEnd +
            CurStepLinePos + RealExp->Settings.SSI_LineWidth, Col);
			CurStepLinePos+=RealExp->Settings.SSI_LineWidth;
		}
	}
	else if(RealExp->Diagram == DM_ITERVALUES)
		hline(Bitmap, M2R_X(Coords, startx), y2, M2R_X(Coords, startx) +
        RealExp->Settings.SSI_LineWidth, Col);
	else if(RealExp->Diagram == DM_STEP || RealExp->Diagram == DM_COMBIWEBSTEP) {
		hline(Bitmap, 50 + CurStepLinePos, y2, 50 + CurStepLinePos +
		RealExp->Settings.SSI_LineWidth, Col);
		CurStepLinePos+=RealExp->Settings.SSI_LineWidth;
	}
    release_bitmap(Bitmap);
    vsync();

    /* Restore old values. */

    if(RealExp->Diagram == DM_COMBIWEBSTEP) {
        Coords.XEnd = XEnd;
        Coords.YStart = YStart;
    }
    return(CurStepLinePos);
}

/*!\brief
**  Step-by-step iteration, all diagrams.
**
**  This function returns the current step's line position.
**
**\param    Bitmap Points to an instance of the Allegro BITMAP structure.
**\param    RealExp Points to an instance of the TRealExp    
**          struct, see CreateGraphPar and CreateRealExp for more info.	
**\param    ValueMode      
**\param    HelpText A pointer to strings with helptexts for the current graph.
*/

int DoRealStep(BITMAP *Bitmap, TRealExp *RealExp, int ValueMode, 
    char **HelpText)
{
	double			    x, y, xx, yy, xr, yr, TempWidth;
    int                 XWidth;
    unsigned int        xa, ya, CurStepLinePos = 0, Col, CurCol, ColUsed, 
                        TempValueMode, Key, ExtKey, Values = 1, 
                        XMickeys, YMickeys, XEnd, YStart;
	unsigned long int	Iter;
	char			    v1[40], v2[40];
    TCoords             Coords;
    TRGraphPar		    GraphPar;

    /* Some shortcuts */

	Coords = *(RealExp->GraphPar[RealExp->Graph].Coords);
	GraphPar = RealExp->GraphPar[RealExp->Graph];

    /* Determine the number of startingvalues */

    Values = (GraphPar.xbegin1 != GraphPar.xbegin2) + 1;

    /* Set the number of digits after the dot */

	TempWidth = (Coords.XMax - Coords.XMin);
	XWidth = 0;
    while(((int) TempWidth) == 0) { TempWidth *= 10.0; XWidth++; }
    XWidth+=2;

	/*
	**	For 'Webdiagram' and 'Combination: web- and stepdiagram'
    **	the selected function is drawn and the x=y line.
	*/

    DrawFunction(Bitmap, RealExp);

    /* Display help-text, user has to hit a key to continue */

	if(HelpText != NULL) InfoBox(HelpText, 2);

    /* Store the valuemode */

    TempValueMode = ValueMode;

	/*
    **  In case Values is set to 0 (selecting startingvalue with the mouse),
    **  the mouse is set to the location of the 1st startingvalue in the
    **  parametermenu.
    */

   	if(Values == 0)
        position_mouse((int) M2R_X(Coords,
        RealExp->GraphPar[RealExp->Graph].xbegin1),
		(int) M2R_Y(Coords, RealExp->GraphPar[RealExp->Graph].xbegin1));

    /* Start main-loop */

	CurCol = 1;
	do {

        /*
        **  In case 1 or 2 startingvalues are selected the initial values are
        **  set, otherwise the mouse-position is the new initial value.
        */

    	if(Values == 1 || Values == 2) {
			ValueMode = 2;
			x = y = RealExp->GraphPar[RealExp->Graph].xbegin1;
			xx = yy = RealExp->GraphPar[RealExp->Graph].xbegin2;
		}
		if(Values == 0) {
	        poll_mouse();
			x = xx = y = yy = R2M_X(Coords, mouse_x);
		}

        /*
        **  Set starting iteration number,
        **  line position  for step-diagram is 0.
        */

        Iter = 0L;
		CurStepLinePos = 0;

        /* Start iteration-loop */

        while(Iter <= MAXINT) {

			if(Values == 0) {
				Values = 0;
                unscare_mouse();
				while(!keypressed()) {
					poll_mouse();

                    if(RealExp->Diagram == DM_COMBIWEBSTEP) {
                        XEnd = Coords.XEnd;
                        YStart = Coords.YStart;
                        Coords.XEnd = Coords.XEnd / 2;
                        Coords.YStart = Coords.YStart + (Coords.YEnd / 2);
                    }

                    xr = R2M_X(Coords, mouse_x);
					if(RealExp->Diagram == DM_WEB ||
                    RealExp->Diagram == DM_COMBIWEBSTEP)
						yr = xr;
					xa = (int) M2R_X(Coords, xr);
					if(RealExp->Diagram == DM_WEB ||
                    RealExp->Diagram == DM_COMBIWEBSTEP)
						ya = (int) M2R_Y(Coords, yr);
                    
                    if(RealExp->Diagram == DM_COMBIWEBSTEP) {
                        Coords.XEnd = XEnd;
                        Coords.YStart = YStart;
                    }

                    get_mouse_mickeys(&XMickeys, &YMickeys);
                    if((XMickeys != 0) || (YMickeys != 0)) {
                        if(RealExp->Diagram == DM_WEB)
                            position_mouse(xa, ya);
                        else if(RealExp->Diagram == DM_COMBIWEBSTEP) {
                            position_mouse(xa, ya);
                            /*
                            printf("O: x: %d, y: %d N: x: %d, y: %d\n",
                            xa, ya, xa / 2, Coords.YEnd / 2 + ya  / 2);
                            */
                        }
                        else
                            position_mouse(xa, Coords.YEnd);

						sprintf(v1, "x = %2.*f", XWidth, xr);
                        StatusBar(RealExp->Settings.LAY_Statusline, 
                        RealExp->Graph, -1, v1);
					    unscare_mouse();

                        /*
                        **  Since a new startingvalue is selected, the
                        **  starting-iteration-number and current line position
                        **  for stepdiagram are reset to 0 again.
                        */

						Iter = 0L;
						CurStepLinePos = 0;

                        /*
                        **  Since this is the select-with-mouse mode,
                        **  Only one startingvalue is used (xbegin1 and xbegin2
                        **  will be made equal).
                        */

						x = y = RealExp->GraphPar[RealExp->Graph].xbegin1 =
						RealExp->GraphPar[RealExp->Graph].xbegin2 = xr;
					}
				}
                scare_mouse();
			}
			else {
				do {
					poll_mouse();
					get_mouse_mickeys(&XMickeys, &YMickeys);
                    if((XMickeys != 0) || (YMickeys != 0)) {
						Values = 0;
						ValueMode = TempValueMode;
						clear_keybuf();
						show_mouse(screen);
						break;
					}
				} while((!keypressed()) && XMickeys == 0 && YMickeys == 0);
			}
			if(keypressed() && (Iter % (long int) GraphPar.IterSteps) == 0) {
				Key = readkey();
                ExtKey= Key >> 8;
            }

			if(Iter == 0 && ValueMode == 0) CurCol++;

			if(ExtKey == KEY_ESC) {
				scare_mouse();
				return 1;
			}

			if(ExtKey == KEY_DEL) {
				if(Values == 0) scare_mouse();
                rectfill(Bitmap, Coords.XStart, Coords.YStart,
                Coords.XEnd, Coords.YEnd, 0);
                DrawFunction(Bitmap, RealExp);
				if(Values == 0) unscare_mouse();
				break;
			}

            /* Show help-text */

			if(ExtKey == KEY_F1) {
				if(HelpText != NULL) InfoBox(HelpText, 2);
				continue;
			}

			/* Draw an iterationstep (or multiple at once) */

            if(ExtKey == KEY_ENTER && Iter >= GraphPar.IterSkip) {

                /*
                **  Print text next to each iterationstep
                */

                /*
				if((Iter == 0L || l == GraphPar.IterSkip) &&
				(RealExp->Diagram == DM_ITERVALUES ||
				RealExp->Diagram == DM_STEP) &&
				(RealExp->Settings.SSI_Text)) {
					settextstyle(SMALL_FONT, HORIZ_DIR,
					RealExp->Settings->SSI_FontSize);
					sprintf(v1, "%2.*f", XWidth, x);
					outtextxy(M2R_X(*(GraphPar.Coords), x),
                    M2R_Y(*(GraphPar.Coords), x), v1);
					if(Values == 2) {
						sprintf(v1, "%2.*f", XWidth, xx);
						outtextxy(M2R_X(*(GraphPar.Coords), xx),
								  M2R_Y(*(GraphPar.Coords), xx), v1);
					}
				}
                */

				/*
				**	Select the colors which are used for drawing the
				**	iterationstep(s).
				*/

				switch(ValueMode) {
					case 0:
						if(CurCol == 0 ||
                        CurCol == RealExp->Settings.COL_Function
						|| CurCol == RealExp->Settings.COL_XYLine)
							CurCol++;
						if(CurCol == 15)
							CurCol = 1;
                        Col = RealExp->Settings.COL_List[CurCol %
                        RealExp->Settings.COL_Max];
						break;
					case 1:
						ColUsed = (int) (Iter % (long)
                        RealExp->Settings.COL_Max);
						if(ColUsed == 0 ||
						ColUsed == RealExp->Settings.COL_Function
						|| ColUsed == RealExp->Settings.COL_XYLine)
							CurCol++;
						if(ColUsed == 15) ColUsed = 1;
                        Col = RealExp->Settings.COL_List[ColUsed];
						break;
					case 2:
                    	Col =
						RealExp->Settings.COL_List[RealExp->Settings.COL_Iter1];
						break;
				}

                /* Draw the lines */

                CurStepLinePos = DrawIterStep(Bitmap, RealExp, CurStepLinePos,
                RealExp->GraphPar[RealExp->Graph].xbegin1, x, y, Col);

                sprintf(v1, "n=%-4ld x=%.*f", Iter, 6, y);
                
                if(Values != 2) 
                    StatusBar(RealExp->Settings.LAY_Statusline, RealExp->Graph, 
                    -1, v1);
                
                if(Values == 2) {
                    CurStepLinePos = DrawIterStep(Bitmap, RealExp,
                    CurStepLinePos, RealExp->GraphPar[RealExp->Graph].xbegin2,
                    xx, yy, RealExp->Settings.COL_Iter2);

                    sprintf(v2, " x=%.*f", 6, yy);
                    strcat(v1, v2);
                    StatusBar(RealExp->Settings.LAY_Statusline, RealExp->Graph, 
                    -1, v1);
			    }
            }
			x = y;
			if(Values == 2)
				xx = yy;

			if(abs(x) <= GraphPar.EscapeValue) {
        		y = IterateFn(RealExp->FunctionStr, RealExp->Function,
				GraphPar.Iterated, x, GraphPar.c);
			}
			else if(Values < 2) {
				/*sound(1000); delay(dt); nosound(); dt = 20;*/
				continue;
			}
			else
				{ /*sound(1000); delay(dt); nosound(); dt = 20;*/ }

			if(Values == 2) {
				if(abs(xx) <= GraphPar.EscapeValue) {
        			yy = IterateFn(RealExp->FunctionStr, RealExp->Function,
					GraphPar.Iterated, xx, GraphPar.c);
				}
				else
					{ /*sound(1000); delay(dt); nosound(); dt = 20;*/ }
			}
			Iter++;
		}
	} while(TRUE);
}

/*
**	DrawRealOrbits	-	Draw an orbitdiagram for variable c or variable x.
**
**	This function requires iterfun.c which requires parser.c.
**
**	DrawRealOrbits returns 0 when completed, and 1 when calculations are
**	interrupted by the user due to pressing the escape key.
*/

int DrawRealOrbits(BITMAP *Bitmap, TRealExp *RealExp)
{
	TRGraphPar			GraphPar;
    TCoords             Coords;
	unsigned int		ExitValue = 0, Col, XPos, YPos, y0;
	unsigned long int	i;
	double				ca, xa, EscapeValue;

	/* Shortcuts for some very often used parameters */

	Coords = *(RealExp->GraphPar[RealExp->Graph].Coords);
	GraphPar = RealExp->GraphPar[RealExp->Graph];

	if(RealExp->Graph == GR_R_ORBITS_X) EscapeValue = GraphPar.EscapeValue;

	/*
	**	Store the original value for c, since that is used for the
	**	EscapeValue calculation for each c.
	*/

	ca = RealExp->GraphPar[RealExp->Graph].c;

	/* Convert the 0.0 position on the Y-axe to a screen coordinate */

	y0 = M2R_Y(Coords, 0.0);

	/* Start a loop for each X position */

	for(XPos = 0; XPos < Coords.XEnd; XPos++) {
        
		/*
		**	Orbitdiagram for variable x.
		*/

		if(RealExp->Graph == GR_R_ORBITS_X)
			xa = R2M_X(*(GraphPar.Coords), XPos);

		/*
		**	Orbitdiagram for variable c.
		**
		**	For 'Orbitdiagram for variable c' we calculate the escape-
		**	value for each c.
		*/

		if(RealExp->Graph == GR_R_ORBITS_C) {
			RealExp->GraphPar[RealExp->Graph].c =
			R2M_X(*(GraphPar.Coords), XPos);
			EscapeValue = CalculateRealFnEscapeValue(RealExp);
			xa = GraphPar.xbegin1;
		}

		/* Iterationloop */

		for(i = 0; i < GraphPar.IterMax; i++) {

			/*
			**	Calculate Y position only when i is larger than the
			**	number of skipped iterations.
			*/

			if(i >= GraphPar.IterSkip) {
				YPos = M2R_Y(Coords, xa);
				if(YPos <= Coords.YEnd)
					putpixel(Bitmap, XPos, YPos,
					RealExp->Settings.COL_List[RealExp->Settings.COL_Function]);
			}

			/*
			**	Call routine which iterates the function.
			**	For the orbitdiagrams only the 1st iterated is calculated.
			*/

			xa = IterateFn(RealExp->FunctionStr, RealExp->Function, 1,
			xa, RealExp->GraphPar[RealExp->Graph].c);

			/*
			**	Plot the escape colors when the absolute value of the
			**  function result is larger than the escape value.
			*/

			if(fabs(xa) > EscapeValue) {
				if(i == 0)
					Col = RealExp->Settings.COL_Max;
				else if(i == GraphPar.IterMax)
					Col = 0;
				else {
					Col = RealExp->Settings.COL_Max -
                    (i % RealExp->Settings.COL_Max);
					if(Col == 0)
						Col++;
					else if(Col == RealExp->Settings.COL_Max)
						Col--;
				}
				vline(Bitmap, XPos, y0 - 3, y0 + 3,
                RealExp->Settings.COL_List[Col]);
				break;
			}
            
            /* Check for keypress */
            
            if(keypressed()) 
                if((readkey() >> 8) == KEY_ESC) 
                    ExitValue = 1;
		}
	}

	/* Restore original value for c */

	RealExp->GraphPar[RealExp->Graph].c = ca;

	/* Draw grid */

	DrawGrid(Bitmap, font, &Coords);
	return ExitValue;
}

/*
**	IterateFn
**
**	Iterate a function.
**
**	FunctionStr			- 	char, Function string for the parser.
**
**	Function			- 	int, Function-number.
**
**	n					-	int, n-th iterated of function.
**
**	x					- 	double, x-parameter.
**
**	c					- 	double, c-parameter.
**
**	The routine returns the function-result.
*/

double IterateFn(char *FunctionStr, int Function, int n, double x, double c)
{
	int		i;
	double 	y;

	switch(Function) {

		/*
		**	x->x*x+c
		*/

		case 0:
			for(i = 1; i <= n; i++) {
				y = x * x + c;
				x = y;
			}
			break;

		/*
		**	x->frac(cx)
		*/

		case 1:
			for(i = 1; i <= n; i++) {
				y = c * x - floor(c*x);
				x = y;
			}
			break;

		/*
		**	x->c * cos(x)
		*/

		case 2:
			for(i = 1; i <= n; i++) {
				y = c * cos(x);
				x = y;
			}
			break;

		/*
		**	x->cx(1-x)
		*/

		case 3:
			for(i = 1; i <= n; i++) {
				y = c * x * (1.0 - x);
				x = y;
			}
			break;

		/*
		**	x->-cx^3+(c+1)x^2
		*/

		case 4:
			for(i = 1; i <= n; i++) {
				y = -c * x * x * x + (c + 1.0) * x * x;
				x = y;
			}
			break;

		/*
		**	x->c(1-|2x-1|)
		*/

		case 5:
			for(i = 1; i <= n; i++) {
				y = c * (1.0 - fabs(2.0 * x - 1.0));
				x = y;
			}
			break;

		/*
		**	userdefined function
		*/

		case 6:
			for(i = 1; i <= n; i++) {
				/*
                y = evalx(FunctionStr, 2, 'X', x, 'C', c);
				x = y;
                */
			}
			break;
	}
	return(y);
}

/*
**  This routine can be used to jump from one graph to another.
**  For the current (source/original) graph the routine uses the member in 
**  TRealExp, DestGraph is the graph to which should be jumped. 
**  Dialog points to a DIALOG structure which contains fields to enter the
**  value for x or c manually (not yet implemented).
**  InfoStr points to a text which should display the key to hit for help,
**  as HelpText should point to an array of lines for the actual help-text.
**  On 'cancel' the routine returns -1, on 'ok' it returns the selected graph. 
*/

int RealJump(TRealExp *RealExp, int DestGraph, DIALOG *Dialog, 
const char *InfoStr, char *HelpText[])
{
    char        Str[80], *TmpInfoStr;
    int         Key, AltKey, XMickeys, YMickeys, tmp_xp;
    TCoords     Coords;
        
    Coords = *(RealExp->GraphPar[RealExp->Graph].Coords);

    if((RealExp->Graph == GR_R_ORBITS_X || RealExp->Graph == GR_R_ORBITS_C)) {
        
        if((TmpInfoStr = (char *) malloc(strlen(InfoStr) + 50)) == NULL) 
            return -2;

        strncpy(TmpInfoStr, InfoStr, strlen(InfoStr));
        
        if(DestGraph == GR_R_STEP) strcat(TmpInfoStr, " [ x = %2.*f ]");
        if(DestGraph == GR_R_ORBITS_X) strcat(TmpInfoStr, " [ c = %2.*f ]");
        if(DestGraph == GR_R_ORBITS_C) strcat(TmpInfoStr, " [ x = %2.*f ]");
        
        show_mouse(screen);
	    poll_mouse();
	    tmp_xp = mouse_x;
        do {
            poll_mouse();
            get_mouse_mickeys(&XMickeys, &YMickeys);
            if((XMickeys != 0) || (YMickeys != 0)) {
                tmp_xp = mouse_x;
                sprintf(Str, TmpInfoStr, 4, R2M_X(Coords, mouse_x));
                StatusBar(RealExp->Settings.LAY_Statusline, RealExp->Graph, -1,
                Str);
            }
            if(keypressed()) {
                Key = readkey();
                AltKey = Key >> 8;
                if(HelpText != NULL) 
			        if(AltKey == KEY_F1) InfoBox(HelpText, 2);
                if(Dialog != NULL)
                    if(AltKey == KEY_F2) popup_dialog(Dialog, 0);
            }
        } while(!(mouse_b & 1) && AltKey != KEY_ENTER && AltKey != KEY_ESC);
	    show_mouse(NULL);
        
        free(TmpInfoStr);
        
	    if(AltKey == KEY_ESC) return(-1);
    }
  
    /* Step-by-step  ->  Orbit diagram for var. x */
    
    if(RealExp->Graph == GR_R_STEP && DestGraph == GR_R_ORBITS_X) {
        RealExp->GraphPar[RealExp->Graph].c = RealExp->GraphPar[DestGraph].c;
        RealExp->Graph = DestGraph;
    }
    
    /* Orbit diagram for var. x  ->  Orbit diagram for var. c */
    
    if(RealExp->Graph == GR_R_ORBITS_X && DestGraph == GR_R_ORBITS_C) {
        RealExp->Graph = DestGraph;
        RealExp->GraphPar[RealExp->Graph].xbegin1 = R2M_X(Coords, tmp_xp);
    }
    
    /* Orbit diagram for var. x  ->  Step-by-step */
    
    if(RealExp->Graph == GR_R_ORBITS_X && DestGraph == GR_R_STEP) {
        RealExp->GraphPar[DestGraph].c = RealExp->GraphPar[RealExp->Graph].c; 
        RealExp->GraphPar[DestGraph].xbegin1 = 
        RealExp->GraphPar[DestGraph].xbegin2 = R2M_X(Coords, tmp_xp);
        RealExp->Graph = DestGraph;
    }
    
    /* Orbit diagram for var. c  ->  Orbit diagram for var. x */
    
    if(RealExp->Graph == GR_R_ORBITS_C && DestGraph == GR_R_ORBITS_X) {
        RealExp->Graph = DestGraph;
        RealExp->GraphPar[DestGraph].c = R2M_X(Coords, tmp_xp); 
    }
    
    /* Recalculate escapevalue */
        
    RealExp->GraphPar[RealExp->Graph].EscapeValueChanged = FALSE;
    RealExp->GraphPar[RealExp->Graph].EscapeValue =
    CalculateRealFnEscapeValue(RealExp);

    return(RealExp->Graph);
}

int RealDraw(TRealExp *RealExp, char **RealStepHelpStr) 
{
    BITMAP              *Bitmap;
    
    Bitmap = create_bitmap(SCREEN_W, SCREEN_H);
    clear_bitmap(Bitmap);

    if(RealExp->Graph < GR_R_ORBITS_X) {
        clear_bitmap(screen);
        DoRealStep(screen, RealExp, 0, RealStepHelpStr);
    }
    else {
        if(Bitmap != NULL) {
            DrawRealOrbits(Bitmap, RealExp);
            blit(Bitmap, screen, 0,0, 0, 0, SCREEN_W, SCREEN_H);
        }
        else
            DrawRealOrbits(screen, RealExp);
    }
    StatusBar(RealExp->Settings.LAY_Statusline, RealExp->Graph, -1, NULL);
    destroy_bitmap(Bitmap);
    return 0;
}

void SelectRealStdCoords(TCoords *Coords, int Function, int Graph)
{
    switch(Function) {

        /*
        **  x -> x * x + c
        */

        case 0:
            switch(Graph) {
                case GR_R_STEP:
                    SetCoords(Coords, -2, -2, 2, 2);
                    break;
                case GR_R_ORBITS_X:
                    SetCoords(Coords, -2.5, -2, 2.5, 2);
                    break;
                case GR_R_ORBITS_C:
                    SetCoords(Coords, -2.5, -2, 0.5, 2);
                    break;
                default:
                    break;
            }
            break;

        /*
        **  x -> frac(cx)
        */

        case 1:
            switch(Graph) {
                case GR_R_STEP:
                    SetCoords(Coords, 0, 0, 1, 1);
                    break;
                case GR_R_ORBITS_X:
                    SetCoords(Coords, -1, 0, 2, 2);
                    break;
                case GR_R_ORBITS_C:
                    SetCoords(Coords, -2, 0, 2, 1);
                    break;
                default:
                    break;
            }
            break;

        /*
        **  x -> c * cos(x)
        */

        case 2:
            switch(Graph) {
                case GR_R_STEP:
                    SetCoords(Coords, -5, -5, 5, 5);
                    break;
                case GR_R_ORBITS_X:
                    SetCoords(Coords, -4, -5, 4, 5);
                    break;
                case GR_R_ORBITS_C:
                    SetCoords(Coords, -4, -5, 4, 5);
                    break;
                default:
                    break;
            }
            break;

        /*
        **  x -> cx(1 - x)
        */

        case 3:
            switch(Graph) {
                case GR_R_STEP:
                    SetCoords(Coords, 0, 0, 1, 1);
                    break;
                case GR_R_ORBITS_X:
                    SetCoords(Coords, -5, -1, 1.5, 2);
                    break;
                case GR_R_ORBITS_C:
                    SetCoords(Coords, 0, 0, 5, 1.25);
                    break;
                default:
                    break;
            }
            break;

        /*
        **  x -> -cx ^ 3 + (c + 1) x ^ 2
        */

        case 4:
            switch(Graph) {
                case GR_R_STEP:
                    SetCoords(Coords, -0.5, -0.5, 1.5, 1.5);
                    break;
                case GR_R_ORBITS_X:
                    SetCoords(Coords, -0.3, -1.5, 1.3, 2);
                    break;
                case GR_R_ORBITS_C:
                    SetCoords(Coords, 0, -0.5, 10, 2);
                    break;
                default:
                    break;
            }
            break;

        /*
        **  x -> c(1 - |2x - 1|)
        */

        case 5:
            switch(Graph) {
                case GR_R_STEP:
                    SetCoords(Coords, -2, -2, 1.5, 1.5);
                    break;
                case GR_R_ORBITS_X:
                    SetCoords(Coords, -0.2, -2, 1.2, 2);
                    break;
                case GR_R_ORBITS_C:
                    SetCoords(Coords, -2, -2, 2, 2);
                    break;
                default:
                    break;
            }
            break;

        default:
            break;
    }
}

void SelectExperiment(TRealExp *RealExp, int Mode, int n)
{
    int             OldGraph = RealExp->Graph;

    SelectRealStdCoords(RealExp->GraphPar[GR_R_STEP].Coords, n, 
        GR_R_STEP);
    SelectRealStdCoords(RealExp->GraphPar[GR_R_ORBITS_X].Coords, n, 
        GR_R_ORBITS_X);
    SelectRealStdCoords(RealExp->GraphPar[GR_R_ORBITS_C].Coords, n, 
        GR_R_ORBITS_C);

    switch(n) {

        /*
        **  x -> x * x + c
        */

        case 0:
            if(Mode) {
                SetRealGraphPar(&RealExp->GraphPar[GR_R_STEP],
                    RealExp->GraphPar[GR_R_STEP].Coords,
                    -0.7, 1.4, 1.4, 0, 1, 250, 0, 1);
                SetRealGraphPar(&RealExp->GraphPar[GR_R_ORBITS_X],
                    RealExp->GraphPar[GR_R_ORBITS_X].Coords,
                    -1.41, -2.5, 0, 0, 1, 250, 0, 1);
                SetRealGraphPar(&RealExp->GraphPar[GR_R_ORBITS_C],
                    RealExp->GraphPar[GR_R_ORBITS_C].Coords,
                    -2.5, 0, 0, 0, 1, 250, 50, 1);
            }
            break;

        /*
        **  x -> frac(cx)
        */

        case 1:
            if(Mode) {
                SetRealGraphPar(&RealExp->GraphPar[GR_R_STEP],
                    RealExp->GraphPar[GR_R_STEP].Coords,
                    3, 0.1, 0.1, 0, 1, 250, 0, 1);
                SetRealGraphPar(&RealExp->GraphPar[GR_R_ORBITS_X],
                    RealExp->GraphPar[GR_R_ORBITS_X].Coords,
                    1.3, -1, 0, 0, 1, 250, 0, 1);
                SetRealGraphPar(&RealExp->GraphPar[GR_R_ORBITS_C],
                    RealExp->GraphPar[GR_R_ORBITS_C].Coords,
                    -1, 0.2, 0, 0, 1, 250, 0, 1);
            }
            break;

        /*
        **  x -> c * cos(x)
        */

        case 2:
            if(Mode) {
                SetRealGraphPar(&RealExp->GraphPar[GR_R_STEP],
                    RealExp->GraphPar[GR_R_STEP].Coords,
                    1.5, 4, 4, 0, 1, 250, 0, 1);
                SetRealGraphPar(&RealExp->GraphPar[GR_R_ORBITS_X],
                    RealExp->GraphPar[GR_R_STEP].Coords,
                    3, 0.5, 0.5, 0, 1, 250, 0, 1);
                SetRealGraphPar(&RealExp->GraphPar[GR_R_ORBITS_C],
                    RealExp->GraphPar[GR_R_ORBITS_C].Coords,
                0, 0, 0.5, 0, 1, 250, 0, 1);
            }
            break;

        /*
        **  x -> cx(1 - x)
        */

        case 3:
            if(Mode) {
                SetRealGraphPar(&RealExp->GraphPar[GR_R_STEP],
                    RealExp->GraphPar[GR_R_STEP].Coords,
                    3.5, 0.015, 0.015, 0, 1, 250, 0, 1);
                SetRealGraphPar(&RealExp->GraphPar[GR_R_ORBITS_X],
                    RealExp->GraphPar[GR_R_ORBITS_X].Coords,
                    3.5, -0.5, 0, 0, 1, 250, 50, 1);
                SetRealGraphPar(&RealExp->GraphPar[GR_R_ORBITS_C],
                    RealExp->GraphPar[GR_R_ORBITS_C].Coords,
                    0, 0.5, 0.5, 0, 1, 250, 50, 1);
            }
            break;

        /*
        **  x -> -cx ^ 3 + (c + 1) x ^ 2
        */

        case 4:
            if(Mode) {
                SetRealGraphPar(&RealExp->GraphPar[GR_R_STEP],
                    RealExp->GraphPar[GR_R_STEP].Coords,
                    5, 0.3, 0.3, 0, 1, 250, 0, 1);
                SetRealGraphPar(&RealExp->GraphPar[GR_R_ORBITS_X],
                    RealExp->GraphPar[GR_R_ORBITS_X].Coords,
                    5, -0.3, 0, 0, 1, 250, 0, 1);
                SetRealGraphPar(&RealExp->GraphPar[GR_R_ORBITS_C],
                    RealExp->GraphPar[GR_R_ORBITS_C].Coords,
                    0, 0.3, 0, 0, 1, 250, 0, 1);
            }
            break;

        /*
        **  x -> c(1 - |2x - 1|)
        */

        case 5:
            if(Mode) {
                SetRealGraphPar(&RealExp->GraphPar[GR_R_STEP],
                    RealExp->GraphPar[GR_R_STEP].Coords,
                    1.5, 0.075441, 0.075441, 0, 1, 250, 0, 1);
                SetRealGraphPar(&RealExp->GraphPar[GR_R_ORBITS_X],
                    RealExp->GraphPar[GR_R_ORBITS_X].Coords,
                    1.5, -1, 0, 0, 1, 250, 0, 1);
                SetRealGraphPar(&RealExp->GraphPar[GR_R_ORBITS_C],
                    RealExp->GraphPar[GR_R_ORBITS_C].Coords,
                    -2, 0.11111111, 0.3, 0, 1, 250, 0, 1);
            }
            break;

        default:
            break;
    }

    /* Set default escapevalues for the graphs */

	RealExp->Graph = GR_R_ORBITS_C;
	RealExp->Graph = GR_R_STEP;
	RealExp->GraphPar[GR_R_STEP].EscapeValue =
        CalculateRealFnEscapeValue(RealExp);
	RealExp->GraphPar[GR_R_STEP].EscapeValueChanged = FALSE;

	RealExp->Graph = GR_R_ORBITS_X;
	RealExp->GraphPar[GR_R_ORBITS_X].EscapeValue =
        CalculateRealFnEscapeValue(RealExp);
	RealExp->GraphPar[GR_R_ORBITS_X].EscapeValueChanged = FALSE;

	RealExp->Graph = GR_R_STEP;
	RealExp->GraphPar[GR_R_ORBITS_C].EscapeValue =
        CalculateRealFnEscapeValue(RealExp);
	RealExp->GraphPar[GR_R_ORBITS_C].EscapeValueChanged = FALSE;

    RealExp->Graph = OldGraph;
}
