/*
	This file is part of XFuniter.

	realexp.h V2.2.3 - a real experiment (functiontype: from R to R)

	Copyright (C) 1995-2003 Stijn Wolters.
    Original idea: Ernic Kamerich (University of Nijmegen).
    
	See README for contact information.

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version. See

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

	See COPYING for more information.
*/

#ifndef _REALEXP
	#define _REALEXP

	/*!	Include files 	*/

	#include <stdio.h>
	#include <stdlib.h>
    #include <values.h>
	#include <math.h>
    #include <allegro.h>
    #include "misc.h"
	#include "coords.h"
	#include "settings.h"
    #include "help.h"

    #ifndef DBL_DIG
        #define DBL_DIG                         8
    #endif

    #ifndef MAXINT
        #define MAXINT                          65535
    #endif

    #define FT_REAL                             0
    
	/*!	Graph	*/

	#define GR_R_STEP							0
	#define	GR_R_ORBITS_X						1
	#define GR_R_ORBITS_C						2

	/*!	DrawingMethod 	*/

	#define	DM_WEB                              0
	#define DM_ITERVALUES						1
	#define DM_STEP						        2
	#define DM_COMBIWEBSTEP					    3

	/*!\brief
	**	Structure for real graphpar.
	**
	**	Coords					-	Math and screen coordinates.
	**	EscapeFunctionStr		-	String with a userdefined escape-function.
	**	Iterated				-	Nth Iterated of function.
	**	IterMax					-	Maximum number of iterations.
	**	IterSkip				-	Number of skipped iterations.
	**	IterSteps				-	Steps per iteration.
	**	xbegin1					-	First starting value for x.
	**	xbegin2					-	Second starting value for x.
	**	c						-	Value for parameter c.
	**  EscapeValue				-	Value the escape function uses.
    **  EscapeValueChanged      -   Flag indicating an automatic- (FALSE) or
    **                              user- (TRUE) change of the EscapeValue.
	*/

	typedef struct {
		TCoords				*Coords;
		char				EscapeFunctionStr[40];
		unsigned int		Iterated, EscapeValueChanged;
		unsigned long int	IterMax, IterSkip, IterSteps;
		double				xbegin1, xbegin2, c, EscapeValue;
	} TRGraphPar;

	/*
	**	Structure that contains info for one experiment.
	**
	**	FnPar					-	List of functionparameters (see the list
	**								above) for each DiagramType.
	**	FractalFnStr			-	String with userdefined fractal-function.
	**	Graph					-	Graph.
	**	Diagram					-	Type of diagram.
	**	Function       			-	function.
    **  IterFn                  -   Pointer to a function iteration function.
	*/

	typedef struct {
		TRGraphPar          GraphPar[3];
		TSettings			Settings;
		char				FunctionStr[40];
		unsigned int		Graph, Diagram, Function;
        /*
        
        Not sure about this yet. comments welcome :-)
        
        double              (*IterFn)(char *, int, double, double),
                            (*CalcEscFn)(int, int, double, double, double); 
	    */
    } TRealExp;

	/* General */

	TRGraphPar *CreateRealGraphPar(TCoords *Coords, int Iterated,
		long int IterMax, long int IterSkip, long int IterSteps,
		double xbegin1, double xbegin2, double c, double EscapeValue);
    void SetRealGraphPar(TRGraphPar *GraphPar, TCoords *Coords, double c, 
        double xbegin1, double xbegin2, double EscapeValue, 
        unsigned int Iterated, unsigned long int IterMax, 
        unsigned long int IterSkip, unsigned long int IterSteps);
	void FreeRealGraphPar(TRGraphPar *GraphPar);
    
    /* Save real graphpar */
    
    int LoadRealGraphPar(FILE *fp, TRGraphPar *GraphPar);
    int SaveRealGraphPar(FILE *fp, TRGraphPar *GraphPar);
    
	TRealExp *CreateRealExp(unsigned int XStart, unsigned int YStart,
		unsigned int XEnd, unsigned int YEnd);
	void FreeRealExp(TRealExp *RealExp);

    /* Load and save experiments */
    
    int LoadRealExp(TRealExp *RealExp, char *FileName, int Mode);
    int SaveRealExp(TRealExp *RealExp, char *FileName);

    /* Calculate escape-value */
    
	double CalculateRealFnEscapeValue(TRealExp *RealExp);

    /* Step by step */
    
    void DrawFunction(BITMAP *Bitmap, TRealExp *RealExp);
    int DrawIterStep(BITMAP *Bitmap, TRealExp *RealExp, int CurStepLinePos, 
        double startx, double x, double y, int Col);
    int DoRealStep(BITMAP *Bitmap, TRealExp *RealExp, int ValueMode,
        char **HelpText);
	
    /* Orbits */
    
    int DrawRealOrbits(BITMAP *Bitmap, TRealExp *RealExp);

    /* Function */
	
    double IterateFn(char *FunctionStr, int Function, int n, double x,  
        double c);

    /* 
    **  Routine to switch between several graphs with the possibilty to select
    **  values with the mouse for certain parameters.
    */

    int RealJump(TRealExp *RealExp, int DestGraph, DIALOG *Dialog, 
        const char *InfoStr, char *HelpText[]);
    
    /* Draw real experiment */
    
    int RealDraw(TRealExp *RealExp, char **RealStepHelpStr); 
    
    /* Select standard coordinates for the predefined experiments */
    
    void SelectRealStdCoords(TCoords *Coords, int Function, int Graph);
    
    /* Select one of the predefined experiments */
    
    void SelectExperiment(TRealExp *RealExp, int Mode, int n);
#endif
