/*
	This file is part of XFuniter.

	alconio.c V2.2.3 - conio.c emulation for allegro GFX.

	Copyright (C) 1995-2003 Stijn Wolters.
    Original idea: Ernic Kamerich (University of Nijmegen).
    
	See README for contact information.

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version. See

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

	See COPYING for more information.
*/

#include "alconio.h"

TConsole *alconio_init(int cwidth, int cheight, int cstretch, 
int cfgcol, int cbkcol) 
{
    TConsole  *Console = (TConsole *) malloc(sizeof(TConsole));
    
    /* Vertical stretch of font in y-direction */
    
    Console->Stretch = cstretch;
    
    /* Current cursor position inside the virtual screen */
    
    Console->CurX = 0;
    Console->CurY = 0;
    
    /* Set width and height in characters */
    
    Console->CharWidth = cwidth;
    Console->CharHeight = cheight;
    
    /* Determine width and height in pixels instead of chars */
    
    Console->Width = Console->CharWidth << 3;
    Console->Height = text_height(font) * Console->Stretch * 
    Console->CharHeight;
    
    /* Center virtual screen */
    
    Console->XStart = (SCREEN_W - Console->Width) / 2;
    Console->YStart = (SCREEN_H - Console->Height) / 2;
    
    /* Create bitmap for what's currently on screen */
    
    Console->ScreenContents = create_bitmap(Console->Width, Console->Height);
    
    /* Create bitmap for what's under the cursor */
    
    Console->Cursor = create_bitmap(X_C2P(Console, 1), Y_C2P(Console, 1));

    /* Set cursor mode */
    
    Console->CursorMode = _SOLIDCURSOR; 

    /* Set text/background colors */
    
    Console->FgCol = cfgcol; 
    Console->BkCol = cbkcol; 

    /* Store background */
        
    blit(screen, Console->ScreenContents, Console->XStart, Console->YStart, 
    0, 0, Console->Width, Console->Height);

    a_clrscr(Console);
           
    return Console;
}

int a_cprintf(TConsole *Console, const char *format, ...)
{
    va_list     valp;
    char        s[120];

    va_start(valp, format);
    uvsprintf(s, format, valp);
    va_end(valp);

    a_cputs(Console, s);
    
    return 0;
}

int a_cputs(TConsole *Console, const char *s)
{
    int         TextMode;
    BITMAP      *TextBuffer;

    TextBuffer = create_bitmap(X_C2P(Console, strlen(s)), text_height(font));
    
    TextMode = text_mode(-1);
    clear_to_color(TextBuffer, Console->BkCol);
    textprintf(TextBuffer, font, 0, 0, Console->FgCol, "%s", s);
    text_mode(TextMode);
    
    stretch_blit(TextBuffer, screen, 0, 0, gui_strlen(s), text_height(font), 
    X_C2P(Console, Console->CurX) + Console->XStart, 
    Y_C2P(Console, Console->CurY) + Console->YStart, 
    gui_strlen(s), Y_C2P(Console, 1));

    destroy_bitmap(TextBuffer);

    a_gotoxy(Console,  Console->CurX, Console->CurY);

    a_gotoxy(Console, Console->CurX + strlen(s), Console->CurY);

    return 0;
}

int a_putch(TConsole *Console, char c)
{
    char        s[1];
    
    sprintf(s, "%c", c);
    a_cputs(Console, s);
    return 0;
}

int a_clrscr(TConsole *Console)
{
    /* Draw a rectangle in the backgroundcolor to erase screen */
    
    rectfill(screen, Console->XStart, Console->YStart, 
    Console->XStart + Console->Width - 1, 
    Console->YStart + Console->Height - 1, 
    Console->BkCol);
    
    /* Set cursor to 0, 0 and store what is under the cursor */

    Console->CurX = 0;
    Console->CurY = 0;
    
    a_gotoxy(Console, 0, 0);
    return 0;
}

int a_gotoxy(TConsole *Console, int x, int y)
{
    int         x1, y1, x2, y2;

    if(x >= Console->CharWidth) x = Console->CharWidth - 1; 
    if(y >= Console->CharHeight) y = Console->CharHeight - 1; 
        
    x1 = X_C2P(Console, Console->CurX) + Console->XStart;
    y1 = Y_C2P(Console, Console->CurY) + Console->YStart;
    x2 = X_C2P(Console, x) + Console->XStart;
    y2 = Y_C2P(Console, y) + Console->YStart;

    if(Console->CursorMode != _NOCURSOR) {

        /* Erase cursor on old position */
    
        if(!((Console->CurX == x) && (Console->CurY == y))) 
            blit(Console->Cursor, screen, 0, 0, x1, y1, 
            X_C2P(Console, 1), Y_C2P(Console, 1));
    }
    
    Console->CurX = x;
    Console->CurY = y;
    
    if(Console->CursorMode != _NOCURSOR) {
    
        /* Draw cursor at new position */
      
        blit(screen, Console->Cursor, x2, y2, 0, 0,  
        X_C2P(Console, 1), Y_C2P(Console, 1));
    
        xor_mode(TRUE); 
        rectfill(screen, x2, y2, x2 + X_C2P(Console, 1) - 1, 
        y2 + Y_C2P(Console, 1) - 1, makecol(255, 255, 255));
        xor_mode(FALSE);
    }
    
    return 0;
}

int a_wherex(TConsole *Console) { return Console->CurX; }
int a_wherey(TConsole *Console) { return Console->CurY; }

int _a_setcursortype(TConsole *Console, int mode)
{
    if(mode >= 0 && mode < 3) Console->CursorMode = mode;
    return 0;
}

int a_textcolor(TConsole *Console, int Color) 
{ 
    Console->FgCol = Color; 
    return 0; 
}

int a_textbackground(TConsole *Console, int Color) 
{ 
    Console->BkCol = Color; 
    return 0; 
}

int a_getimage(TConsole *Console, BITMAP *Image)
{
    blit(screen, Image, Console->XStart, Console->YStart, 0, 0, 
    Console->Width,
    Console->Height);
    return 0;
}  

int a_putimage(TConsole *Console, BITMAP *Image)
{
    blit(Image, screen, 0, 0, Console->XStart, Console->YStart, Console->Width,
    Console->Height);
    return 0;
}

void alconio_exit(TConsole *Console)
{
    destroy_bitmap(Console->ScreenContents);
    destroy_bitmap(Console->Cursor);
    free(Console);
    return;
}
