/*  
	This file is part of OpenFuniter.

	cplexp.h V2.2.3 - a complex experiment (functiontype: from C to C)

	Copyright (C) 1995-2003 Stijn Wolters.
    Original idea: Ernic Kamerich (University of Nijmegen)

	See README for contact information.

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version. See

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

	See COPYING for more information.
*/

#ifndef __CPLEXP
	#define __CPLEXP

	#include <stdio.h>
	#include <stdlib.h>
	#include <string.h>
	#include <values.h>
	#include <math.h>
	#include <time.h>
	#include <allegro.h>
    #include "misc.h"
	#include "coords.h"
	#include "complex.h"
	#include "settings.h"
	#include "help.h"

    #ifndef DBL_DIG
        #define DBL_DIG                     8
    #endif

    #ifndef MAXINT
        #define MAXINT                      65535
    #endif

    #define FT_COMPLEX          1
    
	/* MirrorState */

    #define	MS_NONE				0
	#define MS_X				2
	#define MS_Y				4

	/* Graph */

	#define GR_C_STEP			0
	#define	GR_C_STEPINV        1
	#define GR_C_ORBITS_Z		2
	#define GR_C_ORBITS_C		3

	/* Shapes */

	#define	SH_POINT 			0
	#define SH_LINE				1
	#define SH_RECT				2
	#define	SH_CIRCLE			3

	/* Diagram */

	#define	DM_FILLED			0
	#define DM_ETD				1
	#define DM_BOUNDARY			2
    #define DM_INVERSE          3
    
    /* Coloring methods */
    
    #define CM_SAME             0
    #define CM_ITERSTEP         1
    #define CM_START            2

	/*
	**	Structure for complex graphpar:
	**
	**	Coords					-	Math and screen coordinates.
	**	zReal, zImag			-	Complex parameter z (orbits for var. c).
	**	cReal, cImag			-	Complex parameter c (orbits for var. z).
	**  EscapeValue				-	Value the escape function uses.
	**	IterMax					-	Maximum number of iterations.
	*/

	typedef struct {
		TCoords			*Coords;
		double 			zReal, zImag, cReal, cImag, EscapeValue;
		unsigned int 	IterSteps, IterSkip, IterMax, EscapeValueChanged;
	} TCGraphPar;

	/*
    **  GraphPar                -   Parameters for each graph.
    **  Settings                -   General settings.
	**	Graph      				-	Type of orbits (see defines).
	**	Diagram      			-	Method of drawing (see defines).
    **  Coloring                -   Coloringmethod used for step-by-step.
	*/

	typedef struct {
		TCGraphPar		GraphPar[4];
		TSettings		Settings;
		int				Function, Graph, Diagram, Coloring, EscValueChanged;
        unsigned int    InvJuliaIterMax;
	} TComplexExp;

    /* General */
    
	TCGraphPar *CreateComplexGraphPar(TCoords *Coords, double zReal,
		double zImag, double cReal, double cImag, double EscapeValue,
		unsigned int IterSteps, unsigned int IterSkip,
		unsigned int IterMax);

	void FreeComplexGraphPar(TCGraphPar *GraphPar);
    
    int LoadComplexGraphPar(FILE *fp, TCGraphPar *GraphPar);
    int SaveComplexGraphPar(FILE *fp, TCGraphPar *GraphPar);

	TComplexExp *CreateComplexExp(unsigned int XStart, unsigned int YStart,
		unsigned int XEnd, unsigned int YEnd);

	void FreeComplexExp(TComplexExp *CplExp);
    
    int LoadComplexExp(TComplexExp *ComplexExp, char *FileName, int Mode);
    int SaveComplexExp(TComplexExp *ComplexExp, char *FileName);

    void SetComplexExpDefaultCoords(TComplexExp *ComplexExp, int Graph);
	
    /* Step by step */

	int IsInList(int *XArray, int *YArray, int x, int y);

	void DrawCircle(BITMAP *Bitmap, TCoords *Coords, int x, int y, int r,
        int Color);

	int StoreLine(BITMAP *Bitmap, TCoords *Coords, int x1, int y1, int x2, 
        int y2, int Color, int PosInLine, TComplex **Points);
	int StoreRectangle(BITMAP *Bitmap, TCoords *Coords, int x1, int y1, int x2, 
        int y2, int Color, TComplex **Points);
	int StoreCircle(BITMAP *Bitmap, int NumPoints, TCoords *Coords, int x, 
        int y, int r, int Color, TComplex **Points);

	TComplex *CreateShape(BITMAP *Bitmap, TComplexExp *ComplexExp, int Color, 
        int *NumPoints, char **HelpText);
    int DefineShape(BITMAP *Bitmap, TComplexExp *ComplexExp, int *x1, int *y1, 
        int *x2, int *y2, int Color, char **HelpText);

	void IPDDrawLine(BITMAP *Bitmap, TComplexExp *ComplexExp, TComplex *Point,
		TComplex *PointLast, int Color, unsigned int Iter, int ClearMode);

	int IPDPlot(BITMAP *Bitmap, TComplexExp *ComplexExp, TComplex **Points,
		TComplex *LastPoint, unsigned int Iter, int Color, int IterColor, 
        int *Size, int KeyPressed);

	int DoComplexStep(BITMAP *Bitmap, BITMAP *JuliaBitmap, 
        TComplexExp *ComplexExp, int JuliaVisible, char **StepHelpText, 
        char **StepInvHelpText, char **JuliaNoMemStr,  char **DefShapeHelpText);
        
	/* Orbits */

	int DrawComplexOrbits(BITMAP *Bitmap, TComplexExp *ComplexExp);

	int DrawComplexOrbitsArea(BITMAP *Bitmap, unsigned int XStart, 
        unsigned int YStart, unsigned int XEnd, unsigned int YEnd, 
        unsigned int XScr, unsigned int YScr, unsigned int Width, 
        unsigned int Height, unsigned int MirrorState, TComplexExp *ComplexExp);

	double CalculateComplexFnEscapeValue(TComplexExp *ComplexExp);

	/* Iteration */

    unsigned int CIterateFn(double zReal, double zImag, double cReal, 
        double cImag, double EscapeValue, double PixelWidth, double PixelHeight, 
        unsigned int MaxPeriod, unsigned int MaxIter, int PeriodCheck, 
        double *RealRes, double *ImagRes, int *EscStatus);

	void IterateInvFn(double *zReal, double *zImag, double cReal,
		double cImag);
    
    /* 
    **  Routine to switch between several graphs with the possibilty to select
    **  values with the mouse for certain parameters.
    */
    
    int ComplexJump(TComplexExp *ComplexExp, int DestGraph, char *InfoStr, 
        char *HelpText[], int Mode);

    /* Draw complex experiment. */

    int ComplexDraw(TComplexExp *ComplexExp, char **StepHelpText, 
        char **StepInvHelpText, char **JuliaNoMemStr, char **DefShapeHelpText);
    
#endif
