/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"

/** The pixmaps for the bookmarks */
PixmapData *folder_pixmap_data = NULL;
PixmapData *folder_open_pixmap_data = NULL;
PixmapData *site_pixmap_data = NULL;

/** The root of the bookmark tree */
BookmarkItem *bookmarks_root = NULL;

/** The root of the temporary bookmark tree */
BookmarkItem *temp_bookmarks_root = NULL;

/** The next id to assign (for alias saving) */
static long maxid = 0;

/** Aliases that have been loaded, but whose real bookmark has not been loaded */
GList *unresolved_aliases = NULL;

/** The favicon pixmap data cache */
GHashTable *favicons_cache = NULL;

/* DnD */
const GtkTargetEntry bookmarks_dnd_targets [] = {
	{ "GALEON_BOOKMARK", 0, DND_TARGET_GALEON_BOOKMARK},
	{ "GALEON_URL", 0, DND_TARGET_GALEON_URL },
	{ "_NETSCAPE_URL", 0, DND_TARGET_NETSCAPE_URL},
	{ "text/uri-list", 0, DND_TARGET_TEXT_URI_LIST},
	{ "STRING", 0, DND_TARGET_STRING}
};
const gint bookmarks_dnd_targets_num_items = (
	sizeof (bookmarks_dnd_targets) / sizeof (GtkTargetEntry));

/* Private prototypes */
static BookmarkItem *bookmarks_xml_read_item (xmlNodePtr item);
static BookmarkItem *bookmarks_xml_read (xmlNodePtr item);
static void bookmarks_add_completion_recursively (BookmarkItem *b);
static void bookmarks_save_recursively (xmlDocPtr doc, xmlNodePtr root,
					BookmarkItem *b);
static BookmarkItem *bookmarks_find_by_nick (const gchar *nick,
					     BookmarkItem *root,
					     gboolean wantsmart);
static void bookmarks_remove_all_alias (BookmarkItem *b);
static void bookmarks_new_alias_do (BookmarkItem *b, BookmarkItem *bookmark);
static BookmarkItem *bookmarks_find_by_id (long id,
					   BookmarkItem *root);

/**
 * bookmarks_load: load bookmarks from xml file
 */
void
bookmarks_load (void)
{
	gchar *bookfile;
	
	bookmarks_load_icons ();
	
	bookfile = g_concat_dir_and_file (g_get_home_dir (), 
					  "/.galeon/bookmarks.xml");
	
	bookmarks_root = bookmarks_load_from (bookfile);
	
	if (!bookmarks_root) {
		bookmarks_root = bookmarks_new_bookmark 
			(BM_CATEGORY, TRUE, _("Bookmarks"), NULL, NULL, NULL, NULL);
	}

	/* generate the autobookmarks category */
	autobookmarks_generate();
	
	bookmarks_add_completion_recursively (bookmarks_root);

	g_free (bookfile);
}

/**
 * bookmarks_load_icons: load the bookmarks icons
 */
void 
bookmarks_load_icons (void) 
{
	folder_pixmap_data = pixmap_data_new_from_file (
						SHARE_DIR "/dir.xpm");
	
	folder_open_pixmap_data = pixmap_data_new_from_file (
						SHARE_DIR "/dir_open.xpm");
	
	site_pixmap_data = pixmap_data_new_from_file (
						SHARE_DIR "/i-bookmark.xpm");
			
	/* TODO: add icons for alias */
}

/**
 * bookmarks_load_from: Read a bookmarks tree
 * @file: The file to read, it should exist
 * Return value: The new allocated tree of bookmarks or NULL if error
 **/
BookmarkItem *
bookmarks_load_from (char *file)
{
	xmlDocPtr doc;
	xmlNodePtr item;
	BookmarkItem *b = NULL;

	if (!(g_file_exists (file))) {
		/* no bookmarks */
		return NULL;
	}

	doc = xmlParseFile (file);	
	
	if (doc) {
		item = doc->root;
		b = bookmarks_xml_read (item);
		xmlFreeDoc (doc);
		return b;
	} else {
		g_warning ("unable to parse bookmarks file: %s", file);
		return NULL;
	}
}

/**
 * bookmarks_xml_read: read all items from xml file
 */
static BookmarkItem * 
bookmarks_xml_read (xmlNodePtr item) 
{
	static BookmarkItem *error_bookmark = NULL;
	BookmarkItem *b, *result;
	item = item->childs;
	
	if (!error_bookmark) 
		error_bookmark = bookmarks_new_bookmark 
			(BM_SITE, TRUE, _("Error"), "about:blank", NULL,
			 _("An error ocurred loading your bookmarks."), NULL);
	
	result = bookmarks_new_bookmark (BM_CATEGORY, TRUE, _("Bookmarks"),
					 NULL, NULL, NULL, NULL);
	
	if (item->next) /* old format */
		while (item != NULL) {
			xmlNodePtr item2 = item;
			b = bookmarks_xml_read_item(item2);
			result->list = g_list_append(result->list, b);
			b->parent = result;
			item = item->next;
		}
	else {
		bookmarks_free_bookmark (result);
		result = bookmarks_xml_read_item (item);
	}

	/* Finish creating the aliases */
	while (unresolved_aliases) {
		BookmarkItem *alias = unresolved_aliases->data;
		BookmarkItem *real = bookmarks_find_by_id (alias->id, result);
		unresolved_aliases = g_list_remove (unresolved_aliases, alias);
		if (real)
			bookmarks_new_alias_do (real, alias);
		else
			bookmarks_new_alias_do (error_bookmark, alias);
	}

	return result;
}

/**
 * bookmarks_xml_read_item: read an item from xml file
 */
static BookmarkItem *
bookmarks_xml_read_item (xmlNodePtr item)
{
	BookmarkItem *b;
	BookmarkItem *b2;
	gchar *ct, *expanded, *ccm;
	
	gchar *name = xmlGetPropISOLatin1 (item, "name");	
	gchar *notes = xmlGetProp (item, "notes");
	gchar *nick = xmlGetProp (item, "nick");
	gchar *pixmap_file = xmlGetProp (item, "pixmap");
	gchar *idstr = xmlGetProp (item, "id");
	gchar *time_added_str = xmlGetProp (item, "time_added");
	gchar *time_modified_str = xmlGetProp (item, "time_modified");
	gchar *time_visited_str = xmlGetProp (item, "time_visited");
  
	g_return_val_if_fail(item != NULL, NULL);

	if (!strcmp (item->name, "category")) {
		b = bookmarks_new_bookmark (BM_CATEGORY, FALSE, name, NULL, nick, notes,
					    pixmap_file);
		ct = xmlGetProp (item, "create_toolbar");
		ccm = xmlGetProp (item, "create_context_menu");
		expanded = xmlGetProp (item, "expanded");
		b->create_toolbar = (ct && !strcmp (ct, "TRUE")) ? TRUE : FALSE;
		b->create_context_menu = (ccm && !strcmp (ccm, "TRUE")) ? TRUE : FALSE;
		b->expanded = (expanded && !strcmp (expanded, "TRUE")) ? TRUE : FALSE;
		xmlFree (ct);
		xmlFree (ccm);
		xmlFree (expanded);
		item = item->childs;
		while (item != NULL) {
			xmlNodePtr item2 = item;
			b2 = bookmarks_xml_read_item (item2);
			b->list = g_list_append (b->list, b2);
			b2->parent = b;
			item = item->next;
		}
	} else if (!strcmp (item->name, "separator")) {
		b = bookmarks_new_bookmark (BM_SEPARATOR, FALSE, name, NULL, nick, notes,
					    pixmap_file);
	} else if (!strcmp (item->name, "autobookmarks")) {
		/* FIXME: remove code duplication */
		b = bookmarks_new_bookmark (BM_AUTOBOOKMARKS, FALSE, name, NULL, nick,
					    notes, pixmap_file);
		ct = xmlGetProp (item, "create_toolbar");
		ccm = xmlGetProp (item, "create_context_menu");
		expanded = xmlGetProp (item, "expanded");
		b->create_toolbar = (ct && !strcmp (ct, "TRUE")) ? TRUE : FALSE;
		b->create_context_menu = (ccm && !strcmp (ccm, "TRUE")) ? TRUE : FALSE;
		b->expanded = (expanded && !strcmp (expanded, "TRUE")) ? TRUE : FALSE;
		xmlFree (ct);
		xmlFree (ccm);
		xmlFree (expanded);
		autobookmarks_root = b;
	} else if (!strcmp (item->name, "alias")) {
		b = bookmarks_new_alias (NULL);
		unresolved_aliases = g_list_prepend (unresolved_aliases, b);
	} else { /* site */
		gchar *url = xmlGetProp (item, "url");
		b = bookmarks_new_bookmark (BM_SITE, FALSE, name, url, nick, notes,
					    pixmap_file);
		ccm = xmlGetProp (item, "create_context_menu");
		b->create_context_menu = (ccm && !strcmp (ccm, "TRUE")) ? TRUE : FALSE;
		xmlFree (ccm);
		if (url) g_free (url);
	}
	if (time_added_str) {
		b->time_added = strtol (time_added_str, NULL, 10);
		g_free (time_added_str);
	}
	if (time_visited_str) {
		b->time_modified = strtol (time_modified_str, NULL, 10);
		g_free (time_modified_str);
	}
	if (time_visited_str) {
		b->time_visited = strtol (time_visited_str, NULL, 10);
		g_free (time_visited_str);
	}
	if (idstr) b->id = atol (idstr);
	if (maxid <= b->id) maxid = b->id + 1;
	if (name) g_free (name);
	if (notes) g_free (notes);
	if (nick) g_free (nick);
	if (pixmap_file) g_free (pixmap_file);
	if (idstr) g_free (idstr);
	
	return b;
}

void
temp_bookmarks_save (void)
{
	char *file = g_strconcat (g_get_home_dir(), "/.galeon/temp_bookmarks.xml", NULL);
	if (temp_bookmarks_root != NULL) {
		bookmarks_save_as (temp_bookmarks_root, file);
	}
	g_free (file);
}

/**
 * bookmarks_save: save all bookmarks to the xml file
 */
void
bookmarks_save (void) 
{
	bookmarks_update_menu ();
	bookmarks_update_tb ();
	if ((!bookmarks_editor) || (bookmarks_editor->dirty)) {
		char *savefile = g_strconcat (g_get_home_dir(), 
					      "/.galeon/bookmarks.xml", NULL);
		bookmarks_save_as (bookmarks_root, savefile);
		if (bookmarks_editor) bookmarks_editor->dirty = FALSE;
		g_free (savefile);
	}
}

/**
 * bookmarks_save_as: Save a set of bookmarks
 * @root: the root category to be saved
 * @file: the filename, if already exists will be overwriten
 **/
/* FIXME: check for errors (permisions...) */
void 
bookmarks_save_as (BookmarkItem *root, char *file)
{
	xmlNodePtr node;
	xmlDocPtr doc = xmlNewDoc ("1.0");
	node = xmlNewDocNode (doc, NULL, "bookmarks", NULL);
	
	bookmarks_save_recursively (doc, node, root);

	xmlDocSetRootElement (doc, node);
	xmlSaveFile (file, doc);
	xmlFreeDoc (doc);
}

/**
 * bookmarks_save_recursively: recursively save a bookmarks tree to the xml file 
 */
static void
bookmarks_save_recursively (xmlDocPtr doc, xmlNodePtr root, BookmarkItem *b)
{
	BookmarkItem *b2;
	xmlNodePtr node;
	GList *li;
	gchar *idstr = g_strdup_printf ("%ld", b->id);
	GString *s = g_string_new ("");

	if (b->alias_of) {
		node = xmlNewDocNode (doc, NULL, "alias", NULL);
		xmlSetProp (node, "id", idstr);
		xmlSetProp (node, "create_toolbar", 
			    b->create_toolbar ? "TRUE" : "FALSE");
		xmlSetProp (node, "create_context_menu", 
			    b->create_context_menu ? "TRUE" : "FALSE");
		g_string_sprintf (s, "%d", b->time_added);
		xmlSetProp (node, "time_added", s->str);
		xmlAddChild (root, node);
	} else {
		switch (b->type) {
		case BM_CATEGORY:
			node = xmlNewDocNode (doc, NULL, "category", NULL);
			xmlSetProp (node, "id", idstr);
			xmlSetPropISOLatin1 (node, "name", b->name);
			xmlSetProp (node, "create_toolbar", 
				    b->create_toolbar ? "TRUE" : "FALSE");
			xmlSetProp (node, "create_context_menu", 
				    b->create_context_menu ? "TRUE" : "FALSE");
			xmlSetProp (node, "expanded", 
				    b->expanded ? "TRUE" : "FALSE");
			xmlSetProp(node, "notes", b->notes);
			if (b->pixmap_file)
				xmlSetProp(node, "pixmap", b->pixmap_file);
			g_string_sprintf (s, "%d", b->time_added);
			xmlSetProp (node, "time_added", s->str);
			g_string_sprintf (s, "%d", b->time_modified);
			xmlSetProp (node, "time_modified", s->str);
			g_string_sprintf (s, "%d", b->time_visited);
			xmlSetProp (node, "time_visited", s->str);
			xmlAddChild (root, node);
			
			for (li = b->list; li != NULL; li = li->next) {
				b2 = li->data;
				bookmarks_save_recursively (doc, node, b2);
		}
			break;
			
		case BM_SEPARATOR:
			node = xmlNewDocNode(doc, NULL, "separator", NULL);
			xmlAddChild(root, node);
			break;

		case BM_SITE:
			node = xmlNewDocNode(doc, NULL, "site", NULL);
			xmlSetProp (node, "id", idstr);
			xmlSetPropISOLatin1 (node, "name", b->name);
			xmlSetProp(node, "url", b->url);    
			xmlSetProp(node, "nick", b->nick);
			xmlSetProp(node, "notes", b->notes);
			xmlSetProp (node, "create_context_menu", 
				    b->create_context_menu ? "TRUE" : "FALSE");
			if (b->pixmap_file)
				xmlSetProp(node, "pixmap", b->pixmap_file);
			g_string_sprintf (s, "%d", b->time_added);
			xmlSetProp (node, "time_added", s->str);
			g_string_sprintf (s, "%d", b->time_modified);
			xmlSetProp (node, "time_modified", s->str);
			g_string_sprintf (s, "%d", b->time_visited);
			xmlSetProp (node, "time_visited", s->str);
			xmlAddChild(root, node);
			break;
			
		case BM_AUTOBOOKMARKS:
			node = xmlNewDocNode(doc, NULL, "autobookmarks", NULL);
			xmlSetProp (node, "id", idstr);
			xmlSetPropISOLatin1 (node, "name", b->name);
			xmlSetProp (node, "create_toolbar", 
				    b->create_toolbar ? "TRUE" : "FALSE");
			xmlSetProp (node, "create_context_menu", 
				    b->create_context_menu ? "TRUE" : "FALSE");
			xmlSetProp (node, "expanded", 
				    b->expanded ? "TRUE" : "FALSE");
			xmlSetProp(node, "notes", b->notes);
			if (b->pixmap_file)
				xmlSetProp(node, "pixmap", b->pixmap_file);
			xmlAddChild(root, node);
			break;
		}
	}
        g_string_free (s, TRUE);
	g_free (idstr);
}

/**
 * bookmarks_remove_recursively: recursively remove a tree of bookmarks items
 * Doesn't update tree items
 */
void 
bookmarks_remove_recursively (BookmarkItem *b) {
	GList *position, *li;
	
	if (b->parent) {
		position = g_list_find (b->parent->list, b);
		b->parent->list = 
			g_list_remove_link (b->parent->list, position);
		bookmarks_update_alias (NULL, b->parent);
	}

	if (!b->alias_of) 
		/* When removing a bookmark, we remove all its alias
		   We should check before removing if it has alias and warn the user */
		bookmarks_remove_all_alias (b);
	else 
		/* we are removing an alias, connect the aliased bookmark with it's
		   next, if any */
		b->alias_of->alias = b->alias;
	
	if ((b->type == BM_CATEGORY || b->type == BM_AUTOBOOKMARKS) && !b->alias_of)
		for (li = b->list; li != NULL; li = li->next) 
			bookmarks_remove_recursively (li->data);

	bookmarks_free_bookmark (b);
}

/**
 * bookmarks_new_bookmark: Creates a new allocated BookmarkItem
 * @type: the type of the bookmark
 * @escape_name: TRUE if the name needs to have _'s escaped, FALSE if they already are
 * @name: the name, if NULL the url is used
 * @url: the url (ignored if type != BM_SITE)
 * @nick: the nickname for the bookmark, if any
 * @notes: the comments aout the bookmark
 * 
 * This function allocates a BookmarkItem and initializes its fields to sane values.
 * Use it instead of g_new (BookmarkItem, 1) and manually initializing the fields. 
 * The returned BookmarkItem has it's parent set to NULL, so you have to add it to 
 * the bookmarks list yourself (if you need it)
 * All the strings passed to this function are g_strduped
 * 
 * Return value: The new BookmarkItem
 **/
BookmarkItem *
bookmarks_new_bookmark (BookmarkType type, gboolean escape_name,
			const char *name, const char *url, 
			const char *nick, const char *notes, 
			const char *pixmap_file)
{
	BookmarkItem *bookmark;

	g_return_val_if_fail (!((type == BM_SITE) && (url == NULL)), NULL);
	g_return_val_if_fail (!((type == BM_CATEGORY) && (name == NULL)), NULL);

	bookmark = g_new0 (BookmarkItem, 1);
	bookmark->type = type;

	switch (bookmark->type) {
	case BM_AUTOBOOKMARKS:
	case BM_CATEGORY:
		if (escape_name)
			bookmark->name = escape_uline_accel (name);
		else
			bookmark->name = g_strdup (name);
		bookmark->url = NULL;
		break;
	case BM_SITE:
		if (name)
		{
			if (escape_name)
				bookmark->name = escape_uline_accel (name);
			else
				bookmark->name = g_strdup (name);
		}
		else
			bookmark->name = escape_uline_accel (url);
		bookmark->url = g_strdup (url);
		break;
	default:
		bookmark->name = bookmark->url = NULL;
		break;
	}
	bookmark->list = NULL;
	bookmark->tree_item = NULL;
	bookmark->parent = NULL;
	bookmark->create_toolbar = FALSE;
	bookmark->create_context_menu = FALSE;
	if (nick) 
		bookmark->nick = g_strdup (nick);
	else
		bookmark->nick = g_strdup ("");
	if (notes)
		bookmark->notes = g_strdup (notes);
	else
		bookmark->notes = g_strdup ("");
	if (pixmap_file != NULL && strlen(pixmap_file) != 0)
	{
		bookmark->pixmap_file = g_strdup (pixmap_file);
		bookmark->pixmap_data = pixmap_data_new_from_file (pixmap_file);
	}
	else
	{
		bookmark->pixmap_file = g_strdup ("");
		bookmark->pixmap_data = NULL;
	}
	bookmark->expanded = TRUE;
	bookmark->alias = NULL;
	bookmark->alias_of = NULL;
	bookmark->id = maxid++;
	bookmark->time_added = 0;
	bookmark->time_modified = 0;
	bookmark->time_visited = 0;
	return bookmark;
}

/**
 * Creates a an alias of a bookmark. If the parameter is NULL, returns bookmark
 * with all fields NULLs, for creating an alias later.
 */
BookmarkItem *
bookmarks_new_alias (BookmarkItem *b)
{
	BookmarkItem *bookmark = g_new0 (BookmarkItem, 1);
	bookmark->tree_item = NULL;
	bookmark->parent = NULL;
	bookmark->create_toolbar = FALSE;
	bookmark->create_context_menu = FALSE;
	if (b) bookmarks_new_alias_do (b, bookmark);
	return bookmark;
}

/**
 * Fill the fields of and alias
 */
static void
bookmarks_new_alias_do (BookmarkItem *b, BookmarkItem *bookmark)
{
	/* Most are the same as the original bookmark */
	/* actually, they point to the same string, so be cautious */
	bookmark->type = b->type;
	bookmark->name = b->name;
	bookmark->url = b->url;
	bookmark->list = b->list;
	bookmark->nick = b->nick;
	bookmark->notes = b->notes;
	bookmark->pixmap_file = b->pixmap_file;
	bookmark->pixmap_data = b->pixmap_data;
	bookmark->expanded = TRUE; /* does not have sense here */
	/* insert in the list */
	bookmark->alias = b->alias;
	bookmark->alias_of = b;
	bookmark->id = b->id;
	b->alias = bookmark;
}

/**
 * Updates all aliases of a bookmark
 */
void
bookmarks_update_alias (BookmarksEditorControls *c, BookmarkItem *b)
{
	BookmarkItem *alias = b->alias;
	if (!alias) return;
	alias->name = b->name;
	alias->url = b->url;
	alias->list = b->list;
	alias->nick = b->nick;
	alias->notes = b->notes;
	alias->pixmap_file = b->pixmap_file;
	alias->pixmap_data = b->pixmap_data;
	alias->id = b->id;
	alias->time_visited = b->time_visited;
	alias->time_modified = b->time_modified;
	if (c)
		bookmarks_editor_update_tree_item (c, alias);
	bookmarks_update_alias (c, alias);
}

/**
 * Finds the real bookmark of an alias
 */
BookmarkItem *
bookmarks_find_real_bookmark (BookmarkItem *b)
{
	if (!b->alias_of) 
		return b;
	else
		return bookmarks_find_real_bookmark (b->alias_of);
}

/**
 * bookmarks_copy_bookmark: copy a bookmark
 * @b: the bookmark
 * 
 * copy a bookmarks and its children
 * 
 * Return value: the new bookmark
 **/
BookmarkItem *
bookmarks_copy_bookmark (BookmarkItem *b)
{
	GList *l;
	/* TODO: if it an alias... */
	BookmarkItem *new = bookmarks_new_bookmark (b->type, FALSE, b->name,
						    b->url, b->nick, b->notes,
						    b->pixmap_file);
	
	new->create_toolbar = b->create_toolbar;
	new->create_context_menu = b->create_context_menu;
	new->expanded = b->expanded;
	/* should these times just copied updated here? should they be updated? */
	new->time_added = b->time_added;
	new->time_modified = b->time_modified;
	new->time_visited = b->time_visited;
	if (b->pixmap_data != NULL)
	{
		new->pixmap_data = b->pixmap_data;
	}
	if (b->type == BM_CATEGORY) {
		for (l = b->list; l != NULL; l = g_list_next (l)) {
			BookmarkItem *child = l->data;
			BookmarkItem *new_child = bookmarks_copy_bookmark (child);
			new_child->parent = new;
			new->list = g_list_append (new->list, new_child);
		}
	}
	return new;
}

/**
 * bookmarks_free_bookmark: free a bookmark
 * @b: the bookmark to be freed, if NULL thenb do nothing
 * 
 * Frees the bookmarks and all the strings referenced by it and the list of children 
 * if it's a BM_CATEGORY (but does not free the children). Use it instead of g_free. 
 **/
void
bookmarks_free_bookmark (BookmarkItem *b)
{
	if (! b) return;

	/* aliases fields point to the real bookmark fields */
	if (!b->alias_of) {
		if (b->name) g_free (b->name);
		if (b->url) g_free (b->url);
		if (b->nick) g_free (b->nick);
		if (b->pixmap_file) g_free (b->pixmap_file);
		if (b->pixmap_data) g_free (b->pixmap_data);
		if (b->notes) g_free (b->notes);
		if (b->list) g_list_free (b->list);
	}
	g_free (b);
}

/**
 * bookmarks_insert_bookmark: insert a bookmark
 * @b: the bookmark to insert
 * @near: a bookmark that should be near @b (a brother or its parent)
 **/
void
bookmarks_insert_bookmark (BookmarkItem *b, BookmarkItem *near,
			   GtkCListDragPos insert_pos)
{
	BookmarkItem *parent;
	GList *position_node;
	gint position;
	
	if (near == NULL)
		parent = bookmarks_root;
	else {
		if (insert_pos == GTK_CLIST_DRAG_INTO &&
		    (near->type == BM_CATEGORY || near->type == BM_AUTOBOOKMARKS))
			parent = near;
		else
			parent = near->parent;
	}
	
	g_assert (parent != NULL);
	b->parent = parent;
	
	position_node = g_list_find (parent->list, near);
	if (!position_node)
		parent->list = g_list_prepend (parent->list, b);
	else {
		position = g_list_position (parent->list, position_node);
		
		switch (insert_pos) {
		case GTK_CLIST_DRAG_NONE:
		case GTK_CLIST_DRAG_BEFORE:
			break;
		case GTK_CLIST_DRAG_INTO:
			if (near == NULL)
				break;
			if (near->type == BM_SITE ||
			    near->type == BM_SEPARATOR)
				position++;
			break;				
		case GTK_CLIST_DRAG_AFTER:
			position++;
			break;
		default:
			break;
		}
		parent->list = g_list_insert (parent->list, b, position);
	}
	bookmarks_update_alias (NULL, parent);
}

/**
 * bookmarks_move_bookmark: Moves a bookmark
 * @controls: the controls of the editor
 * @b: the bookmark to move
 * @where: if 0 then move up, else move down
 * Returns: 1 if bookmark is now first or last in the list, 0 otherwise
 *
 * Moves a bookmark up or down and updates both the edition ctree and the 
 * bookmarks structure 
 **/
gint
bookmarks_move_bookmark (BookmarksEditorControls *controls, 
			 BookmarkItem *b, int where)
{
	BookmarkItem *parent, *grandparent, *other;
	GList *pos, *other_pos;
	
	parent = b->parent;
	if (!parent)
		return 1;

	grandparent = parent->parent;
	pos = g_list_find (parent->list, b);

	if (!where) /* up */
		other_pos = g_list_previous(pos);
	else /* down */
		other_pos = g_list_next(pos);
	
	if (other_pos) {
		other = other_pos->data;
		if (other->type == BM_CATEGORY) {
			/* expand the category and move b into it */
			parent->list = g_list_remove_link (parent->list, pos);
			bookmarks_update_alias (controls, parent);
			gtk_ctree_expand (GTK_CTREE (controls->ctree), other->tree_item);
			
			if (!where) { 
				/* up */
				other->list = g_list_append (other->list, b);
				b->parent = other;
			} else {
				/* down */
				b->parent = other;
				other->list = g_list_prepend (other->list, b);
			}
		} else {
			gint newindex = g_list_position (parent->list, other_pos);
			b->parent->list = g_list_remove_link (b->parent->list, pos);
			b->parent->list = g_list_insert (parent->list, b, newindex);
		}
	} else {
		/* move b to its grandparent*/
		if  (!grandparent)
			return 1;
		
		parent->list = g_list_remove_link(parent->list, pos);
		bookmarks_update_alias (controls, parent);
		
		if (!where) {
			/* up */
			grandparent->list = g_list_insert 
				(grandparent->list, b, g_list_position 
				 (grandparent->list, g_list_find(grandparent->list, 
								 parent)));
			b->parent = grandparent;
		} else {
			/* down */
			GList *next = g_list_find(grandparent->list, parent)->next;
			grandparent->list = g_list_insert (grandparent->list, b, 
							   g_list_position 
							   (grandparent->list, next));
			b->parent = grandparent;
		}
	}
	bookmarks_editor_place_tree_item (controls, b);
	bookmarks_update_alias (controls, b->parent);
	gtk_ctree_select ( GTK_CTREE (controls->ctree), b->tree_item);

	return 0;
}

/**
 * Renames a bookmark (wich has just been added) or deletes it if the user cancelled 
 */
void 
bookmarks_string_request_callback (gchar *string, BookmarkItem *b) 
{
	if (!string) { /* User canceled */
		bookmarks_remove_recursively (b);
	} else { 
		g_free (b->name);
		b->name = string;
		
		if (strlen (b->name) == 0)
		{
			g_free (b->name);
			b->name = escape_uline_accel ( _("Untitled"));
		}

		if (b->url &&
		    gnome_config_get_bool (CONF_GENERAL_FAVICONS_ENABLED))
			get_siteicon (NULL, b->url);
		
		if (bookmarks_editor)
			bookmarks_editor_place_tree_item (bookmarks_editor, b);
		bookmarks_update_menu ();
		bookmarks_save (); 
	}
}

/**
 * add_temp_bookmark: Adds a temporal bookmark
 * @type: The type of the bookmark
 * @name: The name of the bookmark, if NULL or "" the url will be used instead.
 * @url: The url of the bookmark
 * @parent: The parent bookmark or NULL if the bookmark should be added to the 
 * main temp category
 * Return value: The new bookmark
 **/
BookmarkItem *
add_temp_bookmark (BookmarkType type, const gchar *name, const gchar *url, 
		   BookmarkItem *parent)
{
	BookmarkItem *bookmark;

	bookmark = bookmarks_new_bookmark (type, TRUE, name, url, NULL, NULL, NULL);

	if (!temp_bookmarks_root) {
		temp_bookmarks_init ();
	}
	if (!parent) parent = temp_bookmarks_root;
	
	bookmark->parent = parent;
	parent->list = g_list_append (parent->list, bookmark);
	bookmark->time_added = time (NULL);

	if (temp_bookmarks_window != NULL) 
	{
		bookmarks_editor_place_tree_item (temp_bookmarks_window, 
						  bookmark);
		temp_bookmarks_window->dirty = TRUE;
	}

	/* Save the temp bookmarks, for safety in case of a crash */
	temp_bookmarks_save ();

	return bookmark;
}

/**
 * bookmarks_list_all_urls:
 * -----------------------
 * get a list of all bookmarked URLs
 */
GSList *
bookmarks_list_all_urls (BookmarkItem *bookmark)
{
	GList *l;
	GSList * result;

	g_return_val_if_fail(bookmark != NULL, NULL);
	g_return_val_if_fail(bookmark->type == BM_CATEGORY, NULL);

	result = g_slist_alloc();

	for (l = bookmark->list; l != NULL; l = l->next)
	{
		BookmarkItem *b = l->data;
		/* skip alias */
		if (b->alias_of) continue;
		if (b->type == BM_CATEGORY) {
			g_slist_concat(result, bookmarks_list_all_urls(b));
		} else
			if (b->type == BM_SITE) {
				g_slist_append(result, b->url);
			}
	}
	return result;
}

/**
 * get a singly linked list of all the category/folder bookmarks 
 */
GSList * 
bookmarks_list_all_folders (BookmarkItem *bookmark)
{
	GList *l;
	GSList * result;

	g_return_val_if_fail(bookmark != NULL, NULL);
	g_return_val_if_fail(bookmark->type == BM_CATEGORY, NULL);

	result = g_slist_alloc();

	for (l = bookmark->list; l != NULL; l = l->next)
	{
		BookmarkItem *b = l->data;
		if (b->type == BM_CATEGORY) {
			g_slist_append(result, b);
			if (!b->alias_of) 
				g_slist_concat (result, 
						bookmarks_list_all_folders(b));
		}
	}
	return result;
}


/**
 * bookmarks_parse_nick: returns the url of a given nick with one arg
 * @text: the text to try to parse
 * @root: the root bookmark for start the search, if NULL the usual tree will be used
 * 
 * Tries to find a bookmarkitem wich has a nick like the first word of @text and if 
 * found return the URL result of substituting each %s in the BookmarkItem url by the 
 * text after the first " " in @text
 * 
 * Return value: the resulting URL or NULL if not matching nick found
 **/
gchar *
bookmarks_parse_nick (gchar *text, BookmarkItem *root)
{
	gchar *nick;
	gchar *url;
	gchar *space;
	BookmarkItem *b;

	/* remove any leading spaces */
	while (*text == ' ')
		text++;
	if (*text == '\0') 
		return NULL;

	/* find the nick */
	space = strstr (text, " ");
	if (space)  {
		/* split out nickname and arguments */
		nick = g_strndup (text, space - text);

		/* find smart bookmark */
		b = bookmarks_find_by_nick (nick, root, TRUE);
		if (b == NULL) /* try for a normal if no smartbm found*/
			b = bookmarks_find_by_nick (nick, root, FALSE);
		g_free (nick);

		/* check we found it */
		if (b == NULL)
			return NULL;

		/* do substitution */		
		url = bookmarks_substitute_argument (b, space + 1);

		/* return completed url */
		return url;
	} else {
		/* the whole thing is the nick */
		nick = g_strdup (text);

		/* find normal bookmark */
		b = bookmarks_find_by_nick (nick, root, FALSE);
		if (b == NULL) /* try for a smartbm if no normal found*/
			b = bookmarks_find_by_nick (nick, root, TRUE);
		g_free (nick);

		/* check we found it */
		if (b == NULL)
			return NULL;

		/* return copy of the url */
		return g_strdup(b->url);
	}
}

/**
 * bookmarks_substitute_argument:
 **/
gchar *
bookmarks_substitute_argument (BookmarkItem *b, const gchar *arguments)
{
	gchar *ret = NULL;
	const gchar *s = b->url;
	int expected_args = 0;

	/* count %s in the URL */
	while ((s = strstr(s, "%s"))) {
	    ++expected_args;
	    ++s;
	}
	
	ret = g_strdup (b->url);

	if (expected_args <= 1) {
	    /*  do the substitution */
	    while ( (s = strstr (ret, "%s")) ) {
		gchar *new1, *new2;
		new1 = g_strndup (ret, s - ret);
		new2 = g_strconcat (new1, arguments, s + 2, NULL);
		g_free (new1);
		g_free (ret);
		ret = new2;
	    }
	}
	else {
	    /* try to substitute matching arguments */
	    gchar **args;
	    int arg = 0;
	    args = g_strsplit(arguments, " ", -1);
	    while ( (s = strstr (ret, "%s")) ) {
		gchar *new1, *new2;
		if (!args[arg])
		    break;
		new1 = g_strndup (ret, s - ret);
		new2 = g_strconcat (new1, args[arg++], s + 2, NULL);
		g_free (new1);
		g_free (ret);
		ret = new2;
	    }
	    g_strfreev (args);
	}

	return ret;
}

/**
 * bookmarks_find_by_nick: search for a BookmarkItem with a nick
 * @nick: the nick to search for.
 * @r: the root bookmark for start the search, if NULL the usual tree will be used
 * @wantsmart: if we are trying to find a smart bookmark or not
 * 
 * Search a bookmarks category recursivley for a bookmark with the given nick
 * 
 * Return value: the found BookmarkItem or NULL if nothing found
 **/
static BookmarkItem *
bookmarks_find_by_nick (const gchar *nick, BookmarkItem *r, gboolean wantsmart)
{
	GList *l;
	BookmarkItem *b;
	if (!r) r = bookmarks_root;

	if ((r->nick) && !strcmp (r->nick, nick) &&
			(wantsmart ^ (strstr(r->url,"%s")==NULL)))
		return r;
	if ((r->type == BM_CATEGORY) && !r->alias_of)
		for (l = r->list; l != NULL; l = g_list_next (l)) {
			b = bookmarks_find_by_nick (nick, l->data, wantsmart);
			if (b) return b;
		}
	return NULL;
}

/**
 * bookmarks_save_as: convert a bookmark to a string
 * @bookmark: the bookmark to be saved (can be a category)
 *
 * Return value: the bookmarks as a string.
 **/
gchar * 
bookmarks_item_to_string (BookmarkItem *bookmark)
{
	xmlNodePtr node;
	xmlDocPtr doc = xmlNewDoc ("1.0");
	xmlChar *mem;
	int size;

	node = xmlNewDocNode (doc, NULL, "single_bookmark", NULL);

	bookmarks_save_recursively (doc, node, bookmark);

	xmlDocSetRootElement (doc, node);

	xmlDocDumpMemory (doc, &mem, &size);

	xmlFreeDoc (doc);

	return mem;
}

/**
 * bookmarks_load_from_string: Read a bookmark from a string
 * @string: the string to parse.
 * Return value: The new allocated bookmark or NULL if error
 **/
BookmarkItem *
bookmarks_item_from_string (char *string)
{
	xmlDocPtr doc;
	xmlNodePtr item;
	BookmarkItem *b = NULL;

	doc = xmlParseMemory (string, strlen (string));	
	
	if (doc) {
		item = doc->root->childs;
		b = bookmarks_xml_read_item (item);
		xmlFreeDoc (doc);
	} else {
		g_warning ("Error parsing: %s", string);
	}
	return b;
}

/**
 * bookmarks_translate_string
 *   Converts non-alphanumeric characters in a smart bookmark string
 *   to the form %{hex key code}
 *   The returned string should be freed.
 **/
gchar *
bookmarks_translate_string (gchar *smart_bm_string)
{
	gchar *translated_string, *tstring_ptr;
	guchar curr_char;
	gchar key_code[4];

	/* each translated character will have the form %XX, so we'll
	   allocatedenough space for 3*current string length */
	translated_string = g_malloc0(3*strlen(smart_bm_string)+1);
	tstring_ptr = translated_string;

	while (*smart_bm_string)
	{
		curr_char = *(unsigned char *)smart_bm_string;
		if ( (curr_char >= 'a' && curr_char <= 'z') ||
		     (curr_char >= 'A' && curr_char <= 'Z') ||
		     (curr_char >= '0' && curr_char <= '9') )
			*tstring_ptr++ = curr_char;
		else if (curr_char == ' ')
			*tstring_ptr++ = '+';
		else
		{
			sprintf(key_code, "%%%X", curr_char);
			strcat(translated_string, key_code);
			tstring_ptr += 3;
		}
		smart_bm_string++;
	}

	return (translated_string);
}

/**
 * bookmarks_folder_open_all_items: Loads all bookmarks in a folder 
 **/
void
bookmarks_folder_open_all_items (GaleonEmbed *embed, BookmarkItem *bi,
				 gboolean new_window, gboolean reuse)
{
	GList *l = bi->list;
	BookmarkItem *child = NULL;

	while (l) {
		child = l->data;
		if (child->url)
			break;
		l = l->next;
	}

	if (l == NULL)
		return;

	/* reuse the first tab/window if required */
	/* FIXME: reusing the first tab/window often causes a crash, for
	   now a new window/tab is always created  --Josh */
//	if (reuse)
//		embed_load_url (embed, child->url);
//	else
//		embed = embed_create_from_url (embed, child->url, TRUE);

	if (reuse)
		embed = embed_create_from_url (embed, child->url, new_window);
	else
		embed = embed_create_from_url (embed, child->url, TRUE);	

	/* process the remaining bookmarks */
	l = l->next;
	while (l)
	{
		BookmarkItem *child = l->data;
		if (child->url)
			embed_create_from_url (embed, child->url, new_window);
		l = l->next;
	}
}

/* FIXME: Unused? */
/**
 * bookmarks_add_completion_recursively: recursively add bookmarks to the
 * auto completions table
 */
static void bookmarks_add_completion_recursively (BookmarkItem *b)
{
	GList *item;
  
	if (b->alias_of) return; /* the real one should have been added already */
	
	switch (b->type)
	{
	case BM_CATEGORY:
	case BM_AUTOBOOKMARKS:
		for (item = b->list; item != NULL; item = item->next)
		{
			bookmarks_add_completion_recursively (item->data);
		}
		break;

	case BM_SITE:
		auto_completion_add_url (b->url);
		break;

	case BM_SEPARATOR:
		break;
	}

}

/**
 * Removes all alias of a bookmark 
 */
static void 
bookmarks_remove_all_alias (BookmarkItem *b)
{
	BookmarkItem *alias = b->alias;
	if (alias) {
		bookmarks_remove_all_alias (alias);
		bookmarks_remove_recursively (alias);
	}
}

/**
 * Return the bookmark with the given id
 */
static BookmarkItem *bookmarks_find_by_id (long id, BookmarkItem *r)
{
	GList *l;
	BookmarkItem *b;
	if (!r) r = bookmarks_root;
	
	/* Don't return bookmarks that may be not yet created aliases */
	if ((r->id == id) && r->name != NULL) return bookmarks_find_real_bookmark (r);
	if ((r->type == BM_CATEGORY) && !r->alias_of)
		for (l = r->list; l != NULL; l = g_list_next (l)) {
			b = bookmarks_find_by_id (id, l->data);
			if (b) return b;
		}
	return NULL;
}

/**
 * bookmarks_get_siteicon: Parse an http URL ans return the appropriate icon 
 */
PixmapData *
bookmarks_get_siteicon (const gchar *url)
{
	GnomeVFSURI *vfs_uri, *vfs_parent_uri;
	gchar *favicon_url, *favicon_path;
	PixmapData *result = NULL;

	if (!gnome_config_get_bool (CONF_GENERAL_FAVICONS_ENABLED))
	{
		return site_pixmap_data;
	}

	if (url == NULL)
	{
		return site_pixmap_data;
	}
	
	if (favicons_cache != NULL)
	{
		result = g_hash_table_lookup (favicons_cache, url);
		if (result != NULL) return result;
	}
	
	vfs_uri = gnome_vfs_uri_new(url);
	if (vfs_uri == NULL)
	{
		return site_pixmap_data;
	} 
	
	if (gnome_vfs_uri_has_parent (vfs_uri))
	{
		vfs_parent_uri = gnome_vfs_uri_get_parent (vfs_uri);
	} 
	else
	{
		vfs_parent_uri = vfs_uri;
	}
	vfs_parent_uri = gnome_vfs_uri_append_file_name (vfs_parent_uri, 
							 "favicon.ico");
	favicon_url = gnome_vfs_uri_to_string (vfs_parent_uri, 0);

	gnome_vfs_uri_unref (vfs_parent_uri);
	gnome_vfs_uri_unref (vfs_uri);
	
	favicon_path = favicon_filename (favicon_url);
	
	if (g_file_exists (favicon_path))
	{
		GdkPixbuf *pixbuf;
		
		pixbuf = gdk_pixbuf_new_from_file (favicon_path);
		if (pixbuf != NULL)
		{
			if (gdk_pixbuf_get_width (pixbuf) > 16 || 
			    gdk_pixbuf_get_height (pixbuf) > 16)
			{
				GdkPixbuf *scaled_icon = 
					gdk_pixbuf_scale_simple (pixbuf, 
						16, 16, GDK_INTERP_NEAREST);
				gdk_pixbuf_unref (pixbuf);
				pixbuf = scaled_icon;
			}

			result = g_new (PixmapData, 1);
			gdk_pixbuf_render_pixmap_and_mask (pixbuf,
							   &(result->pixmap),
							   &(result->mask),
							   100);
			gdk_pixbuf_unref (pixbuf);
	
			if (favicons_cache == NULL) 
			{
				favicons_cache = g_hash_table_new (g_str_hash,
								   g_str_equal);
			}
			g_hash_table_insert (favicons_cache, (gchar *)url, 
					     result);
		}
		else
		{
			result = site_pixmap_data;
		}
	}
	else
	{
		result = site_pixmap_data;
	}

	g_free (favicon_url);
	g_free (favicon_path);
	
	return result;
}

