/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"

/** The temp bookmarks window */
BookmarksEditorControls *temp_bookmarks_window = NULL;

/** The bookmarks editing controls */
BookmarksEditorControls *bookmarks_editor = NULL;

/* Function prototypes */
BookmarksEditorControls *
bookmarks_editor_controls_init (BookmarkItem *root,
				BookmarksEditorControls_type type);
void bookmarks_editor_remove_tree_item_null (BookmarkItem *b);
void bookmarks_editor_goto_bookmark (BookmarkItem *b);
void bookmarks_editor_add_menubar (GladeXML *gxml, BookmarksEditorControls *editor);

BookmarksEditorControls *
bookmarks_editor_controls_init (BookmarkItem *root,
				BookmarksEditorControls_type type)
{
	GladeXML *gxml = NULL;
	GtkWidget *bookmarks_editor_edit_controls;
	BookmarksEditorControls *editor;
		
	editor = g_new0 (BookmarksEditorControls, 1);	

	bookmarks_load_icons ();
	
	editor->root_bookmark = root;

	switch (type) {
	case STANDARD_BOOKMARKS_EDITOR:
		gxml = glade_widget_new ("bookmarks_editor", 
					 &(editor->dialog), editor);
		editor->toolbar = glade_xml_get_widget 
			(gxml, "bookmarks_editor_toolbar");
		editor->ctree = glade_xml_get_widget
			(gxml, "bookmarks_editor_ctree");
		editor->vpane = NULL;
		editor->edit_frame = glade_xml_get_widget
			(gxml, "bookmarks_editor_edit_frame");
		editor->name_entry = glade_xml_get_widget
			(gxml, "bookmarks_editor_name_entry");
		editor->url_entry = glade_xml_get_widget
			(gxml, "bookmarks_editor_url_entry");
		editor->nick_entry = glade_xml_get_widget
			(gxml, "bookmarks_editor_nick_entry");
		editor->pixmap_entry = glade_xml_get_widget
			(gxml, "bookmarks_editor_pixmap_entry");
		editor->pixmap_file_entry = glade_xml_get_widget
			(gxml, "bookmarks_editor_pixmap_file_entry");
		editor->create_toolbar_toggle = glade_xml_get_widget
			(gxml, "bookmarks_editor_create_toolbar_toggle");
		editor->create_context_menu_toggle = glade_xml_get_widget
			(gxml, "bookmarks_editor_create_context_menu_toggle");
		editor->notes_text = glade_xml_get_widget
			(gxml, "bookmarks_editor_notes_text");
		editor->time_added_entry = glade_xml_get_widget
			(gxml, "bookmarks_editor_time_added_entry");
		editor->time_modified_entry = glade_xml_get_widget
			(gxml, "bookmarks_editor_time_modified_entry");
		editor->time_visited_entry = glade_xml_get_widget
			(gxml, "bookmarks_editor_time_visited_entry");
		bookmarks_editor_add_menubar (gxml, editor);
		g_assert (editor->dialog != NULL);
		g_assert (editor->toolbar != NULL);
		g_assert (editor->name_entry != NULL);
		g_assert (editor->url_entry != NULL);
		g_assert (editor->nick_entry != NULL);
		g_assert (editor->pixmap_entry != NULL);
		g_assert (editor->pixmap_file_entry != NULL);
		g_assert (editor->create_toolbar_toggle != NULL);
		g_assert (editor->create_context_menu_toggle != NULL);
		g_assert (editor->notes_text != NULL);
		g_assert (editor->time_added_entry != NULL);
		g_assert (editor->time_modified_entry != NULL);
		g_assert (editor->time_visited_entry != NULL);
		break;
	case DOCKED_BOOKMARKS_EDITOR:
		gxml = glade_widget_new ("bookmarks_dock", 
					 &(editor->dialog), editor);
		editor->toolbar = glade_xml_get_widget 
			(gxml, "bookmarks_dock_toolbar");
		editor->ctree = glade_xml_get_widget
			(gxml, "bookmarks_dock_ctree");
		editor->vpane = NULL;
		editor->edit_frame = NULL;
		editor->name_entry = glade_xml_get_widget
			(gxml, "bookmarks_dock_name_entry");
		editor->url_entry = glade_xml_get_widget
			(gxml, "bookmarks_dock_url_entry");
		editor->nick_entry = glade_xml_get_widget
			(gxml, "bookmarks_dock_nick_entry");
		editor->pixmap_entry = glade_xml_get_widget
			(gxml, "bookmarks_dock_pixmap_entry");
		editor->pixmap_file_entry = glade_xml_get_widget
			(gxml, "bookmarks_dock_pixmap_file_entry");
		editor->create_toolbar_toggle = glade_xml_get_widget
			(gxml, "bookmarks_dock_create_toolbar_toggle");
		editor->create_context_menu_toggle = glade_xml_get_widget
			(gxml, "bookmarks_dock_create_context_menu_toggle");
		editor->notes_text = glade_xml_get_widget
			(gxml, "bookmarks_dock_notes_text");
		editor->time_added_entry = NULL;
		editor->time_modified_entry = NULL;
		editor->time_visited_entry = NULL;
		g_assert (editor->dialog != NULL);
		g_assert (editor->toolbar != NULL);
		g_assert (editor->name_entry != NULL);
		g_assert (editor->url_entry != NULL);
		g_assert (editor->nick_entry != NULL);
		g_assert (editor->pixmap_entry != NULL);
		g_assert (editor->pixmap_file_entry != NULL);
		g_assert (editor->create_toolbar_toggle != NULL);
		g_assert (editor->create_context_menu_toggle != NULL);
		g_assert (editor->notes_text != NULL);
		break;
	case COMPACT_BOOKMARKS_EDITOR:
		gxml = glade_widget_new ("compact_bookmarks_editor", 
					 &(editor->dialog), editor);
		editor->ctree = glade_xml_get_widget
			(gxml, "compact_bookmarks_editor_ctree");
		editor->edit_frame = glade_xml_get_widget
			(gxml, "compact_bookmarks_editor_edit_frame");
		editor->name_entry = glade_xml_get_widget
			(gxml, "compact_bookmarks_editor_name_entry");
		editor->url_entry = glade_xml_get_widget
			(gxml, "compact_bookmarks_editor_url_entry");
		editor->nick_entry = NULL;
		editor->pixmap_entry = NULL;
		editor->pixmap_file_entry = NULL;
		editor->create_toolbar_toggle = NULL;
		editor->create_context_menu_toggle = NULL;
		editor->notes_text = glade_xml_get_widget
			(gxml, "compact_bookmarks_editor_notes_text");
		g_assert (editor->dialog != NULL);
		g_assert (editor->edit_frame != NULL);
		g_assert (editor->name_entry != NULL);
		g_assert (editor->url_entry != NULL);
		g_assert (editor->nick_entry == NULL);
		g_assert (editor->pixmap_entry == NULL);
		g_assert (editor->pixmap_file_entry == NULL);
		g_assert (editor->create_toolbar_toggle == NULL);
		g_assert (editor->create_context_menu_toggle == NULL);
		g_assert (editor->notes_text != NULL);
		bookmarks_editor_edit_controls = glade_xml_get_widget
			(gxml, "compact_bookmarks_editor_control_buttons");
		gtk_toolbar_set_style (GTK_TOOLBAR (bookmarks_editor_edit_controls),
				       GTK_TOOLBAR_ICONS);
		break;
	default:
		g_error ("Incorrect bookmarks editor dialog type");
	}
	g_assert (editor->ctree != NULL);
	
	editor->selection = NULL;
	editor->last_pressed = NULL;
	editor->lock_edit_controls = FALSE;
	editor->dirty = FALSE;
	editor->window = NULL;
	editor->save_function = NULL;

	/* unref the GladeXML object, we don't use it anymore */
	gtk_object_unref (GTK_OBJECT (gxml));

	gtk_drag_source_set (editor->ctree, GDK_BUTTON1_MASK, 
			     bookmarks_dnd_targets, 
			     bookmarks_dnd_targets_num_items, 
			     GDK_ACTION_COPY | GDK_ACTION_MOVE |
			     GDK_ACTION_LINK);
	gtk_drag_dest_set (editor->ctree,
			   GTK_DEST_DEFAULT_DROP,
			   bookmarks_dnd_targets,
			   bookmarks_dnd_targets_num_items,
			   GDK_ACTION_COPY | GDK_ACTION_MOVE |
			   GDK_ACTION_LINK);

	if (editor->pixmap_entry != NULL)
	{
		gnome_pixmap_entry_set_pixmap_subdir 
			(GNOME_PIXMAP_ENTRY (editor->pixmap_entry), SHARE_DIR);
	}

	bookmarks_editor_place_tree_item (editor, root);
	return editor;
}

void
bookmarks_editor_show_dialog (GaleonWindow *window,
			      BookmarksEditorControls_type type)
{
	gint pane_pos;
	GtkToolbarStyle style;
	BookmarksEditorControls *controls = NULL;
	gint x, y, width, height;
	
	if (type == STANDARD_BOOKMARKS_EDITOR && 
	    (controls = bookmarks_editor) == NULL)
	{
		bookmarks_editor = bookmarks_editor_controls_init
			(bookmarks_root, STANDARD_BOOKMARKS_EDITOR);
		bookmarks_editor->save_function = bookmarks_save;
		controls = bookmarks_editor;

		gnome_config_push_prefix("/galeon/State/");
		x = gnome_config_get_int("bookmarks_editor_x=-1");
		y = gnome_config_get_int("bookmarks_editor_y=-1");
		if (!(x == -1 && y == -1))
			gtk_widget_set_uposition(bookmarks_editor->dialog,
						 x, y);

		width = gnome_config_get_int("bookmarks_editor_width=750");
		height = gnome_config_get_int("bookmarks_editor_height=550");
		gtk_window_set_default_size 
			(GTK_WINDOW (bookmarks_editor->dialog), width, height);
		pane_pos = gnome_config_get_int("bookmarks_editor_vpane=-1");
		if (pane_pos != -1 && bookmarks_editor->vpane != NULL)
			gtk_paned_set_position(
					GTK_PANED(bookmarks_editor->vpane),
					pane_pos);
		gnome_config_pop_prefix ();
	}
	
	if (type == COMPACT_BOOKMARKS_EDITOR &&
	    (controls = temp_bookmarks_window) == NULL)
	{
		if (temp_bookmarks_root == NULL) {
			temp_bookmarks_init ();
		}
		temp_bookmarks_window = bookmarks_editor_controls_init 
			(temp_bookmarks_root, COMPACT_BOOKMARKS_EDITOR);
		temp_bookmarks_window->save_function = temp_bookmarks_save;
		controls = temp_bookmarks_window;

		gnome_config_push_prefix("/galeon/State/");
		x = gnome_config_get_int("temp_bookmarks_x=-1");
		y = gnome_config_get_int("temp_bookmarks_y=-1");
		if (!(x == -1 && y == -1))
			gtk_widget_set_uposition(temp_bookmarks_window->dialog,
						 x, y);

		width = gnome_config_get_int("temp_bookmarks_width=-1");
		height = gnome_config_get_int("temp_bookmarks_height=-1");
		if (!(width == -1 && height == -1))
			gtk_window_set_default_size(
				     GTK_WINDOW(temp_bookmarks_window->dialog),
				     width,height);
		gnome_config_pop_prefix ();
	}
	
	g_return_if_fail (controls != NULL);
	
	if (controls->toolbar != NULL)
	{
		/* read preferences */
		style = gnome_config_get_int 
			("/galeon/Appearance/toolbar_type=0");

		/* setup toolbar */
		gtk_toolbar_set_style 
			(GTK_TOOLBAR (controls->toolbar), style);
	}

	controls->window = window;
	gtk_widget_show (controls->dialog);
	window_set_layer (controls->dialog);
}

void
bookmarks_editor_hide_dialog (BookmarksEditorControls *controls)
{
	gint x, y, width, height;
	GdkWindow *window;

	g_return_if_fail (controls != NULL);

	window = controls->dialog->window;

	gdk_window_get_root_origin(window, &x, &y);
	gdk_window_get_size(window, &width, &height);

	gnome_config_push_prefix("/galeon/State/");

	if (controls == bookmarks_editor)
	{
		gnome_config_set_int("bookmarks_editor_x", x);
		gnome_config_set_int("bookmarks_editor_y", y);
		gnome_config_set_int("bookmarks_editor_width", width);
		gnome_config_set_int("bookmarks_editor_height", height);

		if (controls->vpane != NULL)
		{
			GtkPaned *vpane = GTK_PANED(controls->vpane);
			gint pane_pos = vpane->child1_size;
			gnome_config_set_int("bookmarks_editor_vpane", pane_pos);
		}
	}
	else
	{
		gnome_config_set_int("temp_bookmarks_x", x);
		gnome_config_set_int("temp_bookmarks_y", y);
		gnome_config_set_int("temp_bookmarks_width", width);
		gnome_config_set_int("temp_bookmarks_height", height);
	}
	gnome_config_pop_prefix();
	gnome_config_sync();

	gtk_widget_hide (controls->dialog);
}

void
bookmarks_editor_show_dock (GaleonWindow *window)
{
	bookmarks_editor = bookmarks_editor_controls_init
		(bookmarks_root, DOCKED_BOOKMARKS_EDITOR);
	bookmarks_editor->save_function = bookmarks_save;	

	gnome_app_add_dock_item (GNOME_APP (window->WMain),
				 GNOME_DOCK_ITEM (bookmarks_editor->dialog),
				 GNOME_DOCK_LEFT, 1, 0, 0);
}

void
bookmarks_editor_hide_dock (void)
{
}

/**
 * Inits the temp bookmarks
 */
void
temp_bookmarks_init (void)
{
	char *file = g_strconcat (g_get_home_dir(), "/.galeon/temp_bookmarks.xml", NULL);
	temp_bookmarks_root = bookmarks_load_from (file);
	if (!temp_bookmarks_root) 
		temp_bookmarks_root = bookmarks_new_bookmark 
			(BM_CATEGORY, TRUE, _("Temporary bookmarks"), 
			 NULL, NULL, NULL, NULL);
}
				
/**
 * bookmarks_editor_place_tree_item: place a bookmark in the editor ctree
 * @controls: the set of controls
 * @b: the bookmark
 * 
 * Create the tree items for a bookmark and its children. If it has already
 * been placed, removes the old item.
 * 
 **/
void
bookmarks_editor_place_tree_item (BookmarksEditorControls *controls, BookmarkItem *b)
{
	gchar *text[3];
	gboolean isleaf;
	PixmapData *bm_icon;
	GdkPixmap *pixmap;
	GdkPixmap *pixmap_open;
	GdkPixmap *mask_open;
	GdkBitmap *mask;
	GtkCTreeNode *parent_node;
	GtkCTreeNode *sibling_node;

	text[2] = NULL;
	switch (b->type) {
	case BM_SEPARATOR:
		text[0] = text[1] = "------";
		isleaf = TRUE;
		pixmap = mask = pixmap_open = mask_open = NULL;
		break;
	case BM_SITE:
		text[0] = b->name;
		text[1] = b->url;
		isleaf = TRUE;

		bm_icon = bookmarks_get_siteicon (b->url);
		pixmap = bm_icon->pixmap;
		mask = bm_icon->mask;

		pixmap_open = pixmap;
		mask_open = mask;
		break;
	case BM_CATEGORY:
	case BM_AUTOBOOKMARKS:
		text[0] = b->name;
		text[1] = "";
		isleaf = FALSE;
		pixmap = folder_pixmap_data->pixmap; 
		mask = folder_pixmap_data->mask;
		pixmap_open = folder_open_pixmap_data->pixmap; 
		mask_open = folder_open_pixmap_data->mask;
		break;
	default:
		text[0] = text[1] = "";
		isleaf = TRUE;
		pixmap = mask = pixmap_open = mask_open = NULL;
		break;
	}

	if ((b->parent == NULL) || (b->parent == controls->root_bookmark))
		parent_node = NULL;
	else
		parent_node = b->parent->tree_item;

	sibling_node = NULL;
	if (b->parent != NULL) {
		GList *list_node = g_list_find (b->parent->list, b);
		if (list_node) {
			GList *sibling_list_node = g_list_next (list_node);
			if (sibling_list_node) {
				sibling_node = ((BookmarkItem *) 
					(sibling_list_node->data))->tree_item;
			}
		}
	}
	gtk_clist_freeze (GTK_CLIST (controls->ctree));
	if (b->tree_item)
		bookmarks_editor_remove_tree_item (controls, b);
	if (b != controls->root_bookmark) {
		if (b->alias_of) {
			/* TODO: use a diferent pixmap for aliases 
			   for now, just leave it w/o pixmap */
			pixmap = mask = pixmap_open = mask_open = NULL;
		}
		b->tree_item = gtk_ctree_insert_node
			(GTK_CTREE (controls->ctree), parent_node, sibling_node, text, 
			 2, pixmap, mask, pixmap_open, mask_open, isleaf, b->expanded);
		gtk_ctree_node_set_row_data (GTK_CTREE (controls->ctree), 
					     b->tree_item, b);
	}

	if (!isleaf && !b->alias_of) /* don't display alias' children */ {
		GList *l;
		for (l = b->list; l != NULL; l = g_list_next (l))
			bookmarks_editor_place_tree_item (controls, l->data);
	}
	gtk_clist_thaw (GTK_CLIST (controls->ctree));
}

/**
 * bookmarks_editor_remove_tree_item: remove a bookmark from the editor ctree
 * @controls: the set of controls
 * @b: the bookmark
 * 
 * Removes bookmarks and it's childs from the bookmark editor ctree and sets the 
 * tree_item fields to NULL. This does not delete the bookmarks
 * 
 **/
void
bookmarks_editor_remove_tree_item (BookmarksEditorControls *controls, BookmarkItem *b)
{
	/* this removes the treeitem and its children */
	gtk_ctree_remove_node (GTK_CTREE (controls->ctree), 
			       GTK_CTREE_NODE (b->tree_item));
	/* now, let's NULL the fields */
	bookmarks_editor_remove_tree_item_null (b);
}

/**
 * bookmarks_editor_remove_tree_item_null: Sets to NULL the tree_item field of
 * a bookmarks and its children
 * @b: the bookmark
 * 
 * This is an auxiliar function for bookmarks_editor_remove_tree_item.
 **/
void
bookmarks_editor_remove_tree_item_null (BookmarkItem *b)
{
	GList *l;
	b->tree_item = NULL;
	if ((b->type == BM_CATEGORY || b->type == BM_AUTOBOOKMARKS) && 
	    !b->alias_of) /* alias don't have children */
	{
		for (l = b->list; l != NULL; l = g_list_next (l)) 
			bookmarks_editor_remove_tree_item_null (l->data);
	}
}

/**
 * bookmarks_editor_update_tree_item: update a bookmark in the editor ctree
 * @controls: the set of controls
 * @b: the bookmark
 * 
 * Updates the bookmark (or adds it if not yet added)
 * 
 **/
void
bookmarks_editor_update_tree_item (BookmarksEditorControls *controls, BookmarkItem *b)
{
	gchar *text[3];
	gboolean isleaf;
	PixmapData *bm_icon;
	GdkPixmap *pixmap;
	GdkBitmap *mask;
	GtkCTreeNode *parent_node;
		
	if (!b->tree_item) {
		bookmarks_editor_place_tree_item (controls, b);
		return;
	}

	text[2] = NULL;
	switch (b->type) {
	case BM_SEPARATOR:
		text[0] = text[1] = "------";
		isleaf = TRUE;
		pixmap = mask = NULL;
		break;
	case BM_SITE:
		text[0] = b->name;
		text[1] = b->url;
		isleaf = TRUE;
		bm_icon = bookmarks_get_siteicon (b->url);
		pixmap = bm_icon->pixmap;
		mask = bm_icon->mask;;
		break;
	case BM_CATEGORY:
	case BM_AUTOBOOKMARKS:
		text[0] = b->name;
		text[1] = "";
		isleaf = FALSE;
		pixmap = folder_pixmap_data->pixmap; 
		mask = folder_pixmap_data->mask;
		break;
	default:
		text[0] = text[1] = "";
		isleaf = TRUE;
		pixmap = mask = NULL;
		break;
	}

	if ((b->parent == NULL) || (b->parent == controls->root_bookmark))
		parent_node = NULL;
	else
		parent_node = b->parent->tree_item;

	if (b != controls->root_bookmark) {
		if (b->alias_of) {
			/* TODO: use a diferent pixmap for aliases 
			   for now, just leave it w/o pixmap */
			pixmap = mask = NULL;
		}
		gtk_ctree_node_set_pixtext
			(GTK_CTREE (controls->ctree), b->tree_item, 0, text[0],
			2, pixmap, mask);
		gtk_ctree_node_set_text
			(GTK_CTREE (controls->ctree), b->tree_item, 1, text[1]);
	}
}

/**
 * bookmarks_editor_goto_bookmark : position the bookmark editor 
 * 	at a given bookmark item & focus the name entry
 * @b : bookmark item to focus
 */
void bookmarks_editor_goto_bookmark (BookmarkItem *b)
{
	GtkCTreeNode *pnode;
	GtkCTree *ctree = GTK_CTREE (bookmarks_editor->ctree);
	GtkCList *clist = GTK_CLIST (bookmarks_editor->ctree);
	gint row;
	
	g_return_if_fail (bookmarks_editor != NULL);
	
	pnode = b->tree_item;
	while (pnode)
	{
		gtk_ctree_expand (ctree, pnode);
		pnode = GTK_CTREE_ROW (pnode)->parent;
	}

	gtk_ctree_node_moveto (ctree, b->tree_item, 0, 0, 0);
	gtk_clist_unselect_all (clist);
	gtk_ctree_select (ctree, b->tree_item);
	row = g_list_position (clist->row_list, (GList *) b->tree_item);
	clist->focus_row = row;
	gtk_widget_grab_focus (bookmarks_editor->name_entry);
}

#include "bookmarks_editor_callbacks.h"

void 
bookmarks_editor_add_menubar (GladeXML *gxml, BookmarksEditorControls *editor)
{
	GnomeUIInfo file_menu_uiinfo[] =
	{
		{ GNOME_APP_UI_ITEM, N_("_Go to bookmark target"), NULL,
		  compact_bookmarks_editor_go_clicked_cb, editor, NULL,
		  GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_JUMP_TO,
		  0, 0, NULL },
		GNOMEUIINFO_SEPARATOR,
		GNOMEUIINFO_MENU_SAVE_ITEM (bookmarks_editor_apply_clicked_cb, editor),
		GNOMEUIINFO_MENU_CLOSE_ITEM (bookmarks_editor_accept_clicked_cb, editor),
		GNOMEUIINFO_END
	};
	
	GnomeUIInfo edit_menu_uiinfo[] =
	{
		{ GNOME_APP_UI_ITEM, N_("Move _up"), NULL,
		  bookmarks_editor_move_up_clicked_cb, editor, NULL,
		  GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_UP,
		  0, 0, NULL },
		{ GNOME_APP_UI_ITEM, N_("Move _down"), NULL,
		  bookmarks_editor_move_down_clicked_cb, editor, NULL,
		  GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_DOWN,
		  0, 0, NULL },
		GNOMEUIINFO_SEPARATOR,
		{ GNOME_APP_UI_ITEM, N_("_Refresh icon"), NULL,
		  bookmarks_editor_fetch_icon_clicked_cb, editor, NULL,
		  GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_REFRESH,
		  0, 0, NULL },
		GNOMEUIINFO_SEPARATOR,
		{ GNOME_APP_UI_ITEM, N_("New _item"), NULL,
		  bookmarks_editor_new_item_clicked_cb, editor, NULL,
		  GNOME_APP_PIXMAP_FILENAME, "../galeon/new_item.xpm",
		  0, 0, NULL },
		{ GNOME_APP_UI_ITEM, N_("New _separator"), NULL,
		  bookmarks_editor_new_separator_clicked_cb, editor, NULL,
		  GNOME_APP_PIXMAP_FILENAME, "../galeon/new_separator.xpm",
		  0, 0, NULL },
		{ GNOME_APP_UI_ITEM, N_("New _category"), NULL,
		  bookmarks_editor_new_category_clicked_cb, editor, NULL,
		  GNOME_APP_PIXMAP_FILENAME, "../galeon/new_folder.xpm",
		  0, 0, NULL },
		{ GNOME_APP_UI_ITEM, N_("New _alias"), NULL,
		  bookmarks_editor_add_alias_cb, editor, NULL,
		  GNOME_APP_PIXMAP_NONE, NULL,
		  0, 0, NULL },
		GNOMEUIINFO_SEPARATOR,
		{ GNOME_APP_UI_ITEM, N_("_Remove bookmark"), NULL,
		  bookmarks_editor_remove_clicked_cb, editor, NULL,
		  GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_TRASH,
		  0, 0, NULL },
		GNOMEUIINFO_SEPARATOR,
		{ GNOME_APP_UI_ITEM, N_("Add alias for r_oot bookmark"), NULL,
		  bookmarks_editor_add_alias_root_cb, editor, NULL,
		  GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_TRASH,
		  0, 0, NULL },
		GNOMEUIINFO_END
	};

	GnomeUIInfo view_menu_uiinfo[] =
	{
		{ GNOME_APP_UI_TOGGLEITEM, N_("_Show editing controls"), NULL,
		  compact_bookmarks_editor_edit_toggled_cb , editor, NULL,
		  GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_PROPERTIES,
		  0, 0, NULL },
		GNOMEUIINFO_SEPARATOR,
		{ GNOME_APP_UI_ITEM, N_("_Expand all categories"), NULL,
		  bookmarks_editor_expand_all_cb, editor, NULL,
		  GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_BOOK_OPEN,
		  0, 0, NULL },
		{ GNOME_APP_UI_ITEM, N_("_Collapse all categories"), NULL,
		  bookmarks_editor_collapse_all_clicked_cb, editor, NULL,
		  GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_BOOK_RED,
		  0, 0, NULL },
		/* TODO
		GNOMEUIINFO_SEPARATOR,
		{ GNOME_APP_UI_TOGGLEITEM, N_("Show _date / time information"), NULL, 
		  bookmarks_editor_time_show_cb, editor, NULL, 
		  GNOME_APP_PIXMAP_NONE, NULL,
		  0, 0, NULL },
		*/
		GNOMEUIINFO_END
	};

        /* TODO
	GnomeUIInfo help_menu_uiinfo[] =
	{
		GNOMEUIINFO_END
	};
        */
	GnomeUIInfo menubar_uiinfo[] =
	{
		GNOMEUIINFO_MENU_FILE_TREE (file_menu_uiinfo),
		GNOMEUIINFO_MENU_EDIT_TREE (edit_menu_uiinfo),
		GNOMEUIINFO_MENU_VIEW_TREE (view_menu_uiinfo),
		/*TODO: GNOMEUIINFO_MENU_HELP_TREE (help_menu_uiinfo),*/
		GNOMEUIINFO_END
	};
	GtkWidget *menubar = glade_xml_get_widget (gxml, "bookmarks_editor_menubar");
	GtkWidget *window = glade_xml_get_widget (gxml, "bookmarks_editor");
	GtkAccelGroup *accel = GNOME_APP (window)->accel_group;
	gnome_app_fill_menu (GTK_MENU_SHELL (menubar), menubar_uiinfo, accel, TRUE, 0);
	gtk_widget_show_all (menubar);
	gtk_check_menu_item_set_active 
		(GTK_CHECK_MENU_ITEM (view_menu_uiinfo[0].widget), TRUE);
}

