/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "bookmarks_editor_callbacks.h"

/* internal helper functions */
static void fetch_icon_recursively (BookmarkItem *b,
				    BookmarksEditorControls *controls);
static void bookmarks_editor_apply (BookmarksEditorControls *controls);
static void bookmarks_editor_set_text (GtkWidget *control, gchar *text, 
				       gboolean active);
static void bookmarks_editor_set_name (BookmarksEditorControls *controls, 
				       BookmarkItem *bookmark, 
				       gboolean active);
static void bookmarks_editor_set_url (BookmarksEditorControls *controls, 
				      BookmarkItem *bookmark, gboolean active);
static void bookmarks_editor_set_nick (BookmarksEditorControls *controls, 
				       BookmarkItem *bookmark, 
				       gboolean active);
static void bookmarks_editor_set_create_toolbar (BookmarksEditorControls 
						 *controls, 
						 BookmarkItem *bookmark, 
						 gboolean active);
static void bookmarks_editor_set_create_context_menu (BookmarksEditorControls 
						      *controls, 
						      BookmarkItem *bookmark, 
						      gboolean active);
static void bookmarks_editor_set_notes (BookmarksEditorControls *controls, 
					BookmarkItem *bookmark, 
					gboolean active);
static void bookmarks_editor_set_pixmap (BookmarksEditorControls *controls, 
					 BookmarkItem *bookmark, 
					 gboolean active);
static void bookmarks_editor_set_times (BookmarksEditorControls *controls, 
					 BookmarkItem *bookmark);


void
bookmarks_editor_ctree_tree_select_row_cb (GtkCTree *ctree, GList *node,
					   gint column,
					   BookmarksEditorControls *controls)
{	
	BookmarkItem *b = controls->selection = 
		gtk_ctree_node_get_row_data (ctree, GTK_CTREE_NODE (node));
	if (!b) return;
	controls->lock_edit_controls = TRUE;

	if (controls->notes_text)
	{
		gtk_editable_delete_text (GTK_EDITABLE (controls->notes_text),
					  0, -1);
	}

	switch (b->type)
	{
	case BM_SITE:
		bookmarks_editor_set_name                 (controls, b, TRUE);
		bookmarks_editor_set_url                  (controls, b, TRUE);
		bookmarks_editor_set_nick                 (controls, b, TRUE);
		bookmarks_editor_set_create_toolbar       (controls, b, FALSE);
		bookmarks_editor_set_create_context_menu  (controls, b, TRUE);
		bookmarks_editor_set_notes                (controls, b, TRUE);
		bookmarks_editor_set_pixmap               (controls, b, TRUE);
		break;

	case BM_CATEGORY:
	case BM_AUTOBOOKMARKS:
		bookmarks_editor_set_name                 (controls, b, TRUE);
		bookmarks_editor_set_url                  (controls, b, FALSE);
		bookmarks_editor_set_nick                 (controls, b, FALSE);
		bookmarks_editor_set_create_toolbar       (controls, b, TRUE);
		bookmarks_editor_set_create_context_menu  (controls, b, TRUE);
		bookmarks_editor_set_notes                (controls, b, TRUE);
		bookmarks_editor_set_pixmap               (controls, b, TRUE);
		break;

	case BM_SEPARATOR:
		bookmarks_editor_set_name                 (controls, b, FALSE);
		bookmarks_editor_set_url                  (controls, b, FALSE);
		bookmarks_editor_set_nick                 (controls, b, FALSE);
		bookmarks_editor_set_create_toolbar       (controls, b, FALSE);
		bookmarks_editor_set_create_context_menu  (controls, b, FALSE);
		bookmarks_editor_set_notes                (controls, b, FALSE);
		bookmarks_editor_set_pixmap               (controls, b, FALSE);
		break;
	}
	bookmarks_editor_set_times (controls, b);
	controls->lock_edit_controls = FALSE;
}

void
bookmarks_editor_ctree_tree_unselect_row_cb (GtkCTree *ctree, GList *node,
					     gint column,
					     BookmarksEditorControls *controls)
{	
	controls->lock_edit_controls = TRUE;
	controls->selection = NULL;

	bookmarks_editor_set_name                 (controls, NULL, FALSE);
	bookmarks_editor_set_url                  (controls, NULL, FALSE);
	bookmarks_editor_set_nick                 (controls, NULL, FALSE);
	bookmarks_editor_set_create_toolbar       (controls, NULL, FALSE);
	bookmarks_editor_set_create_context_menu  (controls, NULL, FALSE);
	bookmarks_editor_set_notes                (controls, NULL, FALSE);
	bookmarks_editor_set_pixmap               (controls, NULL, FALSE);
	bookmarks_editor_set_times                (controls, NULL);

	controls->lock_edit_controls = FALSE;
}

void
bookmarks_editor_edited_cb (GtkEditable *editable,
			    BookmarksEditorControls *controls)
{
	if (!controls->lock_edit_controls && controls->selection) {
		BookmarkItem *b_real;
		BookmarkItem *b_alias;
		gchar *name = NULL;
		gchar *url = NULL;
		gchar *nick = NULL;
		gchar *pixmap_file = NULL;
		gboolean create_toolbar = FALSE;
		gboolean create_context_menu = FALSE;
		gchar *notes = NULL;
		/* note that b_alias and b_real may be the same bookmark */
		b_alias = controls->selection;
		b_real = bookmarks_find_real_bookmark (controls->selection);
		if (controls->name_entry)
			name = gtk_editable_get_chars 
				(GTK_EDITABLE (controls->name_entry), 0, -1);
		if (controls->url_entry)
			url = gtk_editable_get_chars 
				(GTK_EDITABLE (controls->url_entry), 0, -1);
		if (controls->nick_entry)
			nick = gtk_editable_get_chars 
				(GTK_EDITABLE (controls->nick_entry), 0, -1);
		if (controls->pixmap_file_entry)
			pixmap_file = gtk_editable_get_chars 
				(GTK_EDITABLE (controls->pixmap_file_entry), 
				 0, -1);
		if (controls->create_toolbar_toggle)
			create_toolbar = gtk_toggle_button_get_active 
				(GTK_TOGGLE_BUTTON (controls->create_toolbar_toggle));
		if (controls->create_context_menu_toggle)
			create_context_menu = gtk_toggle_button_get_active 
				(GTK_TOGGLE_BUTTON (controls->create_context_menu_toggle));
		if (controls->notes_text)
			notes = gtk_editable_get_chars 
				(GTK_EDITABLE (controls->notes_text), 0, -1);
		switch (b_real->type) {
		case BM_SITE:
			if (controls->name_entry) {
				if (b_real->name) g_free (b_real->name);
				b_real->name = name;
			}
			if (controls->url_entry) {
				if (b_real->url) g_free (b_real->url);
				b_real->url = url;
			}
			if (controls->nick_entry) {
				if (b_real->nick) g_free (b_real->nick);
				b_real->nick = nick;
			}
			if (controls->create_context_menu_toggle) {
				b_alias->create_context_menu 
					= create_context_menu;
			}
			/* FIXME: remove duplication */
			if (controls->pixmap_file_entry) {
				if (strcmp (pixmap_file, b_real->pixmap_file) != 0)
				{
					if (b_real->pixmap_data)
						g_free (b_real->pixmap_data);
					b_real->pixmap_data = 
						pixmap_data_new_from_file 
								(pixmap_file);
				}
				if (b_real->pixmap_file) g_free (b_real->pixmap_file);
				b_real->pixmap_file = pixmap_file;
			}
			if (controls->notes_text) {
				if (b_real->notes) g_free (b_real->notes);
				b_real->notes = notes;
			}
			break;
		case BM_CATEGORY:
		case BM_AUTOBOOKMARKS:
			if (controls->name_entry) {
				if (b_real->name) g_free (b_real->name);
				b_real->name = name;
			}
			if (url) g_free (url);
			if (nick) g_free (nick);
			if (controls->create_toolbar_toggle) {
				b_alias->create_toolbar = create_toolbar;
			}
			if (controls->create_context_menu_toggle) {
				b_alias->create_context_menu = create_context_menu;
			}
			/* FIXME: remove duplication */
			if (controls->pixmap_file_entry) {
				if (strcmp (pixmap_file, b_real->pixmap_file) != 0)
				{
					if (b_real->pixmap_data)
						g_free (b_real->pixmap_data);
					b_real->pixmap_data = 
						pixmap_data_new_from_file 
								  (pixmap_file);
				}
				if (b_real->pixmap_file) g_free (b_real->pixmap_file);
				b_real->pixmap_file = pixmap_file;
			}
			if (controls->notes_text) {
				if (b_real->notes) g_free (b_real->notes);
				b_real->notes = notes;
			}
			break;
		default:
			break;
		}

		b_real->time_modified = b_alias->time_modified = time (NULL);
		bookmarks_editor_set_times (controls, b_alias);
		bookmarks_editor_update_tree_item (controls, b_alias);
		bookmarks_update_alias (controls, b_real);
		controls->dirty = TRUE;
	}
}

static void
bookmarks_editor_apply (BookmarksEditorControls *controls)
{
	if (controls->dirty && controls->save_function != NULL)
	{
		controls->save_function();
	}
}

void 
bookmarks_editor_apply_clicked_cb (GtkButton *button,
				   BookmarksEditorControls *controls)
{
	bookmarks_editor_apply (controls);
}

void 
bookmarks_editor_accept_clicked_cb (GtkButton *button,
				    BookmarksEditorControls *controls)
{
	bookmarks_editor_apply (controls);
	bookmarks_editor_hide_dialog (controls);
}

gboolean
bookmarks_editor_delete_event_cb (GtkWidget *widget, GdkEvent *event,
				  BookmarksEditorControls *controls)
{
	bookmarks_editor_accept_clicked_cb (NULL, controls);
	return TRUE;
}

void 
bookmarks_editor_remove_clicked_cb (GtkButton *button,
				    BookmarksEditorControls *controls)
{
	GList *selection_list = NULL, *l;
	GtkCList *clist = GTK_CLIST(controls->ctree);
	GtkCTree *ctree = GTK_CTREE(controls->ctree);
	BookmarkItem *next_to_select = NULL;
	gboolean autobookmarks_deleted = FALSE;

	selection_list = bookmarks_get_selections_sorted(clist);

	if (controls->selection && !g_list_next(clist->selection) &&
	    controls->selection->parent) { 
		/* there was only one bookmark selected */
		GList *next_to_select_pos = g_list_next (
			g_list_find (controls->selection->parent->list,
				     controls->selection));
		if (next_to_select_pos)
			next_to_select = next_to_select_pos->data;
	}

	gtk_clist_freeze (clist);
	for (l = selection_list; l != NULL; l = g_list_next (l)) {
		BookmarkItem *b;

		b =  gtk_ctree_node_get_row_data(ctree, l->data);
		if (b->type == BM_AUTOBOOKMARKS)
			autobookmarks_deleted = TRUE;

		/* TODO: when deleting a bookmark which has aliases, we should wanr the 
		   user, or make one of the aliases the new real bookmark */
		if (b->alias && !b->alias_of) {
			BookmarkItem *alias;
			for (alias = b->alias; alias; alias = alias->alias) {
				if (alias == next_to_select) 
					next_to_select = NULL;
				gtk_ctree_remove_node (ctree, alias->tree_item);
			}
		}

		gtk_ctree_remove_node (ctree, b->tree_item);
		bookmarks_remove_recursively (b);
		controls->dirty = TRUE;
        }
	gtk_clist_thaw (clist);

	if (autobookmarks_deleted)
		autobookmarks_root = NULL;

	if (next_to_select)
		gtk_ctree_select (ctree, next_to_select->tree_item);

	g_list_free (selection_list);
}

void 
bookmarks_editor_move_up_clicked_cb (GtkButton *button,
				     BookmarksEditorControls *controls)
{
	GtkCList *clist;
	GList *selections_sorted, *selection;
	BookmarkItem *b;

	clist = GTK_CLIST(controls->ctree);
	
	selections_sorted = bookmarks_get_selections_sorted(clist);
	selection = selections_sorted;

	gtk_clist_freeze(clist);
	while (selection)
	{
		b = gtk_ctree_node_get_row_data(GTK_CTREE(controls->ctree),
						GTK_CTREE_NODE(selection->data));
		if (bookmarks_move_bookmark(controls, b, 0))
		{
			controls->dirty = TRUE;
			break;
		}
		controls->dirty = TRUE;
		selection = selection->next;
	}
	gtk_clist_thaw(clist);
	
	g_list_free(selections_sorted);
}

void 
bookmarks_editor_move_down_clicked_cb (GtkButton *button,
				       BookmarksEditorControls *controls) 
{
	GtkCList *clist;
	GList *selections_sorted, *selection;
	BookmarkItem *b;

	clist = GTK_CLIST(controls->ctree);
	
	selections_sorted = bookmarks_get_selections_sorted(clist);
	selection = selections_sorted = g_list_reverse(selections_sorted);

	gtk_clist_freeze(clist);
	while (selection)
	{
		b = gtk_ctree_node_get_row_data(GTK_CTREE(controls->ctree),
						GTK_CTREE_NODE(selection->data));
		if (bookmarks_move_bookmark(controls, b, 1))
		{
			controls->dirty = TRUE;
			break;
		}
		controls->dirty = TRUE;
		selection = selection->next;
	}
	gtk_clist_thaw(clist);
	
	g_list_free(selections_sorted);
}

void 
bookmarks_editor_fetch_icon_clicked_cb (GtkButton *button,
					BookmarksEditorControls *controls) 
{
	GtkCList *clist;
	GList *selections_sorted, *selection;
	BookmarkItem *b;

	clist = GTK_CLIST(controls->ctree);
	
	selections_sorted = bookmarks_get_selections_sorted(clist);
	selection = selections_sorted;

	while (selection != NULL)
	{
		b = gtk_ctree_node_get_row_data(GTK_CTREE(controls->ctree),
						GTK_CTREE_NODE(selection->data));
		fetch_icon_recursively (b, controls);
		selection = selection->next;
	}
	
	g_list_free(selections_sorted);
}

static
void fetch_icon_recursively (BookmarkItem *b, 
			     BookmarksEditorControls *controls)
{
	GList *l;

	switch (b->type)
	{
	case BM_CATEGORY:
	case BM_AUTOBOOKMARKS:
		for (l = b->list; l != NULL; l = g_list_next (l))
		{
			fetch_icon_recursively ((BookmarkItem *)(l->data),
						controls);
		}
		break;
		
	case BM_SITE:
		get_siteicon (NULL, b->url);
		controls->dirty = TRUE;
		break;

	case BM_SEPARATOR:
		break;
	}
}

void
compact_bookmarks_editor_edit_toggled_cb (GtkWidget *toggle, 
					  BookmarksEditorControls *controls) 
{
	if (controls->edit_frame) {
		gboolean visible = TRUE;
		if (GTK_IS_TOGGLE_BUTTON (toggle))
			visible = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON 
								(toggle));
		else if (GTK_IS_CHECK_MENU_ITEM (toggle))
			visible = GTK_CHECK_MENU_ITEM (toggle)->active;
		if (visible)
			gtk_widget_show (controls->edit_frame);
		else
			gtk_widget_hide (controls->edit_frame);
	}
}

void
bookmarks_editor_add_alias_cb (GtkButton *button,
			       BookmarksEditorControls *controls) 
{
	BookmarkItem *orig = controls->selection;
	BookmarkItem *new = bookmarks_new_alias (orig);
	BookmarkItem *near = controls->selection;
	/* if the original is a category, add the new boookmark before to avoid loops */
	GtkCListDragPos insert_pos = GTK_CLIST_DRAG_INTO;
	if (new->type != BM_SITE)
		insert_pos = GTK_CLIST_DRAG_BEFORE;
	if (!near) near = controls->root_bookmark;
	new->time_added = time (NULL);
	bookmarks_insert_bookmark (new, near, insert_pos);
	bookmarks_editor_place_tree_item (controls, new);
	controls->dirty = TRUE;
	gtk_clist_unselect_all (GTK_CLIST(controls->ctree));
	gtk_ctree_select (GTK_CTREE (controls->ctree), new->tree_item);
}

void 
bookmarks_editor_new_item_clicked_cb (GtkButton *button,
				      BookmarksEditorControls *controls) 
{
	BookmarkItem *new = bookmarks_new_bookmark 
		(BM_SITE, TRUE, _("New bookmark item"), 
		 _("where://do.you.want.to.go"), NULL, NULL, NULL);
	BookmarkItem *near = controls->selection;
	GtkCListDragPos insert_pos = GTK_CLIST_DRAG_INTO;

	new->time_added = time (NULL);

	if (!near) near = controls->root_bookmark;
	bookmarks_insert_bookmark (new, near, insert_pos);
	bookmarks_editor_place_tree_item (controls, new);
	controls->dirty = TRUE;
	gtk_clist_unselect_all (GTK_CLIST(controls->ctree));
	gtk_ctree_select (GTK_CTREE (controls->ctree), new->tree_item);

	/* Don't grab the focus if we're in the temp bookmarks window and the
	   edit frame isn't visible */
	if (controls->edit_frame)
	{
		if (!GTK_WIDGET_VISIBLE (controls->edit_frame))
			return;
	}
	gtk_widget_grab_focus (controls->name_entry);
}

void 
bookmarks_editor_new_category_clicked_cb (GtkButton *button,
					  BookmarksEditorControls *controls) 
{
	BookmarkItem *new = bookmarks_new_bookmark 
		(BM_CATEGORY, TRUE, _("New bookmark category"), NULL, NULL, NULL, NULL);
	BookmarkItem *near = controls->selection;
	GtkCListDragPos insert_pos = GTK_CLIST_DRAG_INTO;

	new->time_added = time (NULL);

	if (!near) near = controls->root_bookmark;
	bookmarks_insert_bookmark (new, near, insert_pos);
	bookmarks_editor_place_tree_item (controls, new);
	controls->dirty = TRUE;
	gtk_clist_unselect_all (GTK_CLIST(controls->ctree));
	gtk_ctree_select (GTK_CTREE (controls->ctree), new->tree_item);

	/* Don't grab the focus if we're in the temp bookmarks window and the
	   edit frame isn't visible */
	if (controls->edit_frame)
	{
		if (!GTK_WIDGET_VISIBLE (controls->edit_frame))
			return;
	}
	gtk_widget_grab_focus (controls->name_entry);
}

void 
bookmarks_editor_new_separator_clicked_cb (GtkButton *button,
					   BookmarksEditorControls *controls) 
{
	BookmarkItem *new = bookmarks_new_bookmark 
		(BM_SEPARATOR, TRUE, _("Separator"), NULL, NULL, NULL, NULL);
	BookmarkItem *near = controls->selection;
	GtkCListDragPos insert_pos = GTK_CLIST_DRAG_INTO;

	new->time_added = time (NULL);

	if (!near) near = controls->root_bookmark;
	bookmarks_insert_bookmark (new, near, insert_pos);
	bookmarks_editor_place_tree_item (controls, new);
	controls->dirty = TRUE;
	gtk_clist_unselect_all (GTK_CLIST(controls->ctree));
	gtk_ctree_select (GTK_CTREE (controls->ctree), new->tree_item);
}

/* TODO: create bookmarks_editor_new_alias_clicked_cb (after adding the button) */

void
compact_bookmarks_editor_go_clicked_cb (GtkButton *button, 
					BookmarksEditorControls *controls)
{
	GList *pos;
	gchar *url;

	if (!controls->selection || (controls->selection->type != BM_SITE))
		return;

	/* Check if the window that is selected is already (still) open */
	if (controls->window)
	{
		pos = g_list_find (all_windows, controls->window);
		if (!pos)
		{
			controls->window = NULL;
		}
	}

	/* create a new window if needed */
	if (controls->selection)
	{
		url = controls->selection->url;
		controls->selection->time_visited = time (NULL);
		if (!controls->window)
		{
			controls->window = 
				embed_create_from_url (NULL, url, TRUE)->parent_window;
		}
		else
		{
			window_load_url (controls->window, url);
		}
	}
}

void
compact_bookmarks_editor_go_forward_clicked_cb (GtkButton *button,
						BookmarksEditorControls *controls)
{
	if (controls->selection && controls->selection->parent) {
		GList *dest_pos = g_list_next (
			g_list_find (controls->selection->parent->list, 
				     controls->selection));
		if (dest_pos) {
			BookmarkItem *dest = dest_pos->data;
			if (dest) {
				gtk_ctree_unselect (GTK_CTREE (controls->ctree),
						controls->selection->tree_item);
				gtk_ctree_select (GTK_CTREE (controls->ctree),
						  dest->tree_item);
				compact_bookmarks_editor_go_clicked_cb (button, controls);
			}
		}
	}
}

void
compact_bookmarks_editor_go_back_clicked_cb (GtkButton *button,
					     BookmarksEditorControls *controls)
{
	if (controls->selection && controls->selection->parent) {
		GList *dest_pos = g_list_previous (
			g_list_find (controls->selection->parent->list, 
				     controls->selection));
		if (dest_pos) {
			BookmarkItem *dest = dest_pos->data;
			if (dest) {
				gtk_ctree_unselect (GTK_CTREE (controls->ctree),
						controls->selection->tree_item);
				gtk_ctree_select (GTK_CTREE (controls->ctree),
						  dest->tree_item);
				compact_bookmarks_editor_go_clicked_cb (button, controls);
			}
		}
	}
}

gboolean
bookmarks_editor_copy_button_press_cb (GtkWidget *widget, GdkEventButton *event,
				       BookmarksEditorControls *controls)
{
	if (event->button == 3) {
		bookmarks_editor_copy_clicked_cb (GTK_BUTTON (widget), controls);
	}
	return FALSE;
}

void
bookmarks_editor_copy_clicked_cb (GtkButton *button,
				  BookmarksEditorControls *controls)
{
	if (controls->selection) {
		GtkMenu *destination_menu;
		if (!bookmarks_editor)
			bookmarks_editor_show_dialog (
						NULL,
						STANDARD_BOOKMARKS_EDITOR);
		destination_menu = 
			bookmarks_create_copy_menu (bookmarks_editor, 
						    controls->selection);
		gnome_popup_menu_do_popup_modal
			(GTK_WIDGET (destination_menu), menu_position_under_widget, 
			 GTK_WIDGET (button), NULL, NULL);
		gtk_widget_destroy (GTK_WIDGET (destination_menu));
		bookmarks_save ();
	}
}

void 
bookmarks_editor_ctree_tree_move_cb (GtkCTree *ctree, GtkCTreeNode *node,
				     GtkCTreeNode *new_parent_node, 
				     GtkCTreeNode *new_sibling_node, 
				     BookmarksEditorControls *controls)
{
	BookmarkItem *source = gtk_ctree_node_get_row_data (ctree, node);
	BookmarkItem *new_parent = gtk_ctree_node_get_row_data (ctree, new_parent_node);
	BookmarkItem *new_sib = gtk_ctree_node_get_row_data (ctree, new_sibling_node);
	GList *new_sib_list_node;
	gint new_pos = 0;
	
	g_return_if_fail (source && source->parent);

	if (!new_parent)
		new_parent = controls->root_bookmark;
		
	source->parent->list = g_list_remove (source->parent->list, source);
	bookmarks_update_alias (controls, source->parent);
	source->parent = new_parent;
	
	new_sib_list_node = g_list_find (new_parent->list, new_sib);
	if (new_sib_list_node) {
		new_pos = g_list_position (new_parent->list, new_sib_list_node);
		new_parent->list = g_list_insert (new_parent->list, source, new_pos);
	} else
		new_parent->list = g_list_append (new_parent->list, source);

	bookmarks_update_alias (controls, source->parent);
	/* DEBUG only */
#if 0
	g_assert (g_list_find (source->parent->list, source));
	g_assert (g_list_position 
		  (source->parent->list, g_list_find (source->parent->list, source)) 
		  == new_pos);
#endif
	controls->dirty = TRUE;
}

void
bookmarks_editor_bookmark_copy_cb (GtkMenuItem *menuitem, BookmarkItem *dest)
{
	BookmarkItem *item = gtk_object_get_user_data (GTK_OBJECT (menuitem));
	BookmarkItem *new;
	g_assert (item != NULL);
	g_assert (dest->type == BM_CATEGORY);
	new = bookmarks_copy_bookmark (item);
	new->parent = dest;
	dest->list = g_list_append (dest->list, new);
	bookmarks_editor->dirty = TRUE;
	bookmarks_editor_place_tree_item (bookmarks_editor, new);
}

gboolean 
bookmarks_editor_ctree_button_press_event_cb (GtkWidget *widget,
					      GdkEventButton  *event, 
					      BookmarksEditorControls *controls)
{
	gint row, column;
	gtk_clist_get_selection_info (GTK_CLIST (widget), event->x, event->y,
				      &row, &column);
	controls->last_pressed = gtk_ctree_node_get_row_data 
		(GTK_CTREE (widget), gtk_ctree_node_nth (GTK_CTREE (widget), row));

	/* Double click goes to the link */
	if (event->type == GDK_2BUTTON_PRESS && 
	    controls->last_pressed != NULL)
	{
		/* make sure that this entry is selected */
		gtk_ctree_select (GTK_CTREE (controls->ctree),
				  controls->last_pressed->tree_item);
		compact_bookmarks_editor_go_clicked_cb(NULL, controls);
	}

	return FALSE;
}

gboolean
bookmarks_editor_ctree_key_press_event_cb (GtkWidget *widget,
					   GdkEventKey *event,
					   BookmarksEditorControls *controls)
{
	if (event->state == 0)
	{
		if (event->keyval == GDK_Delete ||
		    event->keyval == GDK_KP_Delete)
			bookmarks_editor_remove_clicked_cb(NULL, controls);
	}
	else if ( (event->state & GDK_LOCK_MASK ||
		   event->state & GDK_MOD2_MASK ||
		   event->state & GDK_MOD5_MASK) &&
		 !(event->state & GDK_SHIFT_MASK ||
		   event->state & GDK_MOD1_MASK  ||
		   event->state & GDK_CONTROL_MASK) )
	{
		if (event->keyval == GDK_Delete)
			bookmarks_editor_remove_clicked_cb(NULL, controls);
	}

	return FALSE;
}

void 
bookmarks_editor_tree_expand_cb (GtkCTree *ctree, GtkCTreeNode *node, 
				 BookmarksEditorControls *controls)
{
	BookmarkItem *b = gtk_ctree_node_get_row_data (ctree, node);
	if (b) {
		b->expanded = TRUE;
		controls->dirty = TRUE;
	}
}

void 
bookmarks_editor_tree_collapse_cb (GtkCTree *ctree, GtkCTreeNode *node, 
				   BookmarksEditorControls *controls)
{
	BookmarkItem *b = gtk_ctree_node_get_row_data (ctree, node);
	if (b) {
		b->expanded = FALSE;
		controls->dirty = TRUE;
	}
}

void 
bookmarks_editor_expand_all_cb (GtkButton *button,
				BookmarksEditorControls *controls) 
{
	GList *l;
	gtk_clist_freeze (GTK_CLIST (controls->ctree));
	for (l = controls->root_bookmark->list; l != NULL; l = l->next) {
		BookmarkItem *b = l->data;
		gtk_ctree_expand_recursive (GTK_CTREE (controls->ctree), 
					    b->tree_item);
	}
	gtk_clist_thaw (GTK_CLIST (controls->ctree));
}

void 
bookmarks_editor_collapse_all_clicked_cb (GtkButton *button,
					  BookmarksEditorControls *controls)
{
	GList *l;
	gtk_clist_freeze (GTK_CLIST (controls->ctree));
	for (l = controls->root_bookmark->list; l != NULL; l = l->next) {
		BookmarkItem *b = l->data;
		gtk_ctree_collapse_recursive (GTK_CTREE (controls->ctree), 
					      b->tree_item);
	}
	gtk_clist_thaw (GTK_CLIST (controls->ctree));
}

/**
 * bookmarks_editor_set_text: generic bookmarks editor text field controller
 */
static void
bookmarks_editor_set_text (GtkWidget *control, gchar *text, gboolean active)
{
	/* skip if no control in this editor */
	if (control == NULL)
		return;

	/* set the text if active, or nullify */
	gtk_entry_set_text (GTK_ENTRY (control), active && text ? text : "");

	/* set widget sensitivity */
	gtk_widget_set_sensitive  (GTK_WIDGET (control), active);
}

/**
 * bookmarks_editor_set_name: set name field in bookmarks editor
 */
static void
bookmarks_editor_set_name (BookmarksEditorControls *controls, 
			   BookmarkItem *bookmark, gboolean active)
{
	bookmarks_editor_set_text (controls->name_entry, 
				   bookmark ? bookmark->name : NULL, 
				   active);
}

/**
 * bookmarks_editor_set_url: set url field in bookmarks editor
 */
static void
bookmarks_editor_set_url (BookmarksEditorControls *controls, 
			  BookmarkItem *bookmark, gboolean active)
{
	bookmarks_editor_set_text (controls->url_entry, 
				   bookmark ? bookmark->url : NULL, 
				   active);
}

/**
 * bookmarks_editor_set_nick: set nick field in bookmarks editor
 */
static void
bookmarks_editor_set_nick (BookmarksEditorControls *controls, 
			   BookmarkItem *bookmark, gboolean active)
{
	bookmarks_editor_set_text (controls->nick_entry, 
				   bookmark ? bookmark->nick : NULL, 
				   active);
}

/**
 * bookmarks_editor_set_create_toolbar: set create toolbar toggle button
 * in bookmarks editor
 */
static void
bookmarks_editor_set_create_toolbar (BookmarksEditorControls *controls, 
				     BookmarkItem *bookmark, gboolean active)
{
	GtkWidget *control = controls->create_toolbar_toggle;

	/* check the control is present */
	if (control == NULL)
		return;

	/* set toggle button */
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (control),
				      active && bookmark &&
				      bookmark->create_toolbar);

	/* set widget sensitivity */
	gtk_widget_set_sensitive  (GTK_WIDGET (control), active);
}

/**
 * bookmarks_editor_set_create_context_menu: set create context menuu toggle
 * button in bookmarks editor
 */
static void
bookmarks_editor_set_create_context_menu (BookmarksEditorControls *controls, 
					  BookmarkItem *bookmark,
					  gboolean active)
{
	GtkWidget *control = controls->create_context_menu_toggle;

	/* check the control is present */
	if (control == NULL)
		return;

	/* set toggle button */
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (control),
				      active && bookmark &&
				      bookmark->create_context_menu);

	/* set widget sensitivity */
	gtk_widget_set_sensitive  (GTK_WIDGET (control), active);
}

/**
 * bookmarks_editor_set_notes: set notes field in bookmarks editor
 */
static void
bookmarks_editor_set_notes (BookmarksEditorControls *controls, 
			    BookmarkItem *bookmark, gboolean active)
{
	GtkWidget *control = controls->notes_text;

	/* check the control is present */
	if (control == NULL)
		return;

	/* set the text */
	gtk_editable_delete_text (GTK_EDITABLE (control), 0, -1);
	if (active && bookmark && bookmark->notes)
	{
		gtk_text_insert (GTK_TEXT (control), NULL, NULL, NULL,
				 bookmark->notes, strlen (bookmark->notes));
	}

	/* set widget sensitivity */
	gtk_widget_set_sensitive  (GTK_WIDGET (control), active);
}

/**
 * bookmarks_editor_set_pixmap: set pixmap field in bookmarks editor
 */
static void
bookmarks_editor_set_pixmap (BookmarksEditorControls *controls, 
			     BookmarkItem *bookmark, gboolean active)
{
	bookmarks_editor_set_text (controls->pixmap_file_entry, 
				   bookmark ? bookmark->pixmap_file : NULL, 
				   active);
}

static void 
bookmarks_editor_set_times (BookmarksEditorControls *controls, BookmarkItem *bookmark)
{
	gchar *text;
	if (controls->time_added_entry) {
		text = bookmark ? time_to_string (bookmark->time_added) : g_strdup ("");
		gtk_entry_set_text (GTK_ENTRY (controls->time_added_entry), text);
		g_free (text);
	}
	if (controls->time_modified_entry) {
		text = bookmark ? time_to_string (bookmark->time_modified) : g_strdup ("");
		gtk_entry_set_text (GTK_ENTRY (controls->time_modified_entry), text);
		g_free (text);
	}
	if (controls->time_visited_entry) {
		text = bookmark ? time_to_string (bookmark->time_visited) : g_strdup ("");
		gtk_entry_set_text (GTK_ENTRY (controls->time_visited_entry), text);
		g_free (text);
	}
}

void
bookmarks_editor_add_alias_root_cb (GtkButton *button,
			       BookmarksEditorControls *controls) 
{
	BookmarkItem *orig = controls->root_bookmark;
	BookmarkItem *new = bookmarks_new_alias (orig);
	BookmarkItem *near = controls->selection;
	GtkCListDragPos insert_pos = GTK_CLIST_DRAG_INTO;
	if (!near) near = controls->root_bookmark;
	new->time_added = time (NULL);
	bookmarks_insert_bookmark (new, near, insert_pos);
	bookmarks_editor_place_tree_item (controls, new);
	controls->dirty = TRUE;
	gtk_clist_unselect_all (GTK_CLIST(controls->ctree));
	gtk_ctree_select (GTK_CTREE (controls->ctree), new->tree_item);
}
