/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"

static GtkWidget *bookmarks_folder_context_menu_new (gboolean create_toolbar);
void bookmarks_import_mozilla_activate_cb (GtkMenuItem *menuitem,
					   GaleonWindow *window);
void bookmarks_import_other_activate_cb (GtkMenuItem *menuitem,
					 GaleonWindow *window);
void bookmarks_import_other_ok_cb (const gchar *file, gpointer user_data);
void bookmarks_export_other_ok_cb (const gchar *file, gpointer user_data);
void bookmarks_export_netscape_activate_cb (GtkMenuItem *menuitem, 
					    GaleonWindow *window);
void bookmarks_export_mozilla_activate_cb (GtkMenuItem *menuitem,
					   GaleonWindow *window);
void bookmarks_export_other_activate_cb (GtkMenuItem *menuitem,
					 GaleonWindow *window);
void bookmarks_export_xbel_activate_cb (GtkMenuItem *menuitem,
					 GaleonWindow *window);
void bookmarks_export_xbel_ok (const gchar *file, gpointer data);
void bookmarks_import_netscape_style(GaleonWindow * window, GString * fname);
void bookmark_export_netscape_message (GaleonWindow *window, gchar *fname);
void bookmark_export_netscape_do (gint reply, gchar *fname);


/**
 * Open bookmark in same window if it's a site and the user
 * clicked with button 1
 * Open the bookmark in a new window/tab if it's a site and the user 
 * clicked with button 2
 * Open all the bookmarks in the folder if user clicked with button 2
 * on a folder
 */
gboolean
bookmarks_button_release_event_cb (GtkWidget *widget, GdkEventButton *event,
				   BookmarkItem *bi)
{
	/* FIXME: ugh */
	GtkWidget *WMain = window_lookup_widget (widget, "WMain");
	GaleonWindow *window =  gtk_object_get_data(GTK_OBJECT (WMain), 
						    "GaleonWindow");
	gpointer add_item = gtk_object_get_data (GTK_OBJECT (widget),
						 "add_item");
	gpointer open_all_item = gtk_object_get_data (GTK_OBJECT (widget),
						      "open_all_item");
	gpointer create_toolbar_item = gtk_object_get_data 
						(GTK_OBJECT (widget),
						 "create_toolbar_item");
	GtkWidget *menu;
	static gboolean handled = FALSE;
	gboolean tabbed_mode = gnome_config_get_bool (CONF_APPEARANCE_TABBED);
	gint opts_location = gnome_config_get_int
			       ("/galeon/General/bm_extra_items_location=1");

	return_val_if_not_window (window, FALSE);

	/* Check to see if we're supposed to handle this event */
	/* These checks are here just to make sure tearoff menus get
	   activated correctly */
	if (handled)
	{
		handled = FALSE;
		return FALSE;
	}

	/* If bi is NULL, don't process the event */
	if (!bi)
	{
		handled = TRUE;
		return FALSE;
	}

	/* Get the submenu widget */
	if (GTK_IS_MENU(widget))
		menu = widget;
	else
		menu = widget->parent;

	if (bi->type == BM_SITE)
	{
		bi->time_visited = time (NULL);
		gtk_menu_shell_deactivate (GTK_MENU_SHELL(menu));
		if (strstr(bi->url, "%s"))
		{
			bookmarks_smart_bm_dialog (window, bi,
						   (event->button == 2));
		}
		else
		{
			if (event->button == 2)
			{
				if (event->state & GDK_SHIFT_MASK)
					tabbed_mode = !tabbed_mode;
				embed_create_from_url (window->active_embed,
						       bi->url,
						       !tabbed_mode);
			}
			else
			{
				window_load_url (window, bi->url);
			}
		}
	}
	else if (bi->type == BM_AUTOBOOKMARKS || bi->type == BM_CATEGORY)
	{
		/* The "Add bookmark ..." item was clicked */
		if (add_item)
		{
			gtk_menu_shell_deactivate (GTK_MENU_SHELL(menu));
			bookmarks_file_bookmark_cb (widget, bi);
			return TRUE;
		}

		/* The "Open all ..." item was clicked. The current
		   window/tab will be reused when opening the bookmarks */
		if (open_all_item)
		{
			gtk_menu_shell_deactivate (GTK_MENU_SHELL(menu));
			bookmarks_folder_open_all_items (window->active_embed,
							 bi, !tabbed_mode,
							 TRUE);
			return TRUE;
		}

		/* The "Create toolbar ..." item was clicked. */
		if (create_toolbar_item)
		{
			gtk_menu_shell_deactivate (GTK_MENU_SHELL(menu));
			bi->create_toolbar = !bi->create_toolbar;
			bookmarks_save ();
			return TRUE;
		}

		/* FIXME: I've had to remove the following behaviour
		 * because of an odd problem: when selecting a bookmarks
		 * folder, and then clicking the middle button *not* on
		 * folder or any item in it, it was still getting the
		 * callback and ending up here. This was causing people
		 * some major grief, so I've disabled it for now until
		 * someone can debug it properly. -- MattA 26/04/2001 */
#if 0
		/* A middle click is handled the same way when clicking on
		   the menuitem or directly on a bookmark folder. In tabbed
		   mode, a new window will be opened with new tabs for each
		   bookmark. In new window mode, a new window will be opened
		   for each bookmark.  */
		if ((event->button == 2))
		{
			gtk_menu_shell_deactivate (GTK_MENU_SHELL(menu));
			if (event->state & GDK_SHIFT_MASK)
				tabbed_mode = !tabbed_mode;
			bookmarks_folder_open_all_items (window->active_embed,
							 bi, !tabbed_mode,
							 FALSE);
		}
#endif

		/* Display the context menu, if necessary */
		if (event->button == 3 && opts_location == 2)
		{
			GtkWidget *popup = bookmarks_folder_context_menu_new 
						(bi->create_toolbar);
			gint item;

			item = gnome_popup_menu_do_popup_modal (popup, NULL,
								NULL, NULL,
								NULL);
			gtk_widget_destroy (popup);
			gtk_menu_shell_deactivate (GTK_MENU_SHELL(menu));
			switch (item)
			{
			case 0:
				bookmarks_file_bookmark_cb (widget, bi);
				break;
			case 1:
				bookmarks_folder_open_all_items 
						(window->active_embed, bi,
						 !tabbed_mode, TRUE);
				break;
			case 2:
				bi->create_toolbar = !bi->create_toolbar;
				bookmarks_save ();
				break;
			default:
				break;
			}
		}
	}
	return TRUE;
}

/**
 * handle "standard" activation of bookmarks (eg, with keyboard)
 */
void
bookmarks_activate_cb (GtkWidget *widget, BookmarkItem *bi)
{
	GtkWidget *WMain = window_lookup_widget (widget, "WMain");
	GaleonWindow *window =  gtk_object_get_data(GTK_OBJECT (WMain), 
						    "GaleonWindow");

	return_if_not_window (window);
	g_return_if_fail (bi != NULL);
	
	if (bi->type == BM_SITE)
	{
		bi->time_visited = time (NULL);

		if (strstr (bi->url, "%s"))
		{
			bookmarks_smart_bm_dialog (window, bi, FALSE);
		}
		else
		{
			window_load_url (window, bi->url);
		}
	}
	else if (bi->type == BM_AUTOBOOKMARKS || bi->type == BM_CATEGORY)
	{
		gpointer add_item = gtk_object_get_data (GTK_OBJECT (widget),
							 "add_item");
		gpointer open_all_item = gtk_object_get_data 
				(GTK_OBJECT (widget), "open_all_item");
		gpointer create_toolbar_item = gtk_object_get_data 
						    (GTK_OBJECT (widget),
						     "create_toolbar_item");
		GtkWidget *menu;
		gboolean tabbed_mode = gnome_config_get_bool 
				(CONF_APPEARANCE_TABBED);
		
		/* Get the submenu widget */
		if (GTK_IS_MENU(widget))
			menu = widget;
		else
			menu = widget->parent;


		/* The "Add bookmark ..." item was clicked */
		if (add_item)
		{
			gtk_menu_shell_deactivate (GTK_MENU_SHELL(menu));
			bookmarks_file_bookmark_cb (widget, bi);
			return;
		}

		/* The "Open all ..." item was clicked. The current
		   window/tab will be reused when opening the bookmarks */
		if (open_all_item)
		{
			gtk_menu_shell_deactivate (GTK_MENU_SHELL(menu));
			bookmarks_folder_open_all_items (window->active_embed,
							 bi, !tabbed_mode,
							 TRUE);
			return;
		}

		/* The "Create toolbar ..." item was clicked. */
		if (create_toolbar_item)
		{
			gtk_menu_shell_deactivate (GTK_MENU_SHELL(menu));
			bi->create_toolbar = !bi->create_toolbar;
			bookmarks_save ();
			return;
		}
	}
}

static GtkWidget *
bookmarks_folder_context_menu_new (gboolean create_toolbar)
{
	GnomeUIInfo folder_context_menu_uiinfo[] =
	{
		GNOMEUIINFO_ITEM_STOCK (N_("Add bookmark to this folder"),
					NULL, NULL,
					GNOME_STOCK_MENU_BLANK),
		GNOMEUIINFO_ITEM_STOCK (N_("Open all bookmarks in this folder"),
					NULL, NULL,
					GNOME_STOCK_MENU_BLANK),
		GNOMEUIINFO_TOGGLEITEM (N_("Create toolbar from this folder"),
					NULL, NULL,
					GNOME_STOCK_MENU_BLANK),
                GNOMEUIINFO_END
        };
	GtkWidget *context_menu;

	context_menu = gnome_popup_menu_new (folder_context_menu_uiinfo);
	
	gtk_check_menu_item_set_active 
		(GTK_CHECK_MENU_ITEM (folder_context_menu_uiinfo[2].widget),
		 create_toolbar);

	return (context_menu);
}

void
bookmarks_file_bookmark_cb (GtkWidget *widget, BookmarkItem *p)
{
	BookmarkItem *new;	
	GaleonEmbed *embed;
	GtkWidget *WMain;
	GaleonWindow *window;

	WMain = window_lookup_widget (widget , "WMain");
	window = gtk_object_get_data (GTK_OBJECT (WMain), "GaleonWindow");
	return_if_not_window (window);

	embed = window->active_embed;
	return_if_not_embed (embed);

	new = bookmarks_new_bookmark (BM_SITE, TRUE, embed->site_title, 
				      embed->site_location, NULL, NULL, NULL);
	new->time_added = time (NULL);

	p->list = g_list_append (p->list, new);
	new->parent = p;
	if (bookmarks_editor) bookmarks_editor->dirty = TRUE;

	if (gnome_config_get_int(CONF_GENERAL_BOOKMARK_TITLE) == 
	    BOOKMARK_PAGE_TITLE)
	{
		if (new->url && 
		    gnome_config_get_bool (CONF_GENERAL_FAVICONS_ENABLED))
			get_siteicon (embed, new->url);

		if (bookmarks_editor)
			bookmarks_editor_place_tree_item (bookmarks_editor,
							  new);
		bookmarks_save ();
	} else {
		gnome_request_dialog 
			(FALSE, _("Please enter the name of the bookmark"),
			 new->name, 100,
			 GTK_SIGNAL_FUNC (bookmarks_string_request_callback),
			 new, GTK_WINDOW (window->WMain));
	}
}

/** 
 * bookmarks_menuitem_drag_data_received_cb::
 */
void
bookmarks_menuitem_drag_data_received_cb (GtkWidget *widget, 
					  GdkDragContext *drag_context,
					  gint x, gint y,
					  GtkSelectionData *selection_data,
					  guint info, guint time,
					  GaleonWindow *window)
{
	BookmarkItem *b = NULL; 
	gchar *mem = selection_data->data;

	switch (info) {
	case DND_TARGET_GALEON_BOOKMARK:
		b = bookmarks_item_from_string (mem);
		break;
	case DND_TARGET_STRING:
	case DND_TARGET_NETSCAPE_URL:
	case DND_TARGET_GALEON_URL:
		b = bookmarks_new_bookmark (BM_SITE, TRUE, NULL, mem, 
					    NULL, NULL, NULL);
		break;
	default:
		g_warning ("Unknown DND type");
		break;
	}
	if (b) {
		/* FIXME */
		GtkMenu *destination_menu;
		if (!bookmarks_editor) bookmarks_editor_show_dialog (
						NULL,
						STANDARD_BOOKMARKS_EDITOR);
	       	destination_menu = 
			bookmarks_create_copy_menu (bookmarks_editor, b);
		gnome_popup_menu_do_popup_modal
			(GTK_WIDGET (destination_menu), NULL, 
			 NULL, NULL, NULL);
		gtk_widget_destroy (GTK_WIDGET (destination_menu));
		bookmarks_save ();
		bookmarks_free_bookmark (b);
	}
}

/* import given netscape-compatible bookmark filename with pass/fail dialog */
void
bookmarks_import_netscape_style (GaleonWindow * window, GString * fname)
{
	/* NB: used to use gnome_app_message here, but there's not a 
	 * good choice of window to associate the message with, so 
	 * forcing a dialog seems like a better idea -- MattA 30-12-2000 */
	if (netscape_import_bookmarks(fname))
	{
		gnome_ok_dialog (_("Importing bookmarks failed"));
	}
	else
	{
		gnome_ok_dialog (_("Bookmarks successfully imported/merged"));
	}
}

void
bookmarks_import_netscape_activate_cb (GtkMenuItem *menuitem,
				       GaleonWindow *window)
{
	GString *fname = g_string_new (NULL);

	g_string_assign (fname, g_get_home_dir ());
	g_string_append (fname, NETSCAPE_BOOKMARKS);
       	
	bookmarks_import_netscape_style (window, fname);

	g_string_free (fname, TRUE);
}

void
bookmarks_import_mozilla_activate_cb (GtkMenuItem *menuitem,
				      GaleonWindow *window)
{
	GString * fname = g_string_new (NULL);
	char *prefs_dir = get_mozilla_prefs_directory ();

	if (!prefs_dir) return;

	g_string_assign (fname, prefs_dir);
	g_string_append (fname, "/bookmarks.html");

	bookmarks_import_netscape_style (window, fname);

	g_string_free (fname, TRUE);
	g_free (prefs_dir);
}

void
bookmarks_import_other_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	misc_ask_file_name (_("Import from..."),
			    _("Import bookmarks from... (Netscape/Mozilla format)"), 
			    "bookmarks.html",
			    bookmarks_import_other_ok_cb, NULL, TRUE, FALSE, window);
}


void
bookmarks_import_other_ok_cb (const gchar *file, gpointer user_data)
{
	GString *temp;
	GaleonWindow *window = (GaleonWindow *) user_data;
	temp = g_string_new (file);
	bookmarks_import_netscape_style (window, temp);
	g_string_free (temp, TRUE);
}

void
bookmarks_export_netscape_activate_cb (GtkMenuItem *menuitem,
				       GaleonWindow *window)
{
	GString *fname = g_string_new (NULL);

	g_string_assign (fname, g_get_home_dir ());
	g_string_append (fname, NETSCAPE_BOOKMARKS);
       	
	bookmark_export_netscape_message (window, fname->str);

	g_string_free (fname, FALSE);
}

void
bookmarks_export_mozilla_activate_cb (GtkMenuItem *menuitem,
				      GaleonWindow *window)
{
	char *fname;
	char *prefs_dir = get_mozilla_prefs_directory();

	fname = g_strconcat (prefs_dir,"/bookmarks.html", NULL);
	g_print (fname);
	bookmark_export_netscape_message (window, fname);

	g_free (prefs_dir);
}

void
bookmarks_export_other_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	misc_ask_file_name (_("Export to..."),
			    _("Export bookmarks to... (Netscape/Mozilla format)"), 
			    "bookmarks.html",
			    bookmarks_export_other_ok_cb, NULL, FALSE, TRUE, window);
}

void
bookmarks_export_other_ok_cb (const gchar *file, gpointer user_data)
{
	GaleonWindow *window = (GaleonWindow *) user_data;
	bookmark_export_netscape_message (window, g_strdup (file));
}

void
bookmarks_export_xbel_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	misc_ask_file_name (_("Export in XBEL"), _("Export in XBEL as..."), 
			    "bookmarks.xbel",
			    bookmarks_export_xbel_ok, NULL, FALSE, TRUE, NULL);
}

void 
bookmarks_export_xbel_ok (const gchar *file, gpointer data)
{
	xbel_export_bookmarks (file, bookmarks_root);
}

void
bookmark_export_netscape_message (GaleonWindow *window, gchar *fname)
{
	gchar *question = g_strdup_printf (_("File %s will be overwritten.\n"
					     "If you have any bookmark in that file, "
					     "it will be lost.\n\n"
					     "Do you want to continue?"), fname);
	gnome_question_dialog_modal (question,
				     (GnomeReplyCallback) bookmark_export_netscape_do,
				     fname);
	g_free (question);
}

void 
bookmark_export_netscape_do (gint reply, gchar *fname)
{
	if (reply != GNOME_YES) return;
	if (netscape_export_bookmarks (fname, bookmarks_root))
		gnome_ok_dialog (_("Exporting bookmarks failed"));
	else 
		gnome_ok_dialog (_("Bookmarks successfully exported"));
	g_free (fname);
}
