/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"

static void bookmarks_create_toolbars (GaleonWindow *window, 
				       BookmarkItem *bi);
static void bookmarks_create_toolbar_from (GaleonWindow *window, 
					   BookmarkItem *bi);
static gint bookmarks_folder_show_menu (GtkWidget *menu);
gboolean
bookmarks_toolbar_item_button_release_event_cb (GtkWidget *item, 
						GdkEventButton *event,
						BookmarkDragItem *bookmark_info);
 
/* this magic number is used to hack around a problem with
 * gnome-libs (I think) which causes multiple toolbars with
 * the same number not to display properly -- MattA
 * This number will also be appended to toolbar names to make sure no two
 * toolbars have the same name.  This keeps the gnome layout routines from
 * getting confused when restoring toolbar positions  --Josh */
static gint magic_number;

/**
 * bookmarks_create_toolbars: create personal toolbars
 */
static void 
bookmarks_create_toolbars (GaleonWindow *window, BookmarkItem *bi)
{
	GList *l;

	/* skip if we're not a category */
	if ((bi->type != BM_CATEGORY) && (bi->type != BM_AUTOBOOKMARKS))
		return;
	
	/* if we're meant to create a toolbar... */
	if (bi->create_toolbar)	{
		/* do so */
		bookmarks_create_toolbar_from (window, bi);
	}

	/* now iterate to find any marked subfolders, unless this is an alias */
	if (! bi->alias_of)
		for (l = bi->list; l != NULL; l = l->next) {
			bookmarks_create_toolbars (window, l->data);
		}
}

/**
 * bookmarks_create_toolbar_from: create a toolbar from a folder
 */
static void 
bookmarks_create_toolbar_from (GaleonWindow *window, BookmarkItem *bi)
{
	GnomeDockItemBehavior props;
	gboolean toolbar_detachable;
	GtkToolbar *tb;
	GList *l;
	gchar *unique_name;

	/* make the toolbar */
	tb = GTK_TOOLBAR(gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL,
					  GTK_TOOLBAR_TEXT));

	/* add the toolbar it to the main window */
	/* this has to be done here for the relief to get set */
	toolbar_detachable = gnome_preferences_get_toolbar_detachable ();
	props = toolbar_detachable ? 0 : GNOME_DOCK_ITEM_BEH_LOCKED;
	props |= GNOME_DOCK_ITEM_BEH_EXCLUSIVE;

	/* make sure each toolbar has a unique name */
	unique_name = g_strdup_printf("%s%d", bi->name, magic_number);

	gnome_app_add_toolbar (GNOME_APP (window->WMain), tb, unique_name,
			       props, GNOME_DOCK_TOP, magic_number++, 0, 0);

	/* set border to match button toolbar */
	gtk_container_set_border_width (GTK_CONTAINER (tb), 2);

	/* override global setting */
	gtk_toolbar_set_space_style (tb, GTK_TOOLBAR_SPACE_EMPTY);
	gtk_toolbar_set_space_size (tb, 6);

	/* iterate over the contents of the folder */
	for (l = bi->list; l != NULL; l = l->next)
	{
		BookmarkItem *b = l->data;

		/* build the appropriate element */
		if (b->type != BM_SEPARATOR)
		{
			bookmarks_create_toolbar_from_bm(window, tb, b, -1);
		} 
		else 
		{
			gtk_toolbar_append_space(tb);
		}
	}

	/* show it */
	gtk_widget_show_all (GTK_WIDGET(tb));

	/* add it to the per-window list of toolbars */
	window->bookmarks_toolbars = 
		g_list_append (window->bookmarks_toolbars, tb);
}

/**
 * bookmarks_create_toolbar_from_bm:
 *
 * drag_info should not be freed
 *
 */
void
bookmarks_create_toolbar_from_bm (GaleonWindow *window, GtkToolbar *tb,
				  BookmarkItem *b, gint insert_pos)
{
	GtkWidget *button, *hbox, *pixmap, *entry;
	GtkReliefStyle relief;
	GtkWidget *dnd_dest;
	BookmarkDragItem *drag_info = g_malloc0 (sizeof(BookmarkDragItem));
	PixmapData *bm_icon;
	
	/* get the button relief setting */
	relief = gtk_toolbar_get_button_relief (tb);

	if (b->type == BM_CATEGORY || b->type == BM_AUTOBOOKMARKS)
	{
		pixmap = gtk_pixmap_new (folder_pixmap_data->pixmap,
					 folder_pixmap_data->mask);
	}
	else
	{
		bm_icon = bookmarks_get_siteicon (b->url);
		pixmap = gtk_pixmap_new (bm_icon->pixmap, bm_icon->mask);
	}

	/* build a box, and pack the pixmap and label inside */
	hbox = gtk_hbox_new (FALSE, 2);
	if (b->pixmap_data == NULL)
	{
		gchar *name = strip_uline_accel (b->name);
		gtk_box_pack_start (GTK_BOX (hbox), pixmap, FALSE, FALSE, 0);
		gtk_box_pack_start (GTK_BOX (hbox), gtk_label_new (name),
				    TRUE, TRUE, 0);
		g_free (name);
	}
	else
	{
		GtkWidget *image  = gtk_pixmap_new (b->pixmap_data->pixmap,
						    b->pixmap_data->mask);
		gtk_box_pack_start (GTK_BOX (hbox), GTK_WIDGET (image),
				    TRUE, FALSE, 0);
	}

	/* is this a normal (i.e. non-smart) bookmark */
	if (b->url == NULL || strstr(b->url, "%s") == NULL)
	{
		/* yes, build a button and pack it */
		dnd_dest = button = gtk_button_new ();
		gtk_container_add(GTK_CONTAINER(button), hbox);
		if (insert_pos == -1)
			gtk_toolbar_append_widget(tb, button, b->url, NULL);
		else
			gtk_toolbar_insert_widget(tb, button, b->url, NULL,
						  insert_pos);
		gtk_button_set_relief (GTK_BUTTON (button), relief);
		GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);
	}
	else
	{
		GtkWidget *button = gtk_button_new ();
		gint entry_width;

		dnd_dest = button;
		entry_width = gnome_config_get_int
			("/galeon/Appearance/smart_bm_entry_width=100");
		gtk_container_add (GTK_CONTAINER (button), 
				   GTK_WIDGET (hbox));
		gtk_button_set_relief (GTK_BUTTON (button), relief);
		GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);
		gtk_container_set_border_width (GTK_CONTAINER (hbox), 0);
		/* otherwise build an entry field and pack it */
		entry = gtk_entry_new ();
		gtk_widget_set_usize(entry, entry_width, -2);
		gtk_toolbar_append_widget (tb, button, 
					   _("Search for entered phrase"),
					   NULL);
		gtk_toolbar_append_widget (tb, entry, b->url, NULL);
		gtk_toolbar_append_space (tb);
		gtk_object_set_data (GTK_OBJECT (button), "entry", entry);
		gtk_object_set_data (GTK_OBJECT (entry), "GaleonWindow", 
				     window);
		gtk_signal_connect (GTK_OBJECT (entry), "activate",
				    bookmarks_nick_entry_activate, b);
	}

	/* dnd */
	drag_info->window = window;
	drag_info->bookmark = b;
	drag_info->toolbar = tb;
	gtk_signal_connect (GTK_OBJECT (dnd_dest), "drag_data_received",
		       GTK_SIGNAL_FUNC (bookmarks_toolbar_drag_data_received_cb),
		       drag_info);
	gtk_drag_dest_set (dnd_dest,
			   GTK_DEST_DEFAULT_ALL,
			   bookmarks_dnd_targets,
			   bookmarks_dnd_targets_num_items,
			   GDK_ACTION_COPY | GDK_ACTION_MOVE |
			   GDK_ACTION_LINK);
			   
	/* the click handler */
	gtk_signal_connect (GTK_OBJECT (dnd_dest), "button-release-event",
		GTK_SIGNAL_FUNC (bookmarks_toolbar_item_button_release_event_cb),
		drag_info);				
}

void
bookmarks_create_tb_global (void)
{
	GList *sl;
	gchar *layout;

	/* get the saved layout string */
	layout = gnome_config_get_string ("/galeon/Placement/Dock");

	for (sl = all_windows; sl != NULL; sl = sl->next) 
	{
		GaleonWindow *window = (GaleonWindow *) sl->data;
		GnomeApp *app = GNOME_APP (window->WMain);
		GList *lp;
		GnomeDockLayoutItem *item;

		/* Create new layout object, so gnome_app_toolbar will add the
		   toolbars to the layout instead of directly to the dock */
		app->layout = gnome_dock_layout_new();

		/* Create the menus and the toolbars for this window */
		bookmarks_create_tb (sl->data);

		/* Restore the saved layout */
		gnome_dock_layout_parse_string (app->layout, layout);

		/* Have to do this manually to get the ordering right */
		lp = app->layout->items;
		while (lp)
		{
			item = lp->data;

			if (item->placement == GNOME_DOCK_FLOATING)
			{
				gnome_dock_add_floating_item
					(GNOME_DOCK (app->dock),
					 item->item,
					 item->position.floating.x,
					 item->position.floating.y,
					 item->position.floating.orientation);
			}
			else
			{
				gnome_dock_add_item
					(GNOME_DOCK (app->dock),
					 item->item,
					 item->placement,
					 item->position.docked.band_num,
					 0,
					 item->position.docked.offset,
					 TRUE);
			}
			lp = lp->next;
		}

		/* Get rid of the layout object, it's no longer needed */
		gtk_object_unref (GTK_OBJECT (app->layout));
		app->layout = NULL;
	}
	g_free (layout);
}

void
bookmarks_remove_tb_global (void)
{
	GList *sl;
	GList *li;

	/* Destroys the toolbars */
	for (sl = all_windows; sl != NULL; sl = sl->next)
	{
		GaleonWindow *window = (GaleonWindow *) sl->data;
		GList *toolbars = window->bookmarks_toolbars;

		/* Destroy the dock band containing the toolbar */
		for (li = toolbars; li != NULL; li = li->next)
		{
			GtkWidget *toolbar = li->data;
			GtkWidget *dock_item, *dock_band;

			/* Probably overkill, but check to be safe */
			if (toolbar == NULL)
				continue;

			dock_item = toolbar->parent;
			if (dock_item == NULL)
			{
				gtk_widget_destroy (toolbar);
				continue;
			}

			dock_band = dock_item->parent;
			if (dock_band == NULL)
			{
				gtk_widget_destroy (dock_item);
				continue;
			}

			gtk_widget_destroy (dock_band);
		}
		g_list_free (toolbars);
		window->bookmarks_toolbars = NULL;
	}
}

/**
 * bookmarks_update_tb: destroy and recreate the boomarks toolbars
 */
void 
bookmarks_update_tb (void) 
{
	bookmarks_remove_tb_global ();
	bookmarks_create_tb_global ();
}

/**
 * bookmarks_tb_update_entries_global: search the bookmarks toolbars for
 *	smart bookmark entry widgets and sync their sizes with the
 *	preferences setting
 */
void
bookmarks_tb_update_entries_global (void)
{
	GList *wlist = all_windows;
	gint entry_width = gnome_config_get_int
				("/galeon/Appearance/smart_bm_entry_width=100");

	while (wlist)
	{
		GaleonWindow *window = (GaleonWindow *) wlist->data;
		GList *toolbars = window->bookmarks_toolbars;

		while (toolbars)
		{
			GtkToolbar *toolbar = GTK_TOOLBAR (toolbars->data);
			GList *children = toolbar->children;

			while (children)
			{
				GtkToolbarChild *child = children->data;

				if (GTK_IS_ENTRY (child->widget))
				{
					gtk_widget_set_usize (child->widget,
							      entry_width, -2);
				}
				children = children->next;
			}
			toolbars = toolbars->next;
		}
		wlist = wlist->next;
	}			
}

void 
bookmarks_create_tb (GaleonWindow *window)
{
	gboolean visible = GTK_CHECK_MENU_ITEM (window->view_toolbar)->active;

	magic_number = 99;
	bookmarks_create_toolbars (window, bookmarks_root);
	bookmarks_toolbar_set_visible (window, visible);
}
void 
bookmarks_toolbar_set_visible (GaleonWindow *window, gboolean visible)
{
	GList *l;
	for (l = window->bookmarks_toolbars; l != NULL; l = l->next) {
		if (GTK_IS_WIDGET (l->data)) {
			GtkWidget *w = NULL;
			if (GTK_IS_WIDGET (GTK_WIDGET (l->data)->parent)) {
				w = GTK_WIDGET (l->data)->parent;
				if (visible) 
					gtk_widget_show (w);
				else
					gtk_widget_hide (w);
			}
		}
	}
}

#define TOOLBAR_ITEM_CONTEXT_MENU_EDIT_POS 0
#define TOOLBAR_ITEM_CONTEXT_MENU_REMOVE_POS 1
#define TOOLBAR_ITEM_CONTEXT_MENU_HIDE_TOOLBAR_POS 3

/** 
 * bookmarks_toolbar_item_button_release_event_cb:
 * Handles clicks on bookmarks toolbar items.
 *
 * Button 1: Load bookmark in current window
 * Button 2: Load bookmark in new tab/window
 * Button 3: Show context menu
 */
gboolean
bookmarks_toolbar_item_button_release_event_cb (GtkWidget *item, 
						GdkEventButton *event,
						BookmarkDragItem *bookmark_info)
{
	static GnomeUIInfo toolbar_item_context_menu_uiinfo[] =
	{
		GNOMEUIINFO_ITEM_STOCK (N_("Edit..."), NULL, NULL,
				        GNOME_STOCK_MENU_PROP),
		GNOMEUIINFO_ITEM_STOCK (N_("Remove"), NULL, NULL,
				        GNOME_STOCK_MENU_CLOSE),
		GNOMEUIINFO_SEPARATOR,
		GNOMEUIINFO_ITEM_STOCK (N_("Hide toolbar"), NULL, NULL,
					GNOME_STOCK_MENU_BLANK),
		GNOMEUIINFO_END
	};
	GtkWidget *WMain = window_lookup_widget (GTK_WIDGET(item), "WMain");
	GaleonWindow *window = gtk_object_get_data(GTK_OBJECT(WMain),
						   "GaleonWindow");
	BookmarkItem *bi = bookmark_info->bookmark;	
	GtkWidget *entry;
	GtkWidget *popup;
	gint item_action;
	gboolean autobookmarks_deleted = FALSE;
	gchar *text;
	
	if (item->window != event->window)
		return FALSE;

	if (event->button == 1)
	{
		/* Load the bookmark or create the folder menu */
		if ((bi->type == BM_SITE) && !GTK_IS_EVENT_BOX(item))
		{
			if (strstr (bi->url, "%s") == NULL)
			{
				/* normal bookmark */
				window_load_url(window, bi->url);
			}
			else
			{
				/* smart bookmark, do search for phrase */
				entry = gtk_object_get_data (GTK_OBJECT (item),
							     "entry");
				text = gtk_entry_get_text (GTK_ENTRY (entry));
				if (text != NULL && strlen (text) > 0)
				{
					find_next (window->active_embed, text);
				}
			}
		}
		else if ((bi->type == BM_CATEGORY) || (bi->type == BM_AUTOBOOKMARKS))
		{
			GtkWidget *menu = gtk_menu_new();
			GtkTooltips *tooltips = gtk_tooltips_new ();

			bookmarks_create_menu_recursively
				(bi, GTK_MENU (menu), NULL, tooltips, 
				 TRUE, TRUE);

			/* attach "WMain" to the menu so it can be looked up */
			gtk_object_set_data (GTK_OBJECT (menu), "WMain", WMain);
			gtk_object_set_data (GTK_OBJECT (menu), "tooltips",
					     tooltips);
			gtk_object_set_data (GTK_OBJECT (menu), "widget", item);

			/* show the menu in a timeout function so GTK+ will
			   clean up any pointer grabs before displaying the
			   popup menu */
			gtk_timeout_add (50,
				(GtkFunction) bookmarks_folder_show_menu,
				menu);
		}
	}
	else if (event->button == 2)
	{
		gboolean tabbed_mode;

		tabbed_mode = gnome_config_get_bool (CONF_APPEARANCE_TABBED);
		if (event->state & GDK_SHIFT_MASK)
			tabbed_mode = !tabbed_mode;

		/* Load the bookmark in a new tab/window */
		if ((bi->type == BM_SITE) && !GTK_IS_EVENT_BOX(item))
		{
			embed_create_from_url (window->active_embed, bi->url,
					       !tabbed_mode);

		}
		else if (bi->type == BM_AUTOBOOKMARKS || bi->type == BM_CATEGORY)
		{
			bookmarks_folder_open_all_items (window->active_embed,
							 bi, !tabbed_mode,
							 FALSE);
		}
	}
	else if (event->button == 3)
	{
		popup = gnome_popup_menu_new(toolbar_item_context_menu_uiinfo);
		item_action = gnome_popup_menu_do_popup_modal(popup, NULL,
							      NULL, NULL, NULL);
		gtk_widget_destroy(popup);
		
		switch (item_action) {
		case TOOLBAR_ITEM_CONTEXT_MENU_EDIT_POS:
			bookmarks_editor_show_dialog (
						window, 
						STANDARD_BOOKMARKS_EDITOR);
			bookmarks_editor_goto_bookmark (bookmark_info->bookmark);
			break;
		case TOOLBAR_ITEM_CONTEXT_MENU_REMOVE_POS:
			if (bookmark_info->bookmark->type == BM_AUTOBOOKMARKS)
				autobookmarks_deleted = TRUE;
			if (bookmarks_editor)
			{
				bookmarks_editor_remove_tree_item
						       (bookmarks_editor,
							bookmark_info->bookmark);
			}
			bookmarks_remove_recursively (bookmark_info->bookmark);
			if (autobookmarks_deleted)
				autobookmarks_root = NULL;
			bookmarks_save();
			break;
		case TOOLBAR_ITEM_CONTEXT_MENU_HIDE_TOOLBAR_POS:
			bookmark_info->bookmark->parent->create_toolbar = FALSE;
			bookmarks_save();
			break;
		}
		return FALSE;
	}
	return TRUE;
}

static int
bookmarks_folder_show_menu (GtkWidget *menu)
{
	GtkTooltips *tooltips = gtk_object_get_data (GTK_OBJECT (menu),
						     "tooltips");
	GtkWidget *widget = gtk_object_get_data (GTK_OBJECT (menu), "widget");

	g_return_val_if_fail (menu != NULL, FALSE);

	gnome_popup_menu_do_popup_modal (menu, menu_position_under_widget,
					 widget, NULL, NULL);
	gtk_widget_destroy (menu);
	if (tooltips)
	{
		gtk_object_destroy (GTK_OBJECT (tooltips));
	}

	return FALSE;
}

/**
 * bookmarks_nick_entry_activate: called when the user hits return
 * on an entry field on the toolbar (i.e. one created by a bookmark
 * which has %s in the url)
 */
void 
bookmarks_nick_entry_activate (GtkEntry *entry, BookmarkItem *bookmark)
{
	gchar *text, *url, *translated_text;
	GaleonWindow *window;

	/* find the window */
	window = gtk_object_get_data (GTK_OBJECT (entry), "GaleonWindow");

	/* get the entry text: DON'T free the returned text! */
	text = gtk_entry_get_text (entry);

	/* translate non-alphanumeric characters into %{} format */
	translated_text = bookmarks_translate_string(text);

	/* get a completed url */
	url = bookmarks_substitute_argument (bookmark, translated_text);

	/* load it in this window */
	window_load_url (window, url);

	/* free all allocated strings */
	g_free (url);
	g_free (translated_text);
}

/** 
 * bookmarks_toolbar_drag_data_received_cb::
 */
void
bookmarks_toolbar_drag_data_received_cb (GtkWidget *widget, 
					 GdkDragContext *drag_context,
					 gint x, gint y,
					 GtkSelectionData *selection_data,
					 guint info, guint time,
					 BookmarkDragItem *drag_info)
{
	BookmarkItem *b = NULL; 
	gchar *mem = selection_data->data;
	
	if ( !drag_info ) return; /* should not happen */

	switch (info) {
	case DND_TARGET_GALEON_BOOKMARK:
		b = bookmarks_item_from_string (mem);
		break;
	case DND_TARGET_STRING:
	case DND_TARGET_NETSCAPE_URL:
	case DND_TARGET_GALEON_URL:
		b = bookmarks_new_bookmark (BM_SITE, TRUE, NULL, mem, NULL, NULL, NULL);
		break;
	default:
		g_warning ("Unknown DND type");
		break;
	}
	if (b) {
		extern BookmarksEditorControls *bookmarks_editor; /*FIXME!*/
		gchar *savefile;

		bookmarks_insert_bookmark (b, drag_info->bookmark,
					   GTK_CLIST_DRAG_INTO);
		if (bookmarks_editor)
			bookmarks_editor_place_tree_item(bookmarks_editor, b);

		if (!(drag_info->bookmark->type == BM_CATEGORY ||
		      drag_info->bookmark->type == BM_AUTOBOOKMARKS))
		{
			GList *children = drag_info->toolbar->children;
			GtkToolbarChild *child = NULL;
			gint insert_pos = 0;

			/* Find the position of the widget we're dragging onto
			   in the toolbar and insert after it */
			while (children)
			{
				insert_pos++;
				child = children->data;

				if (child->widget == widget)
					break;

				children = children->next;
			}

			/* Make sure the new bookmark is inserted after a
			   smart bookmark entry widget */
			if (strstr(drag_info->bookmark->url, "%s"))
				insert_pos++;

			bookmarks_create_toolbar_from_bm (drag_info->window,
							  drag_info->toolbar,
							  b, insert_pos);
		}
		gtk_widget_show_all(GTK_WIDGET(drag_info->toolbar));

		/* Save the bookmarks */
		savefile = g_strconcat (g_get_home_dir(), 
					"/.galeon/bookmarks.xml", NULL);
		bookmarks_save_as (bookmarks_root, savefile);
		g_free (savefile);

		/* update the menus */
		bookmarks_update_menu ();
	 }
}

