/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"

#define CONTEXT_MENU_ITEM(label, stock_id) \
 { GNOME_APP_UI_ITEM, label, NULL, NULL, NULL, NULL,  \
 GNOME_APP_PIXMAP_STOCK, stock_id, 0, 0, NULL}
#define CONTEXT_MENU_TOGGLEITEM(label, stock_id) \
 { GNOME_APP_UI_TOGGLEITEM, label, NULL, NULL, NULL, NULL, \
 GNOME_APP_PIXMAP_STOCK, stock_id, 0, 0, NULL}

/* FIXME like hell...  we have a type just like this in galeon_types.h!?!? */
typedef enum
{
	DOC_MENU,
	LINK_MENU,
	IMAGE_MENU,
	LINK_IMAGE_MENU,
	FRAMED_DOC_MENU
} LocalContextMenuType;

/* local functions */
static void context_show_framed_doc_menu (GaleonEmbed *embed);
static void context_show_img_menu (GaleonEmbed *embed, char *img);
static void context_show_link_menu (GaleonEmbed *embed, char *href);
static void context_show_link_img_menu (GaleonEmbed *embed, char *img, 
					char *href);
static void context_show_doc_menu (GaleonEmbed *embed);
static void set_back_forward_buttons(LocalContextMenuType menu_type,
				     GaleonEmbed *embed,
				     GtkWidget *popup);
static void context_menu_add_bookmarks (GtkMenu *menu, GaleonEmbed *embed);
static BookmarkItem *context_menu_create_fake_bookmarks_root (void);
static void context_menu_create_fake_bookmarks_root_recursively 
            (BookmarkItem *b, GList **l);
static void context_menu_bookmarks_cleanup (GtkObject *o, GaleonWindow *window);

/* This positions are used later to set the sensitiviness of the menu entries */
#define GO_BACK_POS    0
#define GO_FORWARD_POS 1

/* this is an ugly workaround for a bug in gettext */
/* strings in #define may not be put in po files */
/* even if they are marked as translatable */
/* check http://bugzilla.eazel.com/show_bug.cgi?id=5180 */
/* Thanks to the #i18n guys for giving me this url!! */
static char BACK_STR[]          = N_("Back");
static char FORWARD_STR[]       = N_("Forward");
static char RELOAD_STR[]        = N_("Reload");
static char OPEN_STR[]          = N_("Open");
static char OPEN_NEW_WIN_STR[]  = N_("Open in a new window");
static char OPEN_NEW_TAB_STR[]  = N_("Open in a new tab");
static char DOWNLOAD_FILE_STR[] = N_("Download file");
static char COPY_LINK_LOC_STR[] = N_("Copy link location");
static char ADD_TEMP_BM_STR[]   = N_("Add temporary bookmark");
static char OPEN_IMG_STR[]      = N_("Open image");
static char OPEN_IMG_WIN_STR[]  = N_("Open image in a new window");
static char OPEN_IMG_TAB_STR[]  = N_("Open image in a new tab");
static char SAVE_IMG_STR[]      = N_("Save image");
static char SAVE_IMG_AS_STR[]   = N_("Save image as...");
static char USE_IMG_AS_BG_STR[] = N_("Use image as background");
static char COPY_IMG_LOC_STR[]  = N_("Copy image location");
static char BLOCK_IMG_SRC_STR[] = N_("Block image source");

/* duplicated menus */
#define NAVIGATION_CONTEXT_MENU \
  CONTEXT_MENU_ITEM(BACK_STR,                     GNOME_STOCK_MENU_BACK),   \
  CONTEXT_MENU_ITEM(FORWARD_STR,                  GNOME_STOCK_MENU_FORWARD),\
  CONTEXT_MENU_ITEM(RELOAD_STR,                   GNOME_STOCK_MENU_REFRESH),
#define LINK_CONTEXT_MENU \
  CONTEXT_MENU_ITEM(OPEN_STR,                     GNOME_STOCK_MENU_OPEN),   \
  CONTEXT_MENU_ITEM(OPEN_NEW_WIN_STR,             GNOME_STOCK_MENU_NEW),    \
  CONTEXT_MENU_ITEM(OPEN_NEW_TAB_STR,             GNOME_STOCK_MENU_NEW),    \
  CONTEXT_MENU_ITEM(DOWNLOAD_FILE_STR,            GNOME_STOCK_MENU_SAVE),   \
  CONTEXT_MENU_ITEM(COPY_LINK_LOC_STR,            GNOME_STOCK_MENU_COPY),   \
  CONTEXT_MENU_ITEM(ADD_TEMP_BM_STR,              GNOME_STOCK_MENU_BLANK),
#define IMAGE_CONTEXT_MENU \
  CONTEXT_MENU_ITEM(OPEN_IMG_STR,                 GNOME_STOCK_MENU_OPEN),   \
  CONTEXT_MENU_ITEM(OPEN_IMG_WIN_STR,             GNOME_STOCK_MENU_NEW),    \
  CONTEXT_MENU_ITEM(OPEN_IMG_TAB_STR,             GNOME_STOCK_MENU_NEW),    \
  CONTEXT_MENU_ITEM(SAVE_IMG_STR,                 GNOME_STOCK_MENU_SAVE),   \
  CONTEXT_MENU_ITEM(SAVE_IMG_AS_STR,              GNOME_STOCK_MENU_SAVE_AS),\
  CONTEXT_MENU_ITEM(USE_IMG_AS_BG_STR,            GNOME_STOCK_MENU_BLANK),  \
  CONTEXT_MENU_ITEM(COPY_IMG_LOC_STR,             GNOME_STOCK_MENU_COPY),   \
  CONTEXT_MENU_ITEM(BLOCK_IMG_SRC_STR,            GNOME_STOCK_MENU_BLANK),


static GnomeUIInfo contextual_link_menu_uiinfo[] =
{
  NAVIGATION_CONTEXT_MENU
  GNOMEUIINFO_SEPARATOR,
  LINK_CONTEXT_MENU
  GNOMEUIINFO_END
};

static GnomeUIInfo contextual_img_menu_uiinfo[] =
{
  NAVIGATION_CONTEXT_MENU
  GNOMEUIINFO_SEPARATOR,
  IMAGE_CONTEXT_MENU
  GNOMEUIINFO_END
};

/* index of the "Full Screen View" item in contextual_doc_menu */
#define FULL_SCREEN_VIEW 5

static GnomeUIInfo contextual_framed_doc_menu_uiinfo[] =
{
  NAVIGATION_CONTEXT_MENU
  GNOMEUIINFO_SEPARATOR,
  CONTEXT_MENU_ITEM(N_("Save As..."),		     GNOME_STOCK_MENU_SAVE_AS),
  CONTEXT_MENU_TOGGLEITEM(N_("Full Screen View"),    GNOME_STOCK_MENU_BLANK),
  CONTEXT_MENU_ITEM(N_("Add temporary bookmark"),    GNOME_STOCK_MENU_BLANK),
  GNOMEUIINFO_SEPARATOR,
  CONTEXT_MENU_ITEM(N_("Open frame in window"),      GNOME_STOCK_MENU_OPEN),
  CONTEXT_MENU_ITEM(N_("Open frame in new window"),  GNOME_STOCK_MENU_OPEN),
  CONTEXT_MENU_ITEM(N_("Open frame in new tab"),     GNOME_STOCK_MENU_OPEN),
  CONTEXT_MENU_ITEM(N_("Reload frame"),              GNOME_STOCK_MENU_REFRESH),
  CONTEXT_MENU_ITEM(N_("Copy frame location"),       GNOME_STOCK_MENU_COPY),
  CONTEXT_MENU_ITEM(N_("Add temporary bookmark for frame"), GNOME_STOCK_MENU_BLANK),
  CONTEXT_MENU_ITEM(N_("View frame source"),         GNOME_STOCK_MENU_BLANK),
  GNOMEUIINFO_END
};

static GnomeUIInfo contextual_doc_menu_uiinfo[] =
{
  NAVIGATION_CONTEXT_MENU
  GNOMEUIINFO_SEPARATOR,
  CONTEXT_MENU_ITEM(N_("Save As..."),		     GNOME_STOCK_MENU_SAVE_AS),
  CONTEXT_MENU_TOGGLEITEM(N_("Full Screen View"),    GNOME_STOCK_MENU_BLANK),
  CONTEXT_MENU_ITEM(N_("Add temporary bookmark"),    GNOME_STOCK_MENU_BLANK),
  GNOMEUIINFO_END
};

static GnomeUIInfo contextual_link_img_menu_uiinfo[] =
{
  NAVIGATION_CONTEXT_MENU
  GNOMEUIINFO_SEPARATOR,
  CONTEXT_MENU_ITEM(N_("Open"),                      GNOME_STOCK_MENU_OPEN),
  CONTEXT_MENU_ITEM(N_("Open in a new window"),      GNOME_STOCK_MENU_NEW),
  CONTEXT_MENU_ITEM(N_("Open in a new tab"),         GNOME_STOCK_MENU_NEW),
  CONTEXT_MENU_ITEM(N_("Download file"),             GNOME_STOCK_MENU_SAVE),
  CONTEXT_MENU_ITEM(N_("Copy link location"),        GNOME_STOCK_MENU_COPY),
  CONTEXT_MENU_ITEM(N_("Add temporary bookmark"),    GNOME_STOCK_MENU_BLANK),
  GNOMEUIINFO_SEPARATOR,
  IMAGE_CONTEXT_MENU
  GNOMEUIINFO_END
};


/**
 * context_show_menu: show appropriate context menu based on the target type
 **/
void 
context_show_menu (GaleonEmbed *embed, WrapperContextInfo *info)
{
	if ((info->context & CONTEXT_IMAGE) &&
	    (info->context & CONTEXT_LINK))
	{
		context_show_link_img_menu(embed, info->img, info->link);
	}
	else if (info->context & CONTEXT_IMAGE)
	{
		context_show_img_menu(embed, info->img);
	}
	else if (info->context & CONTEXT_LINK)
	{
		context_show_link_menu(embed, info->link);
	}
	else if (info->context & CONTEXT_DOCUMENT)
	{
		if (info->framed_page)
			context_show_framed_doc_menu (embed);
		else
			context_show_doc_menu (embed);
	}
}

/**
 * context_show_link_menu: show a popup menu when a link is clicked
 **/
static void 
context_show_link_menu (GaleonEmbed *embed, char *href)
{
	int selected;
	GtkWidget *popup;

	popup = gnome_popup_menu_new (contextual_link_menu_uiinfo);
	set_back_forward_buttons (LINK_MENU, embed, popup);

	selected = gnome_popup_menu_do_popup_modal (popup, NULL, NULL,
							    NULL , NULL);
	/* Wait for the menu to be destroyed if a menu item is selected */
	if (selected != -1) 
		while (gtk_events_pending()) 
			gtk_main_iteration();

	switch (selected) {
	case 0: /* back */
		gtk_moz_embed_go_back (embed->mozEmbed);
		break;
	case 1: /* forward */
		gtk_moz_embed_go_forward (embed->mozEmbed);
		break;
	case 2: /* reload */
		embed_reload (embed);
		break;
	case 4: /* open */
		embed_load_url(embed, href);
		break;
	case 5: /* open in new window */
		embed_create_from_url (embed, href, TRUE);
		break;	
	case 6: /* open in new tab */
		embed_create_from_url (embed, href, FALSE);
		break;	
	case 7: /* save */
		save_url (href);
		break;
	case 8: /* copy link */
		copy_to_clipboard (href, embed);
		break;
	case 9: /* add temp bookmark */
		add_temp_bookmark (BM_SITE, NULL, href, NULL);
		break;
	default:
		break;
	}

	gtk_widget_destroy(popup);
}

/**
 * context_show_doc_menu: show a popup menu when a document is clicked
 **/
static void 
context_show_doc_menu (GaleonEmbed *embed)
{
    	int selected;
	GtkWidget *popup;
	GaleonWindow *window;
	gboolean tabbed_mode;

	return_if_not_embed (embed);
	window = embed->parent_window;
	return_if_not_window (window);

	tabbed_mode = gnome_config_get_bool (CONF_APPEARANCE_TABBED);
	popup = gnome_popup_menu_new (contextual_doc_menu_uiinfo);
	context_menu_add_bookmarks (GTK_MENU (popup), embed);

	/* sync the state of the fullscreen checkbox to the "View" menu */
	if (GTK_CHECK_MENU_ITEM(window->view_fullscreen)->active)
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(
			contextual_doc_menu_uiinfo[FULL_SCREEN_VIEW].widget),
			TRUE);
	else
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(
	                contextual_doc_menu_uiinfo[FULL_SCREEN_VIEW].widget),
		  	FALSE);

	set_back_forward_buttons(DOC_MENU, embed, popup);
	
	selected = gnome_popup_menu_do_popup_modal (popup, NULL, NULL,
							    NULL , NULL);
	/* Wait for the menu to be destroyed if a menu item is selected */
	if (selected != -1) 
		while (gtk_events_pending()) 
			gtk_main_iteration();

	switch (selected) {
	case 0: /* back */
		gtk_moz_embed_go_back (embed->mozEmbed);
		break;
	case 1: /* forward */
		gtk_moz_embed_go_forward (embed->mozEmbed);
		break;
	case 2: /* reload */
		embed_reload (embed);
		break;
	case 4: /* save */
		embed_save_document (embed, FALSE);
		break;
	case 5: /* toggle full screen view */
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(window->view_fullscreen),
				!(GTK_CHECK_MENU_ITEM(window->view_fullscreen))->active);
		break;
	case 6: /* Add temp bookmark */
		add_temp_bookmark (BM_SITE, embed->site_title, 
				   embed->site_location, NULL);
		break;
	default:
		break;
	}

	gtk_widget_destroy(popup);
}

/**
 * context_show_framed_doc_menu: show a popup menu when a framed document is clicked
 **/
static void 
context_show_framed_doc_menu (GaleonEmbed *embed)
{
    	int selected;
	GtkWidget *popup;
	gchar *url;
	GaleonWindow *window;
	gboolean tabbed_mode;

	return_if_not_embed (embed);
	window = embed->parent_window;
	return_if_not_window (window);

	tabbed_mode = gnome_config_get_bool (CONF_APPEARANCE_TABBED);
	popup = gnome_popup_menu_new (contextual_framed_doc_menu_uiinfo);
	context_menu_add_bookmarks (GTK_MENU (popup), embed);

	/* sync the state of the fullscreen checkbox to the "View" menu */
	if (GTK_CHECK_MENU_ITEM(window->view_fullscreen)->active)
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(
			contextual_framed_doc_menu_uiinfo[FULL_SCREEN_VIEW].widget),
			TRUE);
	else
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(
	                contextual_framed_doc_menu_uiinfo[FULL_SCREEN_VIEW].widget),
		  	FALSE);

	set_back_forward_buttons(FRAMED_DOC_MENU, embed, popup);
	
	selected = gnome_popup_menu_do_popup_modal (popup, NULL, NULL,
							    NULL , NULL);
	/* Wait for the menu to be destroyed if a menu item is selected */
	if (selected != -1) 
		while (gtk_events_pending()) 
			gtk_main_iteration();

	switch (selected) {
	case 0: /* back */
		gtk_moz_embed_go_back (embed->mozEmbed);
		break;
	case 1: /* forward */
		gtk_moz_embed_go_forward (embed->mozEmbed);
		break;
	case 2: /* reload */
		embed_reload (embed);
		break;
	case 4: /* save */
		embed_save_document (embed, FALSE);
		break;
	case 5: /* toggle full screen view */
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(window->view_fullscreen),
				!(GTK_CHECK_MENU_ITEM(window->view_fullscreen))->active);
		break;
	case 6: /* Add temp bookmark */
		add_temp_bookmark (BM_SITE, embed->site_title, 
				   embed->site_location, NULL);
		break;
	case 8: /* open frame */
		embed_open_frame (embed, TRUE, FALSE);
		break;
	case 9: /* open frame in new window */
		embed_open_frame (embed, FALSE, TRUE);
		break;
	case 10: /* open frame in new tab */
		embed_open_frame (embed, FALSE, FALSE);
		break;
	case 11: /* reload frame */
		mozilla_reload (embed);
		break;
	case 12: /* copy frame location */
		copy_to_clipboard (mozilla_get_document_url (embed), embed);
		break;
	case 13: /* add temp bookmark for frame */
		url = mozilla_get_document_url (embed);
		if (url)
		{
			add_temp_bookmark (BM_SITE, embed->site_title, url, NULL);
			g_free (url);
		}
		break;
	case 14: /* view source */
		embed_view_source (embed, FALSE, !tabbed_mode);
		break;
	default:
		break;
	}

	gtk_widget_destroy(popup);
}

/**
 * context_show_limg_menu: show a popup menu when an image is clicked
 **/
static void 
context_show_img_menu (GaleonEmbed *embed, char *img)
{
	int selected;
	GtkWidget *popup;

	popup = gnome_popup_menu_new (contextual_img_menu_uiinfo);
	set_back_forward_buttons(IMAGE_MENU, embed, popup);

	selected = gnome_popup_menu_do_popup_modal (popup, NULL, NULL,
						    NULL , NULL);
	/* Wait for the menu to be destroyed if a menu item is selected */
	if (selected != -1) 
		while (gtk_events_pending()) 
			gtk_main_iteration();

	switch (selected) {
	case 0: /* back */
		gtk_moz_embed_go_back (embed->mozEmbed);
		break;
	case 1: /* forward */
		gtk_moz_embed_go_forward (embed->mozEmbed);
		break;
	case 2: /* reload */
		embed_reload (embed);
		break;
	case 4: /* open image */
		embed_load_url(embed, img);
		break;
	case 5: /* open image in new window */
		embed_create_from_url (embed, img, TRUE);
		break;	
	case 6: /* open image in new tab */
		embed_create_from_url (embed, img, FALSE);
		break;	
	case 7: /* save image */
		embed_save_image_default (embed, img);
		break;
	case 8: /* save image as */
		embed_save_image (embed, img);
		break;
	case 9: /* set image as background */
		embed_set_image_as_background (embed, img);
		break;
	case 10: /* copy image location */
		copy_to_clipboard (g_strdup (img), embed);
		break;
	case 11:	/* block image source */
		mozilla_block_url( img );
		break;
	default:
		break;
	}

	gtk_widget_destroy(popup);
}

/**
 * context_show_link_menu: show a popup menu when a linked image is clicked
 **/
static void 
context_show_link_img_menu (GaleonEmbed *embed, char *img, char *href)
{
	int selected;
	GtkWidget *popup;

	popup = gnome_popup_menu_new (contextual_link_img_menu_uiinfo);
	set_back_forward_buttons(LINK_IMAGE_MENU, embed, popup);

	selected = gnome_popup_menu_do_popup_modal (popup, NULL, NULL,
							    NULL , NULL);
	/* Wait for the menu to be destroyed if a menu item is selected */
	if (selected != -1) 
		while (gtk_events_pending()) 
			gtk_main_iteration();

	switch (selected) {
	case 0: /* back */
		gtk_moz_embed_go_back (embed->mozEmbed);
		break;
	case 1: /* forward */
		gtk_moz_embed_go_forward (embed->mozEmbed);
		break;
	case 2: /* reload */
		embed_reload (embed);
		break;

	case 4: /* open */
		embed_load_url(embed, href);
		break;
	case 5: /* open in new window */
		embed_create_from_url (embed, href, TRUE);
		break;	
	case 6: /* open in new tab */
		embed_create_from_url (embed, href, FALSE);
		break;	
	case 7: /* save */
		save_url (href);
		break;
	case 8: /* copy link */
		copy_to_clipboard (href, embed);
		break;
	case 9: /* add temp bookmark */
		add_temp_bookmark (BM_SITE, NULL, href, NULL);
		break;
		
	case 11: /* open image */
		embed_load_url(embed, img);
		break;
	case 12: /* open image in new window */
		embed_create_from_url (embed, img, TRUE);
		break;	
	case 13: /* open image in new tab */
		embed_create_from_url (embed, img, FALSE);
		break;	
	case 14: /* save image */
		embed_save_image_default (embed, img);
		break;
	case 15: /* save image as... */
		embed_save_image (embed, img);
		break;
	case 16: /* set image as background */
		embed_set_image_as_background (embed, img);
		break;
	case 17: /* copy image location */
		copy_to_clipboard (img, embed);
		break;
	case 18:/* block image source */
		mozilla_block_url( img );
		break;

	default:
		break;
	}

	gtk_widget_destroy(popup);
}

/**
 * context_show_bookmark_menu: show a bookmarks popup menu 
 **/
void
context_show_bookmark_menu (GaleonEmbed *embed) 
{
	GtkWidget *menu = gtk_menu_new();
	GtkTooltips *tips = gtk_tooltips_new ();
	GaleonWindow *window;

	return_if_not_embed (embed);
	window = embed->parent_window;
	return_if_not_window (window);

	bookmarks_create_menu_recursively (bookmarks_root, GTK_MENU (menu),
					   NULL, tips, TRUE, TRUE);
	/* warning: ugly code */
	gtk_object_set_data (GTK_OBJECT (menu), "WMain", (window->WMain)); 
	gtk_object_set_data (GTK_OBJECT (window->WMain), "bm_context_menu",
			     menu);
	gnome_popup_menu_do_popup_modal (menu, NULL, NULL,
					 NULL , NULL);

	gtk_object_remove_data (GTK_OBJECT (window->WMain), "bm_context_menu");
	gtk_widget_destroy (menu);
	gtk_object_destroy (GTK_OBJECT (tips));
}

/**
 * context_menu_add_bookmarks: add the bookmarks that have the 
 * create_context_menu flag to a menu
 */
static void
context_menu_add_bookmarks (GtkMenu *menu, GaleonEmbed *embed)
{
	GtkTooltips *tips = gtk_tooltips_new ();
	GaleonWindow *window;
	BookmarkItem *fake_bookmarks_root = 
		context_menu_create_fake_bookmarks_root ();

	return_if_not_embed (embed);
	window = embed->parent_window;
	return_if_not_window (window);

	gtk_menu_append (menu, gtk_menu_item_new ());

	bookmarks_create_menu_recursively (fake_bookmarks_root, 
					   menu, NULL, tips, FALSE, TRUE);
	
	bookmarks_free_bookmark (fake_bookmarks_root);
	
	/* warning: ugly code */
	gtk_object_set_data (GTK_OBJECT (menu), "WMain", (window->WMain)); 
	gtk_object_set_data (GTK_OBJECT (window->WMain), "bm_context_menu",
			     menu);

	gtk_signal_connect_object (GTK_OBJECT (menu), "destroy", 
				   gtk_object_destroy, GTK_OBJECT (tips));
	gtk_signal_connect (GTK_OBJECT (menu), "destroy", 
			    context_menu_bookmarks_cleanup, window);
}

/**
 * Do some cleanups before destroying the context menu
 */
static void
context_menu_bookmarks_cleanup (GtkObject *o, GaleonWindow *window)
{
	gtk_object_remove_data (GTK_OBJECT (window->WMain), "bm_context_menu");
}

/**
 * Create a fake bookmark category that haves all the bookmarks that need to be
 * addeed to context menus
 */
static BookmarkItem *
context_menu_create_fake_bookmarks_root (void)
{
	BookmarkItem *r = bookmarks_new_bookmark 
		(BM_CATEGORY, TRUE, "fake bookmark", NULL, NULL, NULL, NULL);
	context_menu_create_fake_bookmarks_root_recursively 
		(bookmarks_root, &r->list);
	return r;
}

static void
context_menu_create_fake_bookmarks_root_recursively (BookmarkItem *b, 
						     GList **l)
{
	GList *li;
	if (b->create_context_menu) 
		*l = g_list_append (*l, b);
	if (!b->alias_of)
		for (li = b->list; li; li = li->next)
			context_menu_create_fake_bookmarks_root_recursively
				(li->data, l);
}


/**
 * set_back_forward_buttons: set the sensitivity of the back/forward context
 *   menu buttons and attach popup menus, if required
 **/
static void
set_back_forward_buttons (LocalContextMenuType menu_type, GaleonEmbed *embed, 
			  GtkWidget *popup)
{
	GnomeUIInfo *context_menu = NULL;
	gboolean can_go_back, can_go_forward;
	GtkMenu *menu_back, *menu_forward;

	switch (menu_type)
	{
		case DOC_MENU:
			context_menu = contextual_doc_menu_uiinfo;
			break;
		case LINK_MENU:
			context_menu = contextual_link_menu_uiinfo;
			break;
		case IMAGE_MENU:
			context_menu = contextual_img_menu_uiinfo;
			break;
		case LINK_IMAGE_MENU:
			context_menu = contextual_link_img_menu_uiinfo;
			break;
		case FRAMED_DOC_MENU:
			context_menu = contextual_framed_doc_menu_uiinfo;
			break;
		default:
			break;
	}

	g_return_if_fail(context_menu != NULL);

	/* query mozilla */
	can_go_back = gtk_moz_embed_can_go_back(embed->mozEmbed);
	can_go_forward = gtk_moz_embed_can_go_forward(embed->mozEmbed);

	/* set widget status appropriately */
	gtk_widget_set_sensitive(context_menu[GO_BACK_POS].widget,
				 can_go_back);
	gtk_widget_set_sensitive(context_menu[GO_FORWARD_POS].widget,
		 		 can_go_forward);
	
	if (gnome_config_get_bool("/galeon/Mouse/history_popup=FALSE"))
	{
		if (can_go_back)
		{
			menu_back = create_back_menu(embed);
			gtk_signal_connect(
				GTK_OBJECT(menu_back),
				"selection-done",
				(GtkSignalFunc) history_popup_selection_done_cb,
				popup);
			gnome_popup_menu_attach(GTK_WIDGET(menu_back),
						context_menu[GO_BACK_POS].widget,
						NULL);
		}      
		if (can_go_forward) {
			menu_forward = create_forward_menu(embed);
			gtk_signal_connect(
				GTK_OBJECT(menu_forward),
				"selection-done",
				(GtkSignalFunc)history_popup_selection_done_cb,
				popup);
			gnome_popup_menu_attach(
					GTK_WIDGET(menu_forward),
					context_menu[GO_FORWARD_POS].widget,
					NULL);
		}
	}
}
