/*
 *  $Id: cookies.c,v 1.13 2001/04/26 13:17:32 yaneti Exp $
 *
 *  Original Cookie Manager code
 *  Copyright (C) 2001 Daniel Stodden <stodden@in.tum.de>
 *  
 *  Updates for new nsICookieManager interface
 *  Password Handling
 *  General Cleanup
 *  Copyright (C) 2001 Philip Langdale <philipl@mail.utexas.edu>
 *
 *  GTK+based persistent data manager ( cookies, passwords, image blocks )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"

static GtkWidget *dialog;
static GtkWidget *stored_name;
static GtkWidget *stored_value; 
static GtkWidget *stored_host;
static GtkWidget *stored_path;
static GtkWidget *stored_secure;
static GtkWidget *stored_expire;
static GtkWidget *stored_block;

#define foreach_list_elem(elem, list) \
	for (elem = list; elem != NULL; elem = g_list_next(elem))

typedef struct {
	GList		*keep, *remove;
	GtkWidget	*clist;
	GtkWidget	*remove_button, *remove_all_button;
} CookieView;

CookieView *stored, *permissions, *images;
#ifdef CVS_SOURCE
CookieView *passwords, *rejects;
#endif

/* local function prototypes */
static void cookies_clear_stored_fields (void);

static void
view_select_cb (GtkCList *clist, gint row, gint column,
		GdkEventButton *event, gpointer data)
{
	GList *entry = gtk_clist_get_row_data (clist, row);
	CookieBase *c;

	if (entry)
	{
		c = (CookieBase*)entry->data;
		c->remove = TRUE;
	}
}

static void
view_unselect_cb (GtkCList *clist, gint row, gint column,
		  GdkEventButton *event, gpointer data)
{
	GList *entry = gtk_clist_get_row_data (clist, row);
	CookieBase *c = (CookieBase*)entry->data;

	c->remove = FALSE;
}

static void
view_remove_cb (GtkWidget *widget, gpointer data)
{
	CookieView *view = data;
	GList *entry, *next, *remove = NULL;

	for (entry = view->keep; entry != NULL; entry = next)
	{
		CookieBase *b = (CookieBase*)entry->data;
		gint rownum;

		next = g_list_next (entry);

		if (b->remove == FALSE) continue;

		rownum = gtk_clist_find_row_from_data (GTK_CLIST(view->clist),
						       entry);

		remove = g_list_append (remove, GINT_TO_POINTER(rownum));

		view->keep = g_list_remove_link (view->keep, entry);
		view->remove = g_list_concat (view->remove, entry);
		
		gnome_property_box_changed (GNOME_PROPERTY_BOX(dialog));
	}

	cookies_clear_stored_fields ();

	foreach_list_elem (entry, remove)
		gtk_clist_remove (GTK_CLIST(view->clist),
				  GPOINTER_TO_INT(entry->data));
	g_list_free (remove);
}

static void
view_remove_all_cb (GtkWidget *widget, gpointer data)
{
	CookieView *view = data;

	view->remove = g_list_concat (view->remove, view->keep);
	view->keep = NULL;

	gtk_clist_clear (GTK_CLIST(view->clist));

	gnome_property_box_changed (GNOME_PROPERTY_BOX(dialog));

	cookies_clear_stored_fields ();
}

static void
cookies_clear_stored_fields (void)
{
	gtk_editable_delete_text (GTK_EDITABLE (stored_name),   0, -1);
	gtk_editable_delete_text (GTK_EDITABLE (stored_value),  0, -1);
	gtk_editable_delete_text (GTK_EDITABLE (stored_host),   0, -1);
	gtk_editable_delete_text (GTK_EDITABLE (stored_path),   0, -1);
	gtk_editable_delete_text (GTK_EDITABLE (stored_secure), 0, -1);
	gtk_editable_delete_text (GTK_EDITABLE (stored_expire), 0, -1);
}

static gboolean
view_key_press_cb (GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	if (event->state == 0)
 	{
 		if (event->keyval == GDK_Delete ||
 		    event->keyval == GDK_KP_Delete)
 			view_remove_cb (NULL, data);
 	}
	else if ((event->state & GDK_LOCK_MASK ||
		  event->state & GDK_MOD2_MASK ||
		  event->state & GDK_MOD5_MASK) &&
		 !(event->state & GDK_SHIFT_MASK ||
		   event->state & GDK_MOD1_MASK  ||
		   event->state & GDK_CONTROL_MASK))
 	{
 		if (event->keyval == GDK_Delete)
 			view_remove_cb (NULL, data);
 	}
	
 	return FALSE;
}

static void
cookie_view_init (CookieView *view)
{
	gtk_signal_connect (GTK_OBJECT(view->clist), "select-row",
			    view_select_cb, view);
	gtk_signal_connect (GTK_OBJECT(view->clist), "unselect-row",
			    view_unselect_cb, view);
	gtk_signal_connect (GTK_OBJECT(view->remove_button), "clicked",
			    GTK_SIGNAL_FUNC(view_remove_cb), view);
	gtk_signal_connect (GTK_OBJECT(view->remove_all_button), "clicked",
			    GTK_SIGNAL_FUNC(view_remove_all_cb), view);
	gtk_signal_connect (GTK_OBJECT(view->clist), "key_press_event",
			    GTK_SIGNAL_FUNC(view_key_press_cb), view);
}

static void
stored_select_cb (GtkCList *clist, gint row, gint column,
		  GdkEventButton *event, gpointer data)
{
	GList *entry = gtk_clist_get_row_data (clist, row);
	Cookie *c; 

	if (entry)
	{
		c = (Cookie*)entry->data;

		gtk_entry_set_text (GTK_ENTRY(stored_name), c->name);
		gtk_entry_set_text (GTK_ENTRY(stored_value), c->value);
		gtk_entry_set_text (GTK_ENTRY(stored_host), c->base.domain);
		gtk_entry_set_text (GTK_ENTRY(stored_path), c->path);
		gtk_entry_set_text (GTK_ENTRY(stored_secure), c->secure);
		gtk_entry_set_text (GTK_ENTRY(stored_expire), c->expire);
	}
}

static void
init_cookie_clist (CookieView *view, GList *list)
{
	GList *entry;

	gtk_clist_freeze (GTK_CLIST(view->clist));
	gtk_clist_clear (GTK_CLIST(view->clist));

	view->keep = list;

	/* 
	   set selection mode to single, to prevent
	   select/unselect-row emission without valid row data
	*/
	gtk_clist_set_selection_mode (GTK_CLIST(view->clist), 
				      GTK_SELECTION_SINGLE);

	foreach_list_elem (entry, view->keep) {
		Cookie *c = (Cookie*)entry->data;
		gint rownum;
		gchar *row[2];

		c->base.remove = FALSE;

		row[0] = c->base.domain;
		row[1] = c->name;

		rownum = gtk_clist_append (GTK_CLIST(view->clist), row);
		gtk_clist_set_row_data (GTK_CLIST(view->clist), 
					rownum, entry);
	}

	gtk_clist_set_selection_mode(GTK_CLIST(view->clist), 
				     GTK_SELECTION_BROWSE);

	gtk_clist_thaw (GTK_CLIST(view->clist));
}
		
static void
init_server_clist (CookieView *view, GList *list)
{
	GList *entry;

	gtk_clist_freeze (GTK_CLIST(view->clist));
	gtk_clist_clear (GTK_CLIST(view->clist));

	view->keep = list;

	foreach_list_elem (entry, list) {
		CookieBase *b = (CookieBase*)entry->data;
		gint rownum;
		gchar *row[2];
		gchar *status;

		b->remove = FALSE;
		
		row[0] = b->domain;
#ifndef CVS_SOURCE
		if (!strcmp(b->type, "-"))
			status = N_ ("Blocked");
		else if (!strcmp(b->type, "+"))
			status = N_ ("Allowed");
		else
#endif		
			status = b->type;

		row[1] = status;

		rownum = gtk_clist_append (GTK_CLIST(view->clist), row);
		gtk_clist_set_row_data (GTK_CLIST(view->clist), 
					rownum, entry);
	}

	gtk_clist_thaw (GTK_CLIST(view->clist));
}


#ifdef CVS_SOURCE
static void
init_password_clist (CookieView *view, GList *list)
{
	GList *entry;

	gtk_clist_freeze (GTK_CLIST(view->clist));
	gtk_clist_clear (GTK_CLIST(view->clist));

	view->keep = list;

	foreach_list_elem (entry, list) {
		Password *p = (Password*)entry->data;
		gint rownum;
		gchar *row[2];

		p->remove = FALSE;
		
		row[0] = p->host;
		row[1] = p->username;

		rownum = gtk_clist_append (GTK_CLIST(view->clist), row);
		gtk_clist_set_row_data (GTK_CLIST(view->clist), 
					rownum, entry);
	}

	gtk_clist_thaw (GTK_CLIST(view->clist));
}

static void
init_reject_clist (CookieView *view, GList *list)
{
	GList *entry;

	gtk_clist_freeze (GTK_CLIST(view->clist));
	gtk_clist_clear (GTK_CLIST(view->clist));

	view->keep = list;

	foreach_list_elem (entry, list) {
		Password *p = (Password*)entry->data;
		gint rownum;
		gchar *row[1];

		p->remove = FALSE;
		
		row[0] = p->host;

		rownum = gtk_clist_append (GTK_CLIST(view->clist), row);
		gtk_clist_set_row_data (GTK_CLIST(view->clist), 
					rownum, entry);
	}

	gtk_clist_thaw (GTK_CLIST(view->clist));
}
#endif


static void 
init_all_clists (void)
{
	init_cookie_clist (stored, mozilla_list_cookies());
	init_server_clist (permissions,
			   mozilla_get_permissions(COOKIEPERMISSION));
	init_server_clist (images, mozilla_get_permissions(IMAGEPERMISSION));
#ifdef CVS_SOURCE
	init_password_clist (passwords, mozilla_list_passwords(PASSWORD));
	init_reject_clist (rejects, mozilla_list_passwords(REJECT));
#endif	
}

static void
cookie_view_cleanup (CookieView *view)
{
	if (view->keep) {
		g_list_foreach (view->keep, (GFunc)g_free, NULL);
		g_list_free (view->keep);
	}

	if (view->remove) {
		g_list_foreach (view->remove, (GFunc)g_free, NULL);
		g_list_free (view->remove);
	}

	g_free (view);
}
		
		
static void
dialog_destroy_cb (GtkWidget *widget, gpointer data)
{
	cookie_view_cleanup (stored);
	cookie_view_cleanup (permissions);
	cookie_view_cleanup (images);
#ifdef CVS_SOURCE
	cookie_view_cleanup (passwords);
	cookie_view_cleanup (rejects);
#endif
	dialog = NULL;
}



static void
dialog_apply_cb (GtkWidget *widget, gint pagenum)
{
	gboolean block;

	if (pagenum != -1) return;

	block = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(stored_block));
#ifdef CVS_SOURCE
	mozilla_remove_cookies (stored->remove, block);
	mozilla_remove_permissions (permissions->remove, COOKIEPERMISSION);
	mozilla_remove_permissions (images->remove,IMAGEPERMISSION);
	mozilla_remove_passwords (passwords->remove, PASSWORD);
	mozilla_remove_passwords (rejects->remove, REJECT);
	passwords->remove = rejects-> remove = NULL;	
#else	
	mozilla_set_cookies (stored->remove, 
			     permissions->remove, 
			     images->remove, 
			     block);
#endif

	stored->remove = permissions->remove = images->remove = NULL;

	init_all_clists ();
}

static void
dialog_help_cb (GtkWidget *widget, gpointer data)
{
}

GtkWidget *
cookies_show_manager (void)
{
	GladeXML *gxml;

	if (dialog) return dialog;

	gxml = glade_widget_new ("cookies", &dialog, NULL);

	/*
	 * init stored cookies list
	 */
	stored = g_new0 (CookieView, 1);
	stored->clist = glade_lookup_widget (dialog, "stored_clist");
	stored->remove_button = glade_lookup_widget (dialog, "stored_remove");
	stored->remove_all_button = glade_lookup_widget (dialog, 
							 "stored_remove_all");
	gtk_signal_connect (GTK_OBJECT(stored->clist), "select-row",
			    stored_select_cb, stored);

	stored_name = glade_lookup_widget (dialog, "stored_name");
	stored_value = glade_lookup_widget (dialog, "stored_value");
	stored_host = glade_lookup_widget (dialog, "stored_host");
	stored_path = glade_lookup_widget (dialog, "stored_path");
	stored_secure = glade_lookup_widget (dialog, "stored_secure");
	stored_expire = glade_lookup_widget (dialog, "stored_expire");
	stored_block = glade_lookup_widget (dialog, "stored_block");

	cookie_view_init (stored);

	/*
	 * init permissions list
	 */
	permissions = g_new0 (CookieView, 1);
	permissions->clist = glade_lookup_widget (dialog, "permissions_clist");
	permissions->remove_button = 
			glade_lookup_widget (dialog, "permissions_remove");
	permissions->remove_all_button = 
			glade_lookup_widget (dialog, "permissions_remove_all");

	cookie_view_init (permissions);

	/*
	 * init images list
	 */
	images = g_new0 (CookieView, 1);
	images->clist = glade_lookup_widget (dialog, "images_clist");
	images->remove_button = glade_lookup_widget (dialog, "images_remove");
	images->remove_all_button = glade_lookup_widget (dialog, 
							 "images_remove_all");
	
	cookie_view_init (images);
#ifdef CVS_SOURCE	
	/*
	 * init password list
	 */
	passwords = g_new0 (CookieView, 1);
	passwords->clist = glade_lookup_widget (dialog, "passwords_clist");
	passwords->remove_button = glade_lookup_widget (dialog, 
							"passwords_remove");
	passwords->remove_all_button = glade_lookup_widget 
					(dialog, "passwords_remove_all");
					
	cookie_view_init (passwords);

	/*
	 * init rejected password sites list
	 */
	rejects = g_new0 (CookieView, 1);
	rejects->clist = glade_lookup_widget (dialog, "rejects_clist");
	rejects->remove_button = glade_lookup_widget (dialog, "rejects_remove");
	rejects->remove_all_button = glade_lookup_widget (dialog,
							  "rejects_remove_all");
					
	cookie_view_init (rejects);

#endif	
	/*
	 * connect lower buttons
	 */
	gtk_signal_connect (GTK_OBJECT(dialog), "apply",
			    dialog_apply_cb, NULL);
	gtk_signal_connect (GTK_OBJECT(dialog), "help",
			    dialog_help_cb, NULL);
	gtk_signal_connect (GTK_OBJECT(dialog), "destroy", 
			    dialog_destroy_cb, NULL);
	
	gtk_window_set_title (GTK_WINDOW(dialog), 
			      _("Persistent Data Manager"));
	gtk_window_set_modal (GTK_WINDOW(dialog), TRUE);

	init_all_clists ();

	gtk_widget_show (dialog);
	window_set_layer (dialog);

	return dialog;
}

GtkWidget *
cookies_show_page (const CookiePage page)
{
	GtkWidget *dialog, *notebook;

	dialog = cookies_show_manager ();

	notebook = GNOME_PROPERTY_BOX(dialog)->notebook;
	gtk_notebook_set_page (GTK_NOTEBOOK(notebook),page);

	return dialog;
}
