/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"

/* local function prototypes */
static GtkWidget *embed_notebook_tab_new (GaleonEmbed *embed,
					  const gchar *title);
static GaleonEmbed *embed_create (GaleonEmbed *previous, gboolean new_window,
				  gboolean force_jump);
static gchar *embed_get_save_file_name (gchar *url);
static void embed_notebook_tab_update_closebutton (GaleonEmbed *embed,
						   GtkWidget *tab);
GaleonEmbed *embed_create_with_window (GaleonWindow *window, 
				       gboolean force_jump);

/** The global list of all GaleonEmbed structures */
GList *all_embeds = NULL;

/* global character set hash and sorted title list*/
GHashTable *charsets = NULL;
GList *sorted_charset_titles = NULL;

/* global last save image dir */
gchar *image_save_dir = NULL;

/* tab close button global pixmap data */
PixmapData *close_pix = NULL;

/**
 * embed_create_default: create a new browser pointing at the page
 * specified by configuration
 */
GaleonEmbed *
embed_create_default (GaleonEmbed *previous_embed, gboolean new_window)
{
	GaleonEmbed *embed;

	/* create a browser */
	embed = embed_create (previous_embed, new_window, TRUE);

	/* go to the default first page */
	embed_go_default (previous_embed, embed);

	/* finished */
	return embed;
}

/**
 * embed_go_default: go to the default page for a given situation
 */
void
embed_go_default (GaleonEmbed *previous_embed, GaleonEmbed *embed)
{
	gchar *home_page_url;
	gchar *last_page_url = NULL;
	gint page_type;

	/* get location of home page */
	home_page_url = gnome_config_get_string(CONF_GENERAL_HOMEPAGE);

	/* get location of last page: use previous browser if one,
	 * otherwise resort to fetching it from global history */
	if (previous_embed != NULL && previous_embed->site_location != NULL)
	{
		last_page_url = g_strdup (previous_embed->site_location);
	}
	else
	{
		last_page_url = history_get_last_url ();
	}
	
	/* find out where we're supposed to start */
	if (previous_embed == NULL)
	{
		page_type = gnome_config_get_int(CONF_GENERAL_HOMEPAGE_TYPE);
	}
	else
	{
		page_type = gnome_config_get_int(CONF_GENERAL_NEWPAGE_TYPE);
	}

	/* show the embed */
	embed_set_visibility (embed, TRUE);

       	/* go to the appropriate page */
	if (page_type == STARTPAGE_HOME && home_page_url != NULL)
	{
		/* load home page, if set */
		embed_load_url (embed, home_page_url);
	}
	else if (page_type == STARTPAGE_LAST && last_page_url != NULL)
	{
		/* load page pointed at by last browser */
		embed_load_url (embed, last_page_url);

		/* copy history, if any */
		if (previous_embed != NULL)
		{
			mozilla_copy_session_history (previous_embed, 
						      embed);
		}
	}
	else
	{
		/* even in case of error, it's a good default */
		embed_load_url (embed, "about:blank");
	}

	/* free allocated strings */
	if (home_page_url) g_free(home_page_url);
	if (last_page_url) g_free(last_page_url);
}

/**
 * embed_set_visibility: 
 */
void
embed_set_visibility (GaleonEmbed *embed, gboolean visibility)
{
	GaleonWindow *window;

	return_if_not_embed (embed);
	window = embed->parent_window;
	return_if_not_window (window);

	/* see if the status has actually changed */
	if (visibility && !embed->is_visible)
	{
		/* showing */
		window->visible_embeds++;
		embed->is_visible = TRUE;
		if (window->visible_embeds == 1)
		{
			window_set_visibility (window, TRUE);
		}
	}
	else if (!visibility && embed->is_visible)
	{
		/* hiding */
		window->visible_embeds--;
		embed->is_visible = FALSE;
		if (window->visible_embeds == 0)
		{
			window_set_visibility (window, FALSE);
		}
	}
}

/**
 * embed_load_url: interpret and load a URL into a given GaleonEmbed
 */
void
embed_load_url (GaleonEmbed *embed, const gchar *url)
{
	/* check arguments */
	return_if_not_embed (embed);
	g_assert(url != NULL);

	/* error if we can't handle this URL */
	if (handle_foreign_protocols(url))
		return;

	/* update location bar */
	if (embed->site_location != NULL)
	{
		g_free (embed->site_location);
	}
	embed->site_location = g_strdup (url);
	embed_update_page_location (embed);

	/* reset title */
	g_free (embed->site_title);
	embed->site_title = g_strdup (_("Untitled"));
	embed_update_page_title (embed);

	/* load the URL */
	if (strcmp(url, MYPORTAL_URL) == 0)
	{
		gtk_moz_embed_stop_load (embed->mozEmbed);

		/* display my portal */
		portal_render_into_embed (embed);
	}
	else
	{
		if (GTK_BIN(embed->mozEmbed)->child)
		{
			gtk_widget_grab_focus 
				(GTK_BIN(embed->mozEmbed)->child);
		}

		/* load the url */
		gtk_moz_embed_load_url (embed->mozEmbed, url);
	}

	/* initialise embed whenever a document is first loaded */
	if (embed->wrapper == NULL)
	{
		embed_wrapper_init (embed);
	}
}

/**
 * embed_create_from_url: create a browser from a given url string
 */
GaleonEmbed *
embed_create_from_url (GaleonEmbed *previous, const gchar *url, 
		       gboolean new_window)
{
	GaleonEmbed *embed;

	/* check argument */
	g_assert(url != NULL);

	/* error if we can't handle this URL */
	if (handle_foreign_protocols(url))
		return NULL;

	/* create a window */
	embed = embed_create (previous, new_window, FALSE);

	/* show the window */
	embed_set_visibility (embed, TRUE);

	/* load the url */
	embed_load_url (embed, url);

	return embed;
}

/**
 * embed_create_hidden: do a simple embed creation but without showing it
 */
GaleonEmbed *
embed_create_hidden (GaleonEmbed *previous, gboolean new_window)
{
	GaleonEmbed *embed;

	/* create */
	embed = embed_create (previous, new_window, FALSE);

	/* return */
	return embed;
}

/**
 * embed_create_from_url_view_source: create a browser from a given 
 * url string  in view source mode
 */
GaleonEmbed *
embed_create_from_url_view_source (GaleonEmbed *previous, const gchar *url,
				   gboolean new_window)
{
	GaleonEmbed *embed;
#ifdef CVS_SOURCE
	gchar *vs_url;
#endif

	/* check argument */
	g_assert(url != NULL);

	/* error if we can't handle this URL */
	if (handle_foreign_protocols(url))
		return NULL;

	/* create a window */
	embed = embed_create (previous, new_window, FALSE);

	/* load a blank url, to initialize the embed */
	embed_load_url (embed, "about:blank");

	/* show the embed */
	embed_set_visibility (embed, TRUE);

	/* if we're opening in a new window, turn off the menu, toolbars,
	 * and statusbar */
	if (new_window)
	{
		window_statusbar_hide (embed->parent_window);
		window_menubar_hide (embed->parent_window);
		window_toolbar_hide (embed->parent_window);
	}

#ifdef CVS_SOURCE
	/* prepend view-source: protocol */
	vs_url = g_strconcat ("view-source:", url, NULL);

	/* load the url */
	embed_load_url (embed, vs_url);
	g_free (vs_url);
#else
	/* set view source mode */
	mozilla_view_source_mode (embed, TRUE);

	/* load the url */
	embed_load_url (embed, url);
#endif
	/* return completed browser */
	return embed;
}

/* signals to connect on each embed widget */
static const struct
{ 
	char *event; 
	void *func; /* should be a GtkSignalFunc or similar */
}
signal_connections[] =
{
	{ "location",        mozembed_location_changed_cb  },
	{ "title",           mozembed_title_changed_cb     },
	{ "net_start",       mozembed_load_started_cb      },
	{ "net_stop",        mozembed_load_finished_cb     },
	{ "net_state",       mozembed_net_status_change_cb },
	{ "progress",        mozembed_progress_change_cb   },
	{ "link_message",    mozembed_link_message_cb      },
	{ "js_status",       mozembed_js_status_cb         },
	{ "open_uri",        mozembed_open_uri_cb          },
	{ "visibility",      mozembed_visibility_cb        },
	{ "destroy_browser", mozembed_destroy_brsr_cb      },
	{ "dom_mouse_down",  mozembed_dom_mouse_down_cb    },	
	{ "dom_mouse_click", mozembed_dom_mouse_click_cb   },
	{ "dom_key_press",   mozembed_dom_key_press_cb     },
	{ "size_to",         mozembed_size_to_cb           },
	{ "new_window",      mozembed_new_window_cb        },
	/* terminator -- must be last in the list! */
	{ NULL, NULL } 
};

/**
 * embed_create: create a GaleonEmbed structure based on a given Embed,
 * possibly creating a new window if needed or requested.
 */
static GaleonEmbed *
embed_create (GaleonEmbed *previous, gboolean new_window, gboolean force_jump)
{
	GaleonWindow *window = NULL;

	/* get the parent window if we're using it */
	if (!new_window)
	{
		if (previous == NULL)
		{
			/* this shouldnt happen */
			g_warning ("can't open new tab, previous == NULL");
		}
		else
		{
			window = previous->parent_window;
		}
	}

	/* if we haven't been passed a valid window, create one */
	if (window == NULL)
	{
		window = window_create ();
	}

	return embed_create_with_window (window, force_jump);
}

/**
 * embed_create_with_window: create a GaleonEmbed to go in a given
 * GaleonWindow
 */
GaleonEmbed *
embed_create_with_window (GaleonWindow *window, gboolean force_jump)
{
	static gboolean preferences_set = FALSE;
	static gboolean pushed_startup = FALSE;
	GaleonEmbed *embed;
	GtkWidget *tab;
	gint i;

	/* build an embed structure */
	embed = g_new0 (GaleonEmbed, 1);

	/* set parent and store in parents list */
	embed->parent_window = window;
	window->embed_list = g_list_prepend (window->embed_list, embed);

	/* show tabs if more than one embed or if the user has asked for it */
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (window->notebook), 
		(g_list_length (window->embed_list) > 1) |
		gnome_config_get_bool (CONF_APPEARANCE_TABBED_ALWAYS_SHOW));

	/* get ready to kick off mozembeds */
	if (!pushed_startup)
	{
		gtk_moz_embed_push_startup ();
		pushed_startup = TRUE;
	}

	/* make an embedding widget */
	embed->mozEmbed = (GtkMozEmbed *) gtk_moz_embed_new ();

	/* ref/unref widget -- is this really needed? */
	/* NB: apparently it is to get rid of some warnings, but I'm
	 * still not clear why -- MattA 13/4/2001 */
	gtk_widget_ref (GTK_WIDGET (embed->mozEmbed));
	gtk_object_set_data_full (GTK_OBJECT (embed->mozEmbed), "mozEmbed",
				  GTK_WIDGET (embed->mozEmbed),
				  (GtkDestroyNotify) gtk_widget_unref);

	/* setup data so we can always find GaleonEmbed */
	gtk_object_set_data (GTK_OBJECT (embed->mozEmbed),
			     "GaleonEmbed", embed);

	/* setup chrome */
	gtk_moz_embed_set_chrome_mask (embed->mozEmbed, 
				       GTK_MOZ_EMBED_FLAG_ALLCHROME);

	/* connect signals */
	for (i = 0; signal_connections[i].event != NULL; i++)
	{
		gtk_signal_connect_while_alive (GTK_OBJECT(embed->mozEmbed),
						signal_connections[i].event,
						signal_connections[i].func, 
						embed,
						GTK_OBJECT(embed->mozEmbed));
	}
	gtk_signal_connect (GTK_OBJECT (embed->mozEmbed), "destroy",
			    GTK_SIGNAL_FUNC (mozembed_destroy_cb), embed);

	/* set gtkmozembed drag and drop destination */
	gtk_drag_dest_set (GTK_WIDGET(embed->mozEmbed), GTK_DEST_DEFAULT_ALL,
			   drop_types, drop_types_num_items,
			   GDK_ACTION_COPY | GDK_ACTION_MOVE |
			   GDK_ACTION_LINK | GDK_ACTION_ASK );

	/* set gtmozembed drag and drop signals */
	gtk_signal_connect (GTK_OBJECT(embed->mozEmbed), "drag_drop",
			    GTK_SIGNAL_FUNC(mozembed_drag_drop_cb), embed);
	gtk_signal_connect (GTK_OBJECT(embed->mozEmbed), "drag_data_received",
			    GTK_SIGNAL_FUNC(embed_drag_data_received), embed);

	/* set links drag signal */
	gtk_signal_connect (GTK_OBJECT (embed->mozEmbed), "drag_data_get",
			    GTK_SIGNAL_FUNC (window_drag_data_get_cb),
			    embed);

	/* get the charset config from mozilla, the first time */
	if (sorted_charset_titles == NULL)
	{
		mozilla_get_charsets (&charsets, &sorted_charset_titles);
	}

	/* set preferences */
	if (!preferences_set) 
	{
		mozilla_prefs_set ();
		preferences_set = TRUE;
	}

	/* add to global list of all embeds */
	all_embeds = g_list_prepend (all_embeds, embed);

	/* not closing */
	embed->is_closing = FALSE;

	/* no content for the location or title */
	embed->site_location = NULL;
	embed->site_title = g_strdup (_("Untitled"));
	
	/* set magic */
	embed->magic = GALEON_EMBED_MAGIC;

	/* populate the encoding submenu */
	if (!(window->encodings_menu_set))
	{
		window_set_encoding_menu (window, sorted_charset_titles);
		window->encodings_menu_set = TRUE;
	}

	/* add as a tab into the notebook */
	tab = embed_notebook_tab_new (embed, embed->site_title);
	if ((g_list_length (window->embed_list) > 1) &&
	     gnome_config_get_bool (CONF_APPEARANCE_TABBED_INSERT_NEW_TABS))
	{
		gtk_notebook_insert_page (GTK_NOTEBOOK (window->notebook),
				  	  GTK_WIDGET (embed->mozEmbed), tab,
					  gtk_notebook_current_page (
						GTK_NOTEBOOK (window->notebook))					  + 1);
	}
	else
	{
		gtk_notebook_append_page (GTK_NOTEBOOK (window->notebook),
				  	  GTK_WIDGET (embed->mozEmbed), tab);
	}
	embed_set_notebook_label_status (embed, TAB_NEW);

	/* show it */
	gtk_widget_show (GTK_WIDGET (embed->mozEmbed));

	/* switch to page if set in config */
	if (gnome_config_get_bool (CONF_APPEARANCE_TABBED_AUTOJUMP) ||
	    force_jump)
	{
		gint page;

		/* argh, this is laborious! */
		page = gtk_notebook_page_num (GTK_NOTEBOOK (window->notebook),
					      GTK_WIDGET (embed->mozEmbed));
		gtk_notebook_set_page (GTK_NOTEBOOK (window->notebook), page);
	}

	/* return completed structure */
	return embed;
}

/**
 * embed_wrapper_init: call it after the first page is loaded
 */
void embed_wrapper_init (GaleonEmbed *embed)
{
	gboolean event_listener = gnome_config_get_bool(CONF_MOUSE_LINKS_DRAG);
	embed->wrapper = mozilla_wrapper_init (embed, event_listener);
}

/**
 * embed_progress_clear: clear all infos about download progress
 */
void
embed_progress_clear(GaleonEmbed *embed)
{
	/* set all progress values to 0 */ 
	embed->loadPercent = 0;
	embed->bytesLoaded = 0;
	embed->maxBytesLoaded = 0;
}

/**
 * embed_show_find_dialog: show the find dialog
 */
void embed_show_find_dialog (GaleonEmbed *embed)
{
	GladeXML *gxml;
	FindDialog *find;
	
	if (embed->find_dialog == NULL)
	{
		find = g_new0 (FindDialog, 1);
		embed->find_dialog = find;
		
		gxml = glade_widget_new ("dFind", &(find->dialog), embed);
		
		find->gnome_entry = glade_xml_get_widget (gxml, 
							  "find_gnome_entry");
		find->entry = glade_xml_get_widget (gxml, "find_entry");
		find->prev_button = glade_xml_get_widget (gxml, "BPrev");
		find->next_button = glade_xml_get_widget (gxml, "BNext");
		find->case_checkbutton = glade_xml_get_widget (gxml, 
							       "case_check");
		find->string_changed = TRUE;
	}

	gtk_widget_show (embed->find_dialog->dialog);
	window_set_layer (embed->find_dialog->dialog);
}

/**
 * embed_add_temporary_bookmark: add a temporary bookmark
 */
void embed_add_temporary_bookmark (GaleonEmbed *embed)
{
	add_temp_bookmark (BM_SITE, embed->site_title, 
			   embed->site_location, NULL);
}

/**
 * embed_save_image
 */
void embed_save_image (GaleonEmbed *embed, gchar *url)
{
	GtkWidget *fs = NULL;
	GladeXML *gxml;
	gchar *filename, *path;

	gxml = glade_widget_new ("fssave", &fs, embed->parent_window);

	if (image_save_dir == NULL)
	{
		image_save_dir = g_strdup (g_get_home_dir());
	}
	filename = embed_get_save_file_name (url);
	path = g_strconcat (image_save_dir, "/", filename, NULL);
	gtk_file_selection_set_filename (GTK_FILE_SELECTION (fs), path);
	g_free (filename);
	g_free (path);

	gtk_object_set_data (GTK_OBJECT(fs), "image", (gpointer *) url);

	gtk_window_set_modal(GTK_WINDOW(fs), TRUE);
	gtk_widget_show(fs);
	window_set_layer(fs);
}

/**
 * embed_save_image_default
 */
void embed_save_image_default (GaleonEmbed *embed, gchar *url)
{
	gchar *filename, *path, *temp_message;
	gboolean result;
	
	if (image_save_dir == NULL)
	{
		image_save_dir = g_strdup (g_get_home_dir());
	}
	filename = embed_get_save_file_name (url);
	path = g_strconcat (image_save_dir, "/", filename, NULL);
	
	result = mozilla_save_image (embed, url, path);
	if (!result)
	{
		 gnome_error_dialog(_("Cannot save the file"));
	}
	else
	{
		temp_message = g_strconcat(N_("Image saved as "), path, NULL);
		window_update_temp_message (embed->parent_window,
					    temp_message);
		g_free (temp_message);
	}

	g_free (filename);
	g_free (path);
}

/**
 * embed_set_image_as_background
 */
void embed_set_image_as_background (GaleonEmbed *embed, gchar *url)
{
	gchar *file, *path, *command;
	gboolean result;

	/* get the filename and check */
	file = embed_get_save_file_name (url);
	g_return_if_fail (file != NULL);
	g_return_if_fail (strlen (file) != 0);

	/* build a path to save it in */
	path = g_strconcat (g_get_home_dir (), "/.galeon/", file, NULL);

	/* save the image */
	result = mozilla_save_image (embed, url, path);

	/* if successful, use gnome to set the background */
	if (result)
	{
		/* build command */
		command = g_strconcat ("background-properties-capplet "
				       "--init-session-settings --ignore "
				       "--background-image=",
				       path, NULL);

		/* execute it synchronously */
		gnome_execute_shell (g_get_home_dir (), command);

		/* free */
		g_free (command);
	}

	/* free */
	g_free (file);
	g_free (path);
}

/**
 * embed_sav_document
 */
void embed_save_document (GaleonEmbed *embed, gboolean save_main)
{
	GtkWidget *fs = NULL;
	GladeXML *gxml;
	gchar *target = NULL, *filename;

	gxml = glade_widget_new ("fssave", &fs, embed->parent_window);

	if (save_main)
	{
		target = g_strdup (embed->site_location);
	}
	else
	{
		target = mozilla_get_document_url (embed);
	}

	filename = embed_get_save_file_name (target);
	gtk_file_selection_set_filename (GTK_FILE_SELECTION (fs), filename);

	gtk_object_set_data (GTK_OBJECT(fs), "main", (gpointer *) save_main);

	g_free (filename);

	gtk_window_set_modal(GTK_WINDOW(fs), TRUE);
	gtk_widget_show(fs);
	window_set_layer(fs);
}

/**
 * embed_view_source: view web page source 
 */
void embed_view_source (GaleonEmbed *embed, gboolean main, gboolean new_window)
{
	/* FIXME is use /tmp ok ? */
	char *filename = g_strdup("/tmp/galeon-viewsource-XXXXXX");
	gboolean save_result;
	int result;

	/* get a name for the temporary file */
	result = mkstemp (filename);

	if (result == -1) {
		gnome_error_dialog ( _("Could not create a temporary file"));
		return;
	}
		
	/* save the file from mozilla to a temporary file */
	if (main)
	{
		save_result = mozilla_save_main_document (embed, filename);
	}
	else
	{
		save_result = mozilla_save (embed, filename);
	}

	if (!save_result) {
		gnome_error_dialog (_("Something went wrong while saving the "
			"file to a temporary location"));
		g_free(filename);
		return;
	}
		
	if (gnome_config_get_bool (CONF_HANDLERS_USE_EXTERNAL_SOURCE_VIEWER))
	{
		launch_external_viewer (filename);
	}
	else
	{
		gchar *file_url = g_strconcat ("file://", filename, NULL);
		embed_create_from_url_view_source (embed, file_url,
				new_window);
		g_free (file_url);
	}
	
	g_free (filename);
}

/**
 * embed_close: close a GaleonEmbed
 */
void
embed_close (GaleonEmbed *embed)
{
	GaleonWindow *window;

	/* already closing */
	if (embed->is_closing)
	{
		return;
	}
	embed->is_closing = TRUE;

	/* stop any loading */
	gtk_moz_embed_stop_load (embed->mozEmbed);

	/* kill off find dialog if active */
	if (embed->find_dialog != NULL)
	{
		gtk_widget_destroy (GTK_WIDGET (embed->find_dialog->dialog));
	}

	/* set as invisible */
	embed_set_visibility (embed, FALSE);

	/* select the next page */
	if (embed->is_active)
	{
		window = embed->parent_window;
		return_if_not_window (window);
		gtk_notebook_next_page (GTK_NOTEBOOK (window->notebook));
	}

	/* destroy the embedding widget -- this will also destroy 
	 * the notebook tab and it's label, and remove it from 
	 * the relevant lists */
	gtk_widget_destroy (GTK_WIDGET (embed->mozEmbed));
}

/**
 * embed_open_frame: open the frame pointed by the event target 
 */
void 
embed_open_frame (GaleonEmbed *embed, gboolean same_embed, gboolean new_window)
{
	gchar *url = NULL;
	
	url = mozilla_get_document_url (embed);

	if (url != NULL)
	{
		if (same_embed)
		{
			embed_load_url (embed, url);
		} 
		else 
		{
			embed_create_from_url (embed, url, new_window);
		}
	}
}

/*
 * embed_reload: call gtk_moz_embed_reload but first check if 
 * we are not in MyPortal
 */
void
embed_reload (GaleonEmbed *embed)
{
	/* check if it's the portal */
	if (strcmp (embed->site_location, MYPORTAL_URL) == 0)
	{
		/* reload by regenerating */
		gtk_moz_embed_stop_load (embed->mozEmbed);
		portal_render_into_embed (embed);
	}
	else
	{
		/* reload as usual */
		gtk_moz_embed_reload (embed->mozEmbed, 
				      GTK_MOZ_EMBED_FLAG_RELOADNORMAL);
	}
}

/**
 * embed_update_page_location: called if the page location changes, or to
 * bring the GaleonEmbed.site_location field in sync with the currently 
 * viewed page
 */
void
embed_update_page_location (GaleonEmbed *embed)
{
	GaleonWindow *window;
	gchar *new_location;
	GtkWidget *entry;
	PixmapData *drag_pixmap;
	GdkPixmap *pixmap;
	GdkBitmap *mask;

	/* check we're currently being viewed */
	if (!embed->is_active)
		return;

	/* get the parent window */
	window = embed->parent_window;
	return_if_not_window (window);

	/* get the location string */
	entry = window->toolbar_entry;
	new_location = window->active_embed->site_location;

	/* clear text */
	gtk_editable_delete_text(GTK_EDITABLE (entry), 0, -1);
	if (new_location && strcmp(new_location, "about:blank") != 0)
	{
		/* change the url entry text */
		gtk_entry_set_text(GTK_ENTRY (entry), new_location);
		
		/* update the drag location pixmap */
		if (gnome_config_get_bool (CONF_GENERAL_FAVICONS_ENABLED))
		{
			drag_pixmap = bookmarks_get_siteicon (new_location);
			pixmap = drag_pixmap->pixmap;
			mask = drag_pixmap->mask;
			gtk_pixmap_set (GTK_PIXMAP (window->drag_pixmap), 
					pixmap, mask);
		}
	}

	window_update_temp_message (window, NULL);
	window_update_nav_buttons (window);
}

/**
 * embed_update_page_title: called if the page title changes, or to bring
 * the main window title in sync with the currently viewed page
 */
void
embed_update_page_title (GaleonEmbed *embed)
{
	gchar *full_title;
	gchar *title_string;
	GaleonWindow *window;

	/* set notebook label (although this might not be visible) */
	embed_set_notebook_label (embed);

	/* if this page isn't being viewed, get out now */
	if (!embed->is_active)
		return;

	/* get the window */
	window = embed->parent_window;
	return_if_not_window (window);

	/* get the format string */
	title_string = gnome_config_get_string (CONF_APPEARANCE_WINDOWS_TITLE);

	/* format the full title */
	full_title = g_strdup_printf (title_string, embed->site_title);

	/* set the toplevel window title to the document title */
	gtk_window_set_title (GTK_WINDOW(window->WMain), full_title);

	/* free allocated strings */
	if (full_title) g_free (full_title);
	if (title_string) g_free (title_string);  
}

/**
 * embed_set_notebook_label: sets the notebook tab label belonging to embed
 * to the string contained in text.
 */
void
embed_set_notebook_label (GaleonEmbed *embed)
{
	GtkWidget *tab, *label;
	gchar *shortened;
	GaleonWindow *window;
	gint length;

	/* get the parent window */
	window = embed->parent_window;
	return_if_not_window (window);

	/* get the tab widget */
	tab = gtk_notebook_get_tab_label (GTK_NOTEBOOK (window->notebook),
					  GTK_WIDGET (embed->mozEmbed));

	/* get out if theres a problem */
	g_return_if_fail (tab != NULL);

	/* abbreviate the text label */
	length = gnome_config_get_int (CONF_APPEARANCE_TABBED_SHORTEN);
	shortened = shorten_name (embed->site_title, length);
	g_return_if_fail (shortened != NULL);

	/* get the text label */
	label = gtk_object_get_data (GTK_OBJECT (tab), "label");
	if (!GTK_IS_LABEL (label))
		return;

	/* if it's different than the new shortened text, change it */
	if (strcmp (GTK_LABEL (label)->label, shortened))
	{
		gtk_label_set_text (GTK_LABEL (label), shortened);
	}

	/* the menu text is the full text */
	gtk_notebook_set_menu_label_text (GTK_NOTEBOOK (window->notebook),
					  GTK_WIDGET (embed->mozEmbed),
					  embed->site_title);

	/* free allocated strings */
	g_free (shortened);

	/* resize the tabs */
	gtk_widget_queue_resize(window->notebook);
}

/**
 * embed_set_notebook_label_status: sets the status for the notebook tab
 * label belonging to embed.
 */
void
embed_set_notebook_label_status (GaleonEmbed *embed, TabbedStatus status)
{
	GaleonWindow *window;
	GtkWidget *tab, *label;

	return_if_not_embed (embed);
	window = embed->parent_window;
	return_if_not_window (window);

	/* get the tab widget */
	tab = gtk_notebook_get_tab_label (GTK_NOTEBOOK (window->notebook),
					  GTK_WIDGET (embed->mozEmbed));
	g_return_if_fail (tab != NULL);

	label = gtk_object_get_data (GTK_OBJECT (tab), "label");
	if (!GTK_IS_LABEL (label))
		return;

	switch (status)
	{
		case TAB_NORMAL:
			gtk_widget_set_rc_style (label);
			break;
			
		case TAB_NEW:
			gtk_widget_set_style (label, blue_text_style);
			break;
			
		case TAB_LOADING:
			gtk_widget_set_style (label, red_text_style);
			break;
	}
}

/**
 * embed_notebook_tab_update_closebutton (internal function): tab, the
 * notebook tab belonging to embed, is updated (if necessary) to reflect
 * the user's preference in the display of a close button.  if tab is
 * null, it will be found using embed.
 */
static void
embed_notebook_tab_update_closebutton (GaleonEmbed *embed, GtkWidget *tab)
{
	gint use_button;
	GaleonWindow *window;
	GtkWidget *button;

	return_if_not_embed (embed);
	window = embed->parent_window;
	return_if_not_window (window);

	if (!tab)
	{
		/* get the tab widget */
		tab = gtk_notebook_get_tab_label
			(GTK_NOTEBOOK (window->notebook),
				GTK_WIDGET (embed->mozEmbed));
	}
	if (!GTK_IS_CONTAINER (tab)) return;

	use_button = 
		gnome_config_get_bool (CONF_APPEARANCE_TABBED_CLOSEBUTTON);

	button = gtk_object_get_data (GTK_OBJECT (tab), "button");

	if (!use_button && button)
	{
		gtk_widget_destroy (button);
		gtk_object_remove_data (GTK_OBJECT (tab), "button");
		gtk_object_remove_data (GTK_OBJECT (tab), "pixmap");
	}
	else if (use_button && !button)
	{
		GtkWidget *pixmap, *pixmap_plain;
		button = gtk_button_new ();

		if (close_pix == NULL)
		{
			close_pix = pixmap_data_new_from_file
				(SHARE_DIR "/small-close.xpm");
		}
		pixmap = gtk_pixmap_new (close_pix->pixmap, close_pix->mask);
		gtk_widget_show (pixmap);
		gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
		gtk_container_add (GTK_CONTAINER (button), pixmap);

		/* Create another pixmap widget so we can use it when a tab
		   has been made inactive.  We do this because the button has
		   its own GdkWindow and cannot be made transparent, leading
		   to much ugliness.  --Josh */
		pixmap_plain = gtk_pixmap_new (close_pix->pixmap, 
					       close_pix->mask);
		gtk_widget_set_sensitive (pixmap_plain, FALSE);

		gtk_box_pack_start (GTK_BOX (tab), button, FALSE, FALSE, 0);
		gtk_box_pack_start (GTK_BOX (tab), pixmap_plain, FALSE,
				    FALSE, 0);
		gtk_object_set_data (GTK_OBJECT (tab), "button",
				     button);
		gtk_object_set_data (GTK_OBJECT (tab), "pixmap",
				     pixmap_plain);

		gtk_signal_connect (GTK_OBJECT (button), "clicked", 
			GTK_SIGNAL_FUNC (embed_notebook_close_clicked_cb),
			embed);
		gtk_signal_connect (GTK_OBJECT (tab), "draw",
				    GTK_SIGNAL_FUNC (embed_tab_draw_cb),
				    embed);

		/* we don't show the button or pixmap, as that will be
		   handled by embed_label_draw_cb */
	}
}

/**
 * embed_update_notebook_closebutton: adds or removes embed's notebook
 * tab's closebutton, as appropriate
 */
void
embed_update_notebook_closebutton (GaleonEmbed *embed)
{
	embed_notebook_tab_update_closebutton (embed, NULL);
}

GtkWidget *
embed_notebook_tab_new (GaleonEmbed *embed, const gchar *title)
{
	GtkWidget *box = gtk_hbox_new (FALSE, 4);
	GtkWidget *label = gtk_label_new (title);

	gtk_object_set_data (GTK_OBJECT (box), "label",
			     label);
	gtk_drag_dest_set (box, GTK_DEST_DEFAULT_ALL,
			   drop_types, drop_types_num_items,
			   GDK_ACTION_COPY | GDK_ACTION_MOVE |
			   GDK_ACTION_LINK | GDK_ACTION_ASK );
	gtk_signal_connect (GTK_OBJECT (box), "drag_data_received",
			    GTK_SIGNAL_FUNC(embed_drag_data_received), embed);

	gtk_box_pack_start (GTK_BOX (box), label, TRUE, TRUE, 0);
	gtk_widget_show (label);

	embed_notebook_tab_update_closebutton (embed, box);

// FIXME: try something like this?
//	gtk_widget_set_usize (GTK_WIDGET (box), 50, -2);

	return box;
}

/**
 * embed_get_save_file_name: Parse a URL to find a filename for saving
 *
 * The string passed in will be modified. The returned string should be freed.
 */
static gchar *
embed_get_save_file_name (gchar *url)
{
	gchar *filename, *curr;
	gchar *default_name;

	default_name = g_strdup_printf(_("index.html"));

	if (!url)
		return default_name;

	filename = strrchr (url, '/');

	if (!filename)
		return default_name;
	else if (filename && (*(filename + 1) == '\0'))
		return default_name;
	else
	{
		curr = ++filename;
		while (*curr != '\0')
		{
			if (*curr == '?' || *curr == '#')
			{
				*curr = '\0';
				break;
			}
			curr++;
		}
	}
		
	if (*filename == '\0')
		return default_name;
	else
	{
		g_free (default_name);

		filename = g_strdup (filename);
		return filename;
	}
}

/**
 * embed_go_up: go to the nth parent directory
 */
void
embed_go_up (GaleonEmbed *embed, gint levels)
{
	GnomeVFSURI *uri, *up_uri;
	gchar *location;

	/* use gnome-vfs to find parent */
	uri = gnome_vfs_uri_new (embed->site_location);
	if (uri == NULL)
	{
		return;
	}

	/* go upwards to find the nth level up */
	do
	{
		up_uri = gnome_vfs_uri_get_parent (uri);
		gnome_vfs_uri_unref (uri);
		uri = up_uri;

		/* this can happen if Up is selected from the menu */
		if (uri == NULL)
		{
			return;
		}
	}
	while (levels--);

	/* get the location */
	location = gnome_vfs_uri_to_string (uri, 0);
	gnome_vfs_uri_unref (uri);

	/* visit it if apparently valid */
	if (location == NULL)
	{
		return;
	}
	if (strlen (location) != 0)
	{
		embed_load_url (embed, location);
	}
	g_free (location);
}

/**
 * embed_can_go_up: test to see if we can go to a parent directory
 */
gboolean
embed_can_go_up (GaleonEmbed *embed)
{
	GnomeVFSURI *uri, *up_uri;

	/* check embed location is valid */
	if (embed->site_location == NULL || strlen (embed->site_location) == 0)
	{
		return FALSE;
	}

	/* use gnome-vfs to find parent */
	uri = gnome_vfs_uri_new (embed->site_location);
	if (uri == NULL)
	{
		return FALSE;
	}
	up_uri = gnome_vfs_uri_get_parent (uri);
	gnome_vfs_uri_unref (uri);
	if (up_uri == NULL)
	{
		return FALSE;
	}
	else
	{
		gnome_vfs_uri_unref (up_uri);
		return TRUE;
	}
}

