/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <galeon.h>
#include <gtkmozembed_internal.h>

#include "regex.h"
#include "nsIGlobalHistory.h"
#include "nsIDocShellHistory.h"
#include "nsIDocShellTreeItem.h"
#include "nsIDocShellTreeNode.h"
#include "nsIDocShellTreeOwner.h"
#include "nsIDiskDocument.h"
#include "nsIWebBrowserFind.h"
#include "nsIWebBrowserFocus.h"
#include "nsIDocument.h"
#include "nsFileSpec.h"
#include "nsISHEntry.h"
#ifdef CVS_SOURCE
#include "nsISHistoryInternal.h"
#include "nsIWebBrowserPrint.h"
#else
#include "nsINetDataCacheManager.h"
#include "nsICachedNetData.h"
#endif
#include "nsIURI.h"
#include "nsIURL.h"
#include "nsIPresShell.h"
#include "nsTextServicesCID.h"
#include "nsIMarkupDocumentViewer.h"
#include "nsIDOMHTMLElement.h"
#include "nsIDOMHTMLDocument.h"
#include "nsIDOMNodeList.h"
#include "nsIDOMNamedNodeMap.h"
#include "nsIScriptGlobalObject.h"
#include "nsIDOMWindowInternal.h"
#include "nsIDOMHTMLAnchorElement.h"
#include "nsIContentViewerFile.h"
#include "nsICharsetConverterManager.h"
#include "nsICharsetConverterManager2.h"
#include "nsIInterfaceRequestor.h"
#include "nsIChannel.h"
#include "nsILoadGroup.h"
#include "nsIWebBrowserPersist.h"
#include "nsNetUtil.h"
#include "nsPIDOMWindow.h"
#include "nsIChromeEventHandler.h"
#include "galeon-wrapper.h"

static NS_DEFINE_CID(kCTextServicesDocumentCID, NS_TEXTSERVICESDOCUMENT_CID);
static NS_DEFINE_CID(kStandardURLCID, NS_STANDARDURL_CID);

GaleonWrapper::GaleonWrapper ()
{
}

GaleonWrapper::~GaleonWrapper ()
{
}

nsresult GaleonWrapper::Init (GaleonEmbed *galeon_embed)
{
	nsresult result;

	return_val_if_not_embed (galeon_embed, NS_ERROR_FAILURE);

	embed = galeon_embed;
	mGtkMozEmbed = embed->mozEmbed;

	gtk_moz_embed_get_nsIWebBrowser (mGtkMozEmbed,
					 getter_AddRefs(mWebBrowser));
	if (!mWebBrowser) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocShellHistory> dsHistory = do_QueryInterface (DocShell);
	if (!dsHistory) return NS_ERROR_FAILURE;

	NS_WITH_SERVICE(nsIGlobalHistory, history,
			NS_GLOBALHISTORY_CONTRACTID, &result);  
	if (NS_FAILED(result) || !history) return NS_ERROR_FAILURE;
	dsHistory->SetGlobalHistory(history);

#if MOZILLA_MILESTONE != 81
  	mChromeNav = do_QueryInterface(mWebBrowser);
	if (!mChromeNav) return NS_ERROR_FAILURE;
#else
	nsCOMPtr<nsIDOMWindow> domWindow;
	mWebBrowser->GetContentDOMWindow(getter_AddRefs(domWindow));
	if (!domWindow) return NS_ERROR_FAILURE;

	nsCOMPtr<nsPIDOMWindow> domWindowPrivate = do_QueryInterface(domWindow);
	nsCOMPtr<nsIDOMWindowInternal> rootWindow;
	domWindowPrivate->GetPrivateRoot(getter_AddRefs(rootWindow));
  
	nsCOMPtr<nsIChromeEventHandler> chromeHandler;
	nsCOMPtr<nsPIDOMWindow> piWin(do_QueryInterface(rootWindow));
	piWin->GetChromeEventHandler(getter_AddRefs(chromeHandler));
  
	mEventReceiver = do_QueryInterface(chromeHandler);

	mEventListener = new GaleonEventListener();
	mEventListener->Init(this);
#endif
	if (attachListener)
	return AddListener ();
}

#ifdef CVS_SOURCE
nsresult GaleonWrapper::Print (nsIPrintOptions *options)
{
	nsresult rv = NS_OK;

	nsCOMPtr<nsIDocShell> DocShell;
	rv = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(rv) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIContentViewer> contentViewer;
	DocShell->GetContentViewer(getter_AddRefs(contentViewer));
	if (contentViewer) {
		nsCOMPtr<nsIContentViewerFile> contentViewerFile(do_QueryInterface(contentViewer));
		if (contentViewerFile) {
			rv = contentViewerFile->Print(PR_TRUE, nsnull, nsnull);
		}
	}

	return rv;
}
#endif

nsresult GaleonWrapper::GetSHistory (nsISHistory **aSHistory)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebNavigation> ContentNav = do_QueryInterface (DocShell,
								   &result);
	if (!ContentNav) return NS_ERROR_FAILURE;

	nsCOMPtr<nsISHistory> SessionHistory;
	result = ContentNav->GetSessionHistory (getter_AddRefs (SessionHistory));
	if (!SessionHistory) return NS_ERROR_FAILURE;

	*aSHistory = SessionHistory.get();
	NS_IF_ADDREF (*aSHistory);
}

nsresult GaleonWrapper::GetDocShell (nsIDocShell **aDocShell)
{
#if MOZILLA_MILESTONE == 81
	nsCOMPtr<nsIDocShell> DocShell (do_GetInterface(mWebBrowser));
	if (!DocShell) return NS_ERROR_FAILURE;

	*aDocShell = DocShell.get();

	NS_IF_ADDREF(*aDocShell);
#else
	nsCOMPtr<nsIDocShellTreeItem> browserAsItem;
	browserAsItem = do_QueryInterface(mWebBrowser);
	if (!browserAsItem) return NS_ERROR_FAILURE;

	// get the owner for that item
	nsCOMPtr<nsIDocShellTreeOwner> treeOwner;
	browserAsItem->GetTreeOwner(getter_AddRefs(treeOwner));
	if (!treeOwner) return NS_ERROR_FAILURE;

	// get the primary content shell as an item
	nsCOMPtr<nsIDocShellTreeItem> contentItem;
	treeOwner->GetPrimaryContentShell(getter_AddRefs(contentItem));
	if (!contentItem) return NS_ERROR_FAILURE;

	// QI that back to a docshell
	nsCOMPtr<nsIDocShell> DocShell;
	DocShell = do_QueryInterface(contentItem);
	if (!DocShell) return NS_ERROR_FAILURE;

	*aDocShell = DocShell.get();

	NS_IF_ADDREF(*aDocShell);
#endif
}

nsresult GaleonWrapper::Destroy ()
{
 	if (attachListener)
  		return RemoveListener ();
      	mWebBrowser = nsnull;
#if MOZILLA_MILESTONE == 81
	mEventReceiver = nsnull;
#else
	mChromeNav = nsnull;
#endif
}

nsresult GaleonWrapper::AddListener ()
{
	nsresult rv = NS_OK;

#if MOZILLA_MILESTONE == 81
	if (!mEventReceiver || !mEventListener) return NS_ERROR_FAILURE;

	nsIDOMEventListener *eventListener =
		NS_STATIC_CAST(nsIDOMEventListener *,
			       NS_STATIC_CAST(nsIDOMDragListener *, mEventListener));

	rv = mEventReceiver->AddEventListenerByIID(eventListener,
						 NS_GET_IID(nsIDOMDragListener));
	if (NS_FAILED(rv)) return NS_ERROR_FAILURE;
#else
	nsCOMPtr <nsIDOMDocument> domDoc;
	mChromeNav->GetDocument(getter_AddRefs(domDoc));
	if (!domDoc) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMEventReceiver> eventReceiver;
	eventReceiver = do_QueryInterface(domDoc);
	if (!eventReceiver) return NS_ERROR_FAILURE;

	GaleonEventListener *newListener = new GaleonEventListener();
	NS_ADDREF(newListener);
	if (!newListener) return NS_ERROR_FAILURE;

	newListener->Init (this);
	nsCOMPtr<nsIDOMEventListener> mEventListener = 
				do_QueryInterface(NS_STATIC_CAST(nsISupports *,
					NS_STATIC_CAST(nsIDOMDragListener *,
					newListener)));
	if (!mEventListener) return NS_ERROR_FAILURE;
	NS_RELEASE(newListener);

	rv = eventReceiver->AddEventListenerByIID(mEventListener,
						 NS_GET_IID(nsIDOMDragListener));
	if (NS_FAILED(rv)) return NS_ERROR_FAILURE;
#endif
}

nsresult GaleonWrapper::RemoveListener ()
{
	nsresult rv = NS_OK;

#if MOZILLA_MILESTONE == 81
	if (!mEventReceiver || !mEventListener) return NS_ERROR_FAILURE;
	
	nsIDOMEventListener *eventListener =
		NS_STATIC_CAST(nsIDOMEventListener *,
			       NS_STATIC_CAST(nsIDOMDragListener *, mEventListener));

	rv = mEventReceiver->RemoveEventListenerByIID(eventListener,
						      NS_GET_IID(nsIDOMDragListener));
	if (NS_FAILED(rv)) return NS_ERROR_FAILURE;
#else
	if (!mEventListener) return NS_ERROR_FAILURE;

	nsCOMPtr <nsIDOMDocument> domDoc;
	mChromeNav->GetDocument(getter_AddRefs(domDoc));
	if (!domDoc) return NS_ERROR_FAILURE;;

	nsCOMPtr<nsIDOMEventReceiver> eventReceiver;
	eventReceiver = do_QueryInterface(domDoc);
	if (!eventReceiver) return NS_ERROR_FAILURE;

	rv = eventReceiver->RemoveEventListenerByIID(mEventListener,
						NS_GET_IID(nsIDOMDragListener));
	if (NS_FAILED(rv)) return NS_ERROR_FAILURE;

	mEventListener = NULL;
#endif
}

nsresult GaleonWrapper::GoToHistoryIndex (PRInt16 index)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs (DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebNavigation> ContentNav = do_QueryInterface (DocShell,
								   &result);
	if (!ContentNav) return NS_ERROR_FAILURE;

	return  ContentNav->GotoIndex (index);
}

#ifndef CVS_SOURCE

nsresult GaleonWrapper::SetViewSourceMode (PRInt32 mode)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	return DocShell->SetViewMode (mode);
}

#endif

nsresult GaleonWrapper::SetZoom (float aZoom)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIContentViewer> contentViewer;	
	result = DocShell->GetContentViewer (getter_AddRefs(contentViewer));
	if (!NS_SUCCEEDED (result) || !contentViewer) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIMarkupDocumentViewer> mdv = do_QueryInterface(contentViewer,
								  &result);
	if (NS_FAILED(result) || !mdv) return NS_ERROR_FAILURE;

	return mdv->SetTextZoom (aZoom);
}

nsresult GaleonWrapper::GetZoom (float *aZoom)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIContentViewer> contentViewer;	
	result = DocShell->GetContentViewer (getter_AddRefs(contentViewer));
	if (!NS_SUCCEEDED (result) || !contentViewer) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIMarkupDocumentViewer> mdv = do_QueryInterface(contentViewer,
								  &result);
	if (NS_FAILED(result) || !mdv) return NS_ERROR_FAILURE;

	return mdv->GetTextZoom (aZoom);
}

nsresult GaleonWrapper::GetDOMDocument (nsIDOMDocument **aDOMDocument)
{
	nsresult result;

	nsCOMPtr<nsIWebBrowserFocus> focus(do_GetInterface(mWebBrowser));

	nsCOMPtr<nsIDOMWindow> DOMWindow;
	result = focus->GetFocusedWindow (getter_AddRefs(DOMWindow));
	if (NS_FAILED(result) || !DOMWindow) return NS_ERROR_FAILURE;

	return DOMWindow->GetDocument (getter_AddRefs(aDOMDocument));
}

nsresult GaleonWrapper::GetMainDOMDocument (nsIDOMDocument **aDOMDocument)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs (DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIContentViewer> contentViewer;	
	result = DocShell->GetContentViewer (getter_AddRefs(contentViewer));
	if (!NS_SUCCEEDED (result) || !contentViewer) return NS_ERROR_FAILURE;

	return contentViewer->GetDOMDocument (getter_AddRefs(aDOMDocument));
}

nsresult GaleonWrapper::SaveMainDocument (const char* filename)
{
	nsresult result;

	nsCOMPtr<nsIDOMDocument> aDOMDocument;

	GetMainDOMDocument (getter_AddRefs(aDOMDocument));
	if (NS_FAILED(result) || !aDOMDocument) return NS_ERROR_FAILURE;

	return SaveDOMDocument (aDOMDocument, filename);
}

nsresult GaleonWrapper::SaveURI (const char *url, const char *filename)
{
        nsresult result;

	nsCOMPtr<nsIWebBrowserPersist> persist(do_GetInterface(mWebBrowser));
	nsString s;
	s.AssignWithConversion(url);
	nsCOMPtr<nsIURI> linkURI;
	result = NS_NewURI(getter_AddRefs(linkURI), s);
	return persist->SaveURI(linkURI, nsnull, filename);
}

nsresult GaleonWrapper::SaveDocument (const char* filename)
{
	nsresult result;

	nsCOMPtr<nsIDOMDocument> aDOMDocument;

	GetDOMDocument (getter_AddRefs(aDOMDocument));
	if (NS_FAILED(result) || !aDOMDocument) return NS_ERROR_FAILURE;

	return SaveDOMDocument (aDOMDocument, filename);
}

nsresult GaleonWrapper::SaveDOMDocument (nsIDOMDocument *aDOMDocument, const char* filename)
{
	nsresult result;
	nsCOMPtr<nsIWebBrowserPersist> persist(do_GetInterface(mWebBrowser));
	return persist->SaveDocument (aDOMDocument, filename, nsnull);
}

#if MOZILLA_MILESTONE != 81

nsresult GaleonWrapper::SaveURLFromCache (const char *url, 
		const char *dir, const char *filename)
{
	nsresult result;

	nsCOMPtr<nsINetDataCacheManager> ndcm = 
		do_GetService(NS_NETWORK_CACHE_MANAGER_CONTRACTID, &result);

	nsCOMPtr<nsICachedNetData> cnd;
	result = ndcm->GetCachedNetData (url, nsnull, 0, 0, getter_AddRefs(cnd));
	if (NS_FAILED(result)) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIChannel> chan;
	/* FIXME: dunno if this is needed */
	nsCOMPtr<nsILoadGroup> buhLoadGroup = null_nsCOMPtr();
	result = cnd->NewChannel(buhLoadGroup, getter_AddRefs(chan));
	if (NS_FAILED(result)) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIFile> cfile;
	result = chan->GetLocalFile(getter_AddRefs(cfile));
	if (NS_FAILED(result) || !cfile) return NS_ERROR_FAILURE;

	nsCOMPtr<nsILocalFile> localdir = do_CreateInstance(NS_LOCAL_FILE_CONTRACTID);
	result = localdir->InitWithPath(dir);
	if (NS_FAILED(result)) return NS_ERROR_FAILURE;

	result = cfile->CopyTo(localdir, filename);
	if (NS_FAILED(result)) return NS_ERROR_FAILURE;

	return NS_OK;
}

#endif

nsresult GaleonWrapper::GetSHInfo (PRInt32 *count, PRInt32 *index)
{
	nsresult result;

	nsCOMPtr<nsISHistory> SessionHistory;
	result = GetSHistory (getter_AddRefs(SessionHistory));
	if (NS_FAILED(result) || ! SessionHistory) return NS_ERROR_FAILURE;

	SessionHistory->GetCount (count);
	SessionHistory->GetIndex (index);	
}

nsresult GaleonWrapper::GetSHTitleAtIndex (PRInt32 index, PRUnichar **title)
{
	nsresult result;

	nsCOMPtr<nsISHistory> SessionHistory;
	result = GetSHistory (getter_AddRefs(SessionHistory));
	if (NS_FAILED(result) || ! SessionHistory) return NS_ERROR_FAILURE;

#ifdef CVS_SOURCE
	nsCOMPtr<nsIHistoryEntry> he;
#else
	nsCOMPtr<nsISHEntry> he;
#endif
	result = SessionHistory->GetEntryAtIndex (index, PR_FALSE,
						  getter_AddRefs (he));
	if (!NS_SUCCEEDED(result) || (!he)) return NS_ERROR_FAILURE;

	result = he->GetTitle (title);
	if (!NS_SUCCEEDED(result) || (!title)) return NS_ERROR_FAILURE;
}

nsresult GaleonWrapper::Find (const char *exp, PRBool matchcase, 
			      PRBool search_backwards, PRBool* didFind)
{
	nsCOMPtr<nsIWebBrowserFind> finder(do_GetInterface(mWebBrowser));

	nsString searchString;
	searchString.AssignWithConversion(exp);  
	finder->SetSearchString(searchString.GetUnicode());
	finder->SetFindBackwards(search_backwards);
	finder->SetMatchCase(matchcase);
	return finder->FindNext(didFind);
}

nsresult GaleonWrapper::ReloadDocument ()
{
	nsresult result;

	nsCOMPtr<nsIDOMDocument> aDOMDocument;

	GetDOMDocument (getter_AddRefs(aDOMDocument));
	if (NS_FAILED(result) || !aDOMDocument) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocument> doc = do_QueryInterface(aDOMDocument);
	if(!doc) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIPresShell> presShell = getter_AddRefs(doc->GetShellAt(0));
	if(!presShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIPresContext> presContext;
	presShell->GetPresContext(getter_AddRefs(presContext));
	if(!presContext) return NS_ERROR_FAILURE;

	nsCOMPtr<nsISupports> pcContainer;
	presContext->GetContainer(getter_AddRefs(pcContainer));
	if(!pcContainer) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocShell> docshell(do_QueryInterface(pcContainer));
	if(!docshell) return NS_ERROR_FAILURE;


	nsCOMPtr<nsIWebNavigation> wn = do_QueryInterface (docshell, &result);
	if (!wn || !NS_SUCCEEDED (result)) return NS_ERROR_FAILURE;

	result = wn->Reload (nsIWebNavigation::LOAD_FLAGS_BYPASS_CACHE | 
			     nsIWebNavigation::LOAD_FLAGS_BYPASS_PROXY);
	if (!NS_SUCCEEDED (result)) return NS_ERROR_FAILURE;
}

nsresult GaleonWrapper::GetDocumentUrl (char **url)
{
	nsresult result;

	nsCOMPtr<nsIDOMDocument> aDOMDocument;

	GetDOMDocument (getter_AddRefs(aDOMDocument));
	if (NS_FAILED(result) || !aDOMDocument) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocument> doc = do_QueryInterface(aDOMDocument);
	if(!doc) return NS_ERROR_FAILURE;

	nsIURI *uri = doc->GetDocumentURL();
	uri->GetSpec (url);
}

nsresult GaleonWrapper::GetKeyEventContext (nsIDOMKeyEvent *keyEvent, WrapperKeyEventInfo *info)
{
	PRBool mod_key;
	nsresult res;

	/* if we are in a text area do not process keys */
	nsCOMPtr<nsIDOMEventTarget>  targetNode;
	res = keyEvent->GetTarget(getter_AddRefs(targetNode));
	if (NS_FAILED(res) || !targetNode) return NS_ERROR_FAILURE;
	if (targetNode)
	{
		nsCOMPtr<nsIDOMNode> node = do_QueryInterface(targetNode);
		if (node)
		{
	
			nsCOMPtr<nsIDOMHTMLElement> element;
	
			element = do_QueryInterface(node);
			if (element)
			{
				nsAutoString tag;
				element->GetTagName(tag);

				if (!tag.EqualsWithConversion("a", PR_TRUE) &&
				    !tag.EqualsWithConversion ("html",PR_TRUE))
				{
					return NS_ERROR_FAILURE;
				}
			}
		}
	}
	
	/* Get the context */

	res = GetEventContext (targetNode, &info->ctx);
	if (NS_FAILED(res)) return res;
	
	/* Get the key/modifier */

	info->modifier = 0;

	keyEvent->GetCharCode((PRUint32*)&info->key);

	if (info->key==0) 
	{
		keyEvent->GetKeyCode((PRUint32*)&info->key);
		info->modifier |= KEY_CODE;
	}

	keyEvent->GetAltKey(&mod_key);
	if (mod_key) info->modifier |= ALT_KEY;

	keyEvent->GetShiftKey(&mod_key);
	if (mod_key) info->modifier |= SHIFT_KEY;

	keyEvent->GetMetaKey(&mod_key);
	if (mod_key) info->modifier |= META_KEY;
	
	keyEvent->GetCtrlKey(&mod_key);
	if (mod_key) info->modifier |= CTRL_KEY;
}

nsresult GaleonWrapper::GetDOMAttribute (nsIDOMNode *node, char *tag,
					 char **attribute)
{
	nsresult result;
	
	nsCOMPtr<nsIDOMNamedNodeMap> attributes;
	result = node->GetAttributes(getter_AddRefs (attributes));
	if (!NS_SUCCEEDED (result) || !attributes) return NS_ERROR_FAILURE;

	nsAutoString attr; 

	attr.AssignWithConversion (tag);
                                        
	nsCOMPtr<nsIDOMNode> attrNode;
	result = attributes->GetNamedItem (attr, getter_AddRefs (attrNode));
	if (!NS_SUCCEEDED (result) || !attrNode) return NS_ERROR_FAILURE;

	nsAutoString nodeValue;
			
	result = attrNode->GetNodeValue (nodeValue);
	if (!NS_SUCCEEDED (result)) return NS_ERROR_FAILURE;
                                                
	char *cstr = nodeValue.ToNewCString();
	*attribute = g_strdup (cstr);
			
	nsMemory::Free (cstr);
}

nsresult GaleonWrapper::GetEventContext (nsCOMPtr<nsIDOMEventTarget> EventTarget, WrapperContextInfo *info)
{
	nsresult result;
	
	info->img = info->link = NULL;

	nsCOMPtr<nsIDOMNode> node = do_QueryInterface(EventTarget);
	if (!node) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMDocument> domDoc;
	result = node->GetOwnerDocument(getter_AddRefs(domDoc));
	if (!NS_SUCCEEDED (result) || !domDoc) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocument> doc = do_QueryInterface(domDoc);
	if(!doc) return NS_ERROR_FAILURE;

	nsIURI *baseURI;
	result = doc->GetBaseURL(baseURI);
	if (NS_FAILED(result) || !baseURI) return NS_ERROR_FAILURE;
   
	nsString mime;  
	doc->GetContentType(mime);  
	if (mime.EqualsWithConversion ("text/xul"))
		return NS_ERROR_FAILURE;

	info->context = CONTEXT_NONE;
	nsCOMPtr<nsIDOMHTMLElement> element;
	do 
	{
		PRUint16 type;
		node->GetNodeType(&type);

		element = do_QueryInterface(node);
		if (element)
		{
			nsAutoString tag;
			element->GetTagName(tag);

			if (tag.EqualsWithConversion("input", PR_TRUE))
			{
				info->context |= CONTEXT_INPUT;
			}
			else if (tag.EqualsWithConversion("img", PR_TRUE))
			{
				info->context |= CONTEXT_IMAGE;

				char *src;
				result = GetDOMAttribute (node, g_strdup("src"),
							  &src);
				if (NS_FAILED(result) || !src)
					return NS_ERROR_FAILURE;
			
				result = baseURI->Resolve(src, &info->img);
				if (NS_FAILED(result) || !info->img)
				return NS_ERROR_FAILURE;

			}
			else
			{
				info->context |= CONTEXT_OTHER;
			}

			nsCOMPtr<nsIDOMNamedNodeMap> attributes;
			node->GetAttributes(getter_AddRefs(attributes));
			if (attributes)
			{
				nsCOMPtr<nsIDOMNode> hrefNode;
				nsAutoString href; 
				href.AssignWithConversion("href");
				attributes->GetNamedItem(href,
						       getter_AddRefs(hrefNode));
				if (hrefNode)
				{
					info->context |= CONTEXT_LINK;

					char *href;
					result = GetDOMAttribute (node,
							g_strdup("href"), &href);
					if (NS_FAILED(result) || !href)
						return NS_ERROR_FAILURE;
				
					result = baseURI->Resolve(href, &info->link);
					if (NS_FAILED(result) || !info->link)
						return NS_ERROR_FAILURE;
					break;
				}
			}

		}
			nsCOMPtr<nsIDOMNode> parentNode;
			node->GetParentNode(getter_AddRefs(parentNode));
		
			if (!parentNode)
			{
				node = nsnull;
				info->context |= CONTEXT_DOCUMENT;
				break;
			}
			node = parentNode;
	} 
	while (node);

	/* Is page framed ? */
	PRBool framed;
	IsPageFramed (&framed);
	info->framed_page = (gboolean)framed;
	
	return NS_OK;
}

nsresult GaleonWrapper::GetMouseEventContext (nsIDOMMouseEvent *event, WrapperMouseEventInfo *info)
{
	nsresult result;
	nsIDOMMouseEvent *aMouseEvent = (nsIDOMMouseEvent*)event;

	aMouseEvent->GetButton ((PRUint16*)&info->button);

	/* be sure we are not clicking on the scroolbars */

	nsCOMPtr<nsIDOMEventTarget> OriginalTarget;
	result = aMouseEvent->GetOriginalTarget(getter_AddRefs(OriginalTarget));
	if (NS_FAILED(result) || !OriginalTarget) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMNode> OriginalNode = do_QueryInterface(OriginalTarget);
	if (!OriginalNode) return NS_ERROR_FAILURE;

	nsString nodename;
	OriginalNode->GetNodeName(nodename);

	if (nodename.EqualsWithConversion ("xul:thumb") ||
	    nodename.EqualsWithConversion ("xul:slider"))
		return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMEventTarget> EventTarget;
	result = aMouseEvent->GetTarget(getter_AddRefs(EventTarget));
	if (NS_FAILED(result) || !EventTarget) return NS_ERROR_FAILURE;

	result = GetEventContext (EventTarget, &info->ctx);
	if (NS_FAILED(result)) return result;

	/* Get the modifier */

	PRBool mod_key;

	info->modifier = 0;

	aMouseEvent->GetAltKey(&mod_key);
	if (mod_key) info->modifier |= ALT_KEY;

	aMouseEvent->GetShiftKey(&mod_key);
	if (mod_key) info->modifier |= SHIFT_KEY;

	aMouseEvent->GetMetaKey(&mod_key);
	if (mod_key) info->modifier |= META_KEY;
	
	aMouseEvent->GetCtrlKey(&mod_key);
	if (mod_key) info->modifier |= CTRL_KEY;

	return NS_OK;
}

nsresult  GaleonWrapper::CopyHistoryTo (GaleonWrapper *dest)
{
	nsresult result;
	int count,index;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebNavigation> wn_src = do_QueryInterface (DocShell,
							       &result);
	if (!wn_src) return NS_ERROR_FAILURE;
	
	nsCOMPtr<nsISHistory> h_src;
	result = wn_src->GetSessionHistory (getter_AddRefs (h_src));
	if (!NS_SUCCEEDED(result) || (!h_src)) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocShell> destDocShell;
	result = dest->GetDocShell (getter_AddRefs(destDocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebNavigation> wn_dest = do_QueryInterface (destDocShell,
								&result);
	if (!wn_dest) return NS_ERROR_FAILURE;
	
	nsCOMPtr<nsISHistory> h_dest;
	result = wn_dest->GetSessionHistory (getter_AddRefs (h_dest));
	if (!NS_SUCCEEDED (result) || (!h_dest)) return NS_ERROR_FAILURE;
#ifdef CVS_SOURCE
	nsCOMPtr<nsISHistoryInternal> hi_dest = do_QueryInterface (h_dest);
	if (!hi_dest) return NS_ERROR_FAILURE;
#endif

	h_src->GetCount (&count);
	h_src->GetIndex (&index);

	if (count) {
#ifdef CVS_SOURCE
		nsCOMPtr<nsIHistoryEntry> he;
		nsCOMPtr<nsISHEntry> she;
#else		
		nsCOMPtr<nsISHEntry> he;
#endif

		for (PRInt32 i = 0; i < count; i++) {

			result = h_src->GetEntryAtIndex (i, PR_FALSE,
							 getter_AddRefs (he));
			if (!NS_SUCCEEDED(result) || (!he))
				return NS_ERROR_FAILURE;
#ifdef CVS_SOURCE
			she = do_QueryInterface (he);
			if (!she) return NS_ERROR_FAILURE;

			result = hi_dest->AddEntry (she, PR_TRUE);
			if (!NS_SUCCEEDED(result) || (!she))
#else
			result = h_dest->AddEntry (he, PR_TRUE);
			if (!NS_SUCCEEDED(result) || (!he))
#endif
				return NS_ERROR_FAILURE;
		}

		result = wn_dest->GotoIndex(index);
		if (!NS_SUCCEEDED(result)) return NS_ERROR_FAILURE;
	}
}


nsresult GaleonWrapper::SetSitePermission (gboolean block, PRInt32 type)
{
	nsresult result;

#ifdef CVS_SOURCE
	nsCOMPtr<nsIDOMDocument> DOMDocument;
	result = GetDOMDocument (getter_AddRefs(DOMDocument));
	if (NS_FAILED(result) || !DOMDocument) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocument> doc = do_QueryInterface (DOMDocument);
	if (!doc) return NS_ERROR_FAILURE;
	
	nsIURI *uri = doc->GetDocumentURL ();
	char *url;
	uri->GetSpec (&url);
	
	nsCOMPtr<nsIPermissionManager> 
		permissionManager = do_CreateInstance (PERMISSIONMANAGER_ID);
	
	result = permissionManager->Add (url,block ? PR_TRUE : PR_FALSE, type);
#else
	nsCOMPtr<nsIWebBrowserFocus> focus (do_GetInterface(mWebBrowser));

	nsCOMPtr<nsIDOMWindow> DOMWindow;
	result = focus->GetFocusedWindow (getter_AddRefs(DOMWindow));
	if (NS_FAILED(result) || !DOMWindow) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMWindowInternal> 
			DOMWindowInternal = do_QueryInterface (DOMWindow);
	if (!DOMWindowInternal) return NS_ERROR_FAILURE;	

	nsCOMPtr<nsICookieViewer> 
			cookieviewer = do_CreateInstance (COOKIEVIEWER_ID);
	result = cookieviewer->AddPermission (DOMWindowInternal,
					      block ? PR_TRUE : PR_FALSE, 
					      type);
#endif
	return result;
}

nsresult GaleonWrapper::ForceCharacterSet (char *charset) 
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIContentViewer> contentViewer;	
	result = DocShell->GetContentViewer (getter_AddRefs(contentViewer));
	if (!NS_SUCCEEDED (result) || !contentViewer) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIMarkupDocumentViewer> mdv = do_QueryInterface(contentViewer,
								  &result);
	if (NS_FAILED(result) || !mdv) return NS_ERROR_FAILURE;

	nsAutoString charset_str;
	charset_str.AssignWithConversion (charset);
	result = mdv->SetForceCharacterSet (charset_str.ToNewUnicode());

	return result;
}

nsresult GaleonWrapper::IsPageFramed (PRBool *Framed)
{
	nsCOMPtr<nsIDocShellTreeItem> browserAsItem;
	browserAsItem = do_QueryInterface(mWebBrowser);
	if (!browserAsItem) return NS_ERROR_FAILURE;

	// get the owner for that item
	nsCOMPtr<nsIDocShellTreeOwner> treeOwner;
	browserAsItem->GetTreeOwner(getter_AddRefs(treeOwner));
	if (!treeOwner) return NS_ERROR_FAILURE;

	// get the primary content shell as an item
	nsCOMPtr<nsIDocShellTreeItem> contentItem;
	treeOwner->GetPrimaryContentShell(getter_AddRefs(contentItem));
	if (!contentItem) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocShellTreeNode> contentNode;	
	contentNode = do_QueryInterface (contentItem);
	if (!contentNode) return NS_ERROR_FAILURE;

	PRInt32 childCount;
	contentNode->GetChildCount (&childCount);

	*Framed = (childCount>0);
}

GaleonEventListener::GaleonEventListener(void)
{
	NS_INIT_REFCNT();
	wrapper = NULL;
}

GaleonEventListener::~GaleonEventListener(void)
{
}


NS_IMPL_ADDREF(GaleonEventListener)
NS_IMPL_RELEASE(GaleonEventListener)
NS_INTERFACE_MAP_BEGIN(GaleonEventListener)
  NS_INTERFACE_MAP_ENTRY_AMBIGUOUS(nsISupports, nsIDOMDragListener)
  NS_INTERFACE_MAP_ENTRY_AMBIGUOUS(nsIDOMEventListener, nsIDOMDragListener)
  NS_INTERFACE_MAP_ENTRY(nsIDOMDragListener)
NS_INTERFACE_MAP_END

void
GaleonEventListener::Init(GaleonWrapper *galeon_wrapper)
{
	wrapper = galeon_wrapper;
}

NS_IMETHODIMP
GaleonEventListener::HandleEvent(nsIDOMEvent* aEvent)
{
	return NS_OK;
}

NS_IMETHODIMP GaleonEventListener::DragEnter(nsIDOMEvent* aMouseEvent)
{
	return NS_OK;
}
NS_IMETHODIMP GaleonEventListener::DragExit(nsIDOMEvent* aMouseEvent)
{
	return NS_OK;
}
NS_IMETHODIMP GaleonEventListener::DragDrop(nsIDOMEvent* aMouseEvent)
{
	return NS_OK;
}
NS_IMETHODIMP GaleonEventListener::DragGesture(nsIDOMEvent* aMouseEvent)
{
	gchar *old_dragging_link;
	WrapperMouseEventInfo *info;
	gchar *url = NULL;

	info = g_new0 (WrapperMouseEventInfo,1);

	mozilla_get_mouse_event_info(wrapper->embed, aMouseEvent, info);


	if (info->ctx.context & CONTEXT_LINK)
	{
		url = info->ctx.link;
	}
	else if (info->ctx.context & CONTEXT_IMAGE)
	{	
		url = info->ctx.img;
	}
	
	if (url)
	{
		if (!link_drag_types_tl)
			link_drag_types_tl = gtk_target_list_new 
				(link_drag_types, link_drag_types_num_items);

		old_dragging_link = (gchar*)gtk_object_get_data (
					GTK_OBJECT (wrapper->mGtkMozEmbed),
					"dragging_link");
		if (old_dragging_link) g_free (old_dragging_link);
		gtk_object_set_data (GTK_OBJECT(wrapper->mGtkMozEmbed),
				     "dragging_link", g_strdup (url));
		gtk_drag_begin (GTK_WIDGET(wrapper->mGtkMozEmbed),
				link_drag_types_tl, 
				(GdkDragAction)(GDK_ACTION_COPY | 
						GDK_ACTION_LINK),
				1, NULL);
	}

	g_free (info);

	return NS_OK;
}

NS_IMETHODIMP GaleonEventListener::DragOver(nsIDOMEvent* aMouseEvent)
{
	return NS_OK;
}

nsresult GaleonWrapper::GetFaviconURL (char **favicon_url)
{
	nsresult result;
	PRUint32 links_count;
	char *relstr;

	nsCOMPtr<nsIDOMDocument> aDOMDocument;
	GetMainDOMDocument (getter_AddRefs(aDOMDocument));
	if (NS_FAILED(result) || !aDOMDocument) return NS_ERROR_FAILURE;

	nsString strname;
	strname.AssignWithConversion("LINK");

	nsCOMPtr<nsIDOMNodeList> aLinks;
	result = aDOMDocument->GetElementsByTagName (strname, 
						     getter_AddRefs (aLinks));
	if (NS_FAILED (result)) return NS_ERROR_FAILURE;

	result = aLinks->GetLength (&links_count);
	if (NS_FAILED (result)) return NS_ERROR_FAILURE;

	for (PRUint32 i = 0; i < links_count; i++)
	{
		nsCOMPtr<nsIDOMNode> aLink;
		result = aLinks->Item (i, getter_AddRefs (aLink));
		if (NS_FAILED (result)) return NS_ERROR_FAILURE;
		
		result = GetDOMAttribute (aLink, "rel", &relstr);
		if (NS_FAILED (result)) return NS_ERROR_FAILURE;
		
		if (relstr == NULL) continue;

		if (strcmp(relstr, "SHORTCUT ICON") == 0)
		{
			char *link = NULL;
			
			g_free (relstr);
			
			result = GetDOMAttribute (aLink, "href", &link);
			if (NS_FAILED (result)) return NS_ERROR_FAILURE;
			
			if (link == NULL)
			{
				*favicon_url = NULL;
				return NS_OK;
			}
			
			gchar *url = NULL;
			
			nsCOMPtr<nsIDocument> doc = 
				do_QueryInterface (aDOMDocument);
			if(!doc) return NS_ERROR_FAILURE;
			
			nsIURI *uri = doc->GetDocumentURL();
			result = uri->Resolve (link, favicon_url);
			if (NS_FAILED (result)) return NS_ERROR_FAILURE;
			
			g_free (link);
			return NS_OK;
		}
	}

	return NS_ERROR_FAILURE;
}

