/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * This file contains all the global types used within Galeon.
 */

/* forward declarations of types */
typedef struct _GaleonWindow GaleonWindow;
typedef struct _GaleonEmbed  GaleonEmbed;
typedef struct _BookmarkItem BookmarkItem;
typedef struct _FindDialog   FindDialog; /* FIXME: remove */

/**
 * GaleonEmbed: this is the type of a single web view (e.g. a tab):
 * this is effectively Galeon's wrapping of a GtkMozEmbed with extra
 * state information.
 */ 
struct _GaleonEmbed
{
	guint32 magic;

	/* parent window in which this embed is visible */
	GaleonWindow *parent_window;

	/* embedding widget */
	GtkMozEmbed *mozEmbed;

	/* C++ helper, a "GaleonWrapper" */
        gpointer wrapper;

	/* current site location and page title, if any */
	gchar *site_location;
	gchar *site_title;

	gint load_started; /* count of currently active connections */
	gint loadPercent;
	gint bytesLoaded;
	gint maxBytesLoaded;
	const gchar *statusMessage;

	/* true if this is the "active_embed" of the parent window */
	gboolean is_active : 1;

	/* is this browser actually visible */
	gboolean is_visible : 1;

	/* is this a popup browser ? */
	gboolean is_popup : 1;

	/* is this embed in the process of closing? */
	gboolean is_closing : 1;

	/* ignore mime processing of next url */
	gboolean mime_ignore_once : 1;

	/* FIXME: refactor */
	FindDialog *find_dialog;
};

/**
 * GaleonWindow: this is the type of Galeon windows, i.e. a wrapping of
 * the toplevel GnomeApp window. A single window can contain many 
 * GaleonEmbeds.
 */
struct _GaleonWindow
{
	guint32 magic;

	/* window parts */
	GtkWidget *WMain;
	GaleonEmbed *active_embed;
	GList *embed_list; /* list of GaleonEmbeds */
	char *tempMessage;
	gint visible_embeds; /* number of visible GaleonEmbeds, used to
			      * track whether or not the window is visible */

	/* number of embeds which are running (for spinner) */
	gint n_running;

	/* assigned size */
	gboolean set_size : 1;

	/* still running the activity bar? */
	gboolean progress_timeout : 1;

	/* have we set the encodings menu yet? */
	gboolean encodings_menu_set : 1;

	/* GUI components */
	GtkWidget *notebook;

	/* Toolbar components */
	GtkWidget *spinner; /* the GnomeAnimator spinner */
	GtkWidget *location_dockitem;
	GtkWidget *main_dockitem;
	GtkWidget *BNew;
	GtkWidget *BBack;
	GtkWidget *BUp;
	GtkWidget *BForward;
	GtkWidget *BRefresh;
	GtkWidget *BStop;
	GtkWidget *BStart;
	GtkWidget *zoom_spin;
	GtkWidget *toolbar_gnomeentry;
	GtkWidget *toolbar_entry;
	GtkWidget *go_toolbar;
	GtkWidget *dock;

	GtkWidget *appbar;
	GtkWidget *menubar;
	GtkWidget *view_toolbar;
	GtkWidget *view_statusbar;
	GtkWidget *view_menubar;
	GtkWidget *view_fullscreen;
	GtkWidget *view_bookmarks_dock;
	GtkWidget *view_history_dock;
	GtkWidget *drag_pixmap;
	GtkWidget *load_images_always;
	GtkWidget *load_images_from_current_server_only;
	GtkWidget *load_images_never;
	GtkWidget *animate_always;
	GtkWidget *animate_once_through;
	GtkWidget *animate_never;
	GtkWidget *use_own_fonts;
	GtkWidget *use_own_colors;
	GtkWidget *enable_java;
	GtkWidget *enable_javascript;
	GtkWidget *enable_proxy;
	GtkWidget *use_galeon_mime_handling;

	/* dialogs */
	GtkWidget *print_dialog;

	/* list of toolbars */
	GList *bookmarks_toolbars; 
	GtkTooltips *bookmarks_tooltips;

	/* saved window geometry for fullscreen windows */
	gint x, y, width, height;
};

/**
 * Magic numbers used to detect if the structure has been accidentally 
 * free'd or corrupted
 */
#define GALEON_WINDOW_MAGIC   (0x87de130f)
#define GALEON_EMBED_MAGIC    (0x45a0ebed)

/**
 * Checking macros used to validate window and embed pointers and
 * their magic. At some point we'll probably do all of this with
 * GObjects, but for now this is it. This will introduce some (very
 * small) overhead, and we should probably provide a #ifdef DEBUG
 * which disables these (around v1.0).
 */

#define return_if_not_window(window) \
  g_return_if_fail (window != NULL); \
  g_return_if_fail (window->magic == GALEON_WINDOW_MAGIC);

#define return_val_if_not_window(window, val) \
  g_return_val_if_fail (window != NULL, val); \
  g_return_val_if_fail (window->magic == GALEON_WINDOW_MAGIC, val);

#define return_if_not_embed(embed) \
  g_return_if_fail (embed != NULL); \
  g_return_if_fail (embed->magic == GALEON_EMBED_MAGIC);

#define return_val_if_not_embed(embed, val) \
  g_return_val_if_fail (embed != NULL, val); \
  g_return_val_if_fail (embed->magic == GALEON_EMBED_MAGIC, val);

/**
 * KeyCode: key pressed char codes, taken from mozilla
 */
typedef enum
{
	DOM_VK_CANCEL = 3,
	DOM_VK_BACK_SPACE = 8,
	DOM_VK_TAB = 9,
	DOM_VK_CLEAR = 12,
	DOM_VK_RETURN = 13,
	DOM_VK_ENTER = 14,
	DOM_VK_SHIFT = 16,
	DOM_VK_CONTROL = 17,
	DOM_VK_ALT = 18,
	DOM_VK_PAUSE = 19,
	DOM_VK_CAPS_LOCK = 20,
	DOM_VK_ESCAPE = 27,
	DOM_VK_SPACE = 32,
	DOM_VK_PAGE_UP = 33,
	DOM_VK_PAGE_DOWN = 34,
	DOM_VK_END = 35,
	DOM_VK_HOME = 36,
	DOM_VK_LEFT = 37,
	DOM_VK_UP = 38,
	DOM_VK_RIGHT = 39,
	DOM_VK_DOWN = 40,
	DOM_VK_PRINTSCREEN = 44,
	DOM_VK_INSERT = 45,
	DOM_VK_DELETE = 46,
	DOM_VK_0 = 48,
	DOM_VK_1 = 49,
	DOM_VK_2 = 50,
	DOM_VK_3 = 51,
	DOM_VK_4 = 52,
	DOM_VK_5 = 53,
	DOM_VK_6 = 54,
	DOM_VK_7 = 55,
	DOM_VK_8 = 56,
	DOM_VK_9 = 57,
	DOM_VK_SEMICOLON = 59,
	DOM_VK_EQUALS = 61,
	DOM_VK_A = 65,
	DOM_VK_B = 66,
	DOM_VK_C = 67,
	DOM_VK_D = 68,
	DOM_VK_E = 69,
	DOM_VK_F = 70,
	DOM_VK_G = 71,
	DOM_VK_H = 72,
	DOM_VK_I = 73,
	DOM_VK_J = 74,
	DOM_VK_K = 75,
	DOM_VK_L = 76,
	DOM_VK_M = 77,
	DOM_VK_N = 78,
	DOM_VK_O = 79,
	DOM_VK_P = 80,
	DOM_VK_Q = 81,
	DOM_VK_R = 82,
	DOM_VK_S = 83,
	DOM_VK_T = 84,
	DOM_VK_U = 85,
	DOM_VK_V = 86,
	DOM_VK_W = 87,
	DOM_VK_X = 88,
	DOM_VK_Y = 89,
	DOM_VK_Z = 90,
	DOM_VK_a = 97,
	DOM_VK_b = 98,
	DOM_VK_c = 99,
	DOM_VK_d = 100,
	DOM_VK_e = 101,
	DOM_VK_f = 102,
	DOM_VK_g = 103,
	DOM_VK_h = 104,
	DOM_VK_i = 105,
	DOM_VK_j = 106,
	DOM_VK_k = 107,
	DOM_VK_l = 108,
	DOM_VK_m = 109,
	DOM_VK_n = 110,
	DOM_VK_o = 111,
	DOM_VK_p = 112,
	DOM_VK_q = 113,
	DOM_VK_r = 114,
	DOM_VK_s = 115,
	DOM_VK_t = 116,
	DOM_VK_u = 117,
	DOM_VK_v = 118,
	DOM_VK_w = 119,
	DOM_VK_x = 120,
	DOM_VK_y = 121,
	DOM_VK_z = 122,
	DOM_VK_NUMPAD0 = 96,
	DOM_VK_NUMPAD1 = 97,
	DOM_VK_NUMPAD2 = 98,
	DOM_VK_NUMPAD3 = 99,
	DOM_VK_NUMPAD4 = 100,
	DOM_VK_NUMPAD5 = 101,
	DOM_VK_NUMPAD6 = 102,
	DOM_VK_NUMPAD7 = 103,
	DOM_VK_NUMPAD8 = 104,
	DOM_VK_NUMPAD9 = 105,
	DOM_VK_MULTIPLY = 106,
	DOM_VK_ADD = 107,
	DOM_VK_SEPARATOR = 108,
	DOM_VK_SUBTRACT = 109,
	DOM_VK_DECIMAL = 110,
	DOM_VK_DIVIDE = 111,
	DOM_VK_F1 = 112,
	DOM_VK_F2 = 113,
	DOM_VK_F3 = 114,
	DOM_VK_F4 = 115,
	DOM_VK_F5 = 116,
	DOM_VK_F6 = 117,
	DOM_VK_F7 = 118,
	DOM_VK_F8 = 119,
	DOM_VK_F9 = 120,
	DOM_VK_F10 = 121,
	DOM_VK_F11 = 122,
	DOM_VK_F12 = 123,
	DOM_VK_F13 = 124,
	DOM_VK_F14 = 125,
	DOM_VK_F15 = 126,
	DOM_VK_F16 = 127,
	DOM_VK_F17 = 128,
	DOM_VK_F18 = 129,
	DOM_VK_F19 = 130,
	DOM_VK_F20 = 131,
	DOM_VK_F21 = 132,
	DOM_VK_F22 = 133,
	DOM_VK_F23 = 134,
	DOM_VK_F24 = 135,
	DOM_VK_NUM_LOCK = 144,
	DOM_VK_SCROLL_LOCK = 145,
	DOM_VK_COMMA = 188,
	DOM_VK_PERIOD = 190,
	DOM_VK_SLASH = 191,
	DOM_VK_BACK_QUOTE = 192,
	DOM_VK_OPEN_BRACKET = 219,
	DOM_VK_BACK_SLASH = 220,
	DOM_VK_CLOSE_BRACKET = 221,
	DOM_VK_QUOTE = 222,
	DOM_VK_META = 224,
	DOM_VK_PLUS = 43,
	DOM_VK_MINUS = 45,

	DOM_VK_LEFT_2 = 65361,   /* added */
	DOM_VK_KP_RIGHT = 65363, /* added */
	DOM_VK_KP_LEFT = 65430   /* added */
} KeyCode;

/**
 * FindDialog: structure representing a "find in document" dialog
 */
struct _FindDialog
{
	GtkWidget *dialog;
	GtkWidget *entry;
	GtkWidget *gnome_entry;
	GtkWidget *case_checkbutton;
	GtkWidget *prev_button;
	GtkWidget *next_button;
	gboolean string_changed;
};

/**
 * KeyModifier: key modifiers
 */
typedef enum
{
	ALT_KEY   = (1 << 0),
	CTRL_KEY  = (1 << 1),
	SHIFT_KEY = (1 << 2),
	META_KEY  = (1 << 3),
	KEY_CODE  = (1 << 4)
} KeyModifier;

/**
 * BookmarkTitle: type of behaviour when making a new bookmark
 */
typedef enum 
{ 
	BOOKMARK_PAGE_TITLE,
	BOOKMARK_ASK_TITLE
} BookmarkTitle;

/**
 * StartPageType: start site type
 */
typedef enum
{
	STARTPAGE_HOME,
	STARTPAGE_LAST,
	STARTPAGE_BLANK,
} StartPageType;

/**
 * HomePageType: home page type
 */
typedef enum
{
	HOMEPAGE_URL,
	HOMEPAGE_MYPORTAL
} HomePageType;

/**
 * MimeAction: types of actions to take for opening files
 */
typedef enum
{
	MIME_UNKNOWN,
	MIME_LEAVE_TO_MOZILLA,
	MIME_SAVE_TO_DISK,
	MIME_RUN_PROGRAM,      /* should run the GNOME association */
	MIME_ASK_ACTION
} MimeAction;

/**
 * AutoResume: how to handling resuming crashed sessions
 */
typedef enum
{
	RESUME_NEVER,
	RESUME_ASK,
	RESUME_ALWAYS
} AutoResume;

/**
 * NSItemType: netscape bookmark item type
 */
typedef enum
{
	NS_SITE,
	NS_FOLDER,
	NS_FOLDER_END,
	NS_SEPARATOR,
	NS_UNKNOWN
} NSItemType;

/**
 * ImageLoadingType: various image loading behaviours
 */
typedef enum
{
	IMAGE_LOADING_ALWAYS                   = 0,
	IMAGE_LOADING_FROM_CURRENT_SERVER_ONLY = 1,
	IMAGE_LOADING_NEVER                    = 2
} ImageLoadingType;

/**
 * AnimateType: various image animation behaviours
 */
typedef enum
{
	ANIMATE_ALWAYS       = 0,
	ANIMATE_ONCE_THROUGH = 1,
	ANIMATE_NEVER        = 2
} AnimateType;

/**
 * TabbedStatus: status of a tab (reflected in its colour)
 */
typedef enum
{
	TAB_NORMAL,
	TAB_NEW,
	TAB_LOADING
} TabbedStatus;

/**
 * ViewState: destination for loading URLs
 */
typedef enum
{
	VIEWSTATE_NONE,
	VIEWSTATE_DEFAULT,
	VIEWSTATE_WINDOW,
	VIEWSTATE_TAB
} ViewState;

/**
 * ContextMenuType: various types of context menu
 */
typedef enum
{
	CONTEXT_NONE     = 0,
	CONTEXT_DEFAULT  = (1 << 1),
	CONTEXT_LINK     = (1 << 2),
	CONTEXT_IMAGE    = (1 << 3),
	CONTEXT_DOCUMENT = (1 << 4),
	CONTEXT_INPUT    = (1 << 5),
	CONTEXT_OTHER    = (1 << 6),
	CONTEXT_XUL      = (1 << 7),
} ContextMenuType;

/**
 * MimeItem: a particular MIME type binding
 */
typedef struct
{
	gchar *name; 
	MimeAction action;
} MimeItem;

/**
 * HistoryItem: an item in the history
 */
typedef struct
{
	gchar *title;
	gchar *url;
	GTime first;
	GTime last;
	gint visits;
} HistoryItem;

/**
 * PixmapData: a single pixmap (stored on the X server)
 */
typedef struct {
	GdkPixmap *pixmap;
	GdkBitmap *mask;
} PixmapData;

/**
 * BookmarkType: various types of bookmarks
 */ 
typedef enum
{
	BM_CATEGORY,
	BM_SITE,
	BM_SEPARATOR,
	BM_AUTOBOOKMARKS
} BookmarkType;

/**
 * BookmarkItem: a single entry in bookmarks
 */
struct _BookmarkItem
{
	/* the type of this bookmark */
	BookmarkType type;

	/* the user provided name of the bookmark */
	gchar *name;

	/* the site it points at */
	gchar *url;

	/* nickname used as a shorthand for this bookmark */
	gchar *nick;
	
	/* any user provided comments about the bookmark */
	gchar *notes; 

	/* the pixmap file used to represent it */
	gchar *pixmap_file;

	/* the in-memory binary data of the pixmap */
	PixmapData *pixmap_data;

	/* the list of children if this is a folder */
	GList *list;

	/* a handle to find the bookmark in the editor dialog */
	GtkCTreeNode *tree_item; 

	/* parent category (NULL for root bookmark) */
	BookmarkItem *parent; 

	/* if TRUE and this is a folder, create a toolbar from it */
	gboolean create_toolbar : 1; 

	/* if TRUE, we add this bookmark to the document context menu */
	gboolean create_context_menu : 1;

	/* if TRUE and this is a folder, then it is shown open in the editor */
	gboolean expanded : 1; 

	/* if not NULL, then this is an alias. The pointed bookmark can be an
	 * alias too */
	BookmarkItem *alias_of;

	/* Points to the first alias of this bookmark, if any. If the 
	 * bookmark is an alias, then it points to the next alias */
	BookmarkItem *alias;

	/* a unique identifier */
	glong id;

	/* time info (if they are 0, then they should be ignored) */
	GTime time_added;
	GTime time_modified;
	GTime time_visited;

};

/**
 * BookmarkDragItem: the representation of a bookmark when it's passing
 * through drag-and-drop.
 */
typedef struct
{
	GaleonWindow *window;
	BookmarkItem *bookmark;
	GtkToolbar   *toolbar;
} BookmarkDragItem;

/**
 * BookmarksEditorControls: representation of a bookmarks editor.
 */
typedef struct
{
	/* the root folder being edited here */
	BookmarkItem *root_bookmark;

	/* the widgets of the editor */
	GtkWidget *dialog;
	GtkWidget *toolbar;
	GtkWidget *ctree;
	GtkWidget *vpane;
	GtkWidget *edit_frame;
	GtkWidget *name_entry;
	GtkWidget *url_entry;
	GtkWidget *nick_entry;
	GtkWidget *pixmap_entry;
	GtkWidget *pixmap_file_entry;
	GtkWidget *create_toolbar_toggle;
	GtkWidget *create_context_menu_toggle;
	GtkWidget *notes_text;
	GtkWidget *time_added_entry;
	GtkWidget *time_modified_entry;
	GtkWidget *time_visited_entry;

	/* FIXME: comment */
	BookmarkItem *selection;
	BookmarkItem *last_pressed; /* for dnd */
	gboolean lock_edit_controls;
	gboolean dirty;
	void (*save_function) (void);
	GaleonWindow *window;

} BookmarksEditorControls;

/**
 * BookmarksEditorControls_type types of bookmark editor dialogs
 */
typedef enum {
	STANDARD_BOOKMARKS_EDITOR,
	DOCKED_BOOKMARKS_EDITOR,
	COMPACT_BOOKMARKS_EDITOR,
} BookmarksEditorControls_type;		

/**
 * BookmarksDnDTargetType: various types that can be dropped
 */
enum BookmarksDnDTargetType
{
	DND_TARGET_GALEON_BOOKMARK, 
	DND_TARGET_GALEON_URL,
	DND_TARGET_NETSCAPE_URL, 
	DND_TARGET_TEXT_URI_LIST,
	DND_TARGET_STRING
};

/**
 * CookieBase: FIXME?
 */
typedef struct
{
#ifndef CVS_SOURCE
	gchar *number;
#endif	
	gchar *type;
	gchar *domain;
	gboolean remove;
} CookieBase;

/**
 * Cookie: the type of cookies
 */
typedef struct
{
	CookieBase base;
	gchar *name;
	gchar *value;
	gchar *path;
	gchar *secure;
	gchar *expire;
} Cookie;

/**
 * Password: a password manager entry
 */

#ifdef CVS_SOURCE
typedef struct {
	gchar *host;
	gchar *username;
	
	gboolean remove;
} Password;

/**
 * PasswordType: To distinguish actual passwords from blocked password sites
 */
  
typedef enum {
	PASSWORD,
	REJECT
} PasswordType;
#endif

/**
 * CookiePage: For selecting which page in the Cookie manager is active
 */

typedef enum {
	COOKIES = 0,
	COOKIESITES = 1,
	IMAGESITES = 2,
	PASSWORDS = 3,
	PASSWORDSITES = 4
} CookiePage;

/**
 * MimeAskActionDialog: the representation of dialogs used to ask
 * about actions on MIME types
 */
typedef struct
{
	gchar *url;
	gchar *mime_type;
	GaleonEmbed *embed;
	GladeXML *gxml;
} MimeAskActionDialog;

/**
 * FileCallbackFunc: the type of functions used in file callbacks
 */
typedef void (*FileCallbackFunc) (const gchar *file, gpointer data);
