/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"

typedef struct
{
	char  *host;
	GtkCTreeNode *node;
}
HistoryCategory;


/**
 * The history hash table. A hash table is used because you have to make 
 * an url lookup in history each time you try to add or update a record
 */
GHashTable *history = NULL;

/**
 * The hosts hash table. This maps from server names (strings) to 
 * HistoryCategory structures.
 */
GHashTable *hosts = NULL;

/**
 * is the history dirty, i.e. needs saving to disk
 */
gboolean history_dirty = FALSE;

/**
 * The history dialog
 */
GtkWidget *    dHistory = NULL;

/**
 * The history list
 */
GtkCTree *     clHistory = NULL;

/**
 * Sorting info
 */
GtkSortType history_sort_type = GTK_SORT_ASCENDING;
gint history_sort_column = 0;

/* Search info */
gchar *search_text = NULL;
gboolean host_group = TRUE;
gint search_time = 0;

/* change this to 1 if you have problems with your history 
 * becoming corrupted */
#define USE_OLD_HISTORY_SAVE 0

/* Private prototypes */

GtkWidget *     history_create_recent_menu_item  (HistoryItem *hi, gint num,
						  GtkWidget *menu);

void            history_save_item                (gpointer key, gpointer value,
						  gpointer user_data);

#if USE_OLD_HISTORY_SAVE
void            history_save_ordering            (void);
#else
void            history_save_ordering            (FILE *fp);
#endif

void            history_add_item                 (HistoryItem *hi);

void            history_update_item              (HistoryItem *hi);

void            history_item_strings             (HistoryItem hi, gchar *text[]);

void            history_look_for_newer           (gpointer key, gpointer value, 
						  gpointer user_data);

gint            history_item_age                 (HistoryItem *hi);

gint            history_compare_rows             (GtkCList *clist, 
						  gconstpointer ptr1,
						  gconstpointer ptr2);

static void history_build (const char *xml_root, GtkWidget **widget_ptr, 
			   gpointer data);

static void     history_show_dialog_add_item     (gpointer key, gpointer value,
	       				          gpointer user_data);

static HistoryCategory *
history_get_host_category (gchar *url);

static HistoryCategory *
history_add_host_category (gchar *host);

static void history_fill_ctree (void);

static void history_empty_ctree (void);

static gboolean history_filter_text (gchar *url);

static gboolean history_filter_text (gchar *url);

static void eputs (guchar *text, FILE *fp);

/**
 * Loads the history from the history file (if ir exists). Creates the 
 * history dialog. Does not load expired entries
 */
void
history_load (void)
{
        char *confdir;
        char *histfile;
        xmlDocPtr doc;
        xmlNodePtr item;
        gint expire_days = gnome_config_get_int (CONF_ADVANCED_HISTORY_EXPIRE);

        confdir = g_concat_dir_and_file(g_get_home_dir (), ".galeon");
	histfile = g_concat_dir_and_file(confdir, "history.xml");
	
        history = g_hash_table_new (g_str_hash, g_str_equal);

	if (access (histfile, F_OK) == -1) {
                return;
        }

        doc = xmlParseFile (histfile);

	g_return_if_fail (doc != NULL);
	g_return_if_fail (doc->root != NULL);

        item = doc->root->childs;
        while (item != NULL) {
                if (!strcmp (item->name, "ordering")) {
                        gint column = atoi (xmlGetProp(item, "column"));
                        GtkSortType type = GTK_SORT_ASCENDING;
                        if (!strcmp (xmlGetProp (item, "type"), "descending"))
                                type = GTK_SORT_DESCENDING;
			history_sort_type = type;
			history_sort_column = column;
                } else {
                        HistoryItem *hi = g_new0 (HistoryItem, 1);
                        gchar *s;
			
			s = xmlGetPropISOLatin1 (item, "title");
                        hi->title = g_strdup (s);
                        xmlFree (s);

                        s = xmlGetProp (item, "url");
                        hi->url = g_strdup (s);
                        xmlFree (s);

                        s = xmlGetProp (item, "first_time");
                        hi->first = strtol (s, NULL, 10);
                        xmlFree (s);

                        s = xmlGetProp (item, "last_time");
                        hi->last = strtol (s, NULL, 10);
                        xmlFree (s);

                        s = xmlGetProp (item, "visits");
                        hi->visits = atoi (s);
                        xmlFree (s);

                        if (history_item_age (hi) <= expire_days) {
                                history_add_item (hi);
                        } else {
                                g_free (hi);
                        }
                }
                item = item->next;
        }
	xmlFreeDoc (doc);
	g_free(confdir);
	g_free(histfile);
}

/**
 * history_build: build a history widget (either docked or dialoged)
 */
static void 
history_build (const char *xml_root, GtkWidget **widget_ptr, gpointer data)
{
	GladeXML *gxml;
	GtkWidget *search_optionmenu;
	GtkWidget *host_group_checkbox;

	/* build the widgets */
	gxml = glade_widget_new (xml_root, widget_ptr, data);

	/* connect the optionmenu activate signal */
	search_optionmenu = 
		glade_xml_get_widget (gxml, "history_time_optionmenu");
	gtk_signal_connect
		(GTK_OBJECT (GTK_OPTION_MENU (search_optionmenu)->menu),
		 "deactivate", 
		 GTK_SIGNAL_FUNC (history_time_optionmenu_deactivate_cb),
		 NULL);

	/* fill the history tree */
	clHistory = GTK_CTREE (glade_xml_get_widget (gxml, "clHistory"));

	/* set the host group checkbox according to the pref */
	host_group_checkbox = 
		glade_xml_get_widget (gxml, "history_host_checkbutton");
	host_group = 
		gnome_config_get_bool ("/Galeon/history_group_by_host=TRUE");
	gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (host_group_checkbox));

	/* no longer needed */
	gtk_object_unref (GTK_OBJECT (gxml));	
}

/**
 * history_show_dialog: show the history dialog
 */
void 
history_show_dialog (GaleonWindow *window)
{
	/* check args */
	return_if_not_window (window);

	/* check current state */
	if (dHistory != NULL)
	{
		return;
	}

	/* build and show the history dialog */
	history_build ("dHistory", &dHistory, window);

	/* now fill it */
	history_refresh_ctree ();

	/* show the widget */
	gtk_widget_show (dHistory);
	window_set_layer (dHistory);
}

/**
 * history_hide_dialog: hide the history dialog
 */
void
history_hide_dialog (void)
{
	g_assert (GNOME_IS_DIALOG (dHistory));
	gtk_widget_destroy (dHistory);
	dHistory = NULL;
}

/**
 * history_show_dock: show the history dock
 */
void
history_show_dock (GaleonWindow *window)
{
	GList *l;

	/* check args */
	return_if_not_window (window);

	/* check the current state */
	if (dHistory != NULL)
	{
		if (GNOME_IS_DOCK_ITEM (dHistory))
		{
			return;
		}
		else
		{
			history_hide_dialog ();
		}
	}

	/* build and pack the history dock */
	history_build ("history_dock", &dHistory, window);
	gnome_app_add_dock_item (GNOME_APP (window->WMain),
				 GNOME_DOCK_ITEM (dHistory),
				 GNOME_DOCK_LEFT, 0, 0, 0);

	/* update the state of all the menu entries */
	for (l = all_windows; l != NULL; l = g_list_next (l))
	{
		GaleonWindow *window = (GaleonWindow *)(l->data);
		return_if_not_window (window);

		/* hack because we don't want to emit toggled signal */
		(GTK_CHECK_MENU_ITEM (window->view_history_dock))->active = 
			TRUE;
	}

	/* show the widget */
	gtk_widget_show (dHistory);

	/* now fill it */
	history_refresh_ctree ();
}

/**
 * history_hide_dock: hide the history dock
 */
void
history_hide_dock (void)
{
	GList *l;

	/* check */
	g_assert (GNOME_IS_DOCK_ITEM (dHistory));

	/* update the state of all the menu entries */
	for (l = all_windows; l != NULL; l = g_list_next (l))
	{
		GaleonWindow *window = (GaleonWindow *)(l->data);
		return_if_not_window (window);

		/* hack because we don't want to emit toggled signal */
		(GTK_CHECK_MENU_ITEM (window->view_history_dock))->active = 
			FALSE;
	}

	/* destroy the dock */
	gtk_widget_destroy (GTK_WIDGET (dHistory));
	dHistory = NULL;
}

/*
 * history_fill_ctree: fill the tree with the list of urls
 */
void
history_fill_ctree (void)
{
	/* create a new hosts hashtable */
	hosts = g_hash_table_new (g_str_hash, g_str_equal);

	/* add every history item */	
	g_hash_table_foreach (history, history_show_dialog_add_item, hosts);

	/* destroy table */
	/* FIXME: when do categories get free'd??? */
	g_hash_table_destroy (hosts);
	hosts = NULL;
}

/*
 * history_empty_ctree: remove all the tree urls
 */
void
history_empty_ctree (void)
{
	gtk_clist_clear(GTK_CLIST (clHistory));
}

/*
 * history_refresh_ctree: update the tree urls, applying current criteria
 */
void
history_refresh_ctree (void)
{
	/* finish display */
	while (gtk_events_pending ())
	{
		gtk_main_iteration ();
	}

	gtk_clist_freeze(GTK_CLIST (clHistory));
	history_empty_ctree ();
	history_fill_ctree ();
	gtk_clist_thaw(GTK_CLIST (clHistory));
}

/*
 * history_add_host_category: add an host to the list and to the tree
 */
static HistoryCategory *
history_add_host_category (gchar *host)
{
	HistoryCategory *cat;
	gchar *text[6] = {host, NULL, NULL, NULL, NULL, NULL};
	
	cat = g_new0 (HistoryCategory, 1);

	cat->host = host;
	cat->node = gtk_ctree_insert_node (clHistory, NULL, 
					   NULL, text, 2,
					   folder_pixmap_data->pixmap,
					   folder_pixmap_data->mask, 
					   folder_open_pixmap_data->pixmap,
					   folder_open_pixmap_data->mask, 
					   FALSE, FALSE);

	/* add to the hosts hashtable */
	g_hash_table_insert (hosts, host, cat);
	
	return cat;
}

/*
 * history_get_host_category: get the host category of the url
 */
static HistoryCategory *
history_get_host_category (gchar *url)
{
	HistoryCategory *category;
	GnomeVFSURI *vfs_uri;
	const gchar *host;

	/* check args */
	g_assert (url != NULL);

	/* parse the url as a GnomeVFS uri */
	vfs_uri = gnome_vfs_uri_new (url);
	if (vfs_uri == NULL)
	{
		/* seems to happen with https URLs, GnomeVFS bug? */
		return NULL;
	}

	/* get the host name (from the GnomeVFSURI structure */
	host = gnome_vfs_uri_get_host_name (vfs_uri);
	if (host == NULL)
	{
		/* shouldn't happen? */
		gnome_vfs_uri_unref (vfs_uri);
		return NULL;
	}

	/* lookup in table */
	category = g_hash_table_lookup (hosts, host);
	if (category == NULL)
	{
		/* new category */
		category = history_add_host_category (g_strdup (host));
	}

	/* free */
	gnome_vfs_uri_unref (vfs_uri);

	/* return the appropriate category */
	return category;
}

/*
 * history_filter_text: filter by search text 
 */
static gboolean
history_filter_text (gchar *url)
{
	if (url == NULL)
	{
		return FALSE;
	}
	if (search_text == NULL || search_text[0] == '\0')
	{
		return TRUE;
	}

	return (g_strcasestr (url, search_text) != NULL);
}

/*
 * history_filter_date: filter by date
 */
static gboolean
history_filter_date (GTime atime)
{
	GDate *current_date = g_date_new ();
	GDate *date = g_date_new ();
	gboolean result;

	g_date_set_time(current_date, time(NULL));
	g_date_set_time(date, atime);

	switch (search_time)
	{
		/* Always */
	case 0:
		return TRUE;
		/* Today */
	case 1:
		break;
		/* Yesterday */
	case 2:
		g_date_subtract_days (current_date, 1);
		break;
		/* Two days ago */
	case 3:
		g_date_subtract_days (current_date, 2);
		break;
		/* Week */
	case 4:
		g_date_subtract_days (current_date, 7);
		break;
		/* Month */
	case 5:
		g_date_subtract_months (current_date, 1);
		break;
	default:
		break;
	}

	result = (g_date_compare (date, current_date)>=0);

	g_date_free (date);
	g_date_free (current_date);
	
	return result;
}

/*
 * history_show_dialog_add_item: add an item to tree, used when filling it
 */
static void 
history_show_dialog_add_item (gpointer key, gpointer value, gpointer user_data)
{
	HistoryItem *hi = (HistoryItem *) value;
	gint i;
	gchar *text[6];
	HistoryCategory *host_cat;
	GtkCTreeNode *node, *parent_node = NULL;

	if ((!history_filter_text (hi->url)) 
	    && (!history_filter_text (hi->title)))
		return;

	if (!history_filter_date (hi->last)) return;

	if (host_group)
	{
		host_cat = history_get_host_category (hi->url);
		if (host_cat) 
		{
			parent_node = host_cat->node;
		}
	}

	history_item_strings (*hi, text);

	node = gtk_ctree_insert_node (clHistory, parent_node,
				      NULL, text, 2, 
				      site_pixmap_data->pixmap,
				      site_pixmap_data->mask, 
				      site_pixmap_data->pixmap,
				      site_pixmap_data->mask, 
				      TRUE, TRUE);
		
	gtk_ctree_node_set_row_data (clHistory,
				     node,
				     hi->url);

	for (i = 0; i < 5; i++)
		g_free(text[i]);
}

/**
 * Creates the string array to represent an history item in the clist.
 * The strings should be freed.
 */
void 
history_item_strings (HistoryItem hi, gchar *text[])
{
	text[0] = g_strdup (hi.title);
        text[1] = g_strdup (hi.url);
	text[2] = time_to_string (hi.last);
 	text[3] = time_to_string (hi.first);
	text[4] = g_strdup_printf ("%d", hi.visits);
}

/**
 * Adds (or updates) a new history item to the history and returns
 * the corresponding HistoryItem (which should not be freed)
 */
HistoryItem *
history_visited (const gchar *url, const gchar *title)
{
        HistoryItem *hi = NULL;
        GTime now;

	/* check arguments */
	g_assert (title != NULL);
        if (url == NULL)
	{
		return NULL;
	}

	/* don't add blank pages or portal into history */
	if (!strcmp (url, "about:blank") || !strcmp (url, MYPORTAL_URL))
	{
		return NULL;
	}
	
        now = time (NULL);

        hi = g_hash_table_lookup (history, url);

        if (hi != NULL)
	{
		g_free (hi->title);
                hi->title = g_strdup (title);
                hi->last = now;
                hi->visits++;
                history_update_item (hi);
        } 
	else
	{
                hi = g_new0 (HistoryItem, 1);
                hi->url = g_strdup (url);
                hi->title = g_strdup (title);
                hi->first = now;
                hi->last = now;
                hi->visits = 1;
		history_add_item (hi);
        }

	history_dirty = TRUE;
        return hi;
}

/** Where does the recent menu start? */
#define GO_RECENT_POS 13

/**
 * Clears the recent menu
 */
void
history_clear_recent_menu (void)
{
  GList *w;
  for (w = all_windows; w != NULL; w = g_list_next(w))
    {
      GaleonWindow *window = (GaleonWindow *)(w->data);
      GtkWidget *menu = gtk_object_get_data (GTK_OBJECT(window->WMain),
					     "go_menu");
      GList *children = gtk_container_children (GTK_CONTAINER(menu));
      children = g_list_nth (children, GO_RECENT_POS);

      while (children != NULL)
	{ 
	  gtk_container_remove (GTK_CONTAINER (menu),
				GTK_WIDGET (children->data));
	  gtk_widget_destroy (GTK_WIDGET (children->data));
	  children = g_list_next (children);
	}
    	
      g_list_free(g_list_first(children));
    }
}
/**
 * Adds a new history item to the hashtable and to the history dialog
 */
void 
history_add_item (HistoryItem *hi)
{
        g_hash_table_insert (history, hi->url, hi);
        auto_completion_add_url (hi->url);
}

/**
 * Updates a history item in the hashtable and in the history dialog
 */
void 
history_update_item (HistoryItem *hi)
{
	g_hash_table_insert (history, hi->url, hi);
}

#if USE_OLD_HISTORY_SAVE

/**
 * These two following variables are used to save the history
 */
xmlDocPtr save_hist_doc = NULL;
xmlNodePtr save_hist_node = NULL;

/** 
 * Saves the history
 */
void 
history_save (void)
{
        char *histfile;

	if (!history_dirty)
	{
		return;
	}

	histfile = g_strconcat (g_get_home_dir (), 
				"/.galeon/history.xml", NULL);
        save_hist_doc = xmlNewDoc ("1.0");
        save_hist_node = xmlNewDocNode (save_hist_doc, NULL, "history", NULL);
        g_hash_table_foreach (history, history_save_item, NULL);
        history_save_ordering ();
        xmlDocSetRootElement (save_hist_doc, save_hist_node);
        xmlSaveFile (histfile, save_hist_doc);
	xmlFreeDoc (save_hist_doc);
        g_free (histfile);
	history_dirty = FALSE;
}

/**
 * Saves one item of the history
 */
void 
history_save_item (gpointer key, gpointer value, gpointer user_data)
{
        HistoryItem *hi = (HistoryItem *) value;
        GString *s = g_string_new ("");
        xmlNodePtr node = xmlNewDocNode (save_hist_doc, NULL, "item", NULL);
        xmlSetPropISOLatin1 (node, "title", hi->title);
        xmlSetProp (node, "url", hi->url);
        g_string_sprintf (s, "%d", hi->first);
        xmlSetProp (node, "first_time", s->str);
        g_string_sprintf (s, "%d", hi->last);
        xmlSetProp (node, "last_time", s->str);
        g_string_sprintf (s, "%d", hi->visits);
        xmlSetProp (node, "visits", s->str);
        g_string_free (s, TRUE);
        xmlAddChild (save_hist_node, node);
}

/**
 * Saves the current sort type of the history dialog
 */
void 
history_save_ordering (void)
{
	GString *s = g_string_new ("");
	xmlNodePtr node = xmlNewDocNode (save_hist_doc, NULL, "ordering", NULL);
	g_string_sprintf (s, "%d", history_sort_column);
	xmlSetProp (node, "column", s->str);
	if (history_sort_type == GTK_SORT_DESCENDING) {
		xmlSetProp (node, "type", "descending");
	} else {
		xmlSetProp (node, "type", "ascending");
	}
	xmlAddChild (save_hist_node, node);
	g_string_free (s, TRUE);
}

#else

/** 
 * Saves the history
 */
void 
history_save (void)
{
        gchar *histfile;
	FILE *fp;

	if (!history_dirty)
	{
		return;
	}

	histfile = g_strconcat (g_get_home_dir (), 
				"/.galeon/history.xml", NULL);
	fp = fopen (histfile, "wb");
	g_return_if_fail (fp != NULL);

	fputs ("<?xml version=\"1.0\"?>\n", fp);
	fputs ("<history>\n", fp);
        g_hash_table_foreach (history, history_save_item, fp);
        history_save_ordering (fp);
	fputs ("</history>\n", fp);
	fclose (fp);
	history_dirty = FALSE;
}

static void
eputs (guchar *text, FILE *fp)
{
	gint i, length, last;
	
	last = 0;
	length = strlen (text);
	for (i = 0; i < length; i++)
	{
		if (text[i] >= 127)
		{
			fwrite (text + last, i - last, 1, fp);
			last = i + 1;
			fprintf (fp, "&#%03d;", text[i]);
			continue;
		}

		switch (text[i])
		{
		case '&':
			fwrite (text + last, i - last, 1, fp);
			last = i + 1;
			fputs ("&amp;", fp);
			break;

		case '<':
			fwrite (text + last, i - last, 1, fp);
			last = i + 1;
			fputs ("&lt;", fp);
			break;

		case '>':
			fwrite (text + last, i - last, 1, fp);
			last = i + 1;
			fputs ("&gt;", fp);
			break;

		case '\"':
			fwrite (text + last, i - last, 1, fp);
			last = i + 1;
			fputs ("&quot;", fp);
			break;

		case '\'':
			fwrite (text + last, i - last, 1, fp);
			last = i + 1;
			fputs ("&apos;", fp);
			break;
		}
	}

	/* write out remainder */
	fwrite (text + last, i - last, 1, fp);
}

/**
 * Saves one item of the history
 */
void 
history_save_item (gpointer key, gpointer value, gpointer user_data)
{
        HistoryItem *hi = (HistoryItem *) value;
	FILE *fp = (FILE *) user_data;

	fputs ("  <item title=\"", fp);
	eputs (hi->title, fp);
	fputs ("\" url=\"", fp);
	eputs (hi->url, fp);
	fputs ("\" first_time=\"", fp);
	fprintf (fp, "%d", hi->first);
	fputs ("\" last_time=\"", fp);
	fprintf (fp, "%d", hi->last);
	fputs ("\" visits=\"", fp);
	fprintf (fp, "%d", hi->visits);
	fputs ("\"/>\n", fp);
}

/**
 * Saves the current sort type of the history dialog
 */
void 
history_save_ordering (FILE *fp)
{
	fputs ("  <ordering column=\"", fp);
	fprintf (fp, "%d", history_sort_column);
	fputs ("\" type=\"", fp);
	if (history_sort_type == GTK_SORT_DESCENDING)
	{
		fputs ("descending", fp);
	}
	else
	{
		fputs ("ascending", fp);
	}
	fputs ("\"/>\n", fp);
}

#endif

/** 
 * Returns the url of the last visited page
 */ 
gchar *
history_get_last_url (void)
{
        HistoryItem *last = NULL;
        g_hash_table_foreach (history, history_look_for_newer, &last);
        if (last) {
                return g_strdup (last->url);
        } else {
                /* we should return some "default last page" */
                return g_strdup ("http://galeon.sourceforge.net");
        }
}

/**
 * Helper function to locate the newer HistoryItem
 */
void 
history_look_for_newer (gpointer key, gpointer value, gpointer user_data)
{
        static GTime maxtime = 0;
        HistoryItem *hi = (HistoryItem *) value;
        if (hi->last >= maxtime) {
                HistoryItem **last = (HistoryItem **) user_data;
                *last = hi;
                maxtime = hi->last;
        }
}

/** 
 * Returns the number of days since the history item was visited for the last time
 */
gint
history_item_age (HistoryItem *hi)
{
        GTime now = time (NULL);
        gint days = (now - hi->last) / (24 * 60 * 60);
        return days;
}

/**
 * Adds an item to the recent menu of a given window or moves it to the start.
 */
void
history_add_recent_menu (HistoryItem *hi, GnomeApp *WMain)
{
        GtkWidget *menu = gtk_object_get_data (GTK_OBJECT (WMain), "go_menu");
        GtkWidget *menuitem;
        GList *l = gtk_container_children (GTK_CONTAINER(menu));
        GList *li;
        gint count = 1 - GO_RECENT_POS;
        gint max = 10;  /* should we add a preference for this? */
	HistoryItem *mhi;
	
        for (li = l; li != NULL; li = li->next)
	{
		if (count>0)
		{
			mhi = gtk_object_get_user_data (GTK_OBJECT (li->data));
			g_assert(mhi != NULL);
			gtk_container_remove (GTK_CONTAINER (menu),
					GTK_WIDGET (li->data));
			gtk_widget_destroy (GTK_WIDGET (li->data));
			if (mhi != hi)
			{
				if (count < max) 
				{
					menuitem = history_create_recent_menu_item (mhi, count, menu);
					gtk_menu_insert (GTK_MENU (menu), menuitem, GO_RECENT_POS+count);
					gtk_widget_show (menuitem);
				}
				count++;
			}
		}
		else 
			count++;
	}
        g_list_free(l);
        menuitem = history_create_recent_menu_item (hi, 0, menu);
        gtk_menu_insert (GTK_MENU (menu), menuitem, GO_RECENT_POS);
        gtk_widget_show (menuitem);
}

GtkWidget *
history_create_recent_menu_item (HistoryItem *hi, gint num, GtkWidget *menu)
{
	GtkWidget *hb = gtk_hbox_new (FALSE, 0);
	PixmapData *pixmap_data = bookmarks_get_siteicon (hi->url);
	GtkWidget *pixmap = gtk_pixmap_new (pixmap_data->pixmap, pixmap_data->mask);
	GtkWidget *menuitem = gtk_pixmap_menu_item_new ();
	GtkWidget *l = new_num_accel_label (num, hi->title, FALSE, menu, menuitem);
	
	gtk_box_pack_start (GTK_BOX (hb), l, FALSE, FALSE, 2);
	gtk_container_add (GTK_CONTAINER(menuitem), hb);
	gtk_pixmap_menu_item_set_pixmap (GTK_PIXMAP_MENU_ITEM(menuitem), pixmap);
	
	gtk_widget_show_all (hb);
	if (gnome_preferences_get_menus_have_icons ())
		gtk_widget_show (pixmap);
	
	gtk_widget_ref (menuitem); /* so we can gtk_container_remove, and then
				      gtk_widget_destroy */
	
	gtk_object_set_user_data (GTK_OBJECT (menuitem), hi);
	gtk_signal_connect (GTK_OBJECT(menuitem), "activate",
			    GTK_SIGNAL_FUNC (window_menu_recent_activate_cb),
			    NULL);

	return menuitem;
}
