/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include <dirent.h>

/* external function prototypes imported from libxml1 */
extern int isolat1ToUTF8(unsigned char* out, int outlen,
			 const unsigned char* in, int inlen);
extern int UTF8Toisolat1(unsigned char* out, int outlen,
			 const unsigned char* in, int inlen);

/** Styles for coloured text */
GtkStyle *red_text_style;
GtkStyle *yellow_text_style;
GtkStyle *green_text_style;
GtkStyle *blue_text_style;

/** Module with all the symbols of the program */
GModule *mod_self;

/* list of known protocols */
const char *known_protocols[] =
{
	"http",
	"https",
	"file",
	"about",
	"chrome",
	"resource",
	"javascript",
	"jar",
#ifdef CVS_SOURCE
	"view-source",
#endif
       	NULL /* terminator: must be last! */
};

/* list of known mime types */
const char *known_mime_types[] =
{
	/* standard documents */
	"text/html",
	"text/plain",
	"text/x-perl", /* slashdot login! */

	GNOME_VFS_MIME_TYPE_UNKNOWN,

	/* standard images */
	"image/jpeg",
	"image/gif",
	"image/x-png",
	"image/png",

	/* protocols -- these are meta-mimetypes for when the real
	 * type has not yet been divined, e.g. http://some.site.com/ */
	"x-url/http",
	"x-url/ftp",

	/* meta-mimetypes of generated documents */
	"application/x-cgi",
	"application/x-asp",
	"application/x-php",
	"application/x-cgi",

	/* a misidentified type seen on the BBC website */
	"audio/x-stm", 

	/* terminator, must be last! */
	NULL
};

/* Local prototypes */
static void add_bf_menu (GtkWidget *menu, char* label, int index, 
			 GaleonEmbed *embed, int level);
static void free_string_array (char *array[], int size);
static void unrecognised_protocol_cb(gint reply, gpointer url);
static gboolean is_punctuation (gchar c);
gint file_copy_status (GnomeVFSXferProgressInfo *info,gpointer data);
void favicon_download_completed_cb (gpointer data);
gchar *favicon_filename (gchar *favicon_url);
static void glade_signal_connect_func (const gchar *cb_name, GtkObject *obj, 
				       const gchar *signal_name, 
				       const gchar *signal_data,
				       GtkObject *conn_obj, 
				       gboolean conn_after,
				       gpointer user_data);
static void misc_ask_question_cb (gint reply, gpointer data);
static gchar *new_num_accel_str (gint num, gchar *text, gboolean lettersok);
static GtkWidget * new_num_accel_menu_item (gint num, gchar *origtext, 
					    gboolean lettersok, 
					    GtkWidget *menu);

/**
 * glade_lookup_widget
 */
GtkWidget *
glade_lookup_widget (GtkWidget *widget, const gchar *widget_name)
{
	GtkWidget *found_widget, *parent;
	GladeXML *xml = NULL;

	while (xml == NULL)
	{
		/* the following line is to allow to override the
		 * GladeXML object or to set it to an object which was
		 * not created by libglade. Useful for popup menus */
		xml = gtk_object_get_data (GTK_OBJECT(widget), "widget_tree");
		if (!xml) xml = glade_get_widget_tree(widget);

		if (GTK_IS_MENU(widget))
			parent = gtk_menu_get_attach_widget (GTK_MENU(widget));
		else
			parent = widget->parent;

		if (parent == NULL) 
			break;

		widget = parent;
	}

	found_widget = glade_xml_get_widget(xml, widget_name);

	if (!found_widget)
		g_warning ("Widget not found: %s", widget_name);
 
	return found_widget;
}

/**
 * user_file: returns the pathname of galeon shared files (e.g., galeon.glade)
 *
 * fname: just the filename with no path information
 * critical: critical file? (halt if not found)
 */
gchar *
user_file(gchar *fname, gint critical)
{
	int i;
	gchar *file = NULL;
	gchar *alternative[6];
	gboolean found = FALSE;
	static GHashTable *already_found = NULL;
	
	if (already_found == NULL) {
		already_found = g_hash_table_new (g_str_hash, g_str_equal);
	}

        /* Have we already found this? */
	file = g_hash_table_lookup (already_found, fname);
	if (file != NULL) return g_strdup (file);

	/* try the default */
	file = g_strconcat (g_get_home_dir (), "/.galeon/", fname, NULL);
	
	/* success? */
	if (access (file, R_OK) == 0)
		return file;

	g_free(file);

	/* specify alternate locations in order of precedence */
	alternative[0] = g_strdup (fname);
	alternative[1] = g_strconcat("../", fname, NULL);
	alternative[2] = g_strconcat("ui/", fname, NULL);
	alternative[3] = g_strconcat("../ui/", fname, NULL);
	alternative[4] = g_strconcat(SHARE_DIR "/", fname, NULL);
	alternative[5] = NULL;  /* NULL terminator needed */
	
	/* select one of the alternatives */
	for (i = 0; alternative[i] != NULL; i++)
	{
		file = g_strdup (alternative[i]);
		if (access (file, R_OK) == 0) {
                        /* don't warn for the install default */
			if (i != 4)
				g_warning("Using %s (usually OK)\n", file);
			found = TRUE;
			break;
		}
		g_free (file);
	}

	for (i = 0; alternative[i] != NULL; i++)
		g_free (alternative[i]);

	if (found) {
		/* Add it to the set of found files */
		g_hash_table_insert (already_found, g_strdup (fname), g_strdup (file));
		return file;
	}

	/* if nothing then theres an error */
	if (critical)
		g_error(_("%s not found"), fname);
	else
		file = NULL;

	return NULL;
}

/**
 * glade_widget_new: build a new widget of the provided name, with all
 * signals attached and data set to the provided parameter.
 */
GladeXML *
glade_widget_new (const gchar *widget_name, GtkWidget **root, gpointer data)
{
	GladeXML *gxml;
	static gchar *glade_file = NULL;

	/* find the glade file */
	if (glade_file == NULL)
	{
		glade_file = user_file ("galeon.glade", TRUE);
		g_return_val_if_fail (glade_file != NULL, NULL);
	}

	/* build the widget */
	/* note that libglade automatically caches the parsed file,
	 * so we don't need to worry about the efficiency of this */
	gxml = glade_xml_new (glade_file, widget_name);
	g_return_val_if_fail (gxml != NULL, NULL);

	/* lookup the root widget if requested */
	if (root != NULL)
	{
		*root = glade_xml_get_widget (gxml, widget_name);
	}

	/* connect signals and data */
	glade_xml_signal_autoconnect_full
		(gxml, glade_signal_connect_func, data);

	/* return xml document for subsequent widget lookups */
	return gxml;
}

/*
 * glade_signal_connect_func: used by glade_xml_signal_autoconnect_full
 */
static void
glade_signal_connect_func (const gchar *cb_name, GtkObject *obj, 
			   const gchar *signal_name, const gchar *signal_data,
			   GtkObject *conn_obj, gboolean conn_after,
			   gpointer user_data)
{
	gpointer handler_func;
	
	/*g_print( "glade_signal_connect_func: cb_name = '%s', signal_name = '%s', signal_data = '%s'\n",
	  cb_name, signal_name, signal_data ); */
	
	if (g_module_symbol(mod_self, cb_name, &handler_func))
	{
		/* found callback */
		if (conn_obj)
		{
			if (conn_after)
			{
				gtk_signal_connect_object_after
					(obj, signal_name, 
					 handler_func, conn_obj);
			}
			else
			{
				gtk_signal_connect_object
					(obj, signal_name, 
					 handler_func, conn_obj);
			}
		}
		else
		{
			/* no conn_obj; use standard connect */
			gpointer data = NULL;
			
			data = user_data;
			
			if (conn_after)
			{
				gtk_signal_connect_after
					(obj, signal_name, 
					 handler_func, data);
			}
			else
			{
				gtk_signal_connect
					(obj, signal_name, 
					 handler_func, data);
			}
		}
	}
	else
	{
		g_warning("callback function not found: %s", cb_name);
	}
}

/**
 * handle_foreign_mime_types: work out the mime type of a URL,
 * return TRUE if we handle it separately and FALSE if we want
 * mozilla to have a go
 */
gboolean
handle_foreign_mime_types (const char *url, GaleonEmbed *embed)
{
	const gchar *mime_type;
	GnomeVFSURI *vfs_uri;
	MimeAction action;
	gint i;

	if (embed->mime_ignore_once)
	{
		embed->mime_ignore_once = FALSE;
		return FALSE;
	}

	if (gnome_config_get_bool (CONF_HANDLERS_MIME_ENABLE) == FALSE) 
		return FALSE;

	vfs_uri = gnome_vfs_uri_new (url);
	if (vfs_uri)
	{
		mime_type = gnome_vfs_get_mime_type_from_uri (vfs_uri);
		gnome_vfs_uri_unref (vfs_uri);
	}
	else
	{
		return FALSE;
	}

	/* if we found one check if it's one we know */
	if (mime_type != NULL)
	{
		/* if it's one we know we let mozilla display it */
		for (i = 0; known_mime_types[i] != NULL; i++)
		{
			if (strcmp(mime_type, known_mime_types[i]) == 0)
				return FALSE;
		}
	} else {
		/* perhaps it's something mozilla knows about... */
		return FALSE;
	}

/* autodownload--FIXME */

	/* what we *should* do here is let the user choose between
	 * opening, editing, running (etc) the mime type according
	 * to the gnome settings. However, we don't have any way of
	 * downloading-then-executing something: we'll need the
	 * gtm interface to be extended first. What we can do instead
	 * is to download it and save to disk */

	/* see if we already know what to */
	action = mime_get_action (mime_type);
	switch (action) {
	case MIME_UNKNOWN:
	case MIME_ASK_ACTION:
		mime_ask_action (url, mime_type, embed);
		break;
	case MIME_LEAVE_TO_MOZILLA:
		return FALSE;
		break;
	case MIME_SAVE_TO_DISK:
	       	save_url(url);
		break;
	case MIME_RUN_PROGRAM:
		/* TODO */
		gnome_error_dialog ("Not Implemented!");
		break;
	}

	/* we've handled it */
	return TRUE;
}

/**
 * handle_foreign_protocols: work out what protocol is specified
 * by a URL, return TRUE if we find a handler and execute it,
 * otherwise return FALSE to tell mozilla to try to load it
 */
gboolean
handle_foreign_protocols (const char *url)
{
	gint i, length;
	gchar *protocol;
	gchar *key, *result;
	gchar *url_copy;

	/* ignore any spaces preceding the url */
	while (*url == ' ' && *url != '\0')
		url++;

	/* find the colon */
	length = strlen(url);
	for (i = 0; i < length && url[i] != ':' && url[i] != '.'; i++)
		;
	
	/* did we run over? */
	if (i == length)
	{
		/* don't know what to do here so leave it to the lizard */
		return FALSE;
	}

	/* did we find a period? */
	if (url[i] == '.')
	{
		/* must be a site designation, not a protocol */
		return FALSE;
	}

	/* okay, duplicate this */
	protocol = g_strndup(url, i);

        /* see if it's definitely a known protocol */
        for (i = 0; known_protocols[i] != NULL; i++)
        {
                if (strcasecmp (protocol, known_protocols[i]) == 0)
		{
			/* fine, give it to mozilla */
			g_free(protocol);
                        return FALSE;
		}
        }

	/* if it's FTP, check if the user has setup to use mozilla */
	if (strcasecmp (protocol, "ftp") == 0 &&
	    gnome_config_get_int ("/galeon/Handlers/ftp=0") == 0)
	{
			/* give it to mozilla */
			g_free(protocol);
                        return FALSE;
	}

	/* we don't know the protocol, check to see what GNOME has */
	/* unfortunately there's not an easy function to do this... */

	/* build the config key */
	key = g_strconcat("/Gnome/URL Handlers/", protocol, "-show", NULL);
	g_free(protocol);

	/* find it */
	result = gnome_config_get_string(key);
	g_free(key);
	if (result != NULL && strlen(result) != 0)
	{
		/* we have a GNOME handler for this, pass it through
		 * and tell mozilla not to try showing it */
		g_free(result);
		gnome_url_show(url);
		return TRUE;
	}

	/* free */
	if (result)
		g_free(result);

	/* no luck, so offer the user the option of trying the
	 * default handler -- we don't do this automatically in
	 * case the default handler is erroneously set to galeon */
	result = gnome_config_get_string("/Gnome/URL Handlers/default-show");

	/* check there is a default */
	if (result == NULL || strlen(result) == 0)
	{
		/* throw the error */
		gnome_error_dialog(_("Galeon cannot handle this protocol,\n"
				     "and no GNOME default handler is set"));

		/* free */
		if (result) g_free(result);

		/* don't let mozilla try blindly */
		return TRUE;
	}
 
	/* free */
	g_free(result);

	/* offer the choice */
	url_copy = g_strdup(url);
	gnome_question_dialog(_("The protocol specified is not recognised.\n\n"
				"Would you like to try the GNOME default?"),
			      unrecognised_protocol_cb, (gpointer)url_copy);

	/* keep mozilla out of it */
	return TRUE;
}

/**
 * first_time_cb: called if the user decides to import existing
 * netscape bookmarks or not
 */
static void
unrecognised_protocol_cb(gint reply, gpointer url)
{
        /* use default GNOME URL handler if asked to */
        if (reply == GNOME_YES)
	{
		gnome_url_show((gchar *)url);
	}

	/* free the copied url */
	g_free(url);
}

/**
 * save_url
 */
void
save_url (const gchar* url)
{
	if (gnome_config_get_bool("/galeon/Downloading/"
				  "ask_for_download_dir=true")) {
		ask_dir (url);
	} else if (gnome_config_get_int (CONF_DOWNLOADING_FTP_PROGRAM) == 0) {
		save_url_with_gtm (url);
	} else {
		save_url_with_command_line (url);
	}
}

void 
save_url_with_command_line (const gchar *url)
{
	gchar *dir;
	gchar *command = gnome_config_get_string (CONF_DOWNLOADING_FTP_COMMAND);
	gchar *full_command;
	pid_t pid;

	/* get the download directory */ /* FIXME: use this correctly ... */
	dir = gnome_config_get_string ("/galeon/Downloading/download_dir");

	full_command = g_strdup_printf (command, url, dir);

	/* Fork and invoke the external program. */
	pid = fork ();
	if (pid == 0) {
		/* child process */
		gchar **argv = g_strsplit (full_command, " ", 100);
		execvp (argv[0], argv);
		g_warning (_("Error executing external program"));
		_exit (0);
	} else if (pid < 0) {
		/* error */
		g_error ("Error creating subprocess");
		return;
	} else {
		/* parent process */
		/* do nothing else */
	}

	g_free (full_command);
	g_free (command);
	g_free (dir);
}

/**
 * save_url_with_gtm
 */
void
save_url_with_gtm (const gchar* url)
{  
	gboolean disable_auto_download;
	gchar *dir;

	/* get the download directory */
	dir = gnome_config_get_string ("/galeon/Downloading/download_dir");

	if (gnome_config_get_bool ("/galeon/Downloading/auto_download=true"))
		disable_auto_download = FALSE;
	else
		disable_auto_download = TRUE;

	gtm_add_url (strdup(url), dir, FALSE, disable_auto_download);		
		
	g_free(dir);
}

/**
 * ask_dir
 */
void
ask_dir (const gchar *url)
{
	GtkWidget *dirselect;
	GtkWidget *file_list;
	GladeXML *gxml;
	char *olddir;
	
	/* build the directory selector */
	gxml = glade_widget_new ("dirselection2", &dirselect, NULL);

	/* set it up with the standard download directory */
	olddir = gnome_config_get_string ("/galeon/Downloading/download_dir");
	gtk_file_selection_set_filename (GTK_FILE_SELECTION(dirselect),
					 olddir);
	gtk_object_set_data(GTK_OBJECT(dirselect), "url", g_strdup (url));

	/* don't allow the selection of files */
	file_list = GTK_FILE_SELECTION(dirselect)->file_list;
	gtk_widget_set_sensitive(file_list, FALSE);

	/* show it */
	gtk_widget_show(GTK_WIDGET(dirselect));
}


/*
 * g_str_is_url:
 * 
 * test a string for a loadable URL -- very general, not intended to be picky
 * 
 * returns 1 if string is a URL, otherwise returns 0
 */
int g_str_is_url(gchar * string)
{
	gchar * s;
       
	s = strstr(string, "://");

	if (s) {
		/* if there's characters before and after :// */
		if (strlen(s) > 3 /* length of "://" */  && s != string)
			return 1;
	}

	s = strstr(string, "www.");
	if (s) {
		/* if theres characters after and nothing before */
		if (strlen(s) > 4 /* length of "www." */ && s == string)
			return 1;
	}

	return 0;
}

#ifndef CVS_SOURCE
gchar *
init_cache_dir(void)
{ 
	gchar *confdir = g_strconcat(g_get_home_dir(),"/.galeon",NULL);
	gchar *cachedir = g_strconcat(confdir,"/cache",NULL);

	if (access (cachedir, F_OK) == -1)
	{
		if (mkdir (cachedir, 488) != 0)
			g_error("Could not make directory (%s)", cachedir);
	}

	g_free(confdir);
	return cachedir;
}
#endif

/**
 * Frees an array of strings
 */ 
void 
free_string_array (char *array[], int size)
{
	int i;
	for (i = 0; i < size; i++)
		g_free (array[i]);
	g_free (array);
}

/**
 * Creates a string with a numbered/lettered accel (caller must free)
 * returns NULL if num is out of the range of acceleratable nums/letters
 */
static gchar *
new_num_accel_str (gint num, gchar *text, gboolean lettersok)
{
	gchar *label = NULL;
	if (num < 9)
		label = g_strdup_printf	("_%i. %s", num+1, text);
	else if (num == 9)
		label = g_strdup_printf	("_%i. %s", 0, text);
	else if (num < 36 && lettersok)
		label = g_strdup_printf	("_%c. %s", 'a'+num-10, text);
	return label;
}

/**
 * Sets a labels text with shortcut key handling in the standard _<key> accel way
 */
void
label_set_accel_text (gchar *text, GtkWidget *label, GtkWidget *menu, 
		      GtkWidget *item)
{
	gint tmp_key = gtk_label_parse_uline (GTK_LABEL (label), text);
	GtkAccelGroup *menu_accels = 
		gtk_menu_ensure_uline_accel_group (GTK_MENU (menu));
	gtk_widget_add_accelerator (item, "activate_item", menu_accels,
			tmp_key, 0, 0);
}

/**
 * Strip the _ out of a string like gtk_label_parse_uline would do
 * caller is responsible for freeing the returned string
 */
gchar *
strip_uline_accel (const gchar *text)
{
	GString *out;
	const gchar *u, *cur = text;
	out = g_string_new (NULL);
	while ((u = strchr (cur, '_')))
	{
		if (*(u+1) == '_')
		{
			/* __ in the string is equal to _ in the output 
			 * so include the _ in the output, skip over the 
			 * second _ and continue scanning. */
			g_string_sprintfa (out, "%.*s", u - cur + 1, cur);
			cur = u + 2;
		} else {
			/* copy everything before the _ and skip over it */
			g_string_sprintfa (out, "%.*s", u - cur , cur);
			cur = u + 1;
			/* only one accel per string, so we are done now */
			break;
		}
	}
	if (cur && *cur)
	{
		/* attach rest of string */
		g_string_append (out, cur);
	}

	u = out->str;
	g_string_free (out, FALSE); /* don't free char data, caller must */
	return (gchar *)u;
}

/**
 * Escape _'s in string such that the gtk_label_parse_uline will display as normal
 * caller is responsible for freeing the returned string
 */
gchar *
escape_uline_accel (const gchar *text)
{
	GString *out;
	const gchar *u, *cur = text;
	out = g_string_new (NULL);
	while ((u = strchr (cur, '_')))
	{
		/* All we need to do is double every _, so include the _ in 
		 * the output, add another _, and continue scanning. */
		g_string_sprintfa (out, "%.*s_", u - cur + 1, cur);
		cur = u + 1;
	}
	if (cur && *cur)
	{
		/* attach rest of string */
		g_string_append (out, cur);
	}

	u = out->str;
	g_string_free (out, FALSE); /* don't free char data, caller must */
	return (gchar *)u;
}

/**
 * Creates a label with a numbered/lettered accel
 */
GtkWidget *
new_num_accel_label (gint num, gchar *origtext, gboolean lettersok, 
		     GtkWidget *menu, GtkWidget *item)
{
	gchar *text = new_num_accel_str(num, origtext, lettersok);
	if (text == NULL)
		return gtk_label_new (origtext);
	else
	{
		GtkWidget *label = gtk_label_new ("");
		label_set_accel_text (text, label, menu, item);
		g_free(text);
		return label;
	}
}

/**
 * Creates a menu item with a numbered/lettered accel
 */
static GtkWidget *
new_num_accel_menu_item (gint num, gchar *origtext, gboolean lettersok, 
			 GtkWidget *menu)
{
	gchar *text = new_num_accel_str(num, origtext, lettersok);
	if (text == NULL)
		return gtk_menu_item_new_with_label (text);
	else
	{
		GtkWidget *item = gtk_menu_item_new_with_label ("");
		label_set_accel_text (text, GTK_BIN (item)->child, menu, item);
		g_free(text);
		return item;
	}
}

/**
 * Adds a menuitem to a back/forward history menu
 */
static void
add_bf_menu (GtkWidget *menu, char* label, int index, GaleonEmbed *embed, int level)
{
	GtkWidget *item = new_num_accel_menu_item (level, label, TRUE, menu);

	gtk_widget_show (item);
	gtk_object_set_user_data (GTK_OBJECT (item), embed);
	gtk_menu_append (GTK_MENU (menu), item);
	gtk_signal_connect (GTK_OBJECT (item), "activate",
			    history_menu_menuitem_activate_cb, 
			    GINT_TO_POINTER (index));
}

/** 
 * Creates the back history menu 
 */
GtkMenu *
create_back_menu (GaleonEmbed *embed)
{
	int index, count, i, level;
	char **titles;
	GtkWidget *menu = gtk_menu_new ();

	if (!mozilla_session_history (embed, &titles, &count, &index))
	{
		return NULL;
	}

	for (i = index - 1, level = 0; i >= 0; i--, level++) 
	{
		add_bf_menu (menu, titles[i], i, embed, level);
	}
	
	free_string_array (titles, count);
	return GTK_MENU(menu);
}

/**
 * Creates the forward history menu
 */
GtkMenu *
create_forward_menu (GaleonEmbed *embed)
{
	int index, count, i, level;
	char **titles;
	GtkWidget *menu = gtk_menu_new ();

	if (!mozilla_session_history (embed, &titles, &count, &index))
	{
		return NULL;
	}	

	for (i = index + 1, level = 0; i < count; i++, level++)
	{
		add_bf_menu (menu, titles[i], i, embed, level);
	}
	
	free_string_array (titles, count);
	return GTK_MENU(menu);
}

/** 
 * Creates the multi-level up menu 
 */
GtkMenu *
create_up_menu (GaleonEmbed *embed)
{
	GnomeVFSURI *uri, *up_uri;
	GtkWidget *menu, *item;
	gint level;

	/* check embed location is valid */
	if (embed->site_location == NULL || strlen (embed->site_location) == 0)
	{
		return NULL;
	}

	/* create a vfs entry for this level */
	uri = gnome_vfs_uri_new (embed->site_location);
	if (uri == NULL)
	{
		return NULL;
	}

	/* create the menu */
	menu = gtk_menu_new ();

	/* create each possible up entry */
	for (level = 0;; level++)
	{
		up_uri = gnome_vfs_uri_get_parent (uri);
		gnome_vfs_uri_unref (uri);
		uri = up_uri;
		
		/* get out of loop if no parent */
		if (uri == NULL)
		{
			break;
		}

		/* create the menu entry */
		item = new_num_accel_menu_item (level, uri->text, TRUE, menu);
		gtk_widget_show (GTK_WIDGET (item));
		gtk_menu_append (GTK_MENU (menu), GTK_WIDGET (item));
	}

	/* the menu is completed */
	return GTK_MENU (menu);
}

void
menu_position_under_widget (GtkMenu *menu, gint *x, gint *y, 
			    gpointer user_data)
{
	GtkWidget *w = GTK_WIDGET(user_data);
	gint width, height;
	gint screen_width, screen_height;
	GtkRequisition requisition;

	gdk_window_get_size (w->window, &width, &height);
	gdk_window_get_deskrelative_origin(w->window, x, y);
	*y = *y + height;

	gtk_widget_size_request (GTK_WIDGET (menu), &requisition);
      
	screen_width = gdk_screen_width ();
	screen_height = gdk_screen_height ();
	  
	*x = CLAMP (*x, 0, MAX (0, screen_width - requisition.width));
	*y = CLAMP (*y, 0, MAX (0, screen_height - requisition.height));
}

/**
 * copy_to_clipboard: Copies some text into the clipboard
 **/
void
copy_to_clipboard (gchar *text, GaleonEmbed *embed)
{
	gint have_selection;
	GtkWidget *window = embed->parent_window->WMain;

	/* FIXME: free previous data? */
	gtk_object_set_data (GTK_OBJECT (window),
			     "selection", g_strdup (text));
	have_selection = gtk_selection_owner_set (GTK_WIDGET (window),
					 gdk_atom_intern("CLIPBOARD",FALSE), 
					  GDK_CURRENT_TIME)&&
		         gtk_selection_owner_set (window,
					 GDK_SELECTION_PRIMARY,
					 GDK_CURRENT_TIME);
	if (!have_selection)
	{
		g_warning("Selection not found");
	}
} 

static void
galeon_close_all_windows (void)
{
	GList *wl, *next;

	/* force ourselves out of server mode -- this means that closing
	 * the last window really will quit out of galeon */
	galeon_server_mode = FALSE;

	/* close all windows: this in turn will close all embeds */
	/* at the closing of the last window galeon_exit will be called */
	for (wl = all_windows; wl != NULL;)
	{
		next = wl->next;
		window_close ((GaleonWindow *)(wl->data));
		wl = next;
	}
}

void galeon_quit (GaleonWindow *window)
{
	/* location bar history is saved automatically when the widget is
	   destroyed. we don't want to save the text currently in the entry */
	gtk_editable_delete_text (GTK_EDITABLE (window->toolbar_entry),
				  0, -1);

	/* save window size if we are not in fullscreen and if 
	 * the window is not a popup */
	if (!GTK_CHECK_MENU_ITEM (window->view_fullscreen)->active)
	{	
		gnome_config_set_int("/galeon/Appearance/winwidth", 
				     window->WMain->allocation.width);
		gnome_config_set_int("/galeon/Appearance/winheight", 
				     window->WMain->allocation.height);
	}

	/* close all windows, this will also quit */
	galeon_close_all_windows ();
}

void
galeon_quit_with_session (void)
{
	gchar *filename;

	/* make the session filename string */
	filename = g_strconcat (g_get_home_dir (),
				"/.galeon/session_saved.xml", NULL);

	/* save it out */
	session_save(filename);

	/* set config marker */
	gnome_config_set_int ("/galeon/General/session_saved", 1);
	gnome_config_sync ();

	/* close all windows and exit */
	galeon_close_all_windows ();

}

void galeon_exit (void)
{
	gnome_config_set_int("/galeon/General/crashed", FALSE);

	preferences_save();
	bookmarks_save();
	temp_bookmarks_save ();
	history_save();

	g_assert (g_list_length (all_embeds) == 0);
	g_assert (g_list_length (all_windows) == 0);

	/* absolutely no more mozembeds */
	gtk_moz_embed_pop_startup ();

	gtk_main_quit();
}

/*
 * remove_directory: remove a directory. CREDITS gnome-core and _vicious_ that gave me that suggestion
 */

void
remove_directory(const char *dirname, gboolean just_clean)
{
        DIR *dir;
        struct dirent *dent;
        char *oldcwd;

        dir = opendir (dirname);
        if(!dir) return;
        oldcwd = g_get_current_dir();

        chdir(dirname);
        while((dent = readdir (dir)) != NULL) {
                if(strcmp(dent->d_name,".")==0 ||
                   strcmp(dent->d_name,"..")==0)
                        continue;
                if(g_file_test(dent->d_name,G_FILE_TEST_ISDIR))
                        remove_directory(dent->d_name, FALSE);
                else
                        unlink(dent->d_name);
        }
        closedir(dir);
        chdir(oldcwd);

        if(!just_clean)
                rmdir(dirname);
        g_free(oldcwd);
}

static gboolean
is_punctuation (gchar c)
{
	return (c == ' ' || c == '.' || c == '!' || c == '|' ||
		c == ',' || c == ':' || c == ';');
}

/**
 * shorten_name: try to shorten a page title (ideally to target_length or 
 * less). The heurstics here seems to give pretty good results even down
 * to quite small lengths, generally remaining comprehensible down to
 * around six to eight characters.
 */
gchar *
shorten_name(const gchar *input_name, gint target_length)
{
	gint i, j, length;
	gchar *name;
	char c;

	/* copy and clean name */
	name = g_strdup (input_name);
	g_strstrip (name);
	length = strlen (name);

	/* take care of stupid web pages that have a single non-breaking
	 * space in their titles (yes, i've seen this done) */
	if (length == 0)
	{
		free (name);
		return (g_strdup ("Untitled"));
	}

	/* look for some common prefixes -- should these be translated? */
	if (gnome_config_get_bool (CONF_APPEARANCE_TABBED_PREFIX))
	{
		/* prefixes that only come by themselves */
		if (strncasecmp (name, "index of ", 9) == 0)
		{
			length -= 9;
			memmove(name, name + 9, length + 1);
		}
		else if (strncasecmp (name, "re: ", 4) == 0)
		{
			length -= 4;
			memmove(name, name + 4, length + 1);
		}
		else if (strncasecmp (name, "fwd: ", 5) == 0)
		{
			length -= 5;
			memmove(name, name + 5, length + 1);
		}
		else if (strncasecmp (name, "www.", 4) == 0)
		{
			length -= 4;
			memmove(name, name + 4, length + 1);
		}
		else 
		{
			/* prefixes that can be followed by other
			 * prefixes */
			if (strncasecmp (name, "welcome to ", 11) == 0)
			{
				length -= 11;
				memmove(name, name + 11, length + 1);
			}

			/* prefixes that follow the ones in the
			 * previous group */
			if (strncasecmp (name, "a ", 2) == 0)
			{
				length -= 2;
				memmove(name, name + 2, length + 1);
			}
			else if (strncasecmp (name, "my ", 3) == 0)
			{
				length -= 3;
				memmove(name, name + 3, length + 1);
			}
			else if (strncasecmp (name, "the ", 4) == 0)
			{
				length -= 4;
				memmove(name, name + 4, length + 1);
			}
		}

		/* remove any leading whitespace */
		g_strchug (name);
	}

	/* check length */
	length = strlen (name);
	if (length <= target_length)
	{
		return name;
	}

	/* find in name the first occurence of one of
	 * several common separators, and set it to '\0' */
	if (gnome_config_get_bool (CONF_APPEARANCE_TABBED_SEPARATOR))
	{
		gchar *first;
		gchar *str;

		/* set first to initially point to the terminating '\0'
		 * character */
		first = name + sizeof (gchar) * strlen (name);

		/* search for various separators... we can now search
		 * for ": ", becuase because we have stripped "re:" and
		 * "fwd: " in an earlier test */
		str = strstr (name, " - ");
		if (str != NULL && str < first) first = str;
		str = strstr (name, " -- ");
		if (str != NULL && str < first) first = str;
		str = strstr (name, " | ");
		if (str != NULL && str < first) first = str;
		str = strstr (name, " || ");
		if (str != NULL && str < first) first = str;
		str = strstr (name, ": ");
		if (str != NULL && str < first) first = str;
		str = strstr (name, " :: ");
		if (str != NULL && str < first) first = str;

		*first = '\0';

		/* check length */
		g_strchomp (name);
		length = strlen (name);
		if (length <= target_length)
		{
			return name;
		}
	}

	/* getting tricky, try losing a few vowels */
	/* yes, this is very anglocentric -- I don't know of any strategies
	 * for other languages (particularly non-European) -- MattA */
	if (gnome_config_get_bool (CONF_APPEARANCE_TABBED_VOWEL))
	{
		for (i = length - 1, j = length - 1; i >= 0; i--)
		{
			c = name[i];
			if (length <= target_length ||
			    (c != 'a' && c != 'e' && c != 'i' &&
			     c != 'o' && c != 'u'))
			{
				name[j--] = c;
			}
			else
			{
				length--;
			}
		}
		/* shift */
		memmove(name, name + j + 1, length + 1);

		/* short enough yet? */
		if (length <= target_length)
		{
			return name;
		}
	}

	/* argh -- try chopping out whole words */
	for (i = target_length; i > 0; i--)
	{
		if (is_punctuation (name[i]))
		{
			while (is_punctuation (name[i]))
			{
				i--;
			}
			/* this will do */
			name[i + 1] = '\0';
			g_strchomp (name);
			return name;
		}
	}

	/* just chop off and add ellipsis */
	for (i = 0; i < 3; i++)
	{
		/* don't overflow string length */
		if (name[target_length + i] == '\0')
			break;
		
		name[target_length + i] = '.';
	}
	
	/* terminate string */
	name[target_length + i] = '\0';
	
	/* return it */
	return name;
}

/**
 * initialise_colours: initialise global colour styles
 */
void
initialise_colours (void)
{
	GdkColor red, yellow, green, blue;
	GtkWidget *label;
	GtkStyle  *rcstyle;

	/* create a dummy label so we can fetch the associated rc style */
	label = gtk_label_new("");
	rcstyle = gtk_rc_get_style(label);
	gtk_widget_destroy(label);

	/* this is needed for some (broken?) themes */
	if (rcstyle == NULL)
	{
		rcstyle = gtk_style_new ();
	}
	
	/* build colour structures */
	gdk_color_parse ("#ff0000", &red);
	gdk_color_parse ("#ffff00", &yellow);
	gdk_color_parse ("#00ff00", &green);
	gdk_color_parse ("#0000ff", &blue);

	/* make styles */
	red_text_style = gtk_style_copy(rcstyle);
	yellow_text_style = gtk_style_copy(rcstyle);
	green_text_style = gtk_style_copy(rcstyle);
	blue_text_style = gtk_style_copy(rcstyle);

	/* fill in colours */
	red_text_style->fg[0] = red;
	yellow_text_style->fg[0] = yellow;
	green_text_style->fg[0] = green;
	blue_text_style->fg[0] = blue;
}

/**
 * read_line_from_file: reads a line from an opened file and returns it in a 
 * new allocated string
 */
gchar *
read_line_from_file (FILE *f)
{
	gchar *line = g_strdup ("");
	gchar *t;
	gchar *buf = g_new0 (gchar, 256);
	while ( ! ( strchr (buf, '\n') || feof (f) ) ) {
		fgets(buf, 256, f);
		t = line;
		line = g_strconcat (line, buf, NULL);
		g_free (t);
	}
	return line;
}

/**
 * g_strcasestr: test if a string b is a substring of string a, independent
 * of case.
 */
const gchar *
g_strcasestr (const gchar *a, const gchar *b)
{
	gchar *down_a;
	gchar *down_b;
	gchar *ptr;

	/* copy and lower case the strings */
	down_a = g_strdup (a);
	down_b = g_strdup (b);
	g_strdown (down_a);
	g_strdown (down_b);

	/* compare */
	ptr = strstr (down_a, down_b);

	/* free allocated strings */
	g_free (down_a);
	g_free (down_b);
	
	/* return result of comparison */
	return ptr == NULL ? NULL : (a + (ptr - down_a));
}

static gint 
strcasestr_cb (gconstpointer a, gconstpointer b)
{
	return g_strcasestr (a, b) == NULL ? 1 : 0;
}

/*
 * create the charset titles submenu structure 
 */
void
create_charset_submenus (GtkMenuItem *encoding_menu_item, GList *charset_titles, 
		GaleonWindow *window)
{
	GtkWidget *encoding_menu, *tempw, *tempw2, *tempw3;
	GList *templ, *templ2;
	int i = 0;
	int lgroups_count = get_lgroups_count ();
	
 	encoding_menu = gtk_menu_new();
	gtk_menu_item_set_submenu (GTK_MENU_ITEM(encoding_menu_item), encoding_menu);

	templ = g_list_copy(charset_titles);
	
	for (i = 0;i < lgroups_count;i++) {
		tempw = gtk_menu_item_new_with_label (_(lgroups[i]));
		gtk_menu_append (GTK_MENU(encoding_menu), tempw);
		gtk_widget_show (tempw);
		tempw2 = gtk_menu_new();
		gtk_menu_item_set_submenu (GTK_MENU_ITEM(tempw), tempw2);
		while((templ2 = g_list_find_custom (templ, _(lgroups[i]), strcasestr_cb)) != NULL) {
			tempw3 = gtk_menu_item_new_with_label (templ2->data);
			gtk_menu_append (GTK_MENU(tempw2), tempw3);	
			templ = g_list_remove_link (templ, templ2);
			gtk_widget_show (tempw3);
			gtk_signal_connect (GTK_OBJECT (tempw3), "activate",
				window_menu_encoding_activate_cb, window);
		}
	}

	while (templ != NULL) { 
		tempw = gtk_menu_item_new_with_label (templ->data);
		gtk_menu_append (GTK_MENU(encoding_menu), tempw);
		gtk_widget_show (tempw);
		gtk_signal_connect (GTK_OBJECT (tempw), "activate",
				window_menu_encoding_activate_cb, window);
		templ = g_list_remove_link (templ, templ);
	}
}

gchar* get_mozilla_prefs_directory () 
{
	GnomeVFSResult rc;
	GList *list = NULL;
	GList *node;
	GnomeVFSFileInfo *info;
	char *prefs_dir = NULL;
	char *default_dir;

	default_dir = g_strconcat (g_get_home_dir(), MOZILLA_PREFS_DIR, NULL);
	
	rc = gnome_vfs_directory_list_load(&list,
					   default_dir,
					   (GNOME_VFS_FILE_INFO_GET_MIME_TYPE
					    | GNOME_VFS_FILE_INFO_FORCE_FAST_MIME_TYPE
					    | GNOME_VFS_FILE_INFO_FOLLOW_LINKS),
					   NULL);

	for (node = list; node; node = node->next) {
		info = node->data;

		if (info->type == GNOME_VFS_FILE_TYPE_DIRECTORY)
		{
			prefs_dir = g_strconcat(default_dir, info->name, NULL);
			continue;
		}
	}

	gnome_vfs_file_info_list_free(list);
	g_free (default_dir);

	return prefs_dir;
}

void
launch_external_viewer (const gchar *filename)
{
	pid_t pid;
	g_return_if_fail(filename != NULL);

	/* Fork and invoke the external program. */
	pid = fork ();
	if (pid == 0) {
		/* child process */
		gchar *prog = gnome_config_get_string
			(CONF_HANDLERS_EXTERNAL_SOURCE_VIEWER);
		execlp (prog, prog, filename, NULL);
		g_warning (_("Error executing external program"));
		_exit (0);
	} else if (pid < 0) {
		/* error */
		g_error ("Error creating subprocess");
		return;
	} else {
		/* parent process */
		/* fork again and give the external program time to start  
		   before removing the temp file */
		pid_t pid2 = fork ();
		if (pid2 == 0) {
				/* child */
			sleep (45); /* 45 secs should be enough */
			remove (filename);
			_exit (0);
		} else if (pid2 < 0) {
			g_error ("Error creating subprocess");
		}
	}
}

gint file_copy_status (GnomeVFSXferProgressInfo *info,gpointer data)
{
        /* FIXME show overwrite dialog */
        return TRUE;
}

gboolean
copy_file (const char *src, const char *dest)
{
	gboolean result;
	GnomeVFSURI *src_uri, *dest_uri;

	src_uri = gnome_vfs_uri_new (src);
	dest_uri = gnome_vfs_uri_new (dest);

	if (!src_uri || !dest_uri) return FALSE;

	result = gnome_vfs_xfer_uri(src_uri, dest_uri,
				    GNOME_VFS_XFER_DEFAULT,
				    GNOME_VFS_XFER_ERROR_MODE_QUERY, 
				    GNOME_VFS_XFER_OVERWRITE_MODE_QUERY,
				    (GnomeVFSXferProgressCallback)file_copy_status, 
				    (gpointer)NULL) == GNOME_VFS_OK ? TRUE : FALSE;

	gnome_vfs_uri_unref (src_uri);
	gnome_vfs_uri_unref (dest_uri);

	return result;
}

/**
 * xmlSetPropISOLatin1: set a property in an XML file which has value
 * encoded as ISOLatin1. This works around the strictness of libxml1.
 */
xmlAttrPtr
xmlSetPropISOLatin1 (xmlNodePtr node, 
		     const xmlChar *name, 
		     const xmlChar *value)
{
	xmlChar buffer[4096];
	gint count;

	/* check for nullification */
	if (value == NULL)
	{
		return xmlSetProp (node, name, NULL);
	}

	/* do the conversion */
	count = isolat1ToUTF8 (buffer, 4096, value, strlen (value));
	buffer[count] = '\0';
	
	/* store in the node */
	return xmlSetProp (node, name, g_strdup (buffer));
}

/**
 * xmlGetPropISOLatin1: get a property in an XML file which has value
 * encoded as UTF8 but was originally ISOLatin1. This works around the
 * strictness of libxml1.
 */
xmlChar *
xmlGetPropISOLatin1 (xmlNodePtr node, const xmlChar *name)
{
	xmlChar *value;
	xmlChar buffer[4096]; /* FIXME? */
	gint count;

	/* get the value */
	value = xmlGetProp (node, name);
	if (value == NULL)
	{
		return NULL;
	}

	/* do the conversion */
	count = UTF8Toisolat1 (buffer, 4096, value, strlen (value));

	/* check and do the best we can */
	if (count >= 0)
	{
		/* terminate string */
		buffer[count] = '\0';
		xmlFree (value);
		
		/* copy and return */
		return g_strdup (buffer);
	}
	else
	{
		/* return the undecoded result */
		return value;
	}
}

static void
misc_ask_file_name_ok_cb (GtkButton *button, GtkWidget *dialog) 
{
	FileCallbackFunc f = gtk_object_get_data (GTK_OBJECT (dialog), "callback_ok");
	GtkWidget *fentry = gtk_object_get_data (GTK_OBJECT (dialog), "fentry");
	gchar *fname = gnome_file_entry_get_full_path (GNOME_FILE_ENTRY (fentry), FALSE);
	gpointer data = gtk_object_get_data (GTK_OBJECT (dialog), "user_data");
	gboolean must_exist = GPOINTER_TO_INT 
		(gtk_object_get_data (GTK_OBJECT (dialog), "must_exist"));
	gboolean warn_overwrite = GPOINTER_TO_INT 
		(gtk_object_get_data (GTK_OBJECT (dialog), "warn_overwrite"));
	gboolean proceed = TRUE;
	if (must_exist && !g_file_exists (fname)) {
		GtkWidget *err_dialog = gnome_error_dialog_parented 
			(_("The file must exist.\n"
			   "Please select an existing file or chose \"Cancel\""),
			 GTK_WINDOW (dialog));
		proceed = FALSE;
	        window_set_layer (err_dialog);
	        gnome_dialog_run (GNOME_DIALOG (err_dialog));
	}
	if (warn_overwrite && g_file_exists (fname)) {
		proceed = misc_ask_question
			(_("The file already exists.\n"
			   "Do you want to overwrite it?"),
			 GTK_WINDOW (dialog));
	}
	if (proceed) {
		if (f)
			f (fname, data);
		gtk_widget_destroy (dialog);
	}
	g_free (fname);
}

static void
misc_ask_file_name_cancel_cb (GtkButton *button, GtkWidget *dialog) 
{
	FileCallbackFunc f = gtk_object_get_data (GTK_OBJECT (dialog), 
						  "callback_cancel");
	GtkWidget *fentry = gtk_object_get_data (GTK_OBJECT (dialog), "fentry");
	gchar *fname = gnome_file_entry_get_full_path (GNOME_FILE_ENTRY (fentry), FALSE);
	gpointer data = gtk_object_get_data (GTK_OBJECT (dialog), "user_data");
	if (f)
		f (fname, data);
	g_free (fname);
	gtk_widget_destroy (dialog);
}

/**
 * Show a dialog to ask for a file. When the user selects ok/cancel the given 
 * callbacks will be called.
 * TODO: extend to handle file overwriting, inexistent files...
 */
void
misc_ask_file_name (const gchar *title, const gchar *label_title, const gchar *def,
		    FileCallbackFunc callback_ok, FileCallbackFunc callback_cancel, 
		    gboolean must_exist, gboolean warn_overwrite,
		    gpointer user_data)
{
	GladeXML *gxml = glade_xml_new (user_file ("galeon.glade", TRUE), 
			"file_selection_dialog");
	GtkWidget *ok_button = glade_xml_get_widget 
		(gxml, "file_selection_dialog_ok_button");
	GtkWidget *cancel_button = glade_xml_get_widget 
		(gxml, "file_selection_dialog_cancel_button");
	GtkWidget *fentry = glade_xml_get_widget 
		(gxml, "file_selection_dialog_fileentry");
	GtkWidget *label = glade_xml_get_widget 
		(gxml, "file_selection_dialog_label");
	GtkWidget *dialog = glade_xml_get_widget
		(gxml, "file_selection_dialog");
	gtk_object_set_data (GTK_OBJECT (dialog), "callback_ok", callback_ok);
	gtk_object_set_data (GTK_OBJECT (dialog), "callback_cancel", callback_cancel);
	gtk_object_set_data (GTK_OBJECT (dialog), "must_exist", 
			     GINT_TO_POINTER (must_exist));
	gtk_object_set_data (GTK_OBJECT (dialog), "warn_overwrite", 
			     GINT_TO_POINTER (warn_overwrite));
	gtk_object_set_data (GTK_OBJECT (dialog), "user_data", user_data);
	gtk_object_set_data (GTK_OBJECT (dialog), "fentry", fentry);
	gtk_window_set_title (GTK_WINDOW (dialog), title);
	gtk_label_set_text (GTK_LABEL (label), label_title);
	gnome_file_entry_set_title (GNOME_FILE_ENTRY (fentry), title);
	gtk_entry_set_text (GTK_ENTRY (gnome_file_entry_gtk_entry
				       (GNOME_FILE_ENTRY (fentry))), def);
	gtk_object_unref (GTK_OBJECT (gxml));
	gtk_signal_connect (GTK_OBJECT (ok_button), "clicked", 
			    misc_ask_file_name_ok_cb, dialog);
	gtk_signal_connect (GTK_OBJECT (cancel_button), "clicked", 
			    misc_ask_file_name_cancel_cb, dialog);
}

/** Shared var for the next two functions */
static gboolean misc_ask_question_ret = FALSE;

/**
 * Asks a yes/no question and returns a boolean
 */
gboolean 
misc_ask_question (const gchar *question, GtkWindow *parent)
{
	GtkWidget *dialog = gnome_question_dialog_parented
		(question, misc_ask_question_cb, NULL, parent);
	window_set_layer (dialog);
	gnome_dialog_run (GNOME_DIALOG (dialog));
	return misc_ask_question_ret;
}

static void
misc_ask_question_cb (gint reply, gpointer data) {
	misc_ask_question_ret = !reply;
}

char *messages [5] = {NULL, N_("Looking for a bookmarks icon"), NULL};

/**
 * get_siteicon: Parse an http URL ans get its favicon if there is one
 */
void
get_siteicon (GaleonEmbed *embed, gchar *url)
{
	GnomeVFSURI *vfs_uri;
	gchar *favicon_url, *favicon_path;
	GList *source_uris = NULL, *target_uris = NULL;

	g_return_if_fail (url != NULL);
	
	vfs_uri = gnome_vfs_uri_new (url);
	g_return_if_fail (vfs_uri != NULL);
	
	if (embed == NULL || !mozilla_get_favicon_location (embed, &favicon_url))
	{
		GnomeVFSURI *vfs_favicon_uri;
		
		while (gnome_vfs_uri_has_parent (vfs_uri))
		{
			GnomeVFSURI *vfs_parent_uri;
			vfs_parent_uri = gnome_vfs_uri_get_parent (vfs_uri);
			gnome_vfs_uri_unref (vfs_uri);
			vfs_uri = vfs_parent_uri;
		} 
		vfs_favicon_uri = gnome_vfs_uri_append_file_name (vfs_uri, 
								 "favicon.ico");
		favicon_url = gnome_vfs_uri_to_string (vfs_favicon_uri, 0);
		gnome_vfs_uri_unref (vfs_favicon_uri);
	}
	else if (favicon_url == NULL)
	{
		gnome_vfs_uri_unref (vfs_uri);
		return;
	}

	gnome_vfs_uri_unref (vfs_uri);

	favicon_path = favicon_filename (favicon_url);

	source_uris = g_list_append(source_uris, favicon_url);
        target_uris = g_list_append(target_uris, favicon_path);

        file_operations_copy_move (source_uris, target_uris,
                                   FALSE,
                                   messages, 
				   favicon_download_completed_cb,
				   favicon_path);
	g_free (favicon_url);
}

void favicon_download_completed_cb (gpointer data)
{
	GdkPixbuf *test;
	char *favicon_path = (char*)data;
	
	test = gdk_pixbuf_new_from_file (favicon_path);
	if (test == NULL)
	{
		gnome_vfs_unlink (favicon_path);
	}
	else
	{
		gdk_pixbuf_unref (test);
	}

	bookmarks_save ();
	g_free(data);
}

PixmapData *
pixmap_data_new_from_file (const gchar *file)
{
	GdkPixbuf *pixbuf;
	PixmapData *pixmap_data = g_new (PixmapData, 1);

	pixbuf = gdk_pixbuf_new_from_file (file);
	if (pixbuf == NULL)
	{
		g_warning ("%s does not exist or is not an image file\n", file);
		return NULL;
	}
	
	gdk_pixbuf_render_pixmap_and_mask (pixbuf,
					   &(pixmap_data->pixmap),
					   &(pixmap_data->mask),
					   100);
	gdk_pixbuf_unref (pixbuf);
	
	return pixmap_data; 
}

gchar *
favicon_filename (gchar *favicon_url)
{
	GnomeVFSURI *vfs_uri = gnome_vfs_uri_new (favicon_url);
	gchar *filename, *slashpos, *result;
	
	filename = gnome_vfs_uri_to_string (vfs_uri, 
					    GNOME_VFS_URI_HIDE_USER_NAME |
					    GNOME_VFS_URI_HIDE_PASSWORD |
					    GNOME_VFS_URI_HIDE_TOPLEVEL_METHOD);	
	while ((slashpos = strstr (filename, "/")) != NULL) *slashpos = '_';
	
	result = g_strconcat (g_get_home_dir (), "/.galeon/favicons/",
			      filename, NULL);
	g_free (filename);
	
	return result;
}

gchar *
time_to_string (GTime t)
{
	if (t == 0) 
		return g_strdup ("");
	else {
	        gchar *s = g_new0 (gchar, 250);
		struct tm *stm = g_new0 (struct tm, 1);
		int i;
		gchar *ret;
        
		stm = localtime_r ( (time_t *) &t, stm);
		i = strftime (s, 250, _("%-m/%-d/%Y"), stm);
		ret = g_strndup (s, i);

		g_free (s);
		g_free (stm);
		return ret;
	}
}
