 /*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <galeon.h>

#include "nsString.h"
#include "nsIPref.h"
#include "nsICharsetConverterManager.h"
#include "nsICharsetConverterManager2.h"
#include "nsIUnicodeEncoder.h"
#include "galeon-wrapper.h"
#ifdef CVS_SOURCE
#include <time.h>
#include "nsIPrintOptions.h"
#include "nsGfxCIID.h"
#include "nsICacheService.h"
#include "nsIPassword.h"
#include "nsIPasswordManager.h"
#include "PromptService.h"
#else
#include "nsINetDataCacheManager.h"
#endif
static NS_DEFINE_CID(kPrefCID, NS_PREF_CID);
#ifdef CVS_SOURCE
static NS_DEFINE_CID(kPrintOptionsCID, NS_PRINTOPTIONS_CID);
#endif

/* local function prototypes */
static gboolean mozilla_find_setup(nsITextServicesDocument *aDoc,
				   PRInt32 *outBlockOffset,
				   gboolean search_backwards, gboolean start);
static gchar *mozilla_get_attribute (nsIDOMNode *node, gchar *attribute);
static nsIDocShell *mozilla_get_primary_docshell (GtkMozEmbed *b);
static nsIDocShell *mozilla_get_event_docshell (GtkMozEmbed *b, 
						nsIDOMMouseEvent *event);
static gboolean mozilla_resolve_url (char *base, char *relative,
				     char **resolved);
#ifndef CVS_SOURCE
	static gchar** split_list (gchar *list);
	static gchar* join_numbers (GList *cookies);
#endif
static char  *convert_ns_string_to_c_string (const nsString & ns_string);

/**
 * mozilla_preference_set: set a string mozilla preference
 */
extern "C" gboolean
mozilla_preference_set(const char *preference_name, const char *new_value)
{
  g_return_val_if_fail (preference_name != NULL, FALSE);
  g_return_val_if_fail (new_value != NULL, FALSE);
  nsCOMPtr<nsIPref> pref = do_CreateInstance(NS_PREF_CONTRACTID);

  if (pref)
    {
      nsresult rv = pref->SetCharPref (preference_name, new_value);            
      return NS_SUCCEEDED (rv) ? TRUE : FALSE;
    }
  
  return FALSE;
}

/**
 * mozilla_preference_set_boolean: set a boolean mozilla preference
 */
extern "C" gboolean
mozilla_preference_set_boolean (const char        *preference_name,
				gboolean        new_boolean_value)
{
  g_return_val_if_fail (preference_name != NULL, FALSE);
  
  nsCOMPtr<nsIPref> pref = do_CreateInstance(NS_PREF_CONTRACTID);
  
  if (pref)
    {
      nsresult rv = pref->SetBoolPref (preference_name,
				       new_boolean_value ? PR_TRUE : PR_FALSE);
      
      return NS_SUCCEEDED (rv) ? TRUE : FALSE;
    }
  
  return FALSE;
}

/**
 * mozilla_preference_set_int: set an integer mozilla preference
 */
extern "C" gboolean
mozilla_preference_set_int (const char        *preference_name,
				int        new_int_value)
{
  g_return_val_if_fail (preference_name != NULL, FALSE);
  
  nsCOMPtr<nsIPref> pref = do_CreateInstance(NS_PREF_CONTRACTID);
  
  if (pref)
    {
      nsresult rv = pref->SetIntPref (preference_name,
				       new_int_value);
      
      return NS_SUCCEEDED (rv) ? TRUE : FALSE;
    }
  
  return FALSE;
}

/**
 * mozilla_init_promptService: Initialise PromptService handler.
 */
extern "C" gboolean 
mozilla_init_promptService (void)
{
	/* Create PromptService */
	nsresult rv = NS_OK;
#ifdef CVS_SOURCE

#define NS_PROMPTSERVICE_CID \
   {0xa2112d6a, 0x0e28, 0x421f, {0xb4, 0x6a, 0x25, 0xc0, 0xb3, 0x8, 0xcb, 0xd0}}
	static NS_DEFINE_CID(kPromptServiceCID, NS_PROMPTSERVICE_CID);
     
	nsCOMPtr<nsIFactory> promptFactory;
	rv = NS_NewPromptServiceFactory(getter_AddRefs(promptFactory));
	if (NS_FAILED(rv)) return rv;
	rv = nsComponentManager::RegisterFactory(kPromptServiceCID,
						 "Prompt Service",
						 "@mozilla.org/embedcomp/prompt-service;1",
						 promptFactory,
						 PR_TRUE);  
#endif
	return NS_SUCCEEDED (rv) ? TRUE : FALSE;
}

/**
 * mozilla_wrapper_init: initialize the mozilla wrapper
 * @attach_listener: attach event listener. Currently used for
 * links drag and drop only.
 */
extern "C" gpointer 
mozilla_wrapper_init (GaleonEmbed *embed, gboolean attach_listener)
{
	GaleonWrapper *wrapper = new GaleonWrapper ();
	wrapper->attachListener = attach_listener;
	wrapper->Init (embed);
	return wrapper;
}

/**
 * mozilla_wrapper_destroy: destroy the mozilla wrapper
 */
extern "C" gboolean
mozilla_wrapper_destroy (GaleonEmbed *embed)
{
	nsresult result;
	GaleonWrapper *wrapper;

	wrapper = (GaleonWrapper *)(embed->wrapper);
	result = wrapper->Destroy();
	embed->wrapper = NULL;
	delete wrapper;
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

extern "C" gboolean mozilla_find (GaleonEmbed *embed, WrapperSearchProperties *properties)
{
	GaleonWrapper *wrapper;
	PRBool didFind;

	g_return_val_if_fail (embed != NULL, FALSE);
	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 

	wrapper = (GaleonWrapper *)embed->wrapper;

	wrapper->Find (properties->search_string, properties->match_case, 
		       properties->backwards, &didFind);

	return didFind;
}

extern "C" gboolean mozilla_save_image (GaleonEmbed *embed, const char *image, const char *file_name)
{
#if MOZILLA_MILESTONE == 81
	nsresult result;
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;

	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 

	result = wrapper->SaveURI (image, file_name);

	return NS_SUCCEEDED (result) ? TRUE : FALSE;
#else
	nsresult result;
	gchar *dirname = g_dirname(file_name);
	gchar *basename =  g_basename(file_name);
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;

	/* check if the file is local */
	if (strncmp ((gchar *)image, "file:/", 6) == 0)
	{
		return copy_file (image, file_name);
	}
	else
	{
		result = wrapper->SaveURLFromCache 
			((gchar *)image , dirname, basename);

		/* FIXME use gtm if failed */

		return NS_SUCCEEDED (result) ? TRUE : FALSE;
	}
#endif
}

extern "C" gboolean mozilla_save (GaleonEmbed *embed, const char *file_name)
{
	nsresult result;
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;

	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 

	result = wrapper->SaveDocument (file_name);

	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

extern "C" gboolean mozilla_save_main_document (GaleonEmbed *embed, const char *file_name)
{
	nsresult result;
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;
	char *target;

	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 

	result = wrapper->SaveMainDocument (file_name);

	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

extern "C" gboolean mozilla_reload (GaleonEmbed *embed)
{
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;

	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 

	nsresult result = wrapper->ReloadDocument ();
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

/**
 * mozilla_set_zoom: Sets the zoom factor of a embed
 */
extern "C" gboolean
mozilla_set_zoom (GaleonEmbed *embed, float f)
{
	GaleonWrapper *wrapper;
	gfloat current_zoom;

	g_return_val_if_fail (embed != NULL, FALSE);
	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 
	wrapper = (GaleonWrapper *)embed->wrapper;

	nsresult result = wrapper->GetZoom (&current_zoom);
	if (NS_FAILED (result) || current_zoom == f)
		return FALSE;

	result = wrapper->SetZoom (f);
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

/**
 * mozilla_get_zoom: Gets the zoom factor of a embed
 */
extern "C" gboolean
mozilla_get_zoom (GaleonEmbed *embed, float *f)
{
	GaleonWrapper *wrapper;

	g_return_val_if_fail (embed != NULL, FALSE);
	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 
	wrapper = (GaleonWrapper *)embed->wrapper;

	nsresult result = wrapper->GetZoom (f);
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

/**
 * mozilla_print: print a document
 */
extern "C" gboolean
mozilla_print (GaleonEmbed *embed, PrintInfo *info)
{
#ifdef CVS_SOURCE
	nsresult result;
	GaleonWrapper *wrapper = (GaleonWrapper *)(embed->wrapper);

	NS_WITH_SERVICE(nsIPrintOptions, options, kPrintOptionsCID, &result);
	if (NS_FAILED(result)) return NS_ERROR_FAILURE;

	if (info->pages)
	{
		options->SetStartPageRange (info->from_page);
		options->SetEndPageRange (info->to_page);
	}

	options->SetMarginTop (info->top_margin);
	options->SetMarginBottom (info->bottom_margin);
	options->SetMarginLeft (info->left_margin);
	options->SetMarginRight (info->right_margin);

	nsCString printerString(info->printer);
	nsCString fileString(info->file);
	options->SetToFileName (fileString.ToNewUnicode());
	options->SetPrintCommand (printerString.ToNewUnicode());
	options->SetPrintToFile (info->print_to_file);

	options->SetPrintReversed (info->reversed);
	options->SetPaperSize (info->paper);

	options->SetPrintSilent (PR_TRUE);

	result = wrapper->Print(options);
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
#else
	return FALSE;
#endif
}

/**
 * mozilla_session_history: Gets the session history and current position of a 
 * GtkMozWrapper. On return, *titles will be an array of char * wich must be freed.
 */
extern "C" gboolean
mozilla_session_history (GaleonEmbed *embed, char **titles[], 
			 int *count, int *index)
{
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;

	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 

	wrapper->GetSHInfo (count, index);

	char **t = g_new(char *, *count);

	for (PRInt32 i = 0; i < *count; i++) {

		nsresult result;
		PRUnichar *unicodeTitle;

		result = wrapper->GetSHTitleAtIndex(i, &unicodeTitle);

		if (NS_FAILED(result))
		{
			return NS_OK;
		}
		
		t[i] = mozilla_unicode_to_locale (unicodeTitle);
	}
	*titles = t;

	return TRUE;
}

/**
 * mozilla_session_history_go: Goes to the SHEntry at the given index
 */
extern "C" gboolean
mozilla_session_history_go (GaleonEmbed *embed, int index)
{
	nsresult result;
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;

	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 

	result = wrapper->GoToHistoryIndex (index);
	
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

extern "C" char* 
mozilla_get_document_url (GaleonEmbed *embed)
{
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;
	char *url;

	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 

	wrapper->GetDocumentUrl (&url);
	return url;
}

#ifndef CVS_SOURCE
/**
 * mozilla_view_source_mode: Turn a embed source view mode
 */
extern "C" gboolean 
mozilla_view_source_mode (GaleonEmbed *embed, gboolean view_source)
{
	nsresult result;
	GaleonWrapper *wrapper = (GaleonWrapper *)(embed->wrapper);

	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 

	result = wrapper->SetViewSourceMode (view_source ? 
					     (int)(nsIDocShell::viewSource) : 
					     (int)(nsIDocShell::viewNormal));

	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}
#endif

/**
 * mozilla_get_key_event_info:
 */
extern "C" gboolean
mozilla_get_key_event_info (GaleonEmbed *embed, gpointer event, WrapperKeyEventInfo *info)
{
	nsresult result;
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;

	g_return_val_if_fail (embed->wrapper != NULL, FALSE);
	g_return_val_if_fail (event != NULL, FALSE);

	result = wrapper->GetKeyEventContext ((nsIDOMKeyEvent *)event, info);

	return NS_SUCCEEDED(result) ? TRUE : FALSE;
}

/**
 * 
 */
extern "C" gboolean
mozilla_get_mouse_event_info (GaleonEmbed *embed, gpointer event, WrapperMouseEventInfo *context)
{
	nsresult result;
	return_val_if_not_embed (embed, FALSE);
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;
	g_return_val_if_fail (wrapper, FALSE);
	g_return_val_if_fail (event, FALSE);
	
	result = wrapper->GetMouseEventContext ((nsIDOMMouseEvent*)event, context);

	/* for some reason we get different numbers on PPC, this fixes
	 * that up... -- MattA */
	if (context->button == 65536)
	{
		context->button = 1;
	}
	else if (context->button == 131072)
	{
		context->button = 2;
	}

	return NS_SUCCEEDED(result) ? TRUE : FALSE;
}

/**
 * mozilla_list_cookies: get a list of all saved cookies
 */
extern "C" GList *
mozilla_list_cookies (void)
{
	GList *cookies = NULL;
#ifdef CVS_SOURCE
	nsresult result;

	nsCOMPtr<nsICookieManager> 
			cookieManager = do_CreateInstance (COOKIEMANAGER_ID);
	nsCOMPtr<nsISimpleEnumerator> cookieEnumerator;
	result = 
	    cookieManager->GetEnumerator (getter_AddRefs(cookieEnumerator));
	g_return_val_if_fail (NS_SUCCEEDED(result), NULL);	
	PRBool enumResult;
	for (cookieEnumerator->HasMoreElements(&enumResult) ;
	     enumResult == PR_TRUE ;
	     cookieEnumerator->HasMoreElements(&enumResult))
	{
		Cookie *c;
	
		nsCOMPtr<nsICookie> nsCookie;
		result = cookieEnumerator->GetNext (getter_AddRefs(nsCookie));
		g_return_val_if_fail (NS_SUCCEEDED(result), NULL);

		c = g_new0 (Cookie, 1);

		nsCookie->GetHost (&(c->base.domain));
		nsCookie->GetName (&(c->name));
		nsCookie->GetValue (&(c->value));
		nsCookie->GetPath (&(c->path));
		
		PRBool isSecure;
		nsCookie->GetIsSecure (&isSecure);
		if (isSecure == PR_TRUE) 
			c->secure = g_strdup_printf ("Yes");
		else 
			c->secure = g_strdup_printf ("No");

		PRUint64 dateTime;
		nsCookie->GetExpires (&dateTime);
		c->expire = g_strdup_printf ("%s",ctime((time_t*)&dateTime));
		
		cookies = g_list_append (cookies, c);
	}	
#else
	char *list;
	gchar **split;
	gchar **s;

	nsCOMPtr<nsICookieViewer> 
			cookieviewer = do_CreateInstance (COOKIEVIEWER_ID);
	
	cookieviewer->GetCookieValue (&list);

	split = split_list (list);
	for (s = split; *s != NULL; ++s) {
		Cookie *c = g_new0 (Cookie, 1);

		c->base.number = *(s++);
		c->name = *s++;
		c->value = *s++;
		c->base.type = *s++;
		c->base.domain = *s++;
		c->path = *s++;
		c->secure = *s++;
		c->expire = *s;
                g_strchomp (c->expire);

		cookies = g_list_append (cookies, c);
	}
	g_free (split);

	delete list;
#endif
	return cookies;
}

/**
 * mozilla_get_permissions:
 * @type: Type of permissions to get ( cookie or image )
 */

extern "C" GList *
mozilla_get_permissions (int type)
{
	GList *permissions = NULL;
	nsresult result;
#ifdef CVS_SOURCE

	nsCOMPtr<nsIPermissionManager> 
		permissionManager = do_CreateInstance (PERMISSIONMANAGER_ID);
	nsCOMPtr<nsISimpleEnumerator> permissionEnumerator;
	result = permissionManager->
			GetEnumerator (getter_AddRefs(permissionEnumerator));
	g_return_val_if_fail (NS_SUCCEEDED(result), NULL);
	PRBool enumResult;
	for (permissionEnumerator->HasMoreElements(&enumResult) ;
	     enumResult == PR_TRUE ;
	     permissionEnumerator->HasMoreElements(&enumResult))
	{
		nsCOMPtr<nsIPermission> nsPermission;
		result = permissionEnumerator->
				GetNext (getter_AddRefs(nsPermission));
		g_return_val_if_fail (NS_SUCCEEDED(result), NULL);		

		PRInt32 cType;
		nsPermission->GetType (&cType);
		if (cType == type)
		{
			CookieBase *b = g_new0 (CookieBase, 1);

			nsPermission->GetHost (&(b->domain));

			PRBool cap;
			nsPermission->GetCapability (&cap);
			if (cap == PR_TRUE) 
				b->type = g_strdup_printf ("Allowed");
			else 
				b->type = g_strdup_printf ("Blocked");

			permissions = g_list_append (permissions, b);
		}
	}
#else
	char *list;
	gchar **split, **s;

	nsCOMPtr<nsICookieViewer> 
			cookieviewer = do_CreateInstance (COOKIEVIEWER_ID);
	
	result = cookieviewer->GetPermissionValue (type, &list);
	g_return_val_if_fail (NS_SUCCEEDED(result), NULL);

	split = split_list (list);
	for (s = split; *s != NULL; ++s) {
		CookieBase *b = g_new0 (CookieBase, 1);

		b->number = *s++;

		b->type = g_strndup (*s, 1);
		b->domain = g_strdup (*s+1);
		g_free (*s);

		permissions = g_list_append (permissions, b);
	}
	
	g_free (split);
	delete list;
#endif
	return permissions;
};

/**
 * mozilla_block_url:
 * @imgURL: url of site to block
 */
extern "C" void
mozilla_block_url (const char *imgURL)
{
	nsresult result;
	
#ifdef	CVS_SOURCE
	nsCOMPtr<nsIImgManager> 
		imageManager = do_CreateInstance (IMAGEMANAGER_ID);
	
	result = imageManager->Block (imgURL);
#else
	nsCOMPtr<nsICookieViewer> 
		cookieviewer = do_CreateInstance (COOKIEVIEWER_ID);
	
	result = cookieviewer->BlockImage (imgURL);
#endif
	g_return_if_fail (NS_SUCCEEDED(result));
}

/**
 * mozilla_set_permission:
 * @permit: whether to block or allow
 * @type: type of permission to set ( cookie or image )
 */
 
extern "C" void
mozilla_set_permission (GaleonEmbed *embed, gboolean permit, int type)
{
	nsresult result;
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;

	if (!wrapper) return;

	wrapper->SetSitePermission (permit ? PR_TRUE : PR_FALSE, type);
}

#ifdef CVS_SOURCE

/**
 * mozilla_remove_cookies:
 * @gone: list of cookies which are to be removed
 * @block: whether to block the originating servers in future
 */
extern "C" gboolean 
mozilla_remove_cookies (GList *gone, gboolean block)
{
	nsresult result;
	nsCOMPtr<nsICookieManager> 
		cookieManager = do_CreateInstance (COOKIEMANAGER_ID);

	for (GList *cookies = g_list_first(gone) ; cookies!=NULL ; 
	     cookies = g_list_next(cookies))
	{
		Cookie *c = (Cookie *)cookies->data;
		result = cookieManager->Remove (c->base.domain,c->name,
					 	c->path,
						block ? PR_TRUE : PR_FALSE);
		if (NS_FAILED(result)) return FALSE;
	};
	return TRUE;
}

/**
 * mozilla_remove_permissions:
 * @gone: list of cookie or image permissions which are to be removed
 * @type: type of permissions ( cookies or images )
 */
extern "C" gboolean 
mozilla_remove_permissions (GList *gone, int type)
{
	nsresult result;
	nsCOMPtr<nsIPermissionManager> 
		permissionManager = do_CreateInstance( PERMISSIONMANAGER_ID );

	for (GList *permissions = g_list_first(gone) ; permissions!=NULL;
	     permissions = g_list_next(permissions))
	{
		CookieBase *b = (CookieBase *)permissions->data;
		result = permissionManager->Remove(b->domain,type);
		if (NS_FAILED(result)) return FALSE;
	};
	return TRUE;
}

#else //CVS_SOURCE

/**
 * mozilla_set_cookies: remove a set of cookies from the store
 * @gone_c: cookies to remove
 * @gone_p: servers to remove from cookie permissions
 * @gone_i: servers to remove from image permissions
 * @block:  whether to block the cookie in future
 */
extern "C" gboolean
mozilla_set_cookies (GList *gone_c, GList *gone_p, GList *gone_i,
		     gboolean block)
{
	nsresult result;
	gchar *value;
	gchar *gone_cs, *gone_ps, *gone_is, *block_s;

	if (!gone_c && !gone_p && !gone_i) return TRUE;

	gone_cs = join_numbers (gone_c);
	gone_ps = join_numbers (gone_p);
	gone_is = join_numbers (gone_i);
	if (block)
		block_s = "true";
	else 
		block_s = "false";

	value = g_strconcat ("|goneC|", gone_cs,
			     "|goneP|", gone_ps,
			     "|goneI|", gone_is,
			     "|block|", block_s,
			     "|", NULL);

	g_free (gone_cs);
	g_free (gone_ps);
	g_free (gone_is);

	nsCOMPtr<nsICookieViewer> 
			cookieviewer = do_CreateInstance (COOKIEVIEWER_ID);

	result = cookieviewer->SetValue (value, NULL);

	g_free (value);
	return NS_SUCCEEDED(result) ? TRUE : FALSE;
}

/**
 * split_list: split a list of cookies
 */
static gchar **
split_list (gchar *list)
{
	gchar *delim;
	gchar **split;

	delim = g_strndup (list, 1);
	
	if (!(*list)) return g_new0 (gchar *, 1);

	split = g_strsplit (list+1, delim, -1);

	g_free (delim);

	return split;
}

/**
 * join_numbers:
 */
static gchar*
join_numbers (GList *cookies)
{
	gchar **str_array, *join;
	GList *elem;
	int i = 0;

	int len = g_list_length (cookies);
	str_array = g_new (gchar*, len + 2);

	for (elem = cookies; elem != NULL; elem = g_list_next(elem))
		str_array[i++] = ((CookieBase*)elem->data)->number;
	if (cookies)
		str_array[i++] = "";
	str_array[i] = NULL;

	join = g_strjoinv (",", str_array);

	g_free (str_array);
	return join;
}

#endif //CVS_SOURCE

/**
 * mozilla_copy_session_history: copy the session history to another embed
 */
extern "C" gboolean
mozilla_copy_session_history (GaleonEmbed *embed, GaleonEmbed *dest)
{
	g_return_val_if_fail (embed != NULL, FALSE);
	g_return_val_if_fail (dest != NULL, FALSE);
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;
	GaleonWrapper *dest_wrapper = (GaleonWrapper *)dest->wrapper;
	g_return_val_if_fail (wrapper != NULL, FALSE);
	g_return_val_if_fail (dest_wrapper != NULL, FALSE);
	nsresult result = wrapper->CopyHistoryTo (dest_wrapper);
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

/**
 * mozilla_get_charsets: get a list of charset supported by mozilla
 */
extern "C" void
mozilla_get_charsets (GHashTable **charsets, GList **sorted_charset_titles)
{
	nsresult rv;
	PRUint32 cscount;
	PRUint32 translated_cscount = get_translated_cscount ();

	nsCOMPtr<nsIAtom> docCharsetAtom;
	nsCOMPtr<nsICharsetConverterManager2> ccm2 =
		do_GetService (NS_CHARSETCONVERTERMANAGER_CONTRACTID, &rv);
	if (!NS_SUCCEEDED(rv)) return;

	nsCOMPtr <nsISupportsArray> cs_list;
	rv = ccm2->GetDecoderList(getter_AddRefs(cs_list));
	if (!NS_SUCCEEDED(rv)) return;

	rv = cs_list->Count(&cscount);
	*charsets = g_hash_table_new(g_str_hash,g_str_equal);
	for (PRUint32 i = 0; i < cscount; i++)
	{
		nsCOMPtr<nsISupports> cssupports =
					(dont_AddRef)(cs_list->ElementAt(i));
		nsCOMPtr<nsIAtom> csatom ( do_QueryInterface(cssupports) );
		nsString charset_ns = nsnull, charset_title_ns = nsnull;

		/* charset name */
		rv = csatom->ToString(charset_ns);
		char *charset_str = convert_ns_string_to_c_string (charset_ns);

		/* charset readable title */
		rv = ccm2->GetCharsetTitle2(csatom, &charset_title_ns);
		char *charset_title_str = convert_ns_string_to_c_string 
						(charset_title_ns);

		if (strlen(charset_title_str) == 0)
		{
			charset_title_str = g_strdup(charset_str);
		}
		
		for (PRUint32 j = 0; j < translated_cscount; j++)
		{
			if (g_strcasecmp (charset_title_str, 
					  charset_trans_array[j]) == 0)
			{
				g_free(charset_title_str);
				charset_title_str = (char*)_(charset_trans_array[j]);
				break;
			}
		}

		/* fill the hash and the sorted list */
		g_hash_table_insert (*charsets, charset_title_str, charset_str);
		*sorted_charset_titles = g_list_insert_sorted (
						(GList*)*sorted_charset_titles,
						(gpointer)charset_title_str,
						(GCompareFunc)g_strcasecmp); 
	}
}

/**
 * mozilla_force_character_set: force the embed to use the specified
 * character set
 */
extern "C" void
mozilla_force_character_set (GaleonEmbed *embed, char *force_character_set)
{
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;
	wrapper->ForceCharacterSet (force_character_set);
}

/**
 * mozilla_clear_cache: Clear the global memory/disk cache
 */
extern "C" void
mozilla_clear_cache (unsigned long cache)
{
	nsresult result;

#ifdef CVS_SOURCE
	nsCOMPtr<nsICacheService> 
		CacheService = do_CreateInstance (CACHESERVICE_ID);
	CacheService->EvictEntries(cache);
	
#else
	nsCOMPtr<nsINetDataCacheManager> CacheManager = 
		do_GetService(NS_NETWORK_CACHE_MANAGER_CONTRACTID, &result);
	if (NS_FAILED (result)) return;

	CacheManager->Clear(cache);
#endif
}

/* This nonsense is needed to get the allocators right */
static char *
convert_ns_string_to_c_string (const nsString & ns_string)
{
	char *c_string;
	char *ns_c_string = ns_string.ToNewCString ();

	if (ns_c_string == NULL)
	{
		return NULL;
	}

	c_string = g_strdup (ns_c_string);

	nsMemory::Free (ns_c_string);

	return c_string;
}

extern "C" gboolean
mozilla_get_favicon_location (GaleonEmbed *embed, gchar **url)
{
	nsresult result;
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;
	
	result = wrapper->GetFaviconURL (url);
	
	return NS_SUCCEEDED (result);
}

#ifdef CVS_SOURCE

/**
 * mozilla_list_passwords: get a list of all saved username/password pairs
 */
extern "C" GList *
mozilla_list_passwords (PasswordType type)
{
	GList *passwords = NULL;
	nsresult result;

	nsCOMPtr<nsIPasswordManager> 
		passwordManager = do_CreateInstance (PASSWORDMANAGER_ID);
	nsCOMPtr<nsISimpleEnumerator> passwordEnumerator;
	if (type == PASSWORD)
		result = passwordManager->GetEnumerator 
				(getter_AddRefs(passwordEnumerator));
	else if (type == REJECT)
		result = passwordManager->GetRejectEnumerator 
				(getter_AddRefs(passwordEnumerator));
	g_return_val_if_fail (NS_SUCCEEDED(result), NULL);	

	PRBool enumResult;
	for (passwordEnumerator->HasMoreElements(&enumResult) ;
	     enumResult == PR_TRUE ;
	     passwordEnumerator->HasMoreElements(&enumResult))
	{
		nsCOMPtr<nsIPassword> nsPassword;
		result = passwordEnumerator->GetNext 
					(getter_AddRefs(nsPassword));
		g_return_val_if_fail (NS_SUCCEEDED(result), NULL);

		Password *p = g_new0 (Password, 1);

		nsPassword->GetHost (&(p->host));

		if (type==PASSWORD)
		{
			PRUnichar *unicodeName;
			nsPassword->GetUser (&unicodeName);
			p->username = mozilla_unicode_to_utf8 (unicodeName);
		}
		
		passwords = g_list_append (passwords, p);
	}	
	return passwords;
}

/**
 * mozilla_remove_passwords:
 * @gone: list of passwords/rejcts which are to be removed
 * @type: whether list is passwords or rejects
 */
extern "C" gboolean 
mozilla_remove_passwords (GList *gone, PasswordType type)
{
	nsresult result;
	nsCOMPtr<nsIPasswordManager> 
		passwordManager = do_CreateInstance (PASSWORDMANAGER_ID);

	for (GList *passwords = g_list_first(gone) ; passwords!=NULL ; 
	     passwords = g_list_next(passwords))
	{
		Password *p = (Password *)passwords->data;
		if (type == PASSWORD)
		{
			PRUnichar *unicodeName = mozilla_locale_to_unicode
							(p->username);
			result = passwordManager->RemoveUser (p->host,
							      unicodeName);
			g_free (unicodeName);
		}
		else if (type == REJECT)
		{
			result = passwordManager->RemoveReject (p->host);
		};

		if (NS_FAILED(result)) return FALSE;
	};
	return TRUE;
}

#endif

/**
 * mozilla_unicode_to_utf8: Encodes unicode string to UTF-8
 * @uniStr: The unicode string to encode
 */
extern "C" gchar *
mozilla_unicode_to_utf8 (const PRUnichar *uniStr)
{
	PRInt32 sSize,dSize;
	nsresult result;
#ifdef CVS_SOURCE
	const nsLocalString str = nsLiteralString (uniStr);
	sSize = str.Length ();
#else
	for (sSize = 0; uniStr[sSize]!=0 ; sSize++);
#endif	 
	nsCOMPtr<nsIUnicodeEncoder> 
		unicodeEncoder = do_CreateInstance (UTF8_ENCODER_ID);

	/* GetMaxLength returns a worst case prediction for the size
	   of the returned char*. Using it ensures that Convert will
	   not run out of space */
	result = unicodeEncoder->GetMaxLength (uniStr, sSize, &dSize);
	gchar *utfStr = g_new0 (gchar, dSize + 1);

	/* Convert must be passed the size of unicode string and
	   the size of the char* buffer so that bad things(tm)
	   won't happen. No null termination here. */
	result = unicodeEncoder->Convert (uniStr, &sSize, utfStr, &dSize);
					  
	/* Finish ensures that the encoder is left in a clean state
	   for it's next use. */
	result = unicodeEncoder->Finish (utfStr, &dSize);

	/* Normally a null would need to appended at this point but
	   cStr was initialised to zero, so there's no need */

	return utfStr;
}

/**
 * mozilla_unicode_to_locale: Encodes unicode string to something
 * valid for the current locale (which can then be used in GTK+ labels).
 * @uniStr: The unicode string to encode
 */
extern "C" gchar *
mozilla_unicode_to_locale (const PRUnichar *uniStr)
{
	PRInt32 sSize;
	wchar_t *wide;
	gchar *output;
	gint i, count;

	/* sanity */
	if (uniStr == NULL)
	{
		return NULL;
	}

#ifdef CVS_SOURCE
	const nsLocalString str = nsLiteralString (uniStr);
	sSize = str.Length ();
#else
	for (sSize = 0; uniStr[sSize]!=0 ; sSize++);
#endif	 

	/* allocate a wide string big enough to hold the unicode string,
	 * this is necessary since wchar_t is 32-bits with glibc */
	wide = g_new0 (wchar_t, sSize + 1);
	for (i = 0; i < sSize + 1; i++)
	{
		wide[i] = uniStr[i];
	}

	/* use glibc function to determine the size of this string once
	 * encoded to a locale specfic multibyte string */
	count = wcstombs (NULL, wide, 0);

	/* check for success */
	if (count == -1)
	{
		/* not sure what we should do here, but it seems safe to
		 * fallback to encode_to_utf8 */
		g_free (wide);
		return mozilla_unicode_to_utf8 (uniStr);
	}

	/* allocate a string big enough and do the actual conversion */
	output = g_new0 (gchar, count + 1);
	count = wcstombs (output, wide, count + 1);
	g_assert (count != -1);

	/* free wide version and return */
	g_free (wide);
	return output;
}

/**
 * mozilla_utf8_to_unicode: Decodes UTF-8 string to unicode
 * @utfStr: The unicode string to encode
 */
extern "C" PRUnichar *
mozilla_utf8_to_unicode (const gchar *utfStr)
{
	PRInt32 sSize,dSize;
	nsresult result;
	
	for (sSize = 0; utfStr[sSize] != 0; sSize++);

	nsCOMPtr<nsIUnicodeDecoder> unicodeDecoder = 
					do_CreateInstance (UTF8_DECODER_ID);

	/* Unicode decoding is much the same as encoding as
	   described in mozilla_session_history comments, 
	   but there is no Finish function needed to complete
	   the process */
	result = unicodeDecoder->GetMaxLength (utfStr, sSize, &dSize);
	PRUnichar *uniStr = g_new0 (PRUnichar, dSize + 1);

	result = unicodeDecoder->Convert (utfStr, &sSize, uniStr, &dSize);

	return uniStr;
}

/**
 * mozilla_locale_to_unicode: Decodes a string encoded for the current
 * locale into unicode. Used for getting text entered in a GTK+ entry
 * into a form which mozilla can use.
 * @locStr: The unicode string to encode
 */
extern "C" PRUnichar *
mozilla_locale_to_unicode (const gchar *locStr)
{
	PRUnichar *uniStr;
	wchar_t *wide;
	gchar *output;
	gint i, count;

	/* sanity */
	if (locStr == NULL)
	{
		return NULL;
	}

	/* count the number of wide characters which will be produced */
	count = mbstowcs (NULL, locStr, 0);
	if (count == -1)
	{
		/* hmm, shouldnt happen but fallback to utf8 */
		g_warning ("unusual locale string: [%s]\n", locStr);
		return mozilla_utf8_to_unicode (locStr);
	}

	/* allocate and decode */
	wide = g_new0 (wchar_t, count + 1);
	mbstowcs (wide, locStr, count + 1);

	/* make a unicode string and copy into it */
	uniStr = g_new0 (PRUnichar, count + 1);
	for (i = 0; i < count + 1; i++)
	{
		uniStr[i] = wide[i];
	}

	/* free wide string and return the unicode one */
	g_free (wide);
	return uniStr;
}
