/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * Netscape bookmark importing by Nate Case <nd@kracked.com>
 * Netscape preference importing patch by Andreas Heck <aheck@pro-linux.de>
 * Netscape bookmark exporting by Ricardo Fernndez Pascual <ric@users.sourceforge.net>
 */

#include <galeon.h>

/* local data types */

typedef struct
{
	gchar *homepage;
	gint win_height, win_width;
	gint startup_page;
	gint toolbar_style;
	gboolean enable_java, enable_javascript;
	gchar *http_proxy, *ftp_proxy, *no_proxies;
	gint proxy_type, http_proxy_port, ftp_proxy_port;
} ns_prefs;


/* local function prototypes */
static NSItemType ns_get_bookmark_item (FILE *f, GString *name,
					  GString *url);
static char *ns_parse_bookmark_item (GString *string);
static ns_prefs *get_ns_prefs(gchar *);

void netscape_export_bookmarks_item (FILE *file, BookmarkItem *b);


/**
 * updates list, tree widget, and updates menu 
 * as the intelligence of this has improved, it's also gotten big and complex.
 * FIXME: needs modularization - Nate
 */
gint 
netscape_import_bookmarks (GString *filename)
{
	BookmarkItem *parent, *newcat, *newsite;
	FILE *bf;  /* bookmark file */
	GString *name = g_string_new (NULL);
	gchar *parsedname;
	GString *url = g_string_new (NULL);
	GSList *urls, *l;
	int dup = 0;
	GSList * folders;
	
	if (!(bf = fopen (filename->str, "r"))) {
		g_warning ("Failed to open file: %s\n", filename->str);
		return -1;
	}	
	
	if (bookmarks_root)
	{
		urls = bookmarks_list_all_urls (bookmarks_root);
		folders = bookmarks_list_all_folders (bookmarks_root);
	}
	else
	{
		urls = NULL;
		folders = NULL;
	}
	
	parent = bookmarks_root;
	
	while (!feof (bf)) {
		switch (ns_get_bookmark_item(bf, name, url)) {
		case NS_FOLDER: 

			newcat = NULL;

			for (l = folders; l != NULL; l = l->next)
			{    /* use existing folder if it's already there */
				if (l->data)
				{
					gchar *stripped = strip_uline_accel (
						((BookmarkItem *)(l->data))->name);
					if (!g_strcasecmp(stripped, name->str))
					{
						g_free (stripped);
						newcat = (BookmarkItem *) (l->data);
						break;
					}
					g_free (stripped);
				}
			}
					
			if (!newcat)
			{
				newcat = bookmarks_new_bookmark 
					(BM_CATEGORY, TRUE, name->str, 
					 NULL, NULL, NULL, NULL);
				newcat->parent = parent;
				newcat->create_toolbar = strcmp
					(name->str, 
					 _("Personal Toolbar Folder")) == 0 
					? TRUE : FALSE;
				parent->list = g_list_insert (parent->list, newcat, -1);
				newcat->parent = parent;
				if (bookmarks_editor)
					bookmarks_editor_place_tree_item 
						(bookmarks_editor, newcat);
			}
			
			parent = newcat;
			break;
		case NS_SITE:
			dup = FALSE;
			
			for(l = urls; l != NULL; l = l->next)
			{   /* ignore if duplicate bookmark */
				if (l->data)
					if (!g_strcasecmp(l->data, url->str))
					{
						dup = TRUE;
						break;
					}
			}
			if (dup) break;
			
			parsedname = ns_parse_bookmark_item (name);
			newsite = bookmarks_new_bookmark 
				(BM_SITE, TRUE, parsedname, url->str, 
				 NULL, NULL, NULL);
			g_free (parsedname);
			newsite->parent = parent;
			parent->list = g_list_insert(parent->list, newsite, -1);
			if (bookmarks_editor)
				bookmarks_editor_place_tree_item 
					(bookmarks_editor, newsite);
			break;
		case NS_SEPARATOR:
			/* how to check if a separator is duplicated? */
			newsite = bookmarks_new_bookmark 
				(BM_SEPARATOR, TRUE, NULL, NULL, NULL, NULL, NULL);
			newsite->parent = parent;
			parent->list = g_list_insert(parent->list, newsite, -1);
			if (bookmarks_editor)
				bookmarks_editor_place_tree_item 
					(bookmarks_editor, newsite);
			break;
		case NS_FOLDER_END:
			if (parent != bookmarks_root)
				parent = parent->parent;
			break;
		default:
			break;
		}		
	}
	fclose (bf);
	g_string_free (name, TRUE);
	g_string_free (url, TRUE);
	g_slist_free(urls);
	g_slist_free(folders);
		
	if (bookmarks_editor)
		bookmarks_editor->dirty = TRUE;
	bookmarks_save ();

	return 0;	
}

/** 
 * Parses a line of a mozilla/netscape bookmark file. File must be open.
 */
/* this has been tested fairly well */
static NSItemType 
ns_get_bookmark_item (FILE *f, GString *name, GString *url)
{
	char *line = NULL;
	char *found;

	line = read_line_from_file (f);

	if ((found = (char *) g_strcasestr (line, "<A HREF="))) {  /* declare site? */
		g_string_assign (url, found+9);  /* url=URL+ ADD_DATE ... */
		g_string_truncate (url, strstr(url->str, "\"")-url->str);
		found = (char *) strstr (found+9+url->len, "\">");
		if (!found) return NS_UNKNOWN;
		g_string_assign (name, found+2);
		g_string_truncate (name, g_strcasestr(name->str, "</A>")-name->str);
		return NS_SITE;
	} else if ((found = (char *) g_strcasestr (line, "<DT><H3"))) { /* declare folder? */
		found = (char *) strstr(found+7, ">");
		if (!found) return NS_UNKNOWN;
		g_string_assign (name, found+1);
		g_string_truncate (name, g_strcasestr(name->str, "</H3>")-name->str);
		return NS_FOLDER;		
	} else if ((found = (char *) g_strcasestr (line, "</DL>"))) {     /* end folder? */
		return NS_FOLDER_END;
	} else if ((found = (char *) g_strcasestr (line, "<HR>"))) {    /* separator */
		return NS_SEPARATOR;
	}
	
	return NS_UNKNOWN;
}

/**
 * This function replaces some weird elements
 * like &amp; &le;, etc..
 * More info : http://www.w3.org/TR/html4/charset.html#h-5.3.2
 * NOTE : We don't support &#D or &#xH.
 * Patch courtesy of Almer S. Tigelaar <almer1@dds.nl>
 */
static char *
ns_parse_bookmark_item (GString *string)
{
	char *iterator, *temp;
	int cnt = 0;
	GString *result = g_string_new (NULL);

	g_return_val_if_fail (string != NULL, NULL);
	g_return_val_if_fail (string->str != NULL, NULL);

	iterator = string->str;
	
	for (cnt = 0, iterator = string->str;
	     cnt <= strlen (string->str);
	     cnt++, iterator++) {
		if (*iterator == '&') {
			int jump = 0;
			int i;

			if (g_strncasecmp (iterator, "&amp;", 5) == 0) {
			
				g_string_append_c (result, '&');
				jump = 5;
			} else if (g_strncasecmp (iterator, "&lt;", 4) == 0) {
			
				g_string_append_c (result, '<');
				jump = 4;
			} else if (g_strncasecmp (iterator, "&gt;", 4) == 0) {

				g_string_append_c (result, '>');
				jump = 4;
			} else if (g_strncasecmp (iterator, "&quot;", 6) == 0) {
			
				g_string_append_c (result, '\"');
				jump = 6;
			} else {
				/* It must be some numeric thing now */

				iterator++;

				if (iterator && *iterator == '#') {
					int val;
					char *num, *tmp;
					
					iterator++;
					
					val = atoi (iterator);

					tmp = g_strdup_printf ("%d", val);
					jump = strlen (tmp);
					g_free (tmp);
					
					num = g_strdup_printf ("%c", (char) val);
					g_string_append (result, num);
					g_free (num);
				}
			}
			for (i = jump - 1; i > 0; i--) {
				iterator++;
				if (iterator == NULL)
					break;
			}
		} else 
			g_string_append_c (result, *iterator);
	}
	temp = result->str;
	g_string_free (result, FALSE);
	return temp;
}

/**
 * Exports bookmarks to netscape or mozilla
 * Will overwrite the file!
 */
gint 
netscape_export_bookmarks (gchar *filename, BookmarkItem *root)
{
	FILE *file;
	GList *l;

	g_return_val_if_fail (root->type == BM_CATEGORY, -1);
	g_return_val_if_fail (filename != NULL, -1);

	if (!(file = fopen (filename, "w"))) {
		gchar *error = g_strdup_printf (_("Could not open file: %s"), filename);
		gnome_error_dialog (error);
		g_free (error);
		return -1;
	}

	/* write the headers */
	fputs ("<!DOCTYPE NETSCAPE-Bookmark-file-1>\n", file);
	fputs ("<!-- This file was automatically generated by Galeon\n", file);
	fputs ("It will be read and overwritten.\n", file);
	fputs ("Do Not Edit! -->\n", file);
	fputs ("<TITLE>Bookmarks</TITLE>\n", file);
	fputs ("<H1>Bookmarks</H1>\n", file);
	fputs ("\n", file);
	fputs ("<DL><p>\n", file);

	/* write the bookmarks */
	for (l = root->list; l != NULL; l = l->next) 
		netscape_export_bookmarks_item (file, l->data);
	
	/* finish */
	fputs ("</DL><p>\n", file);

	fclose (file);	
	return 0;
}

void 
netscape_export_bookmarks_item (FILE *file, BookmarkItem *b)
{
	GList *l;
	gchar *strippedname;
	switch (b->type) {
	case BM_SITE:
		fputs ("\t<DT><A HREF=\"", file);
		fputs (b->url, file);
		fputs ("\"", file);
		/* support a mozilla-only feature, but won't harm netscape */
		if (strlen (b->nick) > 0) {
			fputs (" SHORTCUTURL=\"", file);
			fputs (b->nick, file);
			fputs ("\"", file);
		}
		fputs (">", file);
		strippedname = strip_uline_accel (b->name);
		fputs (strippedname, file);
		g_free(strippedname);
		fputs ("</A>\n", file);
		break;
	case BM_CATEGORY:
		fputs ("<DT><H3 ADD_DATE=\"0\">", file); 
		strippedname = strip_uline_accel (b->name);
		fputs (strippedname, file);
		g_free(strippedname);
		fputs ("</H3>\n<DL><p>\n", file);
		/* FIXME: Alias catogories are not exported */
		if (!b->alias_of)
			for (l = b->list; l != NULL; l = l->next) 
				netscape_export_bookmarks_item (file, l->data);
		else
			g_warning ("Detected a bookmark item type that I can't export!");
		fputs ("</DL><p>\n", file);
		break;
	case BM_SEPARATOR:
		fputs ("<HR>\n", file);
		break;
	default:
		g_warning ("Detected a bookmark item type that I can't export!");
		break;
	}
}

static ns_prefs *
get_ns_prefs(gchar *nsprefsfile)
{
	static ns_prefs prefs;
	FILE *fp;
	gchar buffer[512];
	gchar *pos;
	GString *pkey = g_string_new("");
	GString *value = g_string_new("");
	int startpos = 0;

	/* preferences struct init start */

	prefs.startup_page = 1;
	prefs.toolbar_style = 2;
	prefs.enable_java = TRUE;
	prefs.enable_javascript = TRUE;

	/* preferences struct init end */

	if ((fp = fopen(nsprefsfile, "r")) == NULL)
	{
		return NULL;
	}

	while (!feof(fp) && !ferror(fp))
	{
		fgets(buffer, 512, fp);

		if (strstr(buffer, "user_pref"))  /* line is a user_pref definition */
		{
			if ((startpos = strcspn(buffer, "\"")))
			{
				memcpy(buffer, &buffer[startpos], strlen(&buffer[startpos]));

				g_string_assign(pkey, "");

				if ((pos = (char *)strstr(buffer, "\"")))
				{
					g_string_assign(pkey, pos+1);
					pos = (char *)strstr(pkey->str, "\"");
					g_string_truncate(pkey, pos-pkey->str);
				}

				if ((pos = (char *)strstr(buffer, " ")))
				{
					g_string_assign(value, pos+1);  /* get the value */
					
					if ((pos = (char *)strstr(value->str, ")")))
						g_string_truncate(value, pos-value->str);

					while ((pos = (char *)strstr(value->str, "\""))) /* strip " */
					{
						g_string_erase(value, pos-value->str, 1);
					}
				}

				/* g_print("Key: %s\tValue: %s\n", pkey->str, value->str); */

				if (strcmp(pkey->str, "browser.startup.homepage") == 0)
					prefs.homepage = g_strdup(value->str);
				else if (strcmp(pkey->str, "browser.win_height") == 0)
					prefs.win_height = atoi(value->str);
				else if (strcmp(pkey->str, "browser.win_width") == 0)
					prefs.win_width = atoi(value->str);
				else if (strcmp(pkey->str, "security.enable_java") == 0)
				{
					if (strcmp(value->str, "false") == 0)
						prefs.enable_java = FALSE;
				}
				else if (strcmp(pkey->str, "javascript.enabled") == 0)
				{
					if (strcmp(value->str, "false") == 0)
						prefs.enable_javascript = FALSE;
				}
				else if (strcmp(pkey->str, "browser.startup.page") == 0)
				{
					prefs.startup_page = atoi(value->str);

					/*
					 * 0: Browser starts with blank page
					 * 1/No value: Last page visited
					 * 2: Starts with homepage
					 */
				}
				else if (strcmp(pkey->str, "network.proxy.http") == 0)
					prefs.http_proxy = g_strdup(value->str);
				else if (strcmp(pkey->str, "network.proxy.ftp") == 0)
					prefs.ftp_proxy = g_strdup(value->str);
				else if (strcmp(pkey->str, "network.proxy.no_proxies_on") == 0)
				{
					prefs.no_proxies = g_strdup(value->str);
				}
				else if (strcmp(pkey->str, "network.proxy.type") == 0)
				{
					prefs.proxy_type = atoi(value->str);

					/*
					 * 0: no proxy
					 * 1: manual proxy
					 * 2: auto proxy setup
					 */
				}
				else if (strcmp(pkey->str, "network.proxy.http_port") == 0)
					prefs.http_proxy_port = atoi(value->str);
				else if (strcmp(pkey->str, "network.proxy.ftp_port") == 0)
					prefs.ftp_proxy_port = atoi(value->str);
			        else if (strcmp(pkey->str, "browser.chrome.toolbar_style") == 0)
					prefs.toolbar_style = atoi(value->str);
					
					/*
					 * 0: Pictures only toolbar
					 * 1: Text only toolbar
					 * 2: Pictures and text toolbar
					 */
			}
		}
	}

	fclose(fp);
	return &prefs;
}

/* returns TRUE if successful */
gboolean
netscape_import_prefs(void)
{
	ns_prefs *prefs;
	gchar *prefpath;
	gint buffer = 0;

	prefpath = g_strconcat(g_get_home_dir(), "/.netscape/preferences.js",
			NULL);

	if (!(prefs = get_ns_prefs(prefpath)))
		return FALSE;

	/* now we set some of galeon prefs to whats in "ns_prefs *prefs" */

	gnome_config_set_string("/galeon/General/startpage", prefs->homepage);
	switch (prefs->startup_page)
	{
		case 0: buffer = 2; break;
		case 1: buffer = 0; break;
		case 2: buffer = 1; break;
	}
	if (prefs->proxy_type == 1)  /* if manual proxy (auto not yet supported) */
	{
		gnome_config_push_prefix("/galeon/Advanced/");
		gnome_config_set_bool ("proxy_enabled", prefs->proxy_type > 0);
		gnome_config_set_string("http_proxy", prefs->http_proxy);
		gnome_config_set_string("ftp_proxy", prefs->ftp_proxy);
		gnome_config_set_int("http_proxy_port", prefs->http_proxy_port);
		gnome_config_set_int("ftp_proxy_port", prefs->ftp_proxy_port);
		gnome_config_set_string("no_proxies_for", prefs->no_proxies);
		gnome_config_pop_prefix();
	}
	gnome_config_set_int("/galeon/General/startpage_type", buffer);
	gnome_config_set_int("/galeon/Appearance/toolbar_type", prefs->toolbar_style);
	gnome_config_set_int("/galeon/Appearance/winwidth", prefs->win_width);
	gnome_config_set_int("/galeon/Appearance/winheight", prefs->win_height);
	gnome_config_set_int("/galeon/Advanced/java_enabled", prefs->enable_java);
	gnome_config_set_int("/galeon/Advanced/javascript_enabled", prefs->enable_javascript);	
	gnome_config_sync();

/* 	set_toolbar_style_global ();
	mozilla_prefs_set ();	*/
/*	
 *	I decided not to do this here because it's only called when Galeon is run for the
 *	first time and it was getting errors in init_cache() from mozilla_prefs_set().
 *	The quick solution is to remove it because the prefs are gonna be read anyways
 *	by preferences_load() called afterwards.
 *	
 *	FIXME: Make more friendly so it will be callable in other places
 */
	
	return TRUE;
}
