/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"

/* set this to use a file in /tmp rather than generating on the fly  */
#define USE_INTERMEDIATE_FILE 0

/* output stream type */
#if USE_INTERMEDIATE_FILE
typedef FILE OutStream;
#else
typedef GtkMozEmbed OutStream;
#endif

/* local function prototypes */
static void render_from_item(OutStream *stream, BookmarkItem *b);
static void render_from_list(OutStream *stream, GList *l);
static void render_url(OutStream *stream, BookmarkItem *b);
static void render_search_form(OutStream *stream, BookmarkItem *b);

/*
 * FIXME: this is eventually due to work by writing the HTML into a file,
 * which gets recreated when the bookmarks changed. This functionality
 * can then (in theory) be replaced by an arbitrary external program.
 * -- MattA
 */

/* hackish shorthand to render HTML into the browser */
#if USE_INTERMEDIATE_FILE
#define RENDER(str) \
        fwrite (str, strlen (str), 1, stream);
#else
#define RENDER(str) \
        gtk_moz_embed_append_data (stream, str, strlen (str))
#endif

/**
 * portal_render_into_embed: render the user's bookmarks into a GaleonEmbed
 */
void
portal_render_into_embed (GaleonEmbed *embed)
{
	gchar *stylesheet_filename, *default_file;
	gchar *configname, full_path[PATH_MAX];
	OutStream *stream;
#if USE_INTERMEDIATE_FILE
	gchar *filename;
#endif

	/* check bookmarks are loaded */
	g_return_if_fail(bookmarks_root != NULL);
	
	/* select a suitable style sheet for a default */
	default_file = user_file("myportal.css", FALSE);
	
	/* construct the config name and default */
	configname = g_strconcat("/galeon/Appearance/myportal_stylesheet=",
				 default_file, NULL);
	
	/* get the location of the CSS file for My Portal */
	stylesheet_filename = gnome_config_get_string(configname);

	/* open the rendering stream */
#if USE_INTERMEDIATE_FILE
	filename = g_strconcat (g_get_home_dir (), "/.galeon/myportal.html",
				NULL);
	stream = fopen (filename, "w");
	if (stream == NULL)
	{
		g_warning ("unable to create `%s'", filename);
		g_free(filename);
		return;
	}
#else
	stream = embed->mozEmbed;
	gtk_moz_embed_open_stream(stream, MYPORTAL_URL, "text/html");
	mozembed_load_started_cb (NULL, embed);
#endif

	/* render the complete portal */
	RENDER("<html><head>\n");
	RENDER("<link rel=\"stylesheet\" href=\"file:");
	RENDER(realpath(stylesheet_filename, full_path) ?
	       full_path : realpath(default_file, full_path));
	RENDER("\" type=\"text/css\">\n");
	RENDER("<title>");
	RENDER(_("My Portal"));
	RENDER("</title></head>\n");
	RENDER("<body>\n");
	RENDER("<a href=\"http://galeon.sourceforge.net/\">"
	       "<img class=\"logo\" src=\"file:" SHARE_DIR
	       "/logo.png\" alt=\"\"></a>");
	render_from_item(stream, bookmarks_root);
	render_from_item(stream, temp_bookmarks_root);
	RENDER("</body></html>\n");

	/* finish the rendering */
#if USE_INTERMEDIATE_FILE
	fclose (stream);
	embed_load_url (embed, filename);
	g_free (filename);
#else
	gtk_moz_embed_close_stream(stream);
	mozembed_load_finished_cb (NULL, embed);
#endif
}

/**
 * render_from_item: render a single bookmark item
 */
static void
render_from_item(OutStream *stream, BookmarkItem *b)
{
	gchar *strippedname;
	/* could be the case if there are no bookmarks */
	if (b == NULL)
		return;

	/* otherwise do the appropriate thing */
	switch (b->type)
	{
	case BM_CATEGORY:
	case BM_AUTOBOOKMARKS:
		RENDER("<blockquote>\n");
		RENDER("<strong>");
		strippedname = strip_uline_accel (b->name);
		RENDER(strippedname);
		g_free(strippedname);
		RENDER("</strong>\n");
		/* FIXME: if the bookmark is an alias, it should point to the real bookmark */
		if (!b->alias_of)
			render_from_list(stream, b->list);
		RENDER("</blockquote>\n");
		break;
		
	case BM_SITE:
		if (strstr (b->url, "%s"))
		{
			render_search_form(stream, b);
		}
		else
		{
			render_url(stream, b);
		}
		break;
		
	case BM_SEPARATOR:
		RENDER("<br>\n");
		break;
	}
}

/**
 * render_from_list: render a list of bookmark items
 */
static void
render_from_list(OutStream *stream, GList *l)
{
	for (; l != NULL; l = l->next)
	{
		render_from_item(stream, l->data);
	}
}

/**
 * render_url: render a bookmark url
 */
static void
render_url(OutStream *stream, BookmarkItem *b)
{

	RENDER("<a href=\"");

	/* If url has no preceding "http://" or "ftp://" presume "http://" */
	if (!strstr(b->url, "://")) {
		RENDER("http://");
	}
	
	RENDER(b->url);
	RENDER("\">");
	if (b->pixmap_data == NULL)
	{
		gchar *strippedname = strip_uline_accel (b->name);
		RENDER(strippedname);
		g_free(strippedname);
	}
	else
	{
		RENDER("<img src=\"file://");
		RENDER(b->pixmap_file);
		RENDER("\">");
	}
	RENDER("</a>\n");
}

/**
 * render_search_form: render a search form based on a smart bookmark
 */
static void 
render_search_form(OutStream *stream, BookmarkItem *b)
{
	gchar *path;    /* static part of URL */
	gchar **query;  /* query string, broken into parameters */
	gchar *inputs;  /* form entry(s) needed to supply %s's */
	int num_inputs = 0;
	gchar *tmp;
	int i;

	/* parse URL to separate static parts from parts needing user input */
	tmp = strchr(b->url, '?');
	if (!tmp || strlen(b->url) < (tmp - b->url + 2))
		return; /* no query in URL, can't render it */
	path = g_strndup(b->url, tmp - b->url + 1);

	/* count occurrences of %s in b->url */
	tmp = b->url;
	while ((tmp = strstr(tmp+1, "%s")))
	       ++num_inputs;

	query = g_strsplit(b->url + strlen(path), "&", -1);

	inputs = g_strconcat("", NULL);
	for (i = 0; query[i]; ++i) {
		gchar **param;
		param = g_strsplit(query[i], "=", 2);
		if (param[0] && param[1] && 
		    strstr(param[1], "%s") == param[1] &&
		    strlen(param[1]) == 2) {
			/* add a form input */
			tmp = g_strconcat(inputs,
					  num_inputs>1?param[0]:"",
					  "<input type=text name=\"",
					  param[0],
					  "\" value=\"\" size=",
					  num_inputs>1?"10>":"30>",
					  NULL);
			g_free(inputs);
			inputs = tmp;
		}
		else
		{
			/* add to static part of URL */
			tmp = g_strconcat(path, query[i], "&", NULL);
			g_free(path);
			path = tmp;
		}
		g_strfreev(param);
	}
	g_strfreev(query);
	
	RENDER("<form action=\"");
	if (!strstr(b->url, "://"))
	{
		RENDER("http://");
	}
	RENDER(path);
	RENDER("\" method=get>");
	RENDER(inputs);
	if (b->pixmap_data == NULL)
	{
		gchar *strippedname = strip_uline_accel (b->name);
		RENDER("<input type=submit value=\"");
		RENDER(strippedname);
		g_free(strippedname);
	}
	else
	{
		RENDER("<input type=image src=\"file://");
			RENDER(b->pixmap_file);
	}
	RENDER("\"></form>\n");
	
	g_free(inputs);
	g_free(path);
}

#undef RENDER

