/*
 *  Copyright (C) 2000-2001 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <galeon.h>
#include <libgnomevfs/gnome-vfs-mime-handlers.h>

typedef struct
{
	GtkWidget *print_dialog;
	GtkWidget *print_entry;
	GtkWidget *file_entry;
	GtkWidget *print_rb;
	GtkWidget *letter_rb;
	GtkWidget *top_entry;
	GtkWidget *bottom_entry;
	GtkWidget *left_entry;
	GtkWidget *right_entry;
	GtkWidget *all_pages_rb;
	GtkWidget *from_spinbutton;
	GtkWidget *to_spinbutton;
	GtkWidget *first_rb;
	GaleonWindow *window;
}
PrintWidgets;

static void
print_lookup_widgets (GladeXML *gxml, PrintWidgets *pw);

static PrintInfo *
print_get_info (PrintWidgets *pw);

static float
get_float_from_entry (GtkWidget *entry);

static int
get_int_from_entry (GtkWidget *entry);

static int
get_radiobutton_active_index (GtkWidget *radiobutton);

void
print_ok_button_cb (GtkWidget *button, PrintWidgets *widgets);

void
print_cancel_button_cb (GtkWidget *button, PrintWidgets *widgets);

void
print_preview_button_cb (GtkWidget *button, PrintWidgets *widgets);

void
print_destroy_cb (GtkWidget *dialog, PrintWidgets *widgets);

void
print_info_free (PrintInfo *info);

void print_show_dialog (GaleonWindow *window)
{	
	GladeXML *gxml;
	PrintWidgets *pw;

	if (window->print_dialog) return;

	pw = g_new0 (PrintWidgets, 1);

	gxml = glade_widget_new ("print_dialog", &pw->print_dialog, pw);
	
	print_lookup_widgets (gxml, pw);

	gtk_window_set_modal (GTK_WINDOW(pw->print_dialog), TRUE);
	window_set_layer (pw->print_dialog);

	window->print_dialog = pw->print_dialog;
	pw->window = window;

	/* show the print dialog */
	gtk_widget_show (window->print_dialog);
	gtk_object_unref (GTK_OBJECT (gxml));	
}

static void
print_lookup_widgets (GladeXML *gxml, PrintWidgets *pw)
{
	pw->print_entry = glade_xml_get_widget(gxml, "printer_entry");
	pw->file_entry = glade_xml_get_widget(gxml, "file_entry");
	pw->print_rb = glade_xml_get_widget(gxml, "printer_radiobutton");
	pw->letter_rb = glade_xml_get_widget(gxml, "letter_radiobutton");
	pw->top_entry = glade_xml_get_widget(gxml, "top_entry");
	pw->bottom_entry = glade_xml_get_widget(gxml, "bottom_entry");
	pw->left_entry = glade_xml_get_widget(gxml, "left_entry");
	pw->right_entry = glade_xml_get_widget(gxml, "right_entry");
	pw->all_pages_rb = glade_xml_get_widget(gxml, "all_pages_radiobutton");
	pw->first_rb = glade_xml_get_widget(gxml, "first_radiobutton");
	pw->to_spinbutton = glade_xml_get_widget(gxml, "to_spinbutton");
	pw->from_spinbutton = glade_xml_get_widget(gxml, "from_spinbutton");
}

static PrintInfo *
print_get_info (PrintWidgets *pw)
{
	PrintInfo *info;

	info = g_new0 (PrintInfo, 1);

      	info->print_to_file = 
		get_radiobutton_active_index (pw->print_rb);
	info->printer = gtk_editable_get_chars (
		GTK_EDITABLE (pw->print_entry), 0, -1);
	info->file = gtk_editable_get_chars (
		GTK_EDITABLE (pw->file_entry), 0, -1);

	info->top_margin = get_float_from_entry (pw->top_entry);
	info->bottom_margin = get_float_from_entry (pw->bottom_entry);
	info->left_margin = get_float_from_entry (pw->left_entry);
	info->right_margin = get_float_from_entry (pw->right_entry);

	info->from_page = get_int_from_entry (pw->from_spinbutton);
	info->to_page = get_int_from_entry (pw->to_spinbutton);

	info->paper = get_radiobutton_active_index (pw->letter_rb)+1;
	info->pages = get_radiobutton_active_index (pw->all_pages_rb);
	info->reversed = get_radiobutton_active_index (pw->first_rb);

	return info;
}

void
print_info_free (PrintInfo *info)
{
	g_free (info->printer);
	g_free (info->file);
	g_free (info);
}

void
print_ok_button_cb (GtkWidget *button, PrintWidgets *widgets)
{
	PrintInfo *info;

	info = print_get_info (widgets);

	if (!mozilla_print (widgets->window->active_embed, info))
	{
		gnome_error_dialog ("Not yet implemented.");
	}

	print_info_free (info);

	gtk_widget_destroy (widgets->print_dialog);
}

void
print_preview_button_cb (GtkWidget *button, PrintWidgets *widgets)
{
	PrintInfo *info;
	GnomeVFSMimeApplication *app;

	info = print_get_info (widgets);

	g_free (info->printer);

	/* get the postcript handler */

	app = gnome_vfs_mime_get_default_application ("application/postcript");
	
	info->printer = g_strdup ("ggv");
	info->print_to_file = FALSE;

	if (!mozilla_print (widgets->window->active_embed, info))
	{
		gnome_error_dialog ("Not yet implemented.");
	}
	
	print_info_free (info);
	gnome_vfs_mime_application_free (app);

	gtk_widget_destroy (widgets->print_dialog);
}

void
print_cancel_button_cb (GtkWidget *button, PrintWidgets *widgets)
{
	gtk_widget_destroy (widgets->print_dialog);
}

void
print_destroy_cb (GtkWidget *dialog, PrintWidgets *widgets)
{
	widgets->window->print_dialog = NULL;
	g_free (widgets);
}

static float
get_float_from_entry (GtkWidget *entry)
{
	float f;
	char *temp;

	temp = gtk_editable_get_chars (
		GTK_EDITABLE (entry), 0, -1);
	f = strtod(temp, (char **)NULL);
	g_free (temp);

	return f;
}

static int
get_int_from_entry (GtkWidget *entry)
{
	int i;
	char *temp;

	temp = gtk_editable_get_chars (
		GTK_EDITABLE (entry), 0, -1);
	i = strtol(temp, (char **)NULL, 10);
	g_free (temp);

	return i;
}

static int
get_radiobutton_active_index (GtkWidget *radiobutton)
{
	GSList *group;
	int i = 0, selected = -1;

	group = gtk_radio_button_group (GTK_RADIO_BUTTON (radiobutton));

	for (; group != NULL ; group = g_slist_next(group))
	{
		if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (group->data)))
		{
			selected = i;
		}
		
		i++;
	}

	g_slist_free (group);

	return (selected<0) ? -1 : i-selected-1;
}
