/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"

/* the type of toolbar button definitions */
typedef struct
{
	const gchar *label;
	const gchar *tooltip;
	const gchar *stock_icon;
	const gchar *nautilus_icon;
	void (*clicked_cb) (GtkButton *button, 
			    GaleonWindow *window);
	gint (*button_press_event_cb) (GtkButton *button, 
				       GdkEventButton *event,
				       GaleonWindow *window);
	void (*drag_data_received_cb) (GtkWidget *widget, 
				       GdkDragContext *drag_context,
				       gint x, gint y,
				       GtkSelectionData *selection_data,
				       guint info, guint time,
				       GaleonWindow *window);
} ToolbarButton;

/* local function prototypes */
static GtkWidget *toolbar_create_main (GaleonWindow *window);
static void toolbar_create_location_widgets (GaleonWindow *window, 
					     GtkWidget *hbox, 
					     GtkWidget *toolbar);
static GtkWidget *toolbar_create_location (GaleonWindow *window);
static GtkWidget *toolbar_button (GaleonWindow *window,
				  GtkWidget *toolbar,
				  const ToolbarButton *button);
static GtkWidget *toolbar_add_docked (GaleonWindow *window, GtkWidget *hbox, 
				      char *name, int band_num);
static void toolbar_add_drag_event_box (GaleonWindow *window, GtkWidget *hbox, 					GtkWidget *widget);
static gchar *toolbar_get_theme_path (void);
static GtkWidget *toolbar_get_theme_pixmap (const gchar *pixmap);
static gboolean toolbar_theme_use_standard_icons (void);
static void toolbar_update (GaleonWindow *window);

/* theme settings -- setup once per toolbar creation */
static gchar *theme_path;
static gchar *theme_name;
static gboolean use_nautilus_theme;

/* FIXME: this stuff is duplicated several times in Galeon */
/* gtkmozembed type of drag and drop */
static const GtkTargetEntry drop_types[] =
{
	{ "GALEON_URL",    0, DND_TARGET_GALEON_URL   },
	{ "_NETSCAPE_URL", 0, DND_TARGET_NETSCAPE_URL },
	{ "STRING",        0, DND_TARGET_STRING       }
};
static const gint drop_types_num_items = (sizeof (drop_types) / 
					  sizeof (GtkTargetEntry));

/* path to Nautilus theme pref in config space */
#define NAUTILUS_THEME_PREF "/apps/nautilus/preferences/theme"

/* standard toolbar buttons */
const static ToolbarButton new_button =
{
	N_("New"), 
	N_("Open new browser"), 
	GNOME_STOCK_PIXMAP_NEW, 
	"New",
	NULL, /* clicked */
	window_new_button_press_cb,
	window_new_button_drag_data_received_cb
};
const static ToolbarButton back_button =
{
	N_("Back"), 
	N_("Go back - Right button: History"), 
	GNOME_STOCK_PIXMAP_BACK, 
	"Back",
	window_back_button_clicked_cb,
	window_back_forward_button_press_cb,
	NULL /* drag_data_received */
};
const static ToolbarButton up_button =
{
	N_("Up"), 
	N_("Go up a directory level"), 
	GNOME_STOCK_PIXMAP_UP, 
	"Up",
	window_up_button_clicked_cb,		
	window_up_button_press_event_cb,
	NULL /* drag_data_received */
};
const static ToolbarButton forward_button =
{
	N_("Forward"), 
	N_("Go forward - Right button: History"), 
	GNOME_STOCK_PIXMAP_FORWARD,
	"Forward",
	window_forward_button_clicked_cb,
	window_back_forward_button_press_cb,
	NULL /* drag_data_received */
};
const static ToolbarButton reload_button =
{
	N_("Reload"), 
	N_("Display the latest content of the current page"), 
	GNOME_STOCK_PIXMAP_REFRESH, 
	"Refresh",
	window_refresh_button_clicked_cb,
	window_refresh_button_press_event_cb,
	NULL /* drag_data_received */
};
const static ToolbarButton home_button =
{
	N_("Home"), 
	N_("Go to your home page"), 
	GNOME_STOCK_PIXMAP_HOME, 
	"Home",
	window_start_button_clicked_cb,
	window_start_button_press_event_cb,
	NULL /* drag_data_received */
};
const static ToolbarButton stop_button =
{
	N_("Stop"), 
	N_("Stop current data transfer"), 
	GNOME_STOCK_PIXMAP_STOP, 
	"Stop",
	window_stop_button_clicked_cb,
	NULL /* press_event */,
	NULL /* drag_data_received */
};
const static ToolbarButton go_button =
{
	N_("Go"), 
	N_("Go to specified location"), 
	GNOME_STOCK_PIXMAP_JUMP_TO, 
	"Go",
	window_go_button_clicked_cb,
	NULL /* press_event */,
	NULL /* drag_data_received */
};

/*
 * toolbar_create: create toolbars
 */
void
toolbar_create (GaleonWindow *window)
{
	gboolean use_theme_pref = 
		gnome_config_get_bool(CONF_APPEARANCE_NAUTILUS_ICONS);

	use_nautilus_theme = FALSE;

	/* get theme infos */
	if (use_theme_pref)
	{
		theme_path = toolbar_get_theme_path ();
		if (theme_path)
		{
			use_nautilus_theme = !toolbar_theme_use_standard_icons();
		}
	}

	/* create main toolbar */
	window->main_dockitem = toolbar_add_docked
					(window,
					 toolbar_create_main (window),
					 GNOME_APP_TOOLBAR_NAME, 1);

	/* create location toolbar, if necessary */
	if (gnome_config_get_int ("/galeon/Appearance/url_location=1") == 0)
	{
		GtkWidget *location_tb = toolbar_create_location (window);

		window->location_dockitem = toolbar_add_docked (window,
								location_tb,
								"location", 2);
	}

	/* set the visibility of the optional components */
	toolbar_update (window);

	/* free theme infos */
	if (use_theme_pref)
	{
		g_free (theme_path);
		g_free (theme_name);
	}
}

/**
 * toolbar_add_docked: add an horizontal box to the toolbar dock
 */
GtkWidget *
toolbar_add_docked (GaleonWindow *window, GtkWidget *hbox, char *name,
		    int band_num)
{
	gboolean toolbar_relief;
	gboolean toolbar_detachable;
	GnomeDockItemBehavior props;
	gboolean url_location;


	url_location = gnome_config_get_int 
		("/galeon/Appearance/url_location=1");
	toolbar_relief = gnome_preferences_get_toolbar_relief ();
	toolbar_detachable = gnome_preferences_get_toolbar_detachable ();

	props = GNOME_DOCK_ITEM_BEH_EXCLUSIVE;

/* FIXME: use this code once the toolbar is packed properly
	if (url_location == 1)
	{
		props |= GNOME_DOCK_ITEM_BEH_NEVER_VERTICAL;
	}
*/
	props |= GNOME_DOCK_ITEM_BEH_NEVER_VERTICAL;

	if (!toolbar_detachable)
	{
		props |= GNOME_DOCK_ITEM_BEH_LOCKED;
	}

	gnome_app_add_docked (GNOME_APP (window->WMain), hbox, name,
			      props, GNOME_DOCK_TOP, band_num, 0, 0);
	gtk_container_set_border_width (GTK_CONTAINER(hbox->parent), 2);
	gnome_dock_item_set_shadow_type 
		(GNOME_DOCK_ITEM (hbox->parent), 
		 toolbar_relief ? GTK_SHADOW_OUT : GTK_SHADOW_NONE);
	gtk_widget_show (hbox);

	return hbox->parent;
}

/**
 * toolbar_update: update the main toolbar(s)
 */
static void
toolbar_update (GaleonWindow *window)
{
	GtkWidget *main_toolbar;
	GtkToolbarStyle style;
	gboolean toolbar_button_relief;

	gnome_config_push_prefix ("/galeon/Appearance/");

	toolbar_button_relief = gnome_preferences_get_toolbar_relief_btn ();
	style = gnome_config_get_int ("toolbar_type=0");

	main_toolbar = gtk_object_get_data (GTK_OBJECT (window->WMain),
					    "main_toolbar");
	if (main_toolbar != NULL)
	{
		gtk_toolbar_set_style (GTK_TOOLBAR (main_toolbar),
				       style);
		gtk_toolbar_set_button_relief (GTK_TOOLBAR (main_toolbar),
					       toolbar_button_relief ? 
							GTK_RELIEF_NORMAL :
							GTK_RELIEF_NONE);
	}

	gtk_toolbar_set_style (GTK_TOOLBAR (window->go_toolbar),
			       style);
	gtk_toolbar_set_button_relief (GTK_TOOLBAR (window->go_toolbar),
				       toolbar_button_relief ?
						GTK_RELIEF_NORMAL :
						GTK_RELIEF_NONE);

	if (gnome_config_get_bool ("show_new_button=FALSE"))
		gtk_widget_show (window->BNew);
	else
		gtk_widget_hide (window->BNew);

	if (gnome_config_get_bool ("show_up_button=FALSE"))
		gtk_widget_show (window->BUp);
	else
		gtk_widget_hide (window->BUp);

	if (gnome_config_get_bool ("show_go_button=FALSE"))
		gtk_widget_show (window->go_toolbar);
	else
		gtk_widget_hide (window->go_toolbar);
		
	gnome_config_pop_prefix ();
		
	/* FIXME: Why are these prefs in a different section? */
	if (gnome_config_get_bool ("/galeon/General/show_zoom=true"))
		gtk_widget_show (window->zoom_spin);
	else
		gtk_widget_hide (window->zoom_spin);

	/* create/destroy the spinner */
	if (gnome_config_get_bool ("/galeon/General/show_spinner=true"))
	{
		if (window->spinner != NULL)
		{
			gnome_animator_stop (GNOME_ANIMATOR (window->spinner));
			gtk_widget_destroy (window->spinner->parent);
		}
		window->spinner = spinner_create (main_toolbar->parent);
		if (window->spinner != NULL)
		{
			gtk_signal_connect
					(GTK_OBJECT (window->spinner->parent),
					 "clicked",
					 window_spinner_button_clicked_cb,
					 (gpointer) window);
		}
	}
	else
	{
		if (window->spinner != NULL)
		{
			gnome_animator_stop (GNOME_ANIMATOR (window->spinner));
			gtk_widget_destroy (window->spinner->parent);
			window->spinner = NULL;
		}
	}
}

/**
 * toolbar_global_recreate: recreate toolbars of all galeon windows
 */
void
toolbar_global_recreate (void)
{
	GList *l;
	GaleonWindow *window = (GaleonWindow *) all_windows->data;
	gint url_location = gnome_config_get_int
					("/galeon/Appearance/url_location=1");
	gboolean recreate = FALSE;

	/*  FIXME: The layout still gets lost when you switch between
	 *	   having the location entry on the main toolbar or on
	 *	   a separate toolbar.  This is probably not a big deal,
	 *	   as I doubt this setting is changed very often   --Josh
	 */

	/* Check the first window in the list to see if we actually need to
	   recreate the toolbars */
	if ( (url_location == 0 && window->location_dockitem == NULL) ||
	     (url_location == 1 && window->location_dockitem != NULL) )
		recreate = TRUE;

	if (recreate)
		bookmarks_remove_tb_global ();

	for (l = all_windows; l != NULL; l = l->next)
	{
		window = (GaleonWindow *) l->data;

		if (recreate)
		{
			/* recreate the toolbars */
			gtk_editable_delete_text
					(GTK_EDITABLE (window->toolbar_entry),
					 0, -1);
			gtk_widget_destroy (window->main_dockitem->parent);
			if (window->location_dockitem)
			{
				gtk_widget_destroy
					(window->location_dockitem->parent);
				window->location_dockitem = NULL;
			}

			toolbar_create (window);

			gtk_widget_show (window->main_dockitem);
			if (window->location_dockitem)
				gtk_widget_show (window->location_dockitem);
		}
		else
		{
			/* otherwise, just update the existing toolbars */
			toolbar_update (window);
		}

		window_update_nav_buttons (window);
	}

	if (recreate)
		bookmarks_create_tb_global ();

	/* We need to recreate the menus in case the location of the
	   extra menu options has changed */
	bookmarks_update_menu ();

	/* Update the smart bookmarks toolbar entry widgets */
	bookmarks_tb_update_entries_global ();
}

/**
 * toolbar_create_main: create main toolbar (buttons)
 */
GtkWidget*
toolbar_create_main (GaleonWindow *window)
{
	GtkWidget *hbox, *toolbar;
	GtkAdjustment *adj;
	int url_location;

	gnome_config_push_prefix ("/galeon/Appearance/");
	url_location = gnome_config_get_int ("url_location=1");
	gnome_config_pop_prefix ();

	hbox = gtk_hbox_new (FALSE, 0);

	/* create buttons toolbar */
	toolbar = gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL, 
				   GTK_TOOLBAR_BOTH);
	/* the lazy way */
	gtk_object_set_data (GTK_OBJECT (window->WMain), "main_toolbar",
			     toolbar);

	/* pack it in */
	gtk_box_pack_start (GTK_BOX(hbox), toolbar, FALSE, FALSE, 1);

	/* create all of the buttons */
	window->BNew     = toolbar_button (window, toolbar, &new_button);
	window->BBack    = toolbar_button (window, toolbar, &back_button);
	window->BUp      = toolbar_button (window, toolbar, &up_button);
	window->BForward = toolbar_button (window, toolbar, &forward_button);
	window->BRefresh = toolbar_button (window, toolbar, &reload_button);
	window->BStart   = toolbar_button (window, toolbar, &home_button);
	window->BStop    = toolbar_button (window, toolbar, &stop_button);

	/* create zoom spin entry */
	adj = GTK_ADJUSTMENT (gtk_adjustment_new (100, 1, 999, 10, 25, 10));
	window->zoom_spin = GTK_WIDGET (gtk_spin_button_new (adj, 1, 0));
	gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
				    GTK_TOOLBAR_CHILD_WIDGET,
				    GTK_WIDGET (window->zoom_spin),
				    NULL, NULL, NULL, NULL, NULL, NULL);
	gtk_signal_connect (GTK_OBJECT (window->zoom_spin), "changed",
			    window_zoom_spin_changed_cb, window);
	gtk_signal_connect (GTK_OBJECT (window->zoom_spin), "key_press_event",
			    (GtkSignalFunc) window_zoom_spin_key_press_cb, 
			    window);

	/* create the spinner, if necessary */
	if (gnome_config_get_bool ("/galeon/General/show_spinner=true"))
	{
		window->spinner = spinner_create (hbox);

		if (window->spinner != NULL)
		{
			gtk_signal_connect
			    (GTK_OBJECT (window->spinner->parent), "clicked",
			     (GtkSignalFunc) window_spinner_button_clicked_cb,
			     (gpointer) window);
		}
	}
	else
	{
			window->spinner = NULL;
	}

	/* create location related widgets if needed */
	if (url_location == 1)
	{
		toolbar_create_location_widgets (window, hbox, toolbar);
	}

	/* show the box and toolbar */
	gtk_widget_show (toolbar);
	gtk_widget_show (hbox);

	return hbox;
}

void
toolbar_add_drag_event_box (GaleonWindow *window, GtkWidget *toolbar, 
			    GtkWidget *widget)
{
	GtkWidget *event_box = gtk_event_box_new ();

	gtk_container_add (GTK_CONTAINER(event_box), widget);
	gtk_container_set_border_width (GTK_CONTAINER(event_box), 3);
	gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
				    GTK_TOOLBAR_CHILD_WIDGET,
				    GTK_WIDGET (event_box),
				    NULL, _("Drag page location from here"),
				    NULL, NULL, NULL, NULL);
	gtk_signal_connect (GTK_OBJECT(event_box), "drag_data_get",
			    window_drag_pixmap_drag_data_get_cb, window);
	gtk_drag_source_set (GTK_WIDGET (event_box),
			     GDK_BUTTON1_MASK, url_drag_types, 
			     url_drag_types_num_items,
			     GDK_ACTION_COPY|GDK_ACTION_LINK);
	gtk_widget_show_all (event_box);
}

/*
 * toolbar_create_location_widgets: create location related widgets
 */
void
toolbar_create_location_widgets (GaleonWindow *window, 
				 GtkWidget *hbox, GtkWidget *toolbar) 
{
	GtkWidget *button;
	PixmapData *drag_pixmap;
	gboolean toolbar_button_relief;

	/* read preferences */
	toolbar_button_relief = gnome_preferences_get_toolbar_relief_btn ();

	drag_pixmap = pixmap_data_new_from_file (SHARE_DIR "/i-bookmark.xpm");
	window->drag_pixmap = gtk_pixmap_new (drag_pixmap->pixmap,
					      drag_pixmap->mask);
	toolbar_add_drag_event_box (window, toolbar, window->drag_pixmap);

	/* create location entry */
	window->toolbar_gnomeentry = gnome_entry_new ("location_history");
	gnome_entry_load_history (GNOME_ENTRY(window->toolbar_gnomeentry));

	/* connect the popwin handlers */
	gtk_signal_connect
		(GTK_OBJECT(GTK_COMBO (window->toolbar_gnomeentry)->popwin),
		 "show",
		 GTK_SIGNAL_FUNC(window_location_gnomeentry_popwin_cb), 
		 window);
	gtk_signal_connect
		(GTK_OBJECT(GTK_COMBO(window->toolbar_gnomeentry)->popwin),
		 "hide",
		 GTK_SIGNAL_FUNC(window_location_gnomeentry_popwin_cb), 
		 window);
	window->toolbar_entry = gnome_entry_gtk_entry 
		(GNOME_ENTRY (window->toolbar_gnomeentry));
	gtk_box_pack_start (GTK_BOX(hbox), window->toolbar_gnomeentry, 
			    TRUE, TRUE, 1);
	gtk_widget_show (GTK_WIDGET(window->toolbar_gnomeentry));
	gtk_signal_connect_after 
		(GTK_OBJECT(window->toolbar_entry), 
		 "key_press_event",
		 (GtkSignalFunc)window_location_entry_key_press_cb, window);
	gtk_signal_connect 
		(GTK_OBJECT(window->toolbar_entry), "drag_data_received",
		 window_location_entry_drag_data_received_cb, window);

	/* create go button toolbar */
	window->go_toolbar = gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL, 
					      GTK_TOOLBAR_BOTH);
	button = toolbar_button (window, window->go_toolbar, &go_button);
	gtk_box_pack_start (GTK_BOX (hbox), window->go_toolbar,
			    FALSE, FALSE, 1);
}

/*
 * toolbar_create_location: create location toolbar
 */
GtkWidget*
toolbar_create_location (GaleonWindow *window)
{
	GtkWidget *hbox;
	GtkWidget *toolbar;

	hbox = gtk_hbox_new (FALSE, 0);

	toolbar = gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL, 
				   GTK_TOOLBAR_BOTH);
	gtk_box_pack_start (GTK_BOX(hbox), toolbar, FALSE, FALSE, 1);

	toolbar_create_location_widgets (window, hbox, toolbar);

	gtk_widget_show (GTK_WIDGET(toolbar));
	gtk_widget_show (GTK_WIDGET(hbox));

	return hbox;
}

/*
 * toolbar_button: create and append a button to a toolbar
 */
GtkWidget *
toolbar_button (GaleonWindow *window, GtkWidget *toolbar, 
		const ToolbarButton *button)
{
	GtkWidget *pixmap = NULL;
	GtkWidget *widget;

	/* try getting nautilus button */
	if (use_nautilus_theme)
	{
		pixmap = toolbar_get_theme_pixmap (button->nautilus_icon);
	}
	
	/* use GNOME stock icon if not found or not requested */
	if (pixmap == NULL)
	{
		pixmap = gnome_stock_pixmap_widget (window->WMain,
						    button->stock_icon);
	}

	/* make the toolbar button and add it to the toolbar */
	widget = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
					     GTK_TOOLBAR_CHILD_BUTTON,
					     NULL,
					     _(button->label),
					     _(button->tooltip),
					     NULL, pixmap, NULL, NULL);

	/* connect appropriate signals */
	if (button->clicked_cb != NULL)
	{
		gtk_signal_connect (GTK_OBJECT (widget), "clicked", 
				    button->clicked_cb, window);
	}
	if (button->button_press_event_cb != NULL)
	{
		gtk_signal_connect (GTK_OBJECT (widget), "button_press_event", 
				    (GtkSignalFunc)
				    (button->button_press_event_cb), window);
	}
	if (button->drag_data_received_cb != NULL)
	{
		gtk_signal_connect (GTK_OBJECT (widget), "drag_data_received", 
				    (GtkSignalFunc)
				    (button->drag_data_received_cb), window);

		/* set drop types */
		gtk_drag_dest_set (GTK_WIDGET (widget), GTK_DEST_DEFAULT_ALL,
				   drop_types, drop_types_num_items, 
				   GDK_ACTION_COPY);
	}

	/* return the completed button widget */
	return GTK_WIDGET (widget);
}

gboolean toolbar_theme_use_standard_icons (void)
{
	xmlDocPtr doc;
	xmlNodePtr item;
	char *filename;
	gchar *icon_theme;
	gboolean result = FALSE;

	filename = g_strconcat (theme_path, theme_name, ".xml", NULL);
	if (g_file_exists (filename))
	{
		doc = xmlParseFile (filename);
		item = doc->root->childs;
		while (item != NULL) {
			if (!strcmp (item->name, "toolbar")) {
				icon_theme = xmlGetProp (item, "icon_theme");
				result = (strcmp (icon_theme, "standard")==0);
				xmlFree (icon_theme);
			}
			item = item->next;
		}
	}

	g_free (filename);
	return result;
}

GtkWidget *
toolbar_get_theme_pixmap (const gchar *pixmap)
{
	char *filename;
	PixmapData *icon;
	GtkWidget *widget;

	filename = g_strconcat (theme_path, pixmap, ".png", NULL);
	if (!g_file_exists (filename))
	{
		/* fall back to the default icon */
		char *dir; 

		g_free (filename);

		dir = gnome_datadir_file("pixmaps/nautilus/");
		filename = g_strconcat (dir, pixmap, ".png", NULL);
		g_free (dir);
		if (!g_file_exists (filename))
		{
			g_free (filename);
			return NULL;
		}

	}

	icon = pixmap_data_new_from_file (filename);
	g_free (filename);

	if (icon != NULL)
	{
		widget = gtk_pixmap_new (icon->pixmap, icon->mask);
		return GTK_WIDGET (widget);
	}
	else
	{
		return NULL;
	}
}

char *
toolbar_get_theme_path (void)
{
	char *pixmap_path;
	char *theme;
	char *dir;

	theme = gconf_client_get_string (gconf_client, 
					 NAUTILUS_THEME_PREF, NULL);
	if (!theme)
	{
		return NULL;
	}

	theme_name = theme;

	dir = gnome_datadir_file("pixmaps/nautilus/");

	if (strcmp(theme_name, "default") == 0) {
		pixmap_path = g_strconcat (dir, NULL);
	} else {
		pixmap_path = g_strconcat (dir, theme, "/", NULL);
	}

	g_free (dir);
	if (g_file_exists (pixmap_path))
	{
		return pixmap_path;
	}
	else
	{
		g_free (pixmap_path);
	}

	/* try with the user directory */
	pixmap_path = g_strconcat (g_get_home_dir (), "/.nautilus/themes/", 
		theme, "/", NULL);
	if (g_file_exists (pixmap_path))
	{
		return pixmap_path;
	}
	else
	{
		g_free (pixmap_path);
	}

	return NULL;
}

