/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "menubar.h"

/* indicates a window is in fullscreen mode */
gboolean fullscreen_active = FALSE;

/* local function prototypes */
static void window_init_data (GaleonWindow *window);
static void window_set_menu_data (GaleonWindow *window);
static void set_initial_window_properties (GaleonWindow *window);

/* clallbacks used here */
void file_open_ok_button_clicked_cb (const gchar *file, gpointer data);

/** The global list of all GaleonWindow structures */
GList *all_windows = NULL;

/* gtkmozembed type of drag and drop */
const GtkTargetEntry drop_types[] =
{
	{ "GALEON_URL",    0, DND_TARGET_GALEON_URL   },
	{ "_NETSCAPE_URL", 0, DND_TARGET_NETSCAPE_URL },
	{ "STRING",        0, DND_TARGET_STRING       }
};
const gint drop_types_num_items = (sizeof (drop_types) / 
				   sizeof (GtkTargetEntry));

/* url type of drag and drop */
const GtkTargetEntry url_drag_types[] = 
{
	{ "GALEON_BOOKMARK", 0, DND_TARGET_GALEON_BOOKMARK },
	{ "GALEON_URL",      0, DND_TARGET_GALEON_URL      },
	{ "_NETSCAPE_URL",   0, DND_TARGET_NETSCAPE_URL    },
	{ "STRING",          0, DND_TARGET_STRING          }
};
const gint url_drag_types_num_items = (sizeof (url_drag_types) /
				       sizeof (GtkTargetEntry));

typedef enum
{
	STRING,
	COMPOUND_TEXT
} SelType;

/* selection types */
const GtkTargetEntry selection_types[] =
{
	{ "STRING",        0, STRING        },
	{ "COMPOUND_TEXT", 0, COMPOUND_TEXT }
};
const gint selection_types_num_targets = (sizeof (selection_types) /
					  sizeof (GtkTargetEntry));


/**
 * window_load_url: load a URL into the active embed of a window
 */
void
window_load_url (GaleonWindow *window, const gchar *url)
{
	/* check arguments */
	return_if_not_window (window);

	/* load into the active GaleonEmbed */
	embed_load_url (window->active_embed, url);
}
 
static void
set_initial_window_properties (GaleonWindow *window)
{
	gint width, height;

	/* find the set width and height if any */
	if (!window->set_size)
	{
		/* use system defaults */
		width = gnome_config_get_int (CONF_APPEARANCE_WINWIDTH);
		height = gnome_config_get_int (CONF_APPEARANCE_WINHEIGHT);
		gtk_window_set_default_size (GTK_WINDOW (window->WMain), 
					     width, height);
	}

	/* setup other policies */
	gtk_window_set_policy (GTK_WINDOW (window->WMain), 
			       TRUE, TRUE, TRUE);
}

void
window_set_visibility (GaleonWindow *window, gboolean visible)
{
        /* show or hide */
	if (visible == TRUE)
	{
		set_settings_menu_window (window);
		set_initial_window_properties (window);
		gtk_widget_show (window->WMain);
	}
	else
	{
		gtk_widget_hide (window->WMain);
	}
}

/**
 * window_create: create a browser structure and main window, 
 * but without an embedding widget. Should only be called from
 * embed_create
 */
GaleonWindow *
window_create (void)
{
        GaleonWindow *window;
	GnomeEntry *ge;
	GtkWidget *bookmarks_item;
	gchar *layout;

	/* get the saved layout string */
	layout = gnome_config_get_string ("/galeon/Placement/Dock");

	/* allocate and initialise the GaleonWindow structure */
	window = g_new0 (GaleonWindow, 1);

	/* set magic */
	window->magic = GALEON_WINDOW_MAGIC;

	/* create the browser window */
	window->WMain = gnome_app_new ("galeon", _("Galeon"));
	gtk_window_set_wmclass (GTK_WINDOW (window->WMain), "",
				"galeon_browser");

	/* need to set the user_data field in all the GnomeUIInfo structs */
	window_set_menu_data (window);

	/* init dialogs data */
	window->print_dialog = NULL;

	/* create the menus and statusbar */
	gnome_app_create_menus (GNOME_APP(window->WMain), menubar_uiinfo);
	gnome_app_set_statusbar (GNOME_APP(window->WMain),
				 gnome_appbar_new (TRUE, TRUE,
						   GNOME_PREFERENCES_NEVER));
  	window_init_data (window);
	gnome_app_install_menu_hints (GNOME_APP(window->WMain), 
				      menubar_uiinfo);

	/* set mini icon */
	gnome_window_icon_set_from_file (GTK_WINDOW (window->WMain), 
					 gnome_pixmap_file ("galeon.png"));

	/* so widgets can look the data up */
	gtk_object_set_data (GTK_OBJECT (window->WMain), "GaleonWindow",
			     window);

	/* connect the delete signal handler*/
	gtk_signal_connect(GTK_OBJECT(window->WMain), "delete-event",
			   GTK_SIGNAL_FUNC(window_delete_cb),
			   window);

	/* create the toolbar */
	toolbar_create (window);

	/* set the initial state of the View_* CheckMenuItems */
	gtk_check_menu_item_set_active 
		(GTK_CHECK_MENU_ITEM (window->view_menubar), TRUE);
	gtk_check_menu_item_set_active 
		(GTK_CHECK_MENU_ITEM (window->view_toolbar), TRUE);
	gtk_check_menu_item_set_active 
		(GTK_CHECK_MENU_ITEM (window->view_statusbar), TRUE);
//	gtk_check_menu_item_set_active 
//		(GTK_CHECK_MENU_ITEM (window->view_history_dock),
//		 dHistory != NULL && GNOME_IS_DOCK_ITEM (dHistory));
	gtk_check_menu_item_set_active 
		(GTK_CHECK_MENU_ITEM (window->view_history_dock),
		 dHistory != NULL && GNOME_IS_DOCK_ITEM (dHistory));

	/* add it to the list of windows */
	all_windows = g_list_prepend (all_windows, window);

	/* create the bookmarks menu */
	bookmarks_create_menu (window);
	bookmarks_create_tb (window);

	/* restore the toolbar layout */
	gnome_dock_layout_parse_string (GNOME_APP(window->WMain)->layout,
				        layout);
	g_free (layout);

	/* setup the history dropdown menu */
	ge = GNOME_ENTRY(window->toolbar_gnomeentry);
	auto_completion_add_from_entry (GNOME_ENTRY(ge));

	/* set url entry drop destination */
	gtk_drag_dest_set 
		(GTK_WIDGET (gnome_entry_gtk_entry 
			     (GNOME_ENTRY (window->toolbar_gnomeentry))),
		 GTK_DEST_DEFAULT_ALL,
		 drop_types, drop_types_num_items,
		 GDK_ACTION_COPY);

	/* set bookmark menuitem drop destination */
	bookmarks_item = gtk_object_get_data(GTK_OBJECT(window->WMain),
					     "bookmarks");
	gtk_drag_dest_set (bookmarks_item, GTK_DEST_DEFAULT_ALL,
			   url_drag_types, url_drag_types_num_items,
			   GDK_ACTION_COPY);
	gtk_signal_connect (GTK_OBJECT(bookmarks_item), "drag_data_received",
			    GTK_SIGNAL_FUNC
			    (bookmarks_menuitem_drag_data_received_cb),
			    window);

	/* set selection signal */
	gtk_selection_add_targets (GTK_WIDGET(window->WMain),
				   GDK_SELECTION_PRIMARY, 
				   selection_types, selection_types_num_targets);
	gtk_selection_add_targets (GTK_WIDGET(window->WMain),
				   gdk_atom_intern("CLIPBOARD", FALSE),
				   selection_types, selection_types_num_targets);
	gtk_signal_connect (GTK_OBJECT(window->WMain), "selection_received",
			    GTK_SIGNAL_FUNC (window_selection_received_cb),
			    window);
	gtk_signal_connect (GTK_OBJECT(window->WMain), "selection_get",
			    GTK_SIGNAL_FUNC (window_selection_get_cb),
			    NULL);

	/* make the toplevel notebobok */
	window->notebook = gtk_notebook_new ();

	/* set some notebook properties */
	gtk_notebook_popup_enable (GTK_NOTEBOOK (window->notebook));
	gtk_notebook_set_scrollable (GTK_NOTEBOOK (window->notebook), TRUE);
	gtk_signal_connect (GTK_OBJECT (window->notebook), "switch_page",
			    embed_notebook_switch_page_cb, window);
	gtk_notebook_set_tab_pos (GTK_NOTEBOOK (window->notebook),
		gnome_config_get_int (CONF_APPEARANCE_TABBED_POSITION));
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (window->notebook),
		gnome_config_get_int (CONF_APPEARANCE_TABBED_ALWAYS_SHOW));
	gtk_notebook_set_show_border (GTK_NOTEBOOK (window->notebook), FALSE);

	/* insert notebook into toplevel window */
	gnome_app_set_contents (GNOME_APP (window->WMain), 
				GTK_WIDGET (window->notebook));

	/* NB: window isn't shown until we add an embed */
	/* don't know the size at which to show it yet -- if we discover
	 * a size (e.g. for a popup) then we'll use that, otherwise we'll
	 * find a default, but not until the window is made visible */
	window->set_size = FALSE;

	/* return the completed GaleonWindow */
	return window;
}

/**
 * window_update_status_bar: update the status bar of the toplevel window
 */
void
window_update_status_bar (GaleonWindow *window)
{
	GaleonEmbed *embed;
	GtkWidget *progress;
	const gchar *status;
	gchar message[256];

	return_if_not_window (window);
	embed = window->active_embed;
	return_if_not_embed (embed);

	/* get progress bar widget */
	progress = GTK_WIDGET (gnome_appbar_get_progress 
			       (GNOME_APPBAR(window->appbar)));

	/* have we started loading? */
	if (embed->load_started > 0 && embed->bytesLoaded > 0)
	{
		if (embed->loadPercent == 0) 
		{
			/* not really, go into activity mode */
			gtk_progress_set_activity_mode 
				(GTK_PROGRESS (progress), TRUE);
			if (!(window->progress_timeout))
			{
				window->progress_timeout = TRUE;
				g_timeout_add (100, (GSourceFunc)
					       window_progress_action, 
					       window);
			}
		} 
		else
		{
			/* yes, show the progress in progress bar */
			window->progress_timeout = FALSE;
			gtk_progress_set_activity_mode 
				(GTK_PROGRESS (progress), FALSE);
			gnome_appbar_set_progress 
				(GNOME_APPBAR (window->appbar), 
				 embed->loadPercent / 100.0);
		}
	}
	else
	{
		/* go to sleep */
		window->progress_timeout = FALSE;
		gtk_progress_set_activity_mode (GTK_PROGRESS (progress),
						FALSE);
		gnome_appbar_set_progress (GNOME_APPBAR (window->appbar), 0);
	}

	/* get rid of last message */
	gnome_appbar_pop (GNOME_APPBAR (window->appbar));

	/* temporary message takes priority */
	if (window->tempMessage != NULL && (strlen (window->tempMessage) != 0))
	{
		gnome_appbar_push (GNOME_APPBAR (window->appbar),
				   window->tempMessage);
		return;
	}

	/* get a status message */
	status = (embed->statusMessage == NULL ? "" : embed->statusMessage);

	/* make a progress message */
	if (embed->bytesLoaded == 0)
	{
		g_snprintf (message, 255, "%s", status);
	}
	else if (embed->bytesLoaded <= embed->maxBytesLoaded)
	{
		/* fill the status bar text with progress info */
		g_snprintf (message, 255, 
			    _("%s (%d%% complete, %d kB of %d kB loaded)"), 
			    status, 
			    embed->loadPercent, 
			    embed->bytesLoaded / 1024,
			    embed->maxBytesLoaded / 1024);
	}
	else
	{
		/* fill the status bar text with progress info: only kb */
		g_snprintf (message, 255, _("%s (%d kB loaded)"), 
			    status, 
			    embed->bytesLoaded / 1024);
	}

	/* show the new message */
	gnome_appbar_push (GNOME_APPBAR (window->appbar), message);
}

/**
 * window_update_temp_message: update the temporary message
 */
void
window_update_temp_message(GaleonWindow *window, const char *message)
{
	/* free the previous message */
	if (window->tempMessage) 
	{
		g_free(window->tempMessage);
	}
	
	/* fill the new temporary message */
	if (message) 
	{
		window->tempMessage = g_strdup(message);
	}	
	else
	{	
		window->tempMessage = NULL;
	}
	window_update_status_bar (window);
}

/**
 * window_update_nav_buttons: update back and forward toolbar buttons status
 */
void
window_update_nav_buttons (GaleonWindow *window)
{
	gboolean can_go_back, can_go_up, can_go_forward;
	GaleonEmbed *embed;

	/* get active embed */
	embed = window->active_embed;

	/* query mozilla / embed */
	can_go_back    = gtk_moz_embed_can_go_back (embed->mozEmbed);
	can_go_up      = embed_can_go_up (embed);
	can_go_forward = gtk_moz_embed_can_go_forward (embed->mozEmbed);

	/* update back and forward buttons */
	gtk_widget_set_sensitive (GTK_WIDGET (window->BBack), 
				  can_go_back);
	gtk_widget_set_sensitive (GTK_WIDGET (window->BUp), 
				  can_go_up);
	gtk_widget_set_sensitive (GTK_WIDGET (window->BForward), 
				  can_go_forward);

	/* update stop and refresh toolbar buttons */
	gtk_widget_set_sensitive (GTK_WIDGET (window->BStop),
				  embed->load_started > 0);
	gtk_widget_set_sensitive (GTK_WIDGET (window->BRefresh),
				  embed->load_started == 0);
}

/**
 * window_init_data: init GaleonWindow data structure
 */
static void
window_init_data (GaleonWindow *window)
{
	int pos; //A temp hack for backward compatibility
#ifdef CVS_SOURCE
	pos = 5;
#else
	pos = 4;
#endif
	/* these are all nullified */
	window->spinner = NULL;
	window->tempMessage = NULL;
	window->bookmarks_toolbars = NULL;
	window->bookmarks_tooltips = NULL;

	window->dock = GNOME_APP(window->WMain)->dock;
	window->appbar = GNOME_APP(window->WMain)->statusbar;
	window->menubar = GNOME_APP(window->WMain)->menubar;

	window->view_menubar = view_menu_uiinfo[2].widget;
	window->view_toolbar = view_menu_uiinfo[3].widget;
	window->view_statusbar = view_menu_uiinfo[4].widget;
//	window->view_bookmarks_dock = view_menu_uiinfo[5].widget;
	window->view_history_dock = view_menu_uiinfo[5].widget;
	window->view_fullscreen = view_menu_uiinfo[6].widget;
	window->load_images_always = load_images_always_uiinfo[0].widget;
	window->load_images_from_current_server_only = 
					load_images_always_uiinfo[1].widget;
	window->load_images_never = load_images_always_uiinfo[2].widget;
	window->animate_always = animate_always_uiinfo[0].widget;
	window->animate_once_through = animate_always_uiinfo[1].widget;
	window->animate_never = animate_always_uiinfo[2].widget;
	window->use_own_fonts =  settings_menu_uiinfo[pos++].widget;
	window->use_own_colors = settings_menu_uiinfo[pos++].widget;
	window->enable_java = settings_menu_uiinfo[pos++].widget;
	window->enable_javascript = settings_menu_uiinfo[pos++].widget;
	window->enable_proxy = settings_menu_uiinfo[pos++].widget;
	window->use_galeon_mime_handling = settings_menu_uiinfo[pos++].widget;

	gtk_object_set_data(GTK_OBJECT(window->WMain), "WMain",
			    window->WMain);

	gtk_widget_ref(menubar_uiinfo[4].widget);
	gtk_object_set_data_full(GTK_OBJECT(window->WMain), "bookmarks",
					    menubar_uiinfo[4].widget,
					   (GtkDestroyNotify) gtk_widget_unref);

	gtk_widget_ref(GTK_MENU_ITEM(menubar_uiinfo[4].widget)->submenu);
	gtk_object_set_data_full(GTK_OBJECT(window->WMain), "bookmarks_menu",
			GTK_MENU_ITEM(menubar_uiinfo[4].widget)->submenu,
			(GtkDestroyNotify) gtk_widget_unref);

	gtk_widget_ref(GTK_MENU_ITEM(menubar_uiinfo[5].widget)->submenu);
	gtk_object_set_data_full(GTK_OBJECT(window->WMain), "go_menu",
			GTK_MENU_ITEM(menubar_uiinfo[5].widget)->submenu,
			(GtkDestroyNotify) gtk_widget_unref);

	gtk_widget_ref(bookmarks_menu_uiinfo[0].widget);
	gtk_object_set_data_full(GTK_OBJECT(window->WMain), "file_bookmark",
				 bookmarks_menu_uiinfo[0].widget,
				 (GtkDestroyNotify) gtk_widget_unref);

	gtk_widget_ref(bookmarks_menu_uiinfo[5].widget);
	gtk_object_set_data_full(GTK_OBJECT(window->WMain),
				 "bookmarks_separator",
				 bookmarks_menu_uiinfo[5].widget,
				 (GtkDestroyNotify) gtk_widget_unref);

	gtk_widget_ref(view_menu_uiinfo[11].widget);
	gtk_object_set_data_full(GTK_OBJECT(window->WMain),
				 "encoding_menu_item",
				 view_menu_uiinfo[11].widget,
				 (GtkDestroyNotify) gtk_widget_unref);
}

/**
 * window_show_open_dialog: show the open dialog
 */
void window_show_open_dialog (GaleonWindow *window)
{
	misc_ask_file_name (_("Open a file"), _("Select the file to open:"), 
			    gnome_config_get_string (CONF_DIR_OPEN),
			    file_open_ok_button_clicked_cb, NULL, TRUE, FALSE, window);
}

/**
 * window_show_openurl_dialog: show the open url dialog
 */
void window_show_openurl_dialog (GaleonWindow *window)
{
	GtkWidget *dialog = NULL;
	GtkBox *vbox;
	GtkBoxChild *child;
	GList *children;

	dialog = gnome_request_dialog(
		FALSE, _("Enter the URL of the location to open:                    \n"),
		NULL, 100, (GnomeStringCallback) open_url_ok_button_clicked_cb,
		window, NULL
		);
	gtk_window_set_title(GTK_WINDOW(dialog), _("Open URL"));

	window_set_layer(dialog);

	/* find the entry */
	vbox = GTK_BOX(GNOME_DIALOG(dialog)->vbox);
	children = vbox->children;

	while (children)
	{
		child = (GtkBoxChild *) children->data;
		if (GTK_IS_ENTRY(child->widget))
		{
			extern gchar *open_url_str;

			gtk_signal_connect_after(GTK_OBJECT(child->widget),
						 "key_press_event",
						 GTK_SIGNAL_FUNC(
						   window_location_entry_key_press_cb),
						 window);
			if (open_url_str)
			{
				gtk_entry_set_text(GTK_ENTRY(child->widget),
						   open_url_str);
			}
			break;
		}
		children = children->next;
	}

	gnome_dialog_run(GNOME_DIALOG (dialog));
}

/**
 * window_toolbar_show: show the toolbar
 */
void window_toolbar_show (GaleonWindow *window)
{
	gtk_widget_show (window->main_dockitem);	
	if (gnome_config_get_int ("/galeon/Appearance/url_location=1") == 0)
	{
		gtk_widget_show (window->location_dockitem);
	}
	bookmarks_toolbar_set_visible (window, TRUE);

}

/**
 * window_toolbar_hide: hide the toolbar
 */
void window_toolbar_hide (GaleonWindow *window)
{
	gtk_widget_hide (GTK_WIDGET (window->main_dockitem));
	if (window->location_dockitem)
	{
		gtk_widget_hide (GTK_WIDGET (window->location_dockitem));
	}
	bookmarks_toolbar_set_visible (window, FALSE);
	gtk_widget_queue_resize (GTK_WIDGET (window->notebook));
}

/**
 * window_menubar_show: show the menubar
 */
void window_menubar_show (GaleonWindow *window)
{
	gtk_widget_show (GTK_WIDGET (window->menubar->parent));	
}

/**
 * window_menubar_hide: hide the menubar
 */
void window_menubar_hide (GaleonWindow *window)
{
	gtk_widget_hide (GTK_WIDGET (window->menubar->parent));
	gtk_widget_queue_resize (GTK_WIDGET (window->notebook));
}

/**
 * window_statusbar_show: show the statusbar
 */
void window_statusbar_show (GaleonWindow *window)
{
	gtk_widget_show (GTK_WIDGET (window->appbar->parent));
}

/**
 * window_statusbar_hide: hide the statusbar
 */
void window_statusbar_hide (GaleonWindow *window)
{
	gtk_widget_hide (window->appbar->parent);
	gtk_widget_queue_resize (GTK_WIDGET (window->notebook));
}

/**
 * window_close: close a GaleonWindow
 */
void 
window_close (GaleonWindow *window)
{
	GList *e, *next;

	return_if_not_window (window);

	/* hide the window, this make the app seem more responsive */
	gtk_widget_hide (window->WMain);

	/* any children? */
	if (window->embed_list != NULL)
	{
		/* close any embed children */
		for (e = window->embed_list; e != NULL;)
		{
			GaleonEmbed *embed = (GaleonEmbed *)(e->data);
			return_if_not_embed (embed);
			next = g_list_next (e);
			embed_close (embed);
			e = next;
		}

		/* window_close will now be called again automatically */
	}
	else
	{
		/* remove window from list */
		all_windows = g_list_remove (all_windows, window);
		
		/* stop the activity bar */
		window->progress_timeout = FALSE;

		/* destroy the window, and all it contains */
		gtk_widget_destroy (window->WMain);

		/* save the history out to disk -- although this entails a bit
		 * of overhead hopefully it will mean we don't end up losing so
		 * much of the history on a crash... MattA 10/12/2000 */
		history_save ();

		/* quit if this was the last window to go and
		 * we're not in server mode */
		if (g_list_length (all_windows) == 0 && !galeon_server_mode)
		{
			galeon_exit ();
		}
	}
}

/**
 * window_go_home:
 */
void
window_go_home(GaleonWindow *window, gboolean new_embed, gboolean new_window)
{
	gchar *startpage;

	startpage = gnome_config_get_string 
		("/galeon/General/startpage=www.gnome.org");

        if (startpage != NULL && strlen(startpage) != 0)
        {
		if (new_embed)
		{
			embed_create_from_url (window->active_embed,
					       startpage, new_window);
		}
		else
		{
			window_load_url (window, startpage);
		}
	}
        else
	{
		gnome_error_dialog (_("You must specify a Start Page "
				      "in the Preferences dialog!"));
	}

	g_free(startpage);
}

/**
 *  Request that the widget be placed above or below the Gnome panels
 */
void
window_set_layer (GtkWidget *widget)
{
	/* fullscreen mode flag */
	extern gboolean fullscreen_active;
	gboolean raise_windows;

	raise_windows = gnome_config_get_bool(
				"/galeon/Appearance/fullscreen_stacking=TRUE");

	if (fullscreen_active && raise_windows)
		gnome_win_hints_set_layer(widget, WIN_LAYER_ABOVE_DOCK);
	else
		gnome_win_hints_set_layer(widget, WIN_LAYER_NORMAL);

	gdk_window_raise(widget->window);
}

/**
 * populate the View/Encodings menu with the available charsets 
 */
void
window_set_encoding_menu (GaleonWindow *window, GList *charset_titles)
{
	GtkWidget *encoding_menu_item;

	encoding_menu_item = gtk_object_get_data (GTK_OBJECT(window->WMain),
						  "encoding_menu_item");

	if (GTK_MENU_ITEM(encoding_menu_item)->submenu == NULL)
	{
		create_charset_submenus (GTK_MENU_ITEM(encoding_menu_item),
					 charset_titles, window);
	}
}

void
window_set_fullscreen_mode (GaleonWindow *window, gboolean active)
{
	GdkWindow *gdk_window = window->WMain->window;
	gint client_x, client_y, root_x, root_y;
	gint width, height;

	return_if_not_window (window);

	if (!gnome_config_get_bool("/galeon/Appearance/show_menubar_in_fullscreen")) 
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(
					       window->view_menubar),
					       !active);
	if (!gnome_config_get_bool("/galeon/Appearance/show_toolbars_in_fullscreen")) 
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(
					       window->view_toolbar),
					       !active);
	if (!gnome_config_get_bool("/galeon/Appearance/show_statusbar_in_fullscreen")) 
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(
					       window->view_statusbar),
					       !active);

	if (active)
	{
		fullscreen_active = TRUE;

		window_set_layer (window->WMain);

		gdk_window_get_origin (gdk_window, &root_x, &root_y);
		gdk_window_get_geometry (gdk_window, &client_x, &client_y,
					 &width, &height, NULL);

		gdk_window_move_resize (gdk_window, -client_x, -client_y,
					gdk_screen_width () + 1,
					gdk_screen_height () + 1);
		
		window->x = root_x - client_x;
		window->y = root_y - client_y;
		window->width = width;
		window->height = height;
	}
	else
	{
		/* Send an X state change request to place the window
		   back in the normal layer */
		gnome_win_hints_set_layer (window->WMain,
					   WIN_LAYER_NORMAL);

		gdk_window_move_resize (gdk_window, 
					window->x, window->y,
					window->width, 
					window->height);
		fullscreen_active = FALSE;
	}
}

static void 
window_set_menu_data (GaleonWindow *window)
{
	gint i, j;

	for (i = 0; i < num_menus; i++)
	{
		for (j = 0; j < menus_num_items[i]; j++)
		{
			all_menus[i][j].user_data = window;
		}
	}
}

/* Use this function instead of glade_lookup_widget() to find a widget
   you have attached to an object via gtk_object_set_data() from one
   of its child widgets. This is necessary because libglade is no longer
   being used to create the main window */
GtkWidget *
window_lookup_widget (GtkWidget *widget, const char *name)
{
	GtkWidget *found_widget = NULL;
	GtkWidget *parent = NULL;

	g_return_val_if_fail(widget != NULL, NULL);
	while (!found_widget)
	{
		found_widget = gtk_object_get_data (GTK_OBJECT(widget), name);

		if (GTK_IS_MENU(widget))
			parent = gtk_menu_get_attach_widget(GTK_MENU(widget));
		else
			parent = widget->parent;

		if (parent == NULL)
			break;

		widget = parent;
	}

	if (!found_widget)
		g_warning("%s not found.", name);

	return found_widget;
}

/**
 * window_reload_all: reload all tabs in a window
 */
void
window_reload_all (GaleonWindow *window)
{
	GList *l;

	/* check arguments */
	return_if_not_window (window);

	/* reload all tabs in the window */
	for (l = window->embed_list; l != NULL; l = g_list_next (l))
	{
		embed_reload ((GaleonEmbed *)(l->data));
	}
}

