/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "mozilla.h"
#include "misc_string.h"
#include "nsCOMPtr.h"
#include "JSConsoleListener.h"
#include "xpconnect/nsIScriptError.h"

#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include "js_console.h"

/* Implementation file */
NS_IMPL_ISUPPORTS1(JSConsoleListener, nsIConsoleListener)

JSConsoleListener::JSConsoleListener()
{
  NS_INIT_ISUPPORTS();
  /* member initializers and constructor code */
}

JSConsoleListener::~JSConsoleListener()
{
  /* destructor code */
}

/* void observe (in nsIConsoleMessage aMessage); */
NS_IMETHODIMP JSConsoleListener::Observe(nsIConsoleMessage *aMessage)
{
	PRUnichar *utmp;
	PRUint32 itmp;
	guint sline_offset = 0;
	GString *code_ref;
	gchar *ctmp;
	gchar *msg;
	gchar *error_message;

	aMessage->GetMessage (&utmp);
	ctmp = mozilla_unicode_to_locale (utmp);
	msg = misc_string_strip_newline (ctmp);
	g_free (ctmp);

	nsCOMPtr<nsIScriptError> error = do_QueryInterface (aMessage);
	if (error) /* best interface, more infos */
	{
		/* build a message */
		PRUint32 flags;
		JSConsoleMessageType type;
		error->GetFlags (&flags);

		if (flags == nsIScriptError::errorFlag ||
		    flags == nsIScriptError::exceptionFlag ||
		    flags == nsIScriptError::strictFlag)
		{
			type = JS_CONSOLE_ERROR;
		}
		else if (flags == nsIScriptError::warningFlag)
		{
			type = JS_CONSOLE_WARNING;
		}
		else
		{
			type = JS_CONSOLE_MESSAGE;
		}

		code_ref = g_string_new(NULL);

		error->GetSourceName (&utmp);
		if (utmp) 
		{
			ctmp =  mozilla_unicode_to_locale (utmp);
			if (strcmp(ctmp,"") != 0)
			{
				code_ref = g_string_append (code_ref, "In ");
				code_ref = g_string_append (code_ref, ctmp);
				code_ref = g_string_append (code_ref, ", ");
			}
			g_free (ctmp);
		}

		error->GetLineNumber (&itmp);
		if (itmp)
		{
			ctmp = g_strdup_printf ("%d", itmp);
			code_ref = g_string_append (code_ref, "Line ");
			code_ref = g_string_append (code_ref, ctmp);
			code_ref = g_string_append (code_ref, ", ");
			g_free (ctmp);
		}

		error->GetSourceLine (&utmp);
		if (utmp) 
		{
			sline_offset = strlen (code_ref->str);

			ctmp = mozilla_unicode_to_locale (utmp);

			char *tmp;
			tmp = misc_string_strip_newline (ctmp);
			code_ref = g_string_append (code_ref, tmp);
			if (strlen (tmp) > 0)
				code_ref = g_string_append (code_ref, "\n");

			g_free (tmp);
			g_free (ctmp);
		}
		
		sline_offset = strlen (msg) + sline_offset + 1;
		error->GetColumnNumber (&itmp);
		error_message = g_strconcat (msg, 
					     "\n",
					     code_ref->str,
					     NULL);
		js_console_append (error_message, type, 
				   sline_offset + itmp);
		g_free (error_message);

		g_string_free (code_ref, TRUE);
	}
	else
	{
		js_console_append (msg, JS_CONSOLE_MESSAGE, 0);
	}

	g_free (msg);
	
	return NS_OK;
}
