#############################################################################
##
#W  access.gi          GAP share package 'atlasrep'             Thomas Breuer
##
#H  @(#)$Id: access.gi,v 1.35 2001/03/29 16:13:50 gap Exp $
##
#Y  Copyright (C)  2001,  Lehrstuhl D fuer Mathematik,  RWTH Aachen,  Germany
##
##  This file contains functions for accessing data from the ATLAS of Group
##  Representations.
##
Revision.( "atlasrep/gap/access_gi" ) :=
    "@(#)$Id: access.gi,v 1.35 2001/03/29 16:13:50 gap Exp $";


#############################################################################
##
#F  FilenameAtlas( <dirname>, <groupname>, <filename> )
##
InstallGlobalFunction( FilenameAtlas,
    function( dirname, groupname, filename )

    local datadirs,
          len,
          datafile,
          triple,
          script,
          infile,
          outfile,
          perl,
          info,
          server,
          path,
          login,
          password,
          result,
          result2;

    # Get the absolute path name.
    datadirs:= DirectoriesPackageLibrary( "atlasrep", dirname );

    # If the file is in Magma format then look for the {\GAP} format file.
    len:= Length( filename );
    if filename[ len ] = 'M' then
      datafile:= ShallowCopy( filename );
      datafile[ len ]:= 'g';
      datafile:= Filename( datadirs, datafile );
      if datafile <> fail then
        return datafile;
      fi;
    fi;

    # Check whether the file is already stored.
    datafile:= Filename( datadirs, filename );
    if datafile = fail then

      # The file is not yet stored.
      # Try to transfer it.
      if AtlasOfGroupRepresentationsInfo.remote = true then

        # Get the group name info.
        triple:= First( AtlasOfGroupRepresentationsInfo.groupnames,
                        x -> x[3] = groupname );
        if triple = fail then
          Error( "illegal value of <groupname>" );
        fi;

        # Get the necessary files (`perl' and `perlftp.pl').
        script:= Filename( DirectoriesPackageLibrary( "atlasrep", "etc" ),
                           "perlftp.pl" );
        if script = fail then
          Info( InfoAtlasRep, 1, "no script `etc/perlftp.pl' found" );
          return fail;
        fi;
        perl:= Filename( DirectoriesSystemPrograms(), "perl" );
        if perl = fail then
          Info( InfoAtlasRep, 1, "no `perl' executable found" );
          return fail;
        fi;

        # Try to get the file remotely.
        for info in AtlasOfGroupRepresentationsInfo.servers do

          server   := info[1];
          path     := info[2];
          login    := info[3];
          password := info[4];

          # Compose the name of the directory on the server.
          if dirname = "datagens" then
            if filename[ Length( filename ) ] = 'M' then
              dirname:= Concatenation( path, triple[1],"/",triple[2],"/mag/" );
            else
              dirname:= Concatenation( path, triple[1],"/",triple[2],"/mtx/" );
            fi;
          elif dirname = "dataword" then
            dirname:= Concatenation( path, triple[1],"/",triple[2],"/words/" );
          else
            Error( "illegal value of <dirname>" );
          fi;

          # Fetch the file if possible.
          Info( InfoAtlasRep, 2,
                "calling `etc/perlftp.pl' to get `", filename, "'" );
          Process( datadirs[1], perl, InputTextNone(), OutputTextNone(),
             [ script, server, dirname, login, password, "get", filename ] );
          result:= Filename( datadirs, filename );
          if result <> fail then
            datafile:= result;
            break;
          fi;

        od;

        if datafile = fail then
          Info( InfoAtlasRep, 1,
                "no file `", filename, "' found on the servers" );
          return fail;
        fi;

      else

        # The file cannot be made available.
        Info( InfoAtlasRep, 1,
              "no file `", filename, "' found in the local directories" );
        return fail;

      fi;

    fi;

    if filename[ Length( filename ) ] = 'M' then

      # Translate the Magma file to {\GAP} format.
      script:= Filename( DirectoriesPackageLibrary( "atlasrep", "etc" ),
                         "mtog" );
      if script = fail then
        Info( InfoAtlasRep, 1, "no script `etc/mtog' found" );
        return fail;
      fi;
      outfile:= ShallowCopy( filename );
      outfile[ Length( outfile ) ]:= 'g';
      Info( InfoAtlasRep, 2, "calling `etc/mtog' for file `", filename, "'" );
      Process( datadirs[1], script, InputTextNone(), OutputTextNone(),
               [ filename, outfile ] );
      datafile:= Filename( datadirs, outfile );
      if not IsExistingFile( datafile ) then
        Info( InfoAtlasRep, 1, "translation with `etc/mtog' failed" );
        return fail;
      fi;

      # Remove the Magma format file.
      RemoveFile( result );

    fi;

    # Return the filename.
    return datafile;
end );


#############################################################################
##
#F  DirectoryTree( <string> )
##
InstallGlobalFunction( DirectoryTree, function( string )

    local lines, tree, i, len, line, lline, dir, file;

    # Split the string into lines.
    lines:= SplitString( string, "", "\n" );

    # Omit lines of the form `total <n>' that occur for `ls -l' output.
    lines:= Filtered( lines,
                l -> Length( l ) < 6 or l{ [ 1 .. 6 ] } <> "total " );

    # Initialize the result.
    tree:= rec();
    i:= 1;
    len:= Length( lines );

    # The files in the top directory are listed before the first leading dot.
    while i <= len do
      line:= lines[i];
      if line[1] = '.' and line <> ".:" then
        break;
      fi;
      line:= SplitString( line, "", " " );
      tree.( line[ Length( line ) ] ):= rec();
      i:= i+1;
    od;

    # Scan blocks separated by lines of the form `./<path>:',
    # which describe the directory in which the following files reside.
    while i <= len do

      # If we deal with `ls -l' output then cut split the line at blanks.
      line:= SplitString( lines[i], "", " " );

      # Step down the directory hierarchy of the path.
      lline:= SplitString( line[ Length( line ) ], ".:", "/" );
      dir:= tree;
      for file in lline do
        if not IsEmpty( file ) then
          dir:= dir.( file );
        fi;
      od;
      i:= i+1;

      # Store filenames (and `ls -l' info if applicable) in the directory.
      while i <= len do
        line:= lines[i];
        if line[1] = '.' then
          break;
        fi;
        line:= SplitString( line, "", " " );
        dir.( line[ Length( line ) ] ):=
            rec( data := line{ [ 1 .. Length( line ) - 1 ] } );
        i:= i+1;
      od;

    od;

    # Return the result.
    return tree;
end );


#############################################################################
##
#F  AtlasFilesFromDirectoryTree( <tree> )
##
InstallGlobalFunction( AtlasFilesFromDirectoryTree, function( tree )

    local filenames,
          dir,
          sub,
          simpname,
          filename,
          info,
          entries;

    # Check whether the notified directories are there.
    if not ForAll( AtlasOfGroupRepresentationsInfo.dirnames,
                   name -> IsBound( tree.( name ) ) ) then
      Info( InfoAtlasRep, 1,
            "some necessary top level directory is missing" );
      return fail;
    fi;

    # Walk through the tree.
    filenames:= [];
    for dir in AtlasOfGroupRepresentationsInfo.dirnames do

      sub:= tree.( dir );
      for simpname in RecNames( sub ) do
        if sub.( simpname ) <> rec() then

#T replace .M access !!
          # Add the files in the subdirectories `mag', `mtx', and `words'.
          if IsBound( sub.( simpname ).mag ) then
            for filename in RecNames( sub.( simpname ).mag ) do
              if filename[ Length( filename ) ] = 'M' then
                info:= [ dir, simpname, ShallowCopy( filename ) ];
                if IsBound( sub.( simpname ).mag.( filename ).data ) then
                  Add( info, sub.( simpname ).mag.( filename ).data );
                fi;
                Add( filenames, info );
              fi;
            od;
          fi;
          if IsBound( sub.( simpname ).mtx ) then
            for filename in RecNames( sub.( simpname ).mtx ) do
              if IsDigitChar( filename[ Length( filename ) ] ) then
                info:= [ dir, simpname, ShallowCopy( filename ) ];
                if IsBound( sub.( simpname ).mtx.( filename ).data ) then
                  Add( info, sub.( simpname ).mtx.( filename ).data );
                fi;
                Add( filenames, info );
              fi;
            od;
          fi;
          if IsBound( sub.( simpname ).words ) then
            for filename in RecNames( sub.( simpname ).words ) do
              if filename <> "data" then
                info:= [ dir, simpname, ShallowCopy( filename ) ];
                if IsBound( sub.( simpname ).words.( filename ).data ) then
                  Add( info, sub.( simpname ).words.( filename ).data );
                fi;
                Add( filenames, info );
              fi;
            od;
          fi;

        fi;
      od;

    od;

    # Return the list.
    return filenames;
end );


#############################################################################
##
#F  AGRGNAN( <gapname>, <atlasname> )
##
InstallGlobalFunction( AGRGNAN, function( gapname, atlasname )
    if   ForAny( AtlasOfGroupRepresentationsInfo.GAPnames,
                 pair -> gapname = pair[1] ) then
      Error( "cannot notify `", gapname, "' more than once" );
    elif ForAny( AtlasOfGroupRepresentationsInfo.GAPnames,
                 pair -> atlasname = pair[2] ) then
      Error( "ambiguous GAP names for ATLAS name `", gapname, "'" );
    fi;

    Add( AtlasOfGroupRepresentationsInfo.GAPnames, [ gapname, atlasname ] );
end );


#############################################################################
##
#F  GroupNamesOfAtlasOfGroupRepresentations( <list> )
##
InstallGlobalFunction( GroupNamesOfAtlasOfGroupRepresentations,
    function( list )

    local result, triple, name, pos, new;

    # Initialize the result list.
    result:= [];

    # Loop over the triples.
    for triple in list do

      # Extract the group name from the filename.
      name:= triple[3];
      pos:= Position( name, '-' );
      if pos = fail then
        if not name in [ "CVS", "dummy" ] then
          Info( InfoAtlasRep, 3,
                "group name construction: illegal name `", name, "'" );
        fi;
      else

        pos:= pos - 1;
        while 0 < pos and IsDigitChar( name[ pos ] ) do
          pos:= pos - 1;
        od;

        # (There may be a `max<k>W<n>' or `cycW<n>' part.)
        if name[ pos ] = 'W' then
          if pos > 3 and name{ [ pos-3 .. pos ] } = "cycW" then
            pos:= pos - 4;
            while pos > 0 and IsDigitChar( name[ pos ] ) do
              pos:= pos - 1;
            od;
          else
            pos:= pos - 1;
            while pos > 0 and IsDigitChar( name[ pos ] ) do
              pos:= pos - 1;
            od;
            if pos > 2 and name{ [ pos-2 .. pos ] } = "max" then
              pos:= pos - 3;
              while pos > 0 and IsDigitChar( name[ pos ] ) do
                pos:= pos - 1;
              od;
            fi;
          fi;
        fi;

        if pos = 0 or name[ pos ] <> 'G' then
          Info( InfoAtlasRep, 3,
                "group name construction: illegal name `", name, "'" );
        else
          new:= [ triple[1], triple[2], name{ [ 1 .. pos-1 ] } ];
          if not new in result then
            if ForAny( result, tr -> tr[3] = new[3] ) then
              Info( InfoAtlasRep, 1,
                    "group name `", new[3], "' used in different places" );
            else
              AddSet( result, new );
            fi;
          fi;
        fi;

      fi;

    od;

    # Sort the result list.
    Sort( result );

    # Return the result list.
    return result;
end );


#############################################################################
##
#F  AGRFLD( <filename>, <descr> )
##
InstallGlobalFunction( AGRFLD, function( filename, descr )
    local pair;
    pair:= [ filename, descr ];
    if pair in AtlasOfGroupRepresentationsInfo.fieldinfo then
      Info( InfoAtlasRep, 1,
            "pair `", pair, "' cannot be notified more than once" );
    else
      Add( AtlasOfGroupRepresentationsInfo.fieldinfo, [ filename, descr ] );
    fi;
end );


#############################################################################
##
#F  AtlasGeneratorsCharacteristicZeroFile( <filename> )
##
InstallGlobalFunction( AtlasGeneratorsCharacteristicZeroFile,
    function( filename )
    local mats;
    InfoRead1( "#I  reading `", filename, "' started\n" );
    mats:= ReadAsFunction( filename )();
    InfoRead1( "#I  reading `", filename, "' done\n" );
    return mats;
end );


#############################################################################
##
#F  AtlasStringOfFieldOfMatrixEntries( <mats> )
#F  AtlasStringOfFieldOfMatrixEntries( <filename> )
##
InstallGlobalFunction( AtlasStringOfFieldOfMatrixEntries, function( mats )

    local datafile, F, n;

    if IsString( mats ) then
      mats:= AtlasGeneratorsCharacteristicZeroFile( mats );
    fi;

    # Compute the field and the conductor.
    if not IsCyclotomicCollCollColl( mats ) then
      Error( "<mats> must be a nonempty list of matrices of cyclotomics" );
    fi;
    F:= Field( Rationals, Flat( mats ) );
    n:= Conductor( F );

    if DegreeOverPrimeField( F ) = 2 then

      # The field is quadratic,
      # so it is generated by `Sqrt(n)' if $`n' \equiv 1 \pmod{4}$,
      # by `Sqrt(-n)' if $`n' \equiv 3 \pmod{4}$,
      # and by one of `Sqrt(n/4)', `Sqrt(-n/4)' otherwise.
      if   n mod 4 = 1 then
        return Concatenation( "Field([Sqrt(", String( n ), ")])" );
      elif n mod 4 = 3 then
        return Concatenation( "Field([Sqrt(-", String( n ), ")])" );
      elif Sqrt( -n/4 ) in F then
        return Concatenation( "Field([Sqrt(-", String( n/4 ), ")])" );
      else
        return Concatenation( "Field([Sqrt(", String( n/4 ), ")])" );
      fi;

    elif IsCyclotomicField( F ) then

      # The field is not quadratic but cyclotomic.
      return Concatenation( "Field([E(", String( n ), ")])" );

    else
      return String( F );
    fi;
end );


#############################################################################
##
#F  AtlasTableOfContents( \"local\"[, <long>] )
#F  AtlasTableOfContents( \"remote\"[, <long>] )
##
##  If `true' is given as the optional second argument <long>
##  then the `TableOfContents' component of the result record has the
##  additional component `data',
##  which is a list describing the last modification dates of the files
##  in the local or remote directory, respectively.
##  (This option is used in the test performed by
##  `AtlasOfGroupRepresentationsTestTableOfContentsRemoteUpdates'.)
##
InstallGlobalFunction( AtlasTableOfContents, function( arg )

    local string,
          long,
          toc,
          str,
          out,
          name,
          options,
          data,
          script,
          path,
          dir,
          server,
          cmd,
          groupnames,
          AtlasScanFilename,
          result,
          maxstd,
          groupname,
          gens,
          bas,
          reps,
          entry,
          listtosort,
          descr,
          dashes,
          order,
          index;

    toc:= AtlasOfGroupRepresentationsInfo.TableOfContents;

    # Get and check the arguments.
    if   Length( arg ) = 1 and arg[1] in [ "local", "remote" ] then
      string := arg[1];
      long   := false;
    elif Length( arg ) = 2 and arg[1] in [ "local", "remote" ]
                           and IsBool( arg[2] ) then
      string := arg[1];
      long   := arg[2];
    else
      Error( "usage: AtlasTableOfContents( \"remote\"[, <long>] ) or\n",
             "AtlasTableOfContents( \"local\"[, <long>] )" );
    fi;

    if string = "local" then

      # Take the stored version if it is already available.
      groupnames:= AtlasOfGroupRepresentationsInfo.groupnames;
      if IsBound( toc.( "local" ) ) then
        return rec( groupnames      := groupnames,
                    TableOfContents := toc.( "local" ) );
      fi;

      # List the contents of the local data directories.
      str:= "";
      out:= OutputTextString( str, true );
      name:= Filename( DirectoriesSystemPrograms(), "ls" );
      if long then
        options:= [ "-l" ];
      else
        options:= [];
      fi;
      for dir in DirectoriesPackageLibrary( "atlasrep", "datagens" ) do
        Info( InfoAtlasRep, 2,
              "calling `ls' for the local `datagens' directory" );
        Process( dir, name, InputTextNone(), out, options );
      od;
      for dir in DirectoriesPackageLibrary( "atlasrep", "dataword" ) do
        Info( InfoAtlasRep, 2,
              "calling `ls' for the local `dataword' directory" );
        Process( dir, name, InputTextNone(), out, options );
      od;
      CloseStream( out );

      # Split the string into lines.
      str:= SplitString( str, "", "\n" );
      str:= Filtered( str,
                x -> Length( x ) < 6 or x{ [ 1 .. 6 ] } <> "total " );

      # If applicable then split the lines into data and filenames.
      str:= List( str, l -> SplitString( l, "", " " ) );
      data:= List( str, l -> l{ [ 1 .. Length( l ) - 1 ] } );
      str:= List( str, l -> l[ Length( l ) ] );

#T allow .gz files?

    elif string = "remote" then

      # Take the stored version if it is already available.
      if IsBound( toc.remote ) then
        groupnames:= AtlasOfGroupRepresentationsInfo.groupnames;
        return rec( groupnames      := groupnames,
                    TableOfContents := toc.remote );
      fi;

      # List the contents of the remote data directories.
      name:= Filename( DirectoriesSystemPrograms(), "perl" );
      if name = fail then
        Info( InfoAtlasRep, 1, "no `perl' executable found" );
        return fail;
      fi;
      script:= Filename( DirectoriesPackageLibrary( "atlasrep", "etc" ),
                         "perlftp.pl" );
      if script = fail then
        Info( InfoAtlasRep, 1, "no script `etc/perlftp.pl' found" );
        return fail;
      fi;
      for server in AtlasOfGroupRepresentationsInfo.servers do
        str:= "";
        out:= OutputTextString( str, true );
        if long then
          options:= "-lR";
        else
          options:= "-R";
        fi;
        Info( InfoAtlasRep, 2,
              "calling `etc/perlftp.pl' to list all server files" );
        Process( DirectoryCurrent(), name, InputTextNone(), out,
                 [ script, server[1], server[2], server[3], server[4],
                   "ls", options ] );
        CloseStream( out );
        if not IsEmpty( str ) then
          break;
        fi;
      od;

      # Turn the information about directories and files into the
      # appropriate format,
      # and extract the group name info.
      str:= AtlasFilesFromDirectoryTree( DirectoryTree( str ) );
      if str = fail then
        # An error message has been printed already by
        # `AtlasFilesFromDirectoryTree'.
        return fail;
      fi;
      groupnames:= GroupNamesOfAtlasOfGroupRepresentations( str );

      # Save the `ls -l' info if applicable.
      if long then
        data:= List( str, x -> x[4] );
      else
        data:= ListWithIdenticalEntries( Length( str ), [] );
      fi;
      str:= List( str, x -> x[3] );

    else
      Error( "usage: AtlasTableOfContents( \"local\" ) or\n",
             "AtlasTableOfContents( \"remote\" )" );
    fi;

    # Process the filename.
    AtlasScanFilename := function( name, result, maxstd )

      local len,
            groupname,
            pos, pos2, pos3,
            dim,
            id,
            nr,
            modulus,
            fieldsize,
            entry,
            degree,
            subgroupname,
            subgens;

      # 1. Identify the group name and the number of standard generators;
      #    the initial substring of `name' --up to the `-' character--
      #    is of the form `<groupname>G<i>', `<groupname>G<i>cycW<n>',
      #    or `<groupname>G<i>max<k>W<n>'.

      # 1.1 Find the `-' character and the number before it.
      len:= Length( name );
      pos:= Position( name, '-' );
      if pos = fail then
        return false;
      fi;
      pos2:= pos - 1;
      while pos2 > 0 and IsDigitChar( name[ pos2 ] ) do
        pos2:= pos2 - 1;
      od;
      if pos2 = 0 then
        return false;
      fi;

      # 1.2 There may be a `max<k>W<n>' or `cycW<n>' part before the `-'.
      if pos2 > 4 and name{ [ pos2-3 .. pos2 ] } = "cycW" then
        pos:= pos2 - 3;
        pos2:= pos - 1;
        if not IsDigitChar( name[ pos2 ] ) then
          return false;
        fi;
        while pos2 > 0 and IsDigitChar( name[ pos2 ] ) do
          pos2:= pos2 - 1;
        od;
      elif name[ pos2 ] = 'W' then
        pos2:= pos2 - 1;
        if pos2 = 0 or not IsDigitChar( name[ pos2 ] ) then
          return false;
        fi;
        while pos2 > 0 and IsDigitChar( name[ pos2 ] ) do
          pos2:= pos2 - 1;
        od;
        if pos2 > 3 and name{ [ pos2-2 .. pos2 ] } = "max" then
          pos:= pos2 - 2;
          pos2:= pos - 1;
          if pos2 = 0 or not IsDigitChar( name[ pos2 ] ) then
            return false;
          fi;
          while pos2 > 0 and IsDigitChar( name[ pos2 ] ) do
            pos2:= pos2 - 1;
          od;
        fi;
      fi;

      # 1.3 Group name and standard generators number can now be read off.
      if pos2 <= 1 or name[ pos2 ] <> 'G' or pos-1 <= pos2 then
        return false;
      fi;
      groupname:= name{ [ 1 .. pos2 - 1 ] };
      gens:= Int( name{ [ pos2 + 1 .. pos - 1 ] } );

      if not IsBound( result.( groupname ) ) then
        result.( groupname ):= rec( perm    := [],
                                    matff   := [],
                                    matalg  := [],
                                    matint  := [],
                                    matmodn := [],
                                    classes := [],
                                    maxes   := [],
                                    maxstd  := [],
                                    out     := [] );
      fi;

      # 2. Determine the data of the representation.
      #    Now `pos' is the position of the first character after the `-'.
      pos:= pos + 1;
      if   pos < len and name[ pos ] = 'A' and name[ pos+1 ] = 'r'
                     and name[ len-1 ] = '.' then

        # {\GAP} readable representation over an algebraic number field,
        # compute dimension and identifier.
        # `<groupname>G<i>-Ar<dim><id>B<m>.M'
        pos2:= First( [ pos+2 .. len ], i -> not IsDigitChar( name[i] ) );
        if pos2 = fail or pos2 = pos+2 then
          return false;
        fi;
        dim:= Int( name{ [ pos+2 .. pos2-1 ] } );
        pos:= len - 2;
        while pos2 < pos and IsDigitChar( name[ pos ] ) do
          pos:= pos - 1;
        od;
        if name[ pos ] = 'B' and pos+1 <= len-2 then
          bas:= Int( name{ [ pos+1 .. len-2 ] } );
          id:= name{ [ pos2 .. pos-1 ] };
          Add( result.( groupname ).matalg,
               [ gens, dim, id, bas, name ] );
          return true;
        fi;

      elif pos < len and name[ pos ] = 'Z' and name[ pos+1 ] = 'r'
                     and name[ len-1 ] = '.' then

        # {\GAP} readable representation over the integers,
        # compute dimension and identifier.
        # `<groupname>G<i>-Zr<dim><id>B<m>.M'
        pos2:= First( [ pos+2 .. len ], i -> not IsDigitChar( name[i] ) );
        if pos2 = fail or pos2 = pos+2 then
          return false;
        fi;
        dim:= Int( name{ [ pos+2 .. pos2-1 ] } );
        pos:= len - 2;
        while pos2 < pos and IsDigitChar( name[ pos ] ) do
          pos:= pos - 1;
        od;
        if name[ pos ] = 'B' and pos+1 <= len-2 then
          bas:= Int( name{ [ pos+1 .. len-2 ] } );
          id:= name{ [ pos2 .. pos-1 ] };
          Add( result.( groupname ).matint,
               [ gens, dim, id, bas, name ] );
          return true;
        fi;

      elif pos < len and name[ pos ] = 'Z'
                     and name[ len-1 ] = '.' then

        # {\GAP} readable representation over a residue class ring,
        # compute modulus and identifier.
        # `<groupname>G<i>-Z<n>r<dim><id>B<m>.M'
        pos2:= First( [ pos+1 .. len ], i -> not IsDigitChar( name[i] ) );
        if pos2 = fail or pos2 = pos+1 or name[ pos2 ] <> 'r' then
          return false;
        fi;
        modulus:= Int( name{ [ pos+1 .. pos2-1 ] } );
        pos:= pos2;
        pos2:= First( [ pos+1 .. len ], i -> not IsDigitChar( name[i] ) );
        if pos2 = fail or pos2 = pos+1 then
          return false;
        fi;
        dim:= Int( name{ [ pos+1 .. pos2-1 ] } );
        pos:= len - 2;
        while pos2 < pos and IsDigitChar( name[ pos ] ) do
          pos:= pos - 1;
        od;
        if name[ pos ] = 'B' and pos+1 <= len-2 then
          bas:= Int( name{ [ pos+1 .. len-2 ] } );
          id:= name{ [ pos2 .. pos-1 ] };
          Add( result.( groupname ).matmodn,
               [ gens, modulus, dim, id, bas, name ] );
          return true;
        fi;

      elif pos < len and name[ pos ] = 'f' then

        # {\MeatAxe} format file with one matrix generator,
        # compute field size, dimension, identifier, and generator number.
        # `<groupname>G<i>-f<q>r<dim><id>B<m>.m<nr>'
        pos2:= First( [ pos+1 .. len ], i -> not IsDigitChar( name[i] ) );
        if pos2 = fail or pos2 = pos+1 or name[ pos2 ] <> 'r' then
          return false;
        fi;
        fieldsize:= Int( name{ [ pos+1 .. pos2-1 ] } );
        pos:= First( [ pos2+1 .. len ],
                     i -> not IsDigitChar( name[i] ) );
        if pos = fail or pos = pos2+1 then
          return false;
        fi;
        dim:= Int( name{ [ pos2+1 .. pos-1 ] } );
        pos2:= len;
        while pos < pos2 and IsDigitChar( name[ pos2 ] ) do
          pos2:= pos2 - 1;
        od;
        if name[ pos2 ] <> 'm' or name[ pos2-1 ] <> '.' or pos2 = len then
          return false;
        fi;
        nr:= Int( name{ [ pos2+1 .. len ] } );
        pos2:= pos2-2;
        pos3:= pos2;
        while pos < pos3 and IsDigitChar( name[ pos3 ] ) do
          pos3:= pos3 - 1;
        od;
        if name[ pos3 ] <> 'B' or pos3 = pos2 then
          return false;
        fi;
        bas:= Int( name{ [ pos3+1 .. pos2 ] } );
        id:= name{ [ pos .. pos3-1 ] };
        entry:= First( result.( groupname ).matff,
                       x ->     x[1] = gens
                            and x[2] = fieldsize
                            and x[3] = dim
                            and x[4] = id
                            and x[5] = bas );
        if entry = fail then
          entry:= [ gens, fieldsize, dim, id, bas, [] ];
          Add( result.( groupname ).matff, entry );
        fi;
        entry[6][ nr ]:= name;
        return true;

      elif pos < len and name[ pos ] = 'p' then

        # {\MeatAxe} format file with one permutation generator,
        # compute degree, identifier, and generator number.
        # `<groupname>G<i>-p<n><id>B<m>.m<nr>'
        pos2:= First( [ pos+1 .. len ], i -> not IsDigitChar( name[i] ) );
        if pos2 = fail or pos2 = pos+1 then
          return false;
        fi;
        degree:= Int( name{ [ pos+1 .. pos2-1 ] } );
        pos:= pos2;
        pos2:= len;
        while pos < pos2 and IsDigitChar( name[ pos2 ] ) do
          pos2:= pos2 - 1;
        od;
        if name[ pos2 ] <> 'm' or name[ pos2-1 ] <> '.' or pos2 = len then
          return false;
        fi;
        nr:= Int( name{ [ pos2+1 .. len ] } );
        pos2:= pos2-2;
        pos3:= pos2;
        while pos < pos3 and IsDigitChar( name[ pos3 ] ) do
          pos3:= pos3 - 1;
        od;
        if name[ pos3 ] <> 'B' or pos3 = pos2 then
          return false;
        fi;
        bas:= Int( name{ [ pos3+1 .. pos2 ] } );
        id:= name{ [ pos .. pos3-1 ] };
        entry:= First( result.( groupname ).perm,
                       x ->     x[1] = gens
                            and x[2] = degree
                            and x[3] = id
                            and x[4] = bas );
        if entry = fail then
          entry:= [ gens, degree, id, bas, [] ];
          Add( result.( groupname ).perm, entry );
        fi;
        entry[5][ nr ]:= name;
        return true;

      elif pos+2 < len and name{ [ pos .. pos+2 ] } = "max" then

        # straight line program to compute generators of a max. subgroup
        # `<groupname>G<i>-max<k>W<n>'
        pos2:= Position( name, 'W', pos );
        if pos2 <> fail and Int( name{ [ pos2+1 .. len ] } ) <> fail then
          nr:= Int( name{ [ pos+3 .. pos2-1 ] } );
          if nr <> fail and IsPosInt( nr ) then
            if IsBound( result.( groupname ).maxes[ nr ] ) then
              Add( result.( groupname ).maxes[ nr ], [ gens, name ] );
            else
              result.( groupname ).maxes[ nr ]:= [ [ gens, name ] ];
            fi;
            return true;
          fi;
        fi;

      elif pos+3 < len and name{ [ pos .. pos+3 ] } = "cycW" then

        # straight line program to compute representatives of
        # maximally cyclic subgroups
        # `<groupname>G<i>-cycW<n>'
        if Int( name{ [ pos+4 .. len ] } ) <> fail then
          Add( result.( groupname ).classes, [ gens, name ] );
          return true;
        fi;

      elif pos+4 < len and name{ [ pos .. pos+4 ] } = "cclsW" then

        # straight line program to compute representatives of
        # maximally cyclic subgroups
        # `<groupname>G<i>-cclsW<n>'
        if Int( name{ [ pos+5 .. len ] } ) <> fail then
          Add( result.( groupname ).classes, [ gens, name ] );
          return true;
        fi;

      elif pos+9 < len and name{ [ pos-1 .. pos+2 ] } = "cycW" then

        # straight line program to complete representatives of conj. classes
        # `<groupname>G<i>cycW<n>-cclsW<m>'
        pos2:= pos+3;
        pos:= pos2;
        while pos <= len and IsDigitChar( name[ pos ] ) do
          pos:= pos+1;
        od;
        if     pos <= len and pos2 < pos
           and name{ [ pos .. pos+5 ] } = "-cclsW"
           and IsPosInt( Int( name{ [ pos+6 .. len ] } ) ) then
          Add( result.( groupname ).classes, [ gens, name ] );
          return true;
        fi;

      elif pos+1 < len and name{ [ pos-1 .. pos+1 ] } = "max" then

        # straight line program to standardize generators of a max. subgroup
        # `<groupname>G<i>max<k>W<n>-<subgroupname>G<j>W<m>'
        pos2:= len;
        while IsDigitChar( name[ pos2 ] ) do
          pos2:= pos2-1;
        od;
        if pos2 = len or name[ pos2 ] <> 'W' then
          return false;
        fi;
        pos2:= pos2-1;
        pos:= pos2;
        while IsDigitChar( name[ pos2 ] ) do
          pos2:= pos2-1;
        od;
        if pos2 = pos or name[ pos2 ] <> 'G' then
          return false;
        fi;
        subgroupname:= name{ [ Position( name, '-' ) + 1 .. pos2-1 ] };
        subgens:= Int( name{ [ pos2+1 .. pos ] } );

        # For the moment, we memorize the parts;
        # and in the end we check whether `subgroupname' occurs as
        # a group name (including the standard generators).
        Add( maxstd, [ groupname, gens, subgroupname, subgens, name ] );
        return true;

      elif pos+1 < len and name{ [ pos ] } = "a" then

        # straight line program to compute images under an automorphism
        # `<groupname>G<i>-a<outname>W<n>'
        # (We store the triple `[<gens>,<outname>,name]'.)
        pos2:= Position( name, 'W', pos );
        if pos2 = fail then
          return false;
        fi;
        descr:= name{ [ pos+1 .. pos2-1 ] };
        pos:= Position( descr, 'p' );
        if pos = fail then
          dashes:= "";
          pos:= Length( descr ) + 1;
        elif pos = Length( descr ) then
          dashes:= "'";
        else
          dashes:= Int( descr{ [ pos+1 .. Length( descr ) ] } );
          if dashes = fail then
            return fail;
          fi;
          dashes:= ListWithIdenticalEntries( dashes, '\'' );
        fi;
        descr:= descr{ [ 1 .. pos-1 ] };
        pos:= Position( descr, '_' );
        if pos = fail then
          order:= descr;
          index:= "";
        else
          order:= descr{ [ 1 .. pos-1 ] };
          index:= descr{ [ pos+1 .. Length( descr ) ] };
        fi;
        if Int( order ) = fail or Int( index ) = fail then
          return fail;
        elif order = "" then
          order:= "2";
        fi;
        if index <> "" then
          order:= Concatenation( order, "_", index );
        fi;
        order:= Concatenation( order, dashes );
        Add( result.( groupname ).out, [ gens, order, name ] );
        return true;

      fi;

      return false;
    end;

    # Initialize the result record.
    result:= rec( otherfiles:= [] );
    maxstd:= [];

    # Loop over the filenames.
    for name in str do
      if AtlasScanFilename( name, result, maxstd ) = false then
        if not name in [ "CVS", "dummy" ] then
          Info( InfoAtlasRep, 3,
                "t.o.c. construction: illegal name `", name, "'" );
        fi;
        AddSet( result.otherfiles, name );
      fi;
    od;

    # Check the names of the standardization files.
    for entry in maxstd do
      name:= entry[5];
      if IsBound( result.( entry[3] ) ) then
        Add( result.( entry[1] ).maxstd, [ entry[2], name ] );
      else
        Info( InfoAtlasRep, 3,
              "t.o.c. construction: illegal name `", name, "'" );
        AddSet( result.otherfiles, name );
      fi;
    od;

    # Postprocessing.
    # Check the representations with generators in several files
    # whether the list of available generators is ``dense''.
    # Also *sort* the representations as follows:
    # - Permutation representations are sorted according to their degree.
    # - Matrix representations over finite fields are sorted according to
    #   the characteristic and for same characteristic according to their
    #   dimension.
    # - Matrix representations over the integers are sorted according to
    #   their dimension.
    # - Matrix representations over algebraic extension fields are sorted
    #   according to their dimension.
    # - Matrix representations over residue class rings are sorted according
    #   to the modulus, and representations with same modulus are sorted
    #   according to their dimension.
    for groupname in List( groupnames, x -> x[3] ) do
      if IsBound( result.( groupname ) ) then

        reps:= result.( groupname );
        for entry in reps.perm do
          if not ForAll( [ 1 .. Length( entry[5] ) ],
                         i -> IsBound( entry[5][i] ) ) then
#T better check whether the number of generators equals the number of
#T standard generators!
            Info( InfoAtlasRep, 1, "not all generators for ", entry[5] );
            Unbind( entry[1] );
          fi;
        od;
        reps.perm:= Filtered( reps.perm, x -> IsBound( x[1] ) );
        listtosort:= List( reps.perm, x -> x[2] );
        SortParallel( listtosort, reps.perm );

        for entry in reps.matff do
          if not ForAll( [ 1 .. Length( entry[6] ) ],
                         i -> IsBound( entry[6][i] ) ) then
#T better check whether the number of generators equals the number of
#T standard generators!
            Info( InfoAtlasRep, 1, "not all generators for ", entry[6] );
            Unbind( entry[1] );
          fi;
        od;
        reps.matff:= Filtered( reps.matff, x -> IsBound( x[1] ) );
        listtosort:= List( reps.matff, x -> [ x[2], x[3] ] );
        SortParallel( listtosort, reps.matff );

        listtosort:= List( reps.matint, x -> x[2] );
        SortParallel( listtosort, reps.matint );

        listtosort:= List( reps.matalg, x -> x[2] );
        SortParallel( listtosort, reps.matalg );

        listtosort:= List( reps.matmodn, x -> [ x[2], x[3] ] );
        SortParallel( listtosort, reps.matmodn );

      fi;
    od;

    # Store the newly computed table of contents.
    if string = "local" then
      toc.( "local" ):= result;
    else
      toc.remote:= result;
      AtlasOfGroupRepresentationsInfo.groupnames:= groupnames;
    fi;

    # Return the result record.
    if long then
      result.data:= [ str, data ];
    fi;
    return rec( groupnames      := groupnames,
                TableOfContents := result );
end );


#############################################################################
##
#F  ReloadAtlasTableOfContents( \"local\"[, <long>] )
#F  ReloadAtlasTableOfContents( \"remote\"[, <long>] )
##
##  If `true' is given as the optional second argument <long>
##  then also the `data' component of the table of contents is computed.
##  (This option is used by `AtlasTableOfContents' and
##  `AtlasOfGroupRepresentationsTestTableOfContentsRemoteUpdates'.)
##
InstallGlobalFunction( ReloadAtlasTableOfContents, function( arg )

    local string, long, toc, old, new;

    if Length( arg ) = 1 then
      string := arg[1];
      long   := false;
    elif Length( arg ) = 2 and IsBool( arg[2] ) then
      string := arg[1];
      long   := arg[2];
    fi;

    if   string in [ "local", "remote" ] then

      toc:= AtlasOfGroupRepresentationsInfo.TableOfContents;
      if IsBound( toc.( string ) ) then
        old:= toc.( string );
        Unbind( toc.( string ) );
      fi;
      new:= AtlasTableOfContents( string, long );
      if new = fail then
        if old <> 0 then
          toc.( string ):= old;
        fi;
        Error( "could not reload ", string, " table of contents" );
      fi;
      toc.( string ):= new.TableOfContents;
      if string = "remote" then
        AtlasOfGroupRepresentationsInfo.groupnames:= new.groupnames;
      fi;

    else
      Error( "usage: <string> must be \"local\" or \"remote\"" );
    fi;
end );


#############################################################################
##
#F  StringOfAtlasTableOfContents( \"remote\" )
#F  StringOfAtlasTableOfContents( \"local\" )
##
InstallGlobalFunction( StringOfAtlasTableOfContents, function( string )

    local toc, str, triple, groupname, reps, entry, i;

    toc:= AtlasTableOfContents( string ).TableOfContents;
    str:= "";

    for triple in AtlasOfGroupRepresentationsInfo.groupnames do

      # Start with the notifications of the group.
      Append( str, "# " );
      Append( str, triple[3] );
      Append( str, "\n" );
      Append( str, "AGRGRP(\"" );
      Append( str, triple[1] );
      Append( str, "\",\"" );
      Append( str, triple[2] );
      Append( str, "\",\"" );
      Append( str, triple[3] );
      Append( str, "\");\n" );

      # Append the notifications of data entries for the groups.
      groupname:= triple[3];
      if IsBound( toc.( groupname ) ) then

        reps:= toc.( groupname );
        for entry in reps.perm do
          Append( str, "AGRTOC(\"" );
          Append( str, groupname );
          Append( str, "\"," );
          Append( str, String( entry[1] ) );              # <gens>
          Append( str, ",\"p\"," );
          Append( str, String( entry[2] ) );              # <degree>
          Append( str, ",\"" );
          Append( str, entry[3] );                        # <id>
          Append( str, "\"," );
          Append( str, String( entry[4] ) );              # <bas>
          Append( str, ",[\"" );                          # <filenames>
          Append( str, entry[5][1]{ [ 1 .. Length( entry[5][1] )-1 ] } );
          Append( str, "\"," );
          Append( str, String( Length( entry[5] ) ) );
          Append( str, "]);\n" );
        od;

        for entry in reps.matff do
          Append( str, "AGRTOC(\"" );
          Append( str, groupname );
          Append( str, "\"," );
          Append( str, String( entry[1] ) );              # <gens>
          Append( str, ",\"f\"," );
          Append( str, String( entry[2] ) );              # <q>
          Append( str, "," );
          Append( str, String( entry[3] ) );              # <dim>
          Append( str, ",\"" );
          Append( str, entry[4] );                        # <id>
          Append( str, "\"," );
          Append( str, String( entry[5] ) );              # <bas>
          Append( str, ",[\"" );                          # <filenames>
          Append( str, entry[6][1]{ [ 1 .. Length( entry[6][1] )-1 ] } );
          Append( str, "\"," );
          Append( str, String( Length( entry[6] ) ) );
          Append( str, "]);\n" );
        od;

        for entry in reps.matalg do
          Append( str, "AGRTOC(\"" );
          Append( str, groupname );
          Append( str, "\"," );
          Append( str, String( entry[1] ) );              # <gens>
          Append( str, ",\"Ar\"," );
          Append( str, String( entry[2] ) );              # <dim>
          Append( str, ",\"" );
          Append( str, entry[3] );                        # <id>
          Append( str, "\"," );
          Append( str, String( entry[4] ) );              # <bas>
          Append( str, ",\"" );
          Append( str, entry[5] );                        # <filename>
          Append( str, "\");\n" );
        od;

        for entry in reps.matint do
          Append( str, "AGRTOC(\"" );
          Append( str, groupname );
          Append( str, "\"," );
          Append( str, String( entry[1] ) );              # <gens>
          Append( str, ",\"Zr\"," );
          Append( str, String( entry[2] ) );              # <dim>
          Append( str, ",\"" );
          Append( str, entry[3] );                        # <id>
          Append( str, "\"," );
          Append( str, String( entry[4] ) );              # <bas>
          Append( str, ",\"" );
          Append( str, entry[5] );                        # <filename>
          Append( str, "\");\n" );
        od;

        for entry in reps.matmodn do
          Append( str, "AGRTOC(\"" );
          Append( str, groupname );
          Append( str, "\"," );
          Append( str, String( entry[1] ) );              # <gens>
          Append( str, ",\"R\"," );
          Append( str, String( entry[2] ) );              # <modulus>
          Append( str, "," );
          Append( str, String( entry[3] ) );              # <dim>
          Append( str, ",\"" );
          Append( str, entry[4] );                        # <id>
          Append( str, "\"," );
          Append( str, String( entry[5] ) );              # <bas>
          Append( str, ",\"" );
          Append( str, entry[6] );                        # <filename>
          Append( str, "\");\n" );
        od;

        for entry in reps.classes do
          Append( str, "AGRTOC(\"" );
          Append( str, groupname );
          Append( str, "\"," );
          Append( str, String( entry[1] ) );              # <gens>
          Append( str, ",\"C\",\"" );
          Append( str, entry[2] );                        # <filename>
          Append( str, "\");\n" );
        od;

        for i in [ 1 .. Length( reps.maxes ) ] do
          if IsBound( reps.maxes[i] ) then
            for entry in reps.maxes[i] do
              Append( str, "AGRTOC(\"" );
              Append( str, groupname );
              Append( str, "\"," );
              Append( str, String( entry[1] ) );          # <gens>
              Append( str, ",\"M\"," );
              Append( str, String( i ) );                 # <n>
              Append( str, ",\"" );
              Append( str, entry[2] );                    # <filename>
              Append( str, "\");\n" );
            od;
          fi;
        od;

        for entry in reps.maxstd do
          Append( str, "AGRTOC(\"" );
          Append( str, groupname );
          Append( str, "\"," );
          Append( str, String( entry[1] ) );              # <gens>
          Append( str, ",\"MS\",\"" );
          Append( str, entry[2] );                        # <filename>
          Append( str, "\");\n" );
        od;

        for entry in reps.out do
          Append( str, "AGRTOC(\"" );
          Append( str, groupname );
          Append( str, "\"," );
          Append( str, String( entry[1] ) );              # <gens>
          Append( str, ",\"O\",\"" );
          Append( str, entry[2] );                        # <outname>
          Append( str, "\",\"" );
          Append( str, entry[3] );                        # <filename>
          Append( str, "\");\n" );
        od;

        Append( str, "\n" );

      fi;

    od;

    Append( str, "\n" );

    return str;
end );


#############################################################################
##
#F  AGRGRP( <dirname>, <simpname>, <groupname> )
##
InstallGlobalFunction( AGRGRP,
    function( dirname, simpname, groupname )

    local entry;

    if   not dirname in AtlasOfGroupRepresentationsInfo.dirnames then

      # The directory name is not known.
      # (Note that the automatic creation of a table of contents
      # assumes that all relevant directories are stored.)
      Error( "`", dirname, "' is not known" );

    elif ForAll( AtlasOfGroupRepresentationsInfo.GAPnames,
                 pair -> pair[2] <> groupname ) then

      # There is no corresponding {\GAP} name.
      Add( AtlasOfGroupRepresentationsInfo.GAPnames,
           [ groupname, groupname ] );
      Info( InfoAtlasRep, 1,
            "no GAP name known for `", groupname, "'" );

    fi;

    entry:= First( AtlasOfGroupRepresentationsInfo.groupnames,
                   l -> l[3] = groupname );
    if entry <> fail then

      # Check whether the group is already notified.
      if entry[1] = dirname then
        Info( InfoAtlasRep, 1,
              "group with Atlas name `", groupname, "' already notified" );
      else
        Error( "group with Atlas name `", groupname,
               "' notified for different directories!" );
      fi;
      if   entry[2] <> simpname then
        Error( "group with Atlas name `", groupname,
               "' notified for different simple groups!" );
      fi;

    else

      # Notify the group.
      Add( AtlasOfGroupRepresentationsInfo.groupnames,
           [ dirname, simpname, groupname ] );

    fi;
end );


#############################################################################
##
#F  AGRTOC( <arg> )
##
InstallGlobalFunction( AGRTOC, function( arg )

    local groupname,
          record,
          entry,
          name;

    groupname:= arg[1];
    if ForAll( AtlasOfGroupRepresentationsInfo.groupnames,
               x -> x[3] <> groupname ) then
      Info( InfoAtlasRep, 1, "`", groupname, "' is not a valid group name" );
      return;
    fi;

    record:= AtlasTableOfContents( "remote" ).TableOfContents;
    if not IsBound( record.( groupname ) ) then
      record.( groupname ):= rec( perm    := [],
                                  matff   := [],
                                  matalg  := [],
                                  matint  := [],
                                  matmodn := [],
                                  maxes   := [],
                                  classes := [],
                                  maxstd  := [],
                                  out     := [] );
    fi;
    record:= record.( groupname );

    if   arg[3] = "p" and Length( arg ) = 7 then

      # `AGRTOC(<groupname>,<gens>,\"p\",<degree>,<id>,<bas>,<filenames>)'
      entry:= arg{ [ 2, 4, 5, 6, 7 ] };
      entry[5]:= List( [ 1 .. entry[5][2] ],
                       i -> Concatenation( entry[5][1], String( i ) ) );
      if not entry in record.perm then
        Add( record.perm, entry );
      fi;

    elif arg[3] = "f" and Length( arg ) = 8 then

      # `AGRTOC(<groupname>,<gens>,\"f\",<q>,<dim>,<id>,<bas>,<filenames>)'
      entry:= arg{ [ 2, 4, 5, 6, 7, 8 ] };
      entry[6]:= List( [ 1 .. entry[6][2] ],
                       i -> Concatenation( entry[6][1], String( i ) ) );
      if not entry in record.matff then
        Add( record.matff, entry );
      fi;

    elif arg[3] = "Ar" and Length( arg ) = 7 then

      # `AGRTOC(<groupname>,<gens>,\"Ar\",<dim>,<id>,<bas>,<filename>)'
      entry:= arg{ [ 2, 4, 5, 6, 7 ] };
      if not entry in record.matalg then
        Add( record.matalg, entry );
      fi;

      # Check whether the field of the representation is stored.
      name:= entry[5]{ [ 1 .. Length( entry[5] )-2 ] };
      if ForAll( AtlasOfGroupRepresentationsInfo.fieldinfo,
                 pair -> pair[1] <> name ) then
        Info( InfoAtlasRep, 1,
              "no field info for `", entry[5], "'" );
      fi;

    elif arg[3] = "Zr" and Length( arg ) = 7 then

      # `AGRTOC(<groupname>,<gens>,\"Zr\",<dim>,<id>,<bas>,<filename>)'
      entry:= arg{ [ 2, 4, 5, 6, 7 ] };
      if not entry in record.matint then
        Add( record.matint, entry );
      fi;

    elif arg[3] = "R" and Length( arg ) = 8 then

      # `AGRTOC(<groupname>,<gens>,\"R\",<n>,<dim>,<id>,<bas>,<filename>)'
      entry:= arg{ [ 2, 4, 5, 6, 7, 8 ] };
      if not entry in record.matmodn then
        Add( record.matmodn, entry );
      fi;

    elif arg[3] = "M" and Length( arg ) = 5 then

      # `AGRTOC(<groupname>,<gens>,\"M\",<n>,<filename>)'
      entry:= [ arg[2], arg[5] ];
      if IsBound( record.maxes[ arg[4] ] ) then
        if not entry in record.maxes then
          Add( record.maxes[ arg[4] ], entry );
        fi;
      else
        record.maxes[ arg[4] ]:= [ entry ];
      fi;

    elif arg[3] = "MS" and Length( arg ) = 4 then

      # `AGRTOC(<groupname>,<gens>,\"MS\",<filename>)'
      entry:= [ arg[2], arg[4] ];
      if not entry in record.maxstd then
        Add( record.maxstd, entry );
      fi;

    elif arg[3] = "C" and Length( arg ) = 4 then

      # `AGRTOC(<groupname>,<gens>,\"C\",<filename>)'
      entry:= [ arg[2], arg[4] ];
      if not entry in record.classes then
        Add( record.classes, entry );
      fi;

    elif arg[3] = "O" and Length( arg ) = 5 then

      # `AGRTOC(<groupname>,<gens>,\"O\",<descr>,<filename>)'
      entry:= [ arg[2], arg[4], arg[5] ];
      if not entry in record.out then
        Add( record.out, entry );
      fi;

    else
      Info( InfoAtlasRep, 1, "`", arg, "' is not a valid t.o.c. entry" );
    fi;
end );


#############################################################################
##
#F  AtlasOfGroupRepresentationsUpdateData( [<groupnames>] )
##
InstallGlobalFunction( AtlasOfGroupRepresentationsUpdateData, function( arg )

    local file, toc, groupnames, errors, pair, record, name, entry;

    file:= function( dir, groupname, names )
      local name;
      if IsString( names ) then
        return FilenameAtlas( dir, groupname, names ) <> fail;
      else
        for name in names do
          if IsString( name ) then
            if FilenameAtlas( dir, groupname, name ) = fail then
              return false;
            fi;
          fi;
        od;
        return true;
      fi;
    end;

    toc:= AtlasTableOfContents( "remote" ).TableOfContents;

    groupnames:= AtlasOfGroupRepresentationsInfo.GAPnames;
    if Length( arg ) = 1 and IsList( arg[1] ) then
      groupnames:= Filtered( groupnames, pair -> pair[1] in arg[1] );
    fi;

    errors:= [];

    for pair in groupnames do
      if IsBound( toc.( pair[2] ) ) then
        record:= toc.( pair[2] );
        for name in [ "perm", "matff", "matint", "matalg", "matmodn" ] do
          for entry in record.( name ) do
            if not file( "datagens", pair[2], entry[ Length( entry ) ] ) then
              Add( errors, entry[ Length( entry ) ] );
            fi;
          od;
        od;
        for name in [ "maxes", "classes", "maxstd", "out" ] do
          for entry in record.( name ) do
            if not file( "dataword", pair[2], entry[ Length( entry ) ] ) then
              Add( errors, entry[ Length( entry ) ] );
            fi;
          od;
        od;
      fi;
    od;

    for entry in errors do
      if IsString( entry ) then
        Print( "#E  failed to fetch the file ", entry, "\n" );
      else
        Print( "#E  failed to fetch the file ", entry[1], "\n" );
      fi;
    od;
end );


#############################################################################
##
#E

