#############################################################################
##
#W  interfac.gi        GAP share package 'atlasrep'             Thomas Breuer
##
#H  @(#)$Id: interfac.gi,v 1.25 2001/02/28 17:19:26 gap Exp $
##
#Y  Copyright (C)  2001,  Lehrstuhl D fuer Mathematik,  RWTH Aachen,  Germany
##
##  This file contains the implementation part of the ``high level'' {\GAP}
##  interface to the {\ATLAS} of Group Representations.
##
Revision.( "atlasrep/gap/interfac_gi" ) :=
    "@(#)$Id: interfac.gi,v 1.25 2001/02/28 17:19:26 gap Exp $";


#############################################################################
##
#F  DisplayAtlasInfoOverview( <gapnames>, <toc> )
##
BindGlobal( "DisplayAtlasInfoOverview", function( gapnames, toc )

    local len,
          maxname,
          maxnr,
          maxmax,
          numbers,
          maxes,
          classes,
          out,
          i, j,
          new,
          record,
          maxcl1,
          maxcl2,
          maxout,
          str,
          name;

    # Consider only those names for which actually information is available.
    gapnames:= Filtered( gapnames, x -> IsBound( toc.( x[2] ) ) );

    # Compute the number of representations for each group.
    # In the first run, find also the longest group name,
    # and the maximal number of representations stored.
    maxname:= 5;  # the header `group' should fit
    maxnr:= 0;
    maxmax:= 5;
    numbers:= [];
    maxes:= [];
    classes:= [];
    out:= [];
    for i in [ 1 .. Length( gapnames ) ] do

      # Consider the length of the group name.
      new:= Length( gapnames[i][1] );
      if maxname < new then
        maxname:= new;
      fi;

      # Get the info about representations.
      record:= toc.( gapnames[i][2] );
      numbers[i]:=  Length( record.perm )
                  + Length( record.matff )
                  + Length( record.matalg )
                  + Length( record.matint )
                  + Length( record.matmodn );

      # Get the info about scripts for maximal subgroups.
      maxes[i]:= Filtered( [ 1 .. Length( record.maxes ) ],
                           x -> IsBound( record.maxes[x] ) );
      if maxnr < numbers[i] then
        maxnr:= numbers[i];
      fi;
      if Length( maxes[i] ) > 2 and IsRange( maxes[i] )
                                and maxes[i][2] = maxes[i][1] + 1 then
        len:=   Length( String( maxes[i][1] ) )
              + Length( String( maxes[i][ Length( maxes[i] ) ] ) )
              + 2;
        if maxmax < len then
          maxmax:= len;
        fi;
      else
        len:=   Sum( maxes[i], x -> Length( String( x ) ), 0 )
              + Length( maxes[i] ) - 1;
        if maxmax < len then
          maxmax:= len;
        fi;
      fi;

      # Get the info about scripts for class representatives.
      classes[i]:= [
          ForAny( record.classes,
                  pair -> pair[2]{ [ 1 .. 4 ] + Position( pair[2], '-' ) }
                          = "ccls" ),
          ForAny( record.classes,
                  pair -> pair[2]{ [ 1 .. 3 ] + Position( pair[2], '-' ) }
                          = "cyc" )
          ];

      # Get the number of different outer automorphisms.
      out[i]:= Length( Set( List( record.out, x -> x[2] ) ) );

    od;

    # Compute the appropriate column widths.
    maxcl1:= 4;
    maxcl2:= 5;
    maxnr:= Length( String( maxnr ) ) + 2;
    maxname:= maxname + 2;
    maxmax:= maxmax + 5;
    maxout:= 5;
    if maxmax > SizeScreen()[1] then
      maxmax:= SizeScreen()[1]
               - maxname - maxnr - maxcl1 - maxcl2 - maxout - 2;
    fi;

    # Print the header line.
    Print( FormattedString( "group", -maxname ),
           FormattedString( "#", maxnr ),
           FormattedString( "maxes", maxmax ),
           FormattedString( "cl", maxcl1 ),
           FormattedString( "cyc", maxcl2 ),
           FormattedString( "out", maxout ),
           "\n" );
    for i in [ 1 .. maxname + maxnr + maxmax + maxcl1 + maxcl2 + maxout ] do
      Print( "-" );
    od;
    Print( "\n" );

    # Print the information for each group.
    for i in [ 1 .. Length( gapnames ) ] do

      # first column: group name
      Print( FormattedString( gapnames[i][1], -maxname ) );

      # second column: number of representations available
      Print( FormattedString( numbers[i], maxnr ) );

      # third column: numbers of available maxes
      if Length( maxes[i] ) > 2 and IsRange( maxes[i] )
                                and maxes[i][2] = maxes[i][1] + 1 then
        Print( FormattedString( Concatenation( String( maxes[i][1] ),
               "..", String( maxes[i][ Length( maxes[i] ) ] ) ), maxmax ) );
      elif IsEmpty( maxes[i] ) then
        Print( FormattedString( "", maxmax ) );
      else
        str:= "";
        j:= 1;
        while j <= Length( maxes[i] ) do
          new:= String( maxes[i][j] );
          if maxmax < Length( str ) + Length( new ) + 3 then
            Print( FormattedString( str, maxmax ) );
            str:= "";
            Print( "\n" );
            Print( FormattedString( "", maxname + maxnr ) );
          fi;
          Append( str, new );
          if j <> Length( maxes[i] ) then
            Add( str, ',' );
          fi;
          j:= j+1;
        od;
        if not IsEmpty( str ) then
          Print( FormattedString( str, maxmax ) );
        fi;
      fi;

      # fourth column: availability of scripts for class representatives
      if classes[i][1] then
        Print( FormattedString( "+", maxcl1 ) );
      else
        Print( FormattedString( "-", maxcl1 ) );
      fi;

      # fifth column: availability of scripts for cyclic subgroups
      if classes[i][2] then
        Print( FormattedString( "+", maxcl2 ) );
      else
        Print( FormattedString( "-", maxcl2 ) );
      fi;

      # sixth column: scripts for outer automorphisms
      if out[i] <> 0 then
        Print( FormattedString( String( out[i] ), maxout ) );
      else
        Print( FormattedString( "", maxout ) );
      fi;

      Print( "\n" );

    od;
    end );


#############################################################################
##
#F  DisplayAtlasInfoStraightLineProgram( <gapname>, <record>, <std> )
##
BindGlobal( "DisplayAtlasInfoStraightLineProgram",
    function( gapname, record, std )

    local stdavail, nam, list, i, maxes, maxstd, cyc, ccl, pos, rel, out;

    # If the standardization is prescribed then do not mention it.
    # Otherwise if all information refers to the same standardization then
    # print just one line.
    # Otherwise print the standardization for each entry.
    stdavail:= [];
    if std = true or 1 < Length( std ) then
      for i in record.maxes do
        for list in i do
          if std = true or list[1] in std then
            AddSet( stdavail, list[1] );
          fi;
        od;
      od;
      for nam in [ "classes", "out" ] do
        for list in record.( nam ) do
          if std = true or list[1] in std then
            AddSet( stdavail, list[1] );
          fi;
        od;
      od;
    fi;

    # Print the header line.
    Print( "Straight line programs for G = ", gapname, ":" );
    if Length( stdavail ) = 1 then
      Print( "    (all refer to std. generators ", stdavail[1], ")" );
    fi;
    Print( "\n" );
    for i in [ 1 .. Length( gapname ) + 32 ] do
      Print( "-" );
    od;
    Print( "\n" );

    # Print info about maxes.
    maxes  := [];
    maxstd := [];
    for i in [ 1 .. Length( record.maxes ) ] do
      if IsBound( record.maxes[i] ) and
         ( std = true or ForAny( record.maxes[i], x -> x[1] in std ) ) then
        Add( maxes, i );
        Add( maxstd, Set( List( record.maxes[i], x -> x[1] ) ) );
      fi;
    od;
    if 2 < Length( maxes ) then
      ConvertToRangeRep( maxes );
    fi;
    Print( "available maxes of G:" );
    if Length( stdavail ) <= 1 then
      Print( "  ", maxes, "\n" );
    else
      Print( "\n" );
      for i in Union( maxstd ) do
        Print( "  ", maxes{ Filtered( [ 1 .. Length( maxes ) ],
                                      x -> i in maxstd[x] ) },
               "    (w.r.t. std. generators ", i, ")\n" );
      od;
    fi;

    # Print info about representatives of cyclic subgroups.
    cyc:= Filtered( record.classes, x -> ( std = true or x[1] in std ) and
                x[2]{ [ 1 .. 3 ] + Position( x[2], '-' ) } = "cyc" );
    Print( "repres. of cyclic subgroups of G available" );
    if IsEmpty( cyc ) then
      Print( "?  false\n" );
    elif 1 < Length( stdavail ) then
      Print( " for std. generators ", Set( List( cyc, x -> x[1] ) ), "\n" );
    else
      Print( "?  true\n" );
    fi;

    # Print info about class representatives.
    # (They can be stored either directly or via a script in `cyc'.)
    ccl:= Filtered( record.classes, x -> ( std = true or x[1] in std ) and
                x[2]{ [ 1 .. 4 ] + Position( x[2], '-' ) } = "ccls" );
    for i in [ 1 .. Length( ccl ) ] do

      # Check that for scripts of the form `<groupname>G<i>cycW<n>-cclsW<m>',
      # a script of the form `<groupname>G<i>-cycW<n>' is available.
      pos:= PositionSublist( ccl[i][2], "cycW" );
      rel:= Concatenation( ccl[i][2]{ [ 1 .. pos-1 ] }, "-",
          ccl[i][2]{ [ pos .. Position( ccl[i][2], '-' ) - 1 ] } );
      if pos <> fail and ForAll( record.classes, x -> x[2] <> rel ) then
        Unbind( ccl[i] );
      fi;

    od;
    ccl:= Compacted( ccl );
    Print( "class repres. of G available" );
    if IsEmpty( ccl ) then
      Print( "?  false\n" );
    elif 1 < Length( stdavail ) then
      Print( " for std. generators ", Set( List( ccl, x -> x[1] ) ), "\n" );
    else
      Print( "?  true\n" );
    fi;

    # Print info about outer automorphisms.
    out:= [];
    for i in [ 1 .. Length( record.out ) ] do
      if ( std = true or record.out[i][1] in std ) then
        Add( out, record.out[i][2] );
      fi;
    od;
    Print( "available automorphisms:  ", out, "\n" );
    end );


#############################################################################
##
#F  DisplayAtlasInfoGroup( <gapname>, <record>, <std>, <deg>, <char>, <dim> )
##
BindGlobal( "DisplayAtlasInfoGroup",
    function( gapname, record, std, deg, char, dim )

    local stdavail,
          nam,
          list,
          len,
          i,
          nr,
          fieldstring;

    # If the standardization is prescribed then do not mention it.
    # Otherwise if all information refers to the same standardization then
    # print just one line.
    # Otherwise print the standardization for each entry.
    stdavail:= [];
    if std = true or 1 < Length( std ) then
      for nam in [ "perm", "matff", "matalg", "matint", "matmodn" ] do
        for list in record.( nam ) do
          if std = true or list[1] in std then
            AddSet( stdavail, list[1] );
          fi;
        od;
      od;
    fi;

    # Print the header line.
    Print( "Representations for G = ", gapname, ":" );
    if Length( stdavail ) = 1 then
      Print( "    (all refer to std. generators ", stdavail[1], ")" );
    fi;
    Print( "\n" );
    for i in [ 1 .. Length( gapname ) + 25 ] do
      Print( "-" );
    od;
    Print( "\n" );

    # Print the information for the given group.
    nr:= 0;
    len:=   Length( record.perm )
          + Length( record.matff )
          + Length( record.matalg )
          + Length( record.matint )
          + Length( record.matmodn );
    len:= Length( String( len ) );
    for i in record.perm do
      nr:= nr + 1;
      if     ( std = true or i[1] in std )
         and ( deg = true or i[2] in deg ) then
        Print( FormattedString( nr, len ),
               ": G <= Sym(", i[2], i[3], ")" );
        if 1 < Length( stdavail ) then
          Print( ",   w.r.t. std. generators ", i[1] );
        fi;
        Print( "\n" );
      fi;
    od;
    for i in record.matff do
      nr:= nr + 1;
      if     ( std = true or i[1] in std )
         and ( char = true or ( IsList( char ) and ForAny( char,
                                         p -> p <> 0 and i[2] mod p = 0 ) ) )
         and ( dim = true or i[3] in dim ) then
        Print( FormattedString( nr, len ),
               ": G <= GL(", i[3], i[4], ",", i[2], ")" );
        if 1 < Length( stdavail ) then
          Print( ",   w.r.t. std. generators ", i[1] );
        fi;
        Print( "\n" );
      fi;
    od;
    for i in record.matint do
      nr:= nr + 1;
      if     ( std = true or i[1] in std )
         and ( char = true or ( IsList( char ) and 0 in char ) )
         and ( dim = true or i[2] in dim ) then
        Print( FormattedString( nr, len ),
               ": G <= GL(", i[2], i[3], ",Z)" );
        if 1 < Length( stdavail ) then
          Print( ",   w.r.t. std. generators ", i[1] );
        fi;
        Print( "\n" );
      fi;
    od;
    for i in record.matalg do
      nr:= nr + 1;
      if     ( std = true or i[1] in std )
         and ( char = true or ( IsList( char ) and 0 in char ) )
         and ( dim = true or i[2] in dim ) then

        # If the field of the representation is stored then show it.
        fieldstring:= i[5]{ [ 1 .. Length( i[5] )-2 ] };
        fieldstring:= First( AtlasOfGroupRepresentationsInfo.fieldinfo,
                             p -> p[1] = fieldstring );
        if fieldstring = fail then
          fieldstring:= "C";
        else
          fieldstring:= fieldstring[2];
        fi;

        Print( FormattedString( nr, len ),
               ": G <= GL(", i[2], i[3], ",", fieldstring, ")" );
        if 1 < Length( stdavail ) then
          Print( ",   w.r.t. std. generators ", i[1] );
        fi;
        Print( "\n" );
      fi;
    od;
    for i in record.matmodn do
      nr:= nr + 1;
      if     ( std = true or i[1] in std )
         and ( char = true or char = fail )
         and ( dim = true or i[3] in dim ) then
        Print( FormattedString( nr, len ),
               ": G <= GL(", i[3], i[4], "Z/", i[2], "Z)" );
        if 1 < Length( stdavail ) then
          Print( ",   w.r.t. std. generators ", i[1] );
        fi;
        Print( "\n" );
      fi;
    od;

    if deg = true and char = true and dim = true then
      Print( "\n" );
      DisplayAtlasInfoStraightLineProgram( gapname, record, std );
    fi;
    end );


#############################################################################
##
#F  DisplayAtlasInfo(  )
#F  DisplayAtlasInfo( <listofnames> )
#F  DisplayAtlasInfo( <gapname>[, <std>] )
#F  DisplayAtlasInfo( <gapname>[, <std>], IsPermGroup )
#F  DisplayAtlasInfo( <gapname>[, <std>], NrMovedPoints, <n> )
#F  DisplayAtlasInfo( <gapname>[, <std>], IsMatrixGroup )
#F  DisplayAtlasInfo( <gapname>[, <std>][, Characteristic, <p>]
#F                                      [, Dimension, <n>] )
#F  DisplayAtlasInfo( <gapname>[, <std>], IsStraightLineProgram )
##
InstallGlobalFunction( DisplayAtlasInfo, function( arg )

    local toc,
          gapnames,
          pair,
          gapname,
          argpos,
          std,
          i,
          pos,
          groupname,
          deg,
          char,
          dim;

    if AtlasOfGroupRepresentationsInfo.remote = true then
      toc:= AtlasTableOfContents( "remote" ).TableOfContents;
    else
      toc:= AtlasTableOfContents( "local" ).TableOfContents;
    fi;

    if   Length( arg ) = 0 then

      DisplayAtlasInfoOverview( AtlasOfGroupRepresentationsInfo.GAPnames,
                                toc );
      return;

    elif Length( arg ) = 1 and IsList( arg[1] )
                           and ForAll( arg[1], IsString ) then

      gapnames:= [];
      for pair in AtlasOfGroupRepresentationsInfo.GAPnames do
        pos:= Position( arg[1], pair[1] );
        if pos <> fail then
          gapnames[ pos ]:= pair;
        fi;
      od;
      DisplayAtlasInfoOverview( gapnames, toc );
      return;

    elif not IsString( arg[1] ) then

      Error( "usage: DisplayAtlasInfo( [<listofnames>] ),\n",
             "DisplayAtlasInfo( <gapname>[,<std>][,IsPermGroup] ),\n",
             "DisplayAtlasInfo( <gapname>[,<std>][,NrMovedPoints,<n>] ),\n",
             "DisplayAtlasInfo( <gapname>[,<std>][,IsMatrixGroup] ),\n",
             "DisplayAtlasInfo( <gapname>[,<std>][,Characteristic, <p>]\n",
             "                                   [,Dimension, <n>] ),\n",
             "DisplayAtlasInfo( <gapname>[,<std>][,IsStraightLineProgram] )" );
    fi;

    gapname:= arg[1];
    groupname:= First( AtlasOfGroupRepresentationsInfo.GAPnames,
                       pair -> pair[1] = gapname );
    if groupname = fail or not IsBound( toc.( groupname[2] ) ) then
      Info( InfoAtlasRep, 1,
            "DisplayAtlasInfo: no group with GAP name `", gapname, "'" );
      return;
    fi;

    if Length( arg ) = 1 or not ( IsInt( arg[2] ) or IsList( arg[2] ) ) then
      std:= true;
      argpos:= 2;
    else
      std:= arg[2];
      if IsInt( std ) then
        std:= [ std ];
      fi;
      argpos:= 3;
    fi;

    if   Length( arg ) = 1 or ( Length( arg ) = 2 and IsInt( arg[2] ) ) then

      # `DisplayAtlasInfo( <gapname>[, <std>] )'
      DisplayAtlasInfoGroup( gapname, toc.( groupname[2] ),
                             std, true, true, true );

    elif Length( arg ) = argpos and arg[ argpos ] = IsPermGroup then

      # `DisplayAtlasInfo( <gapname>[, <std>], IsPermGroup )'
      DisplayAtlasInfoGroup( gapname, toc.( groupname[2] ),
                             std, true, [], [] );

    elif Length( arg ) = argpos and arg[ argpos ] = IsMatrixGroup then

      # `DisplayAtlasInfo( <gapname>[, <std>], IsMatrixGroup )'
      DisplayAtlasInfoGroup( gapname, toc.( groupname[2] ),
                             std, [], true, true );

    elif Length( arg ) = argpos + 1 and arg[ argpos ] = NrMovedPoints then

      # `DisplayAtlasInfo( <gapname>[, <std>], NrMovedPoints, <n> )'
      deg:= arg[ argpos + 1 ];
      if IsPosInt( deg ) then
        deg:= [ deg ];
      elif not IsList( deg ) and not IsBool( deg ) then
        Error( "degree must be a pos. integer or a list of pos. integers" );
      fi;
      DisplayAtlasInfoGroup( gapname, toc.( groupname[2] ),
                             std, deg, [], [] );

    elif Length( arg ) = argpos + 1 and arg[ argpos ] = Characteristic then

      # `DisplayAtlasInfo( <gapname>[, <std>], Characteristic, <p> )'
      char:= arg[ argpos + 1 ];
      if IsInt( char ) and ( IsPrimeInt( char ) or char = 0 ) then
        char:= [ char ];
      elif not IsList( char ) and not IsBool( char ) then
        Error( "char. must be a prime integer or a list of prime integers" );
      fi;
      DisplayAtlasInfoGroup( gapname, toc.( groupname[2] ),
                             std, [], char, true );

    elif Length( arg ) = argpos + 1 and arg[ argpos ] = Dimension then

      # `DisplayAtlasInfo( <gapname>[, <std>], Dimension, <n> )'
      dim:= arg[ argpos + 1 ];
      if IsPosInt( dim ) then
        dim:= [ dim ];
      elif not IsList( dim ) and not IsBool( dim ) then
        Error( "dim. must be a pos. integer or a list of pos. integers" );
      fi;
      DisplayAtlasInfoGroup( gapname, toc.( groupname[2] ),
                             std, [], true, dim );

    elif Length( arg ) = argpos + 3 and arg[ argpos ] = Characteristic
                                    and arg[ argpos + 2 ] = Dimension then

      # `DisplayAtlasInfo( <gapname>[, <std>], Characteristic, <p>,
      #                                          Dimension, <n> )'
      char:= arg[ argpos + 1 ];
      if IsInt( char ) and ( IsPrimeInt( char ) or char = 0 ) then
        char:= [ char ];
      elif not IsList( char ) and not IsBool( char ) then
        Error( "char. must be a prime integer or a list of prime integers" );
      fi;
      dim:= arg[ argpos + 3 ];
      if IsPosInt( dim ) then
        dim:= [ dim ];
      elif not IsList( dim ) and not IsBool( dim ) then
        Error( "dim. must be a pos. integer or a list of pos. integers" );
      fi;
      DisplayAtlasInfoGroup( gapname, toc.( groupname[2] ),
                             std, [], char, dim );

    elif Length( arg ) = argpos and arg[ argpos ] = IsStraightLineProgram then

      # `DisplayAtlasInfo( <gapname>[, <std>], IsStraightLineProgram )'
      DisplayAtlasInfoStraightLineProgram( gapname, toc.( groupname[2] ),
                                            std );

    else
      Error( "usage: DisplayAtlasInfo( [<listofnames>] ),\n",
             "DisplayAtlasInfo( <gapname>[,<std>][,IsPermGroup] ),\n",
             "DisplayAtlasInfo( <gapname>[,<std>][,NrMovedPoints,<n>] ),\n",
             "DisplayAtlasInfo( <gapname>[,<std>][,IsMatrixGroup] ),\n",
             "DisplayAtlasInfo( <gapname>[,<std>][,Characteristic, <p>]\n",
             "                                   [,Dimension, <n>] ),\n",
             "DisplayAtlasInfo( <gapname>[,<std>][,IsStraightLineProgram] )" );
    fi;
    end );


#############################################################################
##
#F  AtlasGenerators( <gapname>, <repnr> )
#F  AtlasGenerators( <gapname>, <repnr>, <maxnr> )
#F  AtlasGenerators( <identifier> )
##
##  <identifier> is a list containing at the first position the string
##  <gapname>,
##  at the second position a string or a list of strings
##  (describing filenames),
##  at the third position a positive integer denoting the standardization of
##  the representation,
##  at the fourth position a positive integer describing the common ring of
##  the generators,
##  and at the fifth position, if bound, a positive integer denoting the
##  number of the maximal subgroup to which the representation is restricted.
##
InstallGlobalFunction( AtlasGenerators, function( arg )

    local identifier,
          gapname,
          groupname,
          file,
          gens,
          name,
          filename,
          prog,
          toc,
          record,
          repnr,
          descr;

    if Length( arg ) = 1 then

      # `AtlasGenerators( <identifier> )'
      identifier:= arg[1];
      gapname:= identifier[1];
      groupname:= First( AtlasOfGroupRepresentationsInfo.GAPnames,
                         pair -> pair[1] = gapname );
      file:= identifier[2];

      if IsList( file ) and ForAll( file, IsString ) then

        # several files containing one generator each
        gens:= [];
        for name in file do
          filename:= FilenameAtlas( "datagens", groupname[2], name );
          if filename = fail then
            return fail;
          fi;
          Add( gens, ScanMeatAxeFile( filename, identifier[4] ) );
        od;
        if ForAll( gens, x -> IsList( x ) and IsPerm( x[1] ) ) then
          gens:= Concatenation( gens );
        fi;

      elif IsString( file ) then

        # one file containing all generators
        filename:= FilenameAtlas( "datagens", groupname[2], file );
        if filename = fail then
          return fail;
        fi;
        gens:= AtlasGeneratorsCharacteristicZeroFile( filename );

      else
        Error( "<file> must be a string or a list of strings" );
      fi;

      if IsBound( identifier[5] ) then

        # Compute the straight line program for the restriction
        # (w.r.t. the correct standardization).
        prog:= AtlasStraightLineProgram( gapname, identifier[3],
                                         identifier[5] );
        if prog = fail then
          return fail;
        fi;

        # Evaluate the straight line program.
        gens:= ResultOfStraightLineProgram( prog.program, gens );

      fi;

      # Return the result.
      return Immutable( rec( generators      := gens,
                             standardization := identifier[3],
                             identifier      := Immutable( identifier ) ) );

    fi;


    if    ( Length( arg ) = 2 and IsString( arg[1] ) and IsPosInt( arg[2] ) )
       or ( Length( arg ) = 3 and IsString( arg[1] ) and IsPosInt( arg[2] )
                              and IsPosInt( arg[3] ) ) then

      # `AtlasGenerators( <gapname>, <repnr>[, <maxnr>] )'
      if AtlasOfGroupRepresentationsInfo.remote = true then
        toc:= AtlasTableOfContents( "remote" ).TableOfContents;
      else
        toc:= AtlasTableOfContents( "local" ).TableOfContents;
      fi;

      gapname:= arg[1];
      groupname:= First( AtlasOfGroupRepresentationsInfo.GAPnames,
                         pair -> pair[1] = gapname );

      if groupname = fail or not IsBound( toc.( groupname[2] ) ) then
        Info( InfoAtlasRep, 1,
              "AtlasGenerators: no group with GAP name `", gapname, "'" );
        return fail;
      else
        record:= toc.( groupname[2] );
      fi;

      repnr:= arg[2];
      descr:= 0;
      if repnr <= Length( record.perm ) then
        descr:= record.perm[ repnr ];
      else
        repnr:= repnr - Length( record.perm );
        if repnr <= Length( record.matff ) then
          descr:= record.matff[ repnr ];
        else
          repnr:= repnr - Length( record.matff );
          if repnr <= Length( record.matint ) then
            descr:= record.matint[ repnr ];
          else
            repnr:= repnr - Length( record.matint );
            if repnr <= Length( record.matalg ) then
              descr:= record.matalg[ repnr ];
            else
              repnr:= repnr - Length( record.matalg );
              if repnr <= Length( record.matmodn ) then
                descr:= record.matmodn[ repnr ];
              else
                return fail;
              fi;
            fi;
          fi;
        fi;
      fi;

      identifier:= [ gapname, descr[ Length(descr) ], descr[1], descr[2] ];
      if IsBound( arg[3] ) then
        identifier[5]:= arg[3];
      fi;

      # Delegate to the one argument version.
      return AtlasGenerators( identifier );

    else
      Error( "usage: AtlasGenerators( <gapname>,<repnr>[,<maxnr>] ) or\n",
             "       AtlasGenerators( <identifier> )" );
    fi;
    end );


#############################################################################
##
#F  OneAtlasGeneratingSet( <gapname>[, <std>] )
#F  OneAtlasGeneratingSet( <gapname>[, <std>], IsPermGroup )
#F  OneAtlasGeneratingSet( <gapname>[, <std>], NrMovedPoints, <n> )
#F  OneAtlasGeneratingSet( <gapname>[, <std>], IsMatrixGroup )
#F  OneAtlasGeneratingSet( <gapname>[, <std>][, Characteristic, <p>]
#F                                           [, Dimension, <m>] )
#F  OneAtlasGeneratingSet( <gapname>[, <std>][, Ring, <R>]
#F                                           [, Dimension, <m>] )
##
InstallGlobalFunction( OneAtlasGeneratingSet, function( arg )

    local gapname,
          groupname,
          std,
          toc,
          record,
          argpos,
          deg,
          i,
          char,
          repnr,
          dim,
          R,
          exp,
          gens,
          m;

    if 0 < Length( arg ) and IsString( arg[1] ) then
      gapname:= arg[1];
    else
      Error( "first argument must be a group name" );
    fi;

    if AtlasOfGroupRepresentationsInfo.remote = true then
      toc:= AtlasTableOfContents( "remote" ).TableOfContents;
    else
      toc:= AtlasTableOfContents( "local" ).TableOfContents;
    fi;

    groupname:= First( AtlasOfGroupRepresentationsInfo.GAPnames,
                       pair -> pair[1] = gapname );

    if groupname = fail or not IsBound( toc.( groupname[2] ) ) then
      Info( InfoAtlasRep, 1,
            "OneAtlasGeneratingSet: no group with GAP name `", gapname, "'" );
      return fail;
    fi;

    record:= toc.( groupname[2] );

    if 2 <= Length( arg ) and ( IsPosInt( arg[2] ) or IsList( arg[2] ) ) then
      argpos:= 3;
      std:= arg[2];
      if IsPosInt( std ) then
        std:= [ std ];
      fi;
    else
      argpos:= 2;
      std:= true;
    fi;

    if   Length( arg ) = 1 then

      # `OneAtlasGeneratingSet( <gapname> )'
      std  := true;
      deg  := true;
      char := true;
      dim  := true;

    elif Length( arg ) = 2 and argpos = 3 then

      # `OneAtlasGeneratingSet( <gapname>, <std> )'
      deg   := true;
      char  := true;
      dim   := true;

    elif Length( arg ) = argpos and arg[ argpos ] = IsPermGroup then

      # `OneAtlasGeneratingSet( <gapname>[, <std>], IsPermGroup )'
      deg   := true;
      char  := [];
      dim   := [];

    elif Length( arg ) = argpos and arg[ argpos ] = IsMatrixGroup then

      # `OneAtlasGeneratingSet( <gapname>[, <std>], IsMatrixGroup )'
      deg   := [];
      char  := true;
      dim   := true;

    elif Length( arg ) = argpos + 1 and arg[ argpos ] = NrMovedPoints then

      # `OneAtlasGeneratingSet( <gapname>[, <std>], NrMovedPoints, <n> )'
      deg:= arg[ argpos + 1 ];
      if IsPosInt( deg ) then
        deg:= [ deg ];
      elif not IsList( deg ) then
        return fail;
      fi;
      char  := [];
      dim   := [];

    elif Length( arg ) = argpos + 1 and arg[ argpos ] = Characteristic then

      # `OneAtlasGeneratingSet( <gapname>[, <std>], Characteristic, <p> )'
      deg  := [];
      dim  := true;
      char:= arg[ argpos + 1 ];
      if char = 0 or char = fail or IsPosInt( char ) then
        char:= [ char ];
      elif not IsList( char ) then
        return fail;
      fi;

    elif Length( arg ) = argpos + 1 and arg[ argpos ] = Dimension then

      # `OneAtlasGeneratingSet( <gapname>[, <std>], Dimension, <n> )'
      deg := [];
      char := true;
      dim:= arg[ argpos + 1 ];
      if IsPosInt( dim ) then
        dim:= [ dim ];
      elif not IsList( dim ) then
        return fail;
      fi;

    elif Length( arg ) = argpos + 3 and arg[ argpos ] = Characteristic
                                    and arg[ argpos + 2 ] = Dimension then

      # `OneAtlasGeneratingSet( <gapname>[, <std>], Characteristic, <p>,
      #                                               Dimension, <n> )'
      deg:= [];
      char:= arg[ argpos + 1 ];
      if char = 0 or char = fail or IsPosInt( char ) then
        char:= [ char ];
      elif not IsList( char ) then
        return fail;
      fi;
      dim:= arg[ argpos + 3 ];
      if IsPosInt( dim ) then
        dim:= [ dim ];
      elif not IsList( dim ) then
        return fail;
      fi;

    elif Length( arg ) = argpos + 1 and arg[ argpos ] = Ring then

      # `OneAtlasGeneratingSet( <gapname>, Ring, <R> )'
      R   := arg[ argpos + 1 ];
      deg := [];
      dim := true;

    elif Length( arg ) = argpos + 3 and arg[ argpos ] = Ring
                                    and arg[ argpos + 2 ] = Dimension then

      # `OneAtlasGeneratingSet( <gapname>[, <std>], Ring, <R>,
      #                                               Dimension, <n> )'
      R   := arg[ argpos + 1 ];
      deg := [];
      dim := arg[ argpos + 3 ];
      if IsPosInt( dim ) then
        dim:= [ dim ];
      elif not IsList( dim ) then
        return fail;
      fi;

    else
      Error( "<arg> is not an admissible list of arguments" );
    fi;

    # Loop over the relevant representations.
    if IsBound( R ) then

      if   not IsRing( R ) then
        return fail;
      elif IsField( R ) and IsFinite( R ) then
        char:= Characteristic( R );
        exp:= DegreeOverPrimeField( R );
        for i in [ 1 .. Length( record.matff ) ] do
          if     ( std = true or record.matff[i][1] in std )
             and record.matff[i][2] mod char = 0
             and exp mod LogInt( record.matff[i][2], char ) = 0
             and ( dim = true or record.matff[i][3] in dim ) then
            return AtlasGenerators( gapname, Length( record.perm ) + i );
          fi;
        od;
      elif IsCyclotomicCollection( R ) then
        repnr:= Length( record.perm ) + Length( record.matff );
        for i in [ 1 .. Length( record.matint ) ] do
          if     ( std = true or record.matint[i][1] in std )
             and ( dim = true or record.matint[i][2] in dim ) then
            return AtlasGenerators( gapname, repnr + i );
          fi;
        od;
        repnr:= repnr + Length( record.matint );
        if 1 < Conductor( R ) then
          for i in [ 1 .. Length( record.matalg ) ] do
            if     ( std = true or record.matalg[i][1] in std )
               and ( dim = true or record.matalg[i][2] in dim ) then
              gens:= AtlasGenerators( gapname, repnr + i );
#T reading all these representations can be very expensive ...
              if IsSubset( R, Flat( gens.generators ) ) then
                return gens;
              fi;
            fi;
          od;
        fi;
      elif IsZmodnZObjNonprimeCollection( R ) then
        m:= ModulusOfZmodnZObj( One( R ) );
        for i in [ 1 .. Length( record.matmodn ) ] do
          if     ( std = true or record.matmodn[i][1] in std )
             and m = record.matmodn[i][2]
             and ( dim = true or record.matmodn[i][3] in dim ) then
            return AtlasGenerators( gapname, Length( record.perm )
                                          + Length( record.matff )
                                          + Length( record.matint )
                                          + Length( record.matalg ) + i );
          fi;
        od;
      fi;

    else

      for i in [ 1 .. Length( record.perm ) ] do
        if     ( std = true or record.perm[i][1] in std )
           and ( deg = true or record.perm[i][2] in deg ) then
          return AtlasGenerators( gapname, i );
        fi;
      od;

      repnr:= Length( record.perm );
      for i in [ 1 .. Length( record.matff ) ] do
        if     ( std = true or record.matff[i][1] in std )
           and ( dim = true or record.matff[i][3] in dim )
           and ( char = true or ForAny( char,
                        c -> c <> 0 and record.matff[i][2] mod c = 0 ) ) then
          return AtlasGenerators( gapname, repnr + i );
        fi;
      od;

      repnr:= repnr + Length( record.matff );
      if char = true or 0 in char then
        for i in [ 1 .. Length( record.matint ) ] do
          if     ( std = true or record.matint[i][1] in std )
             and ( dim = true or record.matint[i][2] in dim ) then
            return AtlasGenerators( gapname, repnr + i );
          fi;
        od;
      fi;

      repnr:= repnr + Length( record.matint );
      if char = true or 0 in char then
        for i in [ 1 .. Length( record.matalg ) ] do
          if     ( std = true or record.matalg[i][1] in std )
             and ( dim = true or record.matalg[i][2] in dim ) then
            return AtlasGenerators( gapname, repnr + i );
          fi;
        od;
      fi;

      repnr:= repnr + Length( record.matalg );
      if char = true or fail in char then
        for i in [ 1 .. Length( record.matmodn ) ] do
          if     ( std = true or record.matmodn[i][1] in std )
             and ( dim = true or record.matmodn[i][3] in dim ) then
            return AtlasGenerators( gapname, repnr + i );
          fi;
        od;
      fi;

    fi;

    # No representation with the given requirements was found.
    return fail;
    end );


#############################################################################
##
#F  AtlasStraightLineProgram( <gapname>[, <std>], <maxnr> )
#F  AtlasStraightLineProgram( <gapname>[, <std>], "classes" )
#F  AtlasStraightLineProgram( <gapname>[, <std>], "cyclic" )
#F  AtlasStraightLineProgram( <gapname>[, <std>], "automorphism", <autname> )
#F  AtlasStraightLineProgram( <identifier> )
##
##  <identifier> is a list containing at the first position the string
##  <gapname>,
##  at the second position a string or a list of strings
##  (describing the filenames involved),
##  and at the third position a positive integer denoting the standardization
##  of the program.
##
InstallGlobalFunction( AtlasStraightLineProgram, function( arg )

    local identifier,
          gapname,
          groupname,
          file,
          progs,
          name,
          filename,
          prog,
          i,
          outputs,
          toc,
          std,
          maxnr,
          record,
          maxes,
          errstring,
          digits,
          pair,
          pos,
          n,
          pos2,
          pair2,
          filename2,
          result;

    if Length( arg ) = 1 then

      # `AtlasStraightLineProgram( <identifier> )'
      identifier:= arg[1];
      gapname:= identifier[1];
      groupname:= First( AtlasOfGroupRepresentationsInfo.GAPnames,
                         pair -> pair[1] = gapname );
      file:= identifier[2];

      if IsList( file ) and ForAll( file, IsString ) then

        # several programs to be composed
        progs:= [];
        for name in file do
          filename:= FilenameAtlas( "dataword", groupname[2], name );
          if filename = fail then
            return fail;
          fi;
          Add( progs, ScanStraightLineProgram( filename ) );
        od;
        if fail in progs then
          return fail;
        fi;
        prog:= progs[1].program;
        for i in [ 2 .. Length( progs ) ] do
          prog:= CompositionOfStraightLinePrograms( prog, progs[i].program );
          if prog = fail then
            return fail;
          fi;
        od;
        if IsBound( progs[1].outputs ) then
          outputs:= progs[1].outputs;
        fi;

      elif IsString( file ) then

        # one file containing the program
        filename:= FilenameAtlas( "dataword", groupname[2], file );
        if filename = fail then
          return fail;
        fi;
        prog:= ScanStraightLineProgram( filename );
        if prog = fail then
          return fail;
        elif IsBound( prog.outputs ) then
          outputs:= prog.outputs;
        fi;
        prog:= prog.program;

      else
        Error( "<file> must be a string or a list of strings" );
      fi;

      # Return the result.
      result:= rec( program         := prog,
                    standardization := identifier[3],
                    identifier      := identifier );
      if IsBound( outputs ) then
        result.outputs:= outputs;
      fi;
      return Immutable( result );

    fi;

    # Now handle the cases of more than one argument.
    if AtlasOfGroupRepresentationsInfo.remote = true then
      toc:= AtlasTableOfContents( "remote" ).TableOfContents;
    else
      toc:= AtlasTableOfContents( "local" ).TableOfContents;
    fi;

    if     Length( arg ) = 2 and IsString( arg[1] )
       and ( IsPosInt( arg[2] ) or IsString( arg[2] ) ) then
      gapname:= arg[1];
      std:= true;
      maxnr:= arg[2];
    elif   Length( arg ) = 3 and IsString( arg[1] )
       and ( IsPosInt( arg[2] ) or IsList( arg[2] ) )
       and ( IsPosInt( arg[3] ) or IsString( arg[3] ) ) then
      gapname:= arg[1];
      std:= arg[2];
      maxnr:= arg[3];
      if IsInt( std ) then
        std:= [ std ];
      fi;
    elif   Length( arg ) = 3 and IsString( arg[1] )
       and arg[2] = "automorphism"
       and IsString( arg[3] ) then
      gapname:= arg[1];
      std:= true;
      maxnr:= arg[3];
    elif   Length( arg ) = 4 and IsString( arg[1] )
       and ( IsPosInt( arg[2] ) or IsList( arg[2] ) )
       and arg[3] = "automorphism"
       and IsString( arg[4] ) then
      gapname:= arg[1];
      std:= arg[2];
      maxnr:= arg[4];
    else
      Error( "usage: AtlasStraightLineProgram(<gapname>[,<std>],<maxnr>)\n",
             " or AtlasStraightLineProgram(<gapname>[,<std>],\"classes\")\n",
             " or AtlasStraightLineProgram(<gapname>[,<std>],\"cyclic\")\n",
             " or AtlasStraightLineProgram(<gapname>[,<std>],",
             "\"automorphism\",<autname>)\n",
             " or AtlasStraightLineProgram(<identifier>)" );
    fi;

    groupname:= First( AtlasOfGroupRepresentationsInfo.GAPnames,
                       pair -> pair[1] = gapname );

    if groupname = fail or not IsBound( toc.( groupname[2] ) ) then
      Info( InfoAtlasRep, 1,
            "AtlasStraightLineProgram: no group with GAP name `",
            gapname, "'" );
      return fail;
    fi;

    record:= toc.( groupname[2] );

    if IsInt( maxnr ) then

      # Find the straight line program for the <maxnr>-th max. subgroup.
      maxes:= record.maxes;
      if IsBound( maxes[ maxnr ] ) then
        for i in [ 1 .. Length( maxes[ maxnr ] ) ] do
          if std = true or maxes[ maxnr ][i][1] in std then
            return AtlasStraightLineProgram(
                       [ groupname[1], maxes[ maxnr ][i][2],
                                       maxes[ maxnr ][i][1] ] );
          fi;
        od;
        errstring:= Concatenation( Ordinal( maxnr ), "max. subgroup" );
      fi;

    elif maxnr = "classes" then

      # Search for a straight line program for class representatives.
      digits:= "1234567890";
      for pair in record.classes do
        if std = true or pair[1] in std then
          filename:= pair[2];
          pos:= Position( filename, '-' );
          if     pos + 3 < Length( filename )
             and filename{ [ pos+1 .. pos+4 ] } = "ccls" then

            # Either this is already the desired program
            # `<groupname>G<i>-cclsW<n>',
            # or it is a program with name
            # `<groupname>G<i>cycW<n>-cclsW<m>',
            # which has to be composed with the program that computes
            # representatives of cyclic subgroups.
            n:= 0;
            pos:= pos - 1;
            pos2:= pos;
            repeat
              pos2:= pos2 - 1;
            until not filename[ pos2 ] in digits;

            if filename[ pos2 ] = 'G' then

              return AtlasStraightLineProgram( [ groupname[1],
                         filename, pair[1] ] );

            else

              n:= filename{ [ pos2 + 1 .. pos ] };
              for pair2 in record.classes do
                filename2:= pair2[2];
                pos:= Position( filename2, '-' );
                if filename2{ [ pos+1 .. Length( filename2 ) ] }
                   = Concatenation( "cycW", n ) then
                  return AtlasStraightLineProgram( [ groupname[1],
                             [ filename, filename2 ], pair[1] ] );
                fi;
              od;

            fi;
          fi;
        fi;
      od;

      errstring:= "class representatives";

    elif maxnr = "cyclic" then

      # Search for a straight line program for representatives
      # of cyclic subgroups.
      digits:= "1234567890";
      for pair in record.classes do
        if std = true or pair[1] in std then
          filename:= pair[2];
          pos:= Position( filename, '-' );
          if     pos + 3 < Length( filename )
             and filename{ [ pos+1 .. pos+3 ] } = "cyc" then

            # The program has the name `<groupname>G<i>-cycW<n>'.
            return AtlasStraightLineProgram( [ groupname[1],
                       filename, pair[1] ] );

          fi;
        fi;
      od;

      errstring:= "repres. of cyclic subgroups";

    elif IsString( maxnr ) then

      # Search for a straight line program for an outer automorphism.
      for pair in record.out do
        if ( std = true or pair[1] in std ) and maxnr = pair[2] then

            # The program has the name `<groupname>G<i>-a<outname>W<n>'.
            return AtlasStraightLineProgram( [ groupname[1],
                       pair[3], pair[2] ] );

        fi;
      od;

      errstring:= "the automorphism";

    else

      errstring:= Concatenation( "the input `", maxnr, "'" );

    fi;

    # No program was found.
    Info( InfoAtlasRep, 2,
          "no straight line program for ", errstring );
    Info( InfoAtlasRep, 2,
          "of the group with GAP name `", groupname[1], "'" );
    return fail;
end );


#############################################################################
##
#E

